<?php
require_once __DIR__ . '/../auth/functions.php';
require_once __DIR__ . '/../config/fast_email_service.php';
requireRole('admin');

require_once __DIR__ . '/../config/db_connect.php';

// Bridge database connections - Email service needs PDO, but admin uses MySQLi
if (!isset($pdo)) {
    try {
        // Create PDO connection using the same credentials as MySQLi
        $pdo = new PDO("mysql:host=" . DB_SERVER . ";dbname=" . DB_NAME, DB_USERNAME, DB_PASSWORD);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        error_log("PDO connection created successfully for admin accreditation email service");
    } catch (Exception $e) {
        error_log("Failed to create PDO connection for admin accreditation email service: " . $e->getMessage());
        $pdo = null;
    }
}

/**
 * Send assessment approval email to student when admin approves request
 */
function sendAssessmentApprovalEmail($request_id) {
    global $conn;
    
    try {
        // Get request, student and course details  
        $stmt = $conn->prepare("SELECT ar.request_id, ar.payment_amount, ar.assignment_document_path, ar.updated_at, ar.course_id, u.first_name, u.last_name, u.email, u.user_id, c.title as course_title, c.description FROM accreditation_requests ar JOIN users u ON ar.student_id = u.user_id JOIN courses c ON ar.course_id = c.course_id WHERE ar.request_id = ?");
        
        if (!$stmt) {
            error_log("Failed to prepare accreditation email query: " . $conn->error);
            return false;
        }
        
        $stmt->bind_param("i", $request_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $details = $result->fetch_assoc();
        
        if (!$details) {
            error_log("Accreditation request not found for assessment approval email: $request_id");
            return false;
        }
        
        $student_name = $details['first_name'] . ' ' . $details['last_name'];
        $student_email = $details['email'];
        $course_title = $details['course_title'];
        $course_id = $details['course_id'];
        $request_id_formatted = str_pad($request_id, 6, '0', STR_PAD_LEFT);
        $approval_date = date('M j, Y', strtotime($details['updated_at']));
        
        // Create email content
        $subject = "🎉 Assessment Approved: Request #$request_id_formatted is Ready!";
        $message = "
        <div style='max-width:600px; margin:0 auto; font-family:Arial,sans-serif;'>
            <div style='background:#28a745; color:white; padding:20px; text-align:center;'>
                <h1>🎓 Assessment Approved!</h1>
            </div>
            <div style='padding:20px; background:white;'>
                <h2>Congratulations, " . htmlspecialchars($student_name) . "! 🎉</h2>
                <p>Your SITA accreditation request <strong>\"#" . $request_id_formatted . "\"</strong> has been approved and is now ready!</p>
                <div style='background:#f8f9fa; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h3>Request Details:</h3>
                    <p><strong>Course:</strong> " . htmlspecialchars($course_title) . "</p>
                    <p><strong>Request ID:</strong> #" . $request_id_formatted . "</p>
                    <p><strong>Approved:</strong> " . $approval_date . "</p>
                </div>
                <p style='text-align:center;'>
                    <a href='https://panaditeacademy.com/student/course-detail.php?id=" . base64_encode($course_id) . "&tab=certificate' 
                       style='background:#667eea; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block;'>
                        View Assessment
                    </a>
                </p>
            </div>
        </div>";
        
        // Try direct email sending
        if (function_exists('sendEmailNow')) {
            $result = sendEmailNow($student_email, $subject, $message);
            if ($result) {
                return true;
            }
        }
        
        // Fallback to native PHP mail
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
        
        return mail($student_email, $subject, $message, $headers);
        
    } catch (Exception $e) {
        error_log("Error sending assessment approval email: " . $e->getMessage());
        return false;
    }
}

$message = '';
$messageType = '';

// Utility function to format file sizes
function formatBytes($size, $precision = 2) {
    $base = log($size, 1024);
    $suffixes = array('', 'KB', 'MB', 'GB', 'TB');
    return round(pow(1024, $base - floor($base)), $precision) . ' ' . $suffixes[floor($base)];
}

// Handle approval/rejection actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        $request_id = intval($_POST['request_id']);
        $action = $_POST['action'];
        
        if ($action === 'approve_with_assignment') {
            // 🚀 STREAMLINED APPROVAL: Approve + Upload Assignment in one step
            
            try {
                // Begin transaction for atomic operations
                $conn->begin_transaction();
                
                // Validate required fields
                $assignment_instructions = trim($_POST['assignment_instructions'] ?? '');
                $admin_notes = trim($_POST['admin_notes'] ?? '');
                
                if (empty($assignment_instructions)) {
                    throw new Exception('Assignment instructions are required');
                }
                
                if (!isset($_FILES['assignment_files']) || empty($_FILES['assignment_files']['name'][0])) {
                    throw new Exception('At least one assignment file is required');
                }
                
                // Validate admin user exists in users table
                $admin_id = $_SESSION['user_id'] ?? null;
                if (!$admin_id) {
                    throw new Exception('Admin user session not found');
                }
                
                // Check if admin user exists
                $admin_check = $conn->prepare("SELECT user_id FROM users WHERE user_id = ?");
                if (!$admin_check) {
                    throw new Exception('Database prepare failed: ' . $conn->error);
                }
                
                $admin_check->bind_param("i", $admin_id);
                if (!$admin_check->execute()) {
                    throw new Exception('Database query failed: ' . $admin_check->error);
                }
                
                $admin_exists = $admin_check->get_result()->fetch_assoc();
                
                if (!$admin_exists) {
                    // Create admin user record if it doesn't exist (fallback)
                    error_log("Admin user ID $admin_id not found in users table, creating record");
                    
                    $create_admin = $conn->prepare("
                        INSERT INTO users (user_id, username, email, first_name, last_name, role, created_at) 
                        VALUES (?, 'admin999', 'admin@panaditeacademy.com', 'System', 'Administrator', 'admin', NOW())
                        ON DUPLICATE KEY UPDATE role = 'admin'
                    ");
                    
                    if (!$create_admin) {
                        // If users table has issues, just proceed without admin reference
                        error_log("Cannot create admin user record, proceeding with NULL approved_by");
                        $admin_exists = null;
                    } else {
                        $create_admin->bind_param("i", $admin_id);
                        if ($create_admin->execute()) {
                            $admin_exists = ['user_id' => $admin_id]; // Mark as exists
                        } else {
                            error_log("Failed to create admin user: " . $create_admin->error);
                            $admin_exists = null;
                        }
                    }
                }
                
                // Get request details first
                $req_stmt = $conn->prepare("SELECT ar.*, u.email, u.first_name, u.last_name, c.title as course_title FROM accreditation_requests ar 
                                           JOIN users u ON ar.student_id = u.user_id 
                                           JOIN courses c ON ar.course_id = c.course_id 
                                           WHERE ar.request_id = ?");
                $req_stmt->bind_param("i", $request_id);
                $req_stmt->execute();
                $request_data = $req_stmt->get_result()->fetch_assoc();
                
                if (!$request_data) {
                    throw new Exception('Request not found');
                }
                
                // Create assignment directory structure
                $base_upload_dir = __DIR__ . '/../uploads/assignments';
                $request_upload_dir = $base_upload_dir . '/request_' . $request_id;
                
                // Ensure directories exist
                if (!file_exists($base_upload_dir)) {
                    mkdir($base_upload_dir, 0755, true);
                }
                if (!file_exists($request_upload_dir)) {
                    mkdir($request_upload_dir, 0755, true);
                }
                
                // Process file uploads
                $uploaded_files = [];
                $files = $_FILES['assignment_files'];
                
                for ($i = 0; $i < count($files['name']); $i++) {
                    if ($files['error'][$i] === UPLOAD_ERR_OK) {
                        $original_name = $files['name'][$i];
                        $tmp_name = $files['tmp_name'][$i];
                        $file_size = $files['size'][$i];
                        
                        // Validate file size (10MB max)
                        if ($file_size > 10 * 1024 * 1024) {
                            throw new Exception("File '{$original_name}' is too large. Maximum size is 10MB.");
                        }
                        
                        // Validate file type
                        $allowed_extensions = ['pdf', 'doc', 'docx', 'zip', 'jpg', 'jpeg', 'png'];
                        $file_extension = strtolower(pathinfo($original_name, PATHINFO_EXTENSION));
                        
                        if (!in_array($file_extension, $allowed_extensions)) {
                            throw new Exception("File '{$original_name}' has invalid format. Allowed: " . implode(', ', $allowed_extensions));
                        }
                        
                        // Generate safe filename
                        $safe_filename = date('YmdHis') . '_' . $i . '_' . preg_replace('/[^a-zA-Z0-9._-]/', '_', $original_name);
                        $target_path = $request_upload_dir . '/' . $safe_filename;
                        
                        // Move uploaded file
                        if (move_uploaded_file($tmp_name, $target_path)) {
                            $uploaded_files[] = [
                                'original_name' => $original_name,
                                'safe_filename' => $safe_filename,
                                'file_size' => $file_size,
                                'file_path' => $target_path
                            ];
                        } else {
                            throw new Exception("Failed to upload file '{$original_name}'");
                        }
                    } else {
                        throw new Exception("Upload error for file '{$files['name'][$i]}'");
                    }
                }
                
                if (empty($uploaded_files)) {
                    throw new Exception('No files were successfully uploaded');
                }
                
                // Update database with approval and assignment info
                $assignment_file_paths = json_encode($uploaded_files);
                
                // Use validated admin_id or NULL as fallback
                $approved_by_id = $admin_exists ? $admin_id : null;
                
                $stmt = $conn->prepare("
                    UPDATE accreditation_requests 
                    SET status = 'assignment_created', 
                        approved_by = ?, 
                        approved_date = NOW(), 
                        admin_notes = ?,
                        assignment_instructions = ?,
                        assignment_document_path = ?,
                        assignment_created_date = NOW(),
                        updated_at = NOW()
                    WHERE request_id = ?
                ");
                
                if ($approved_by_id) {
                    $stmt->bind_param("isssi", 
                        $approved_by_id, 
                        $admin_notes, 
                        $assignment_instructions, 
                        $assignment_file_paths, 
                        $request_id
                    );
                } else {
                    // Handle NULL approved_by case
                    $stmt = $conn->prepare("
                        UPDATE accreditation_requests 
                        SET status = 'assignment_created', 
                            approved_by = NULL, 
                            approved_date = NOW(), 
                            admin_notes = ?,
                            assignment_instructions = ?,
                            assignment_document_path = ?,
                            assignment_created_date = NOW(),
                            updated_at = NOW()
                        WHERE request_id = ?
                    ");
                    $stmt->bind_param("sssi", 
                        $admin_notes, 
                        $assignment_instructions, 
                        $assignment_file_paths, 
                        $request_id
                    );
                }
                
                if (!$stmt->execute()) {
                    throw new Exception('Failed to update database: ' . $stmt->error);
                }
                
                // Commit transaction
                $conn->commit();
                
                // Send assessment approval email using the standardized function
                if (function_exists('sendAssessmentApprovalEmail')) {
                    $email_sent = sendAssessmentApprovalEmail($request_id);
                    if ($email_sent) {
                        error_log("✅ Assessment approval email sent for request ID: " . $request_id);
                    } else {
                        error_log("❌ Failed to send assessment approval email for request ID: " . $request_id);
                    }
                } else {
                    $email_sent = false;
                    error_log("❌ sendAssessmentApprovalEmail function not found");
                }
                
                // Legacy email functionality (keeping as backup)
                $legacy_email_sent = false;
                
                try {
                    // Try to load email service
                    if (file_exists(__DIR__ . '/../config/fast_email_service.php')) {
                        require_once __DIR__ . '/../config/fast_email_service.php';
                    }
                    
                    $file_list_html = '';
                    foreach ($uploaded_files as $file) {
                        $download_url = "https://" . $_SERVER['HTTP_HOST'] . "/admin/download_assignment.php?request_id={$request_id}&file=" . urlencode($file['safe_filename']);
                        $file_list_html .= "<li><a href='{$download_url}' style='color: #0ea5e9; text-decoration: none;'>{$file['original_name']}</a> (" . formatBytes($file['file_size']) . ")</li>";
                    }
                    
                    $approval_email_subject = "🎓 SITA Assignment Ready - {$request_data['course_title']}";
                    $approval_email_body = "
                    <h2>🎉 Your SITA Accreditation Has Been Approved!</h2>
                    <p>Dear {$request_data['first_name']},</p>
                    <p>Great news! Your SITA accreditation request for <strong>{$request_data['course_title']}</strong> has been approved and your assignment is ready!</p>
                    
                    <div style='background: #f0f9ff; border: 2px solid #0ea5e9; border-radius: 8px; padding: 15px; margin: 20px 0;'>
                        <h3 style='color: #0ea5e9; margin-top: 0;'>📋 Assignment Instructions:</h3>
                        <div style='background: white; padding: 15px; border-radius: 5px; white-space: pre-wrap;'>{$assignment_instructions}</div>
                    </div>
                    
                    <div style='background: #f8f9fa; border: 1px solid #dee2e6; border-radius: 8px; padding: 15px; margin: 20px 0;'>
                        <h3 style='color: #495057; margin-top: 0;'>📎 Assignment Documents:</h3>
                        <ul style='color: #1e293b; line-height: 1.8;'>
                            {$file_list_html}
                        </ul>
                        <p style='color: #6c757d; font-size: 0.9rem; margin-bottom: 0;'>
                            💡 <strong>Tip:</strong> Click on each file to download it individually.
                        </p>
                    </div>
                    
                    <div style='background: #fff3cd; border: 1px solid #ffeaa7; border-radius: 8px; padding: 15px; margin: 20px 0;'>
                        <h4 style='color: #856404; margin-top: 0;'>⏰ Important Deadline:</h4>
                        <p style='color: #856404; margin-bottom: 0;'>
                            Please complete and submit your assignment within <strong>14 days</strong> from today.
                            You can upload your completed documents through your student dashboard.
                        </p>
                    </div>
                    
                    <p>If you have any questions about the assignment, please don't hesitate to contact us.</p>
                    <p>Best of luck with your SITA accreditation!</p>
                    ";
                    
                    // Try different email functions that might exist
                    if (function_exists('sendEmail')) {
                        $email_sent = sendEmail($request_data['email'], $approval_email_subject, $approval_email_body);
                    } elseif (function_exists('send_email')) {
                        $email_sent = send_email($request_data['email'], $approval_email_subject, $approval_email_body);
                    } elseif (function_exists('mail')) {
                        // Fallback to basic PHP mail
                        $headers = "MIME-Version: 1.0" . "\r\n";
                        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
                        $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
                        $email_sent = mail($request_data['email'], $approval_email_subject, $approval_email_body, $headers);
                    } else {
                        error_log("No email function available - assignment approved but email not sent");
                    }
                    
                } catch (Exception $email_error) {
                    error_log("Email error: " . $email_error->getMessage());
                    $email_sent = false;
                }
                
                // Set success message regardless of email status
                if ($email_sent) {
                    $message = "✅ Request approved successfully and assignment uploaded! Student has been notified via email.";
                    $messageType = 'success';
                } else {
                    $message = "✅ Request approved and assignment uploaded successfully! Note: Email notification failed - please contact the student manually at " . $request_data['email'];
                    $messageType = 'warning';
                }
                
            } catch (Exception $e) {
                // Rollback transaction on error
                $conn->rollback();
                
                // Clean up any uploaded files on error
                if (isset($request_upload_dir) && file_exists($request_upload_dir)) {
                    $files_to_clean = glob($request_upload_dir . '/*');
                    foreach ($files_to_clean as $file) {
                        if (is_file($file)) unlink($file);
                    }
                    rmdir($request_upload_dir);
                }
                
                $message = 'Error: ' . $e->getMessage();
                $messageType = 'error';
            }
            
        } elseif ($action === 'approve') {
            // Legacy approval (keep for compatibility)
            $stmt = $conn->prepare("UPDATE accreditation_requests SET status = 'approved', approved_by = ?, approved_date = NOW(), admin_notes = ? WHERE request_id = ?");
            $admin_notes = $_POST['admin_notes'] ?? 'Request approved for SITA assessment';
            $stmt->bind_param("isi", $_SESSION['user_id'], $admin_notes, $request_id);
            
            if ($stmt->execute()) {
                // Get request details for email
                $req_stmt = $conn->prepare("SELECT ar.*, u.email, u.first_name, u.last_name, c.title as course_title FROM accreditation_requests ar 
                                           JOIN users u ON ar.student_id = u.user_id 
                                           JOIN courses c ON ar.course_id = c.course_id 
                                           WHERE ar.request_id = ?");
                $req_stmt->bind_param("i", $request_id);
                $req_stmt->execute();
                $request_data = $req_stmt->get_result()->fetch_assoc();
                
                // Send approval email using existing email service
                require_once __DIR__ . '/../config/fast_email_service.php';
                
                $approval_email_subject = "SITA Accreditation Request Approved - {$request_data['course_title']}";
                $approval_email_body = "
                <h2>🎉 Your SITA Accreditation Request Has Been Approved!</h2>
                <p>Dear {$request_data['first_name']},</p>
                <p>Great news! Your SITA accreditation request for <strong>{$request_data['course_title']}</strong> has been approved by our admin team.</p>
                
                <div style='background: #f0f9ff; border: 2px solid #0ea5e9; border-radius: 8px; padding: 15px; margin: 20px 0;'>
                    <h3 style='color: #0ea5e9; margin-top: 0;'>📋 Next Steps:</h3>
                    <ol style='color: #1e293b; line-height: 1.6;'>
                        <li><strong>SITA Assessment Creation:</strong> We will create your personalized SITA assessment within 1-2 business days</li>
                        <li><strong>Assessment Completion:</strong> You'll receive details to complete the assessment (minimum 70% required)</li>
                        <li><strong>Document Submission:</strong> Submit required documents (CV, qualifications, ID, course assignments)</li>
                        <li><strong>Final Review:</strong> Our team will conduct final review for certificate issuance</li>
                    </ol>
                </div>
                
                <p><strong>Request ID:</strong> #" . str_pad($request_id, 6, '0', STR_PAD_LEFT) . "</p>
                <p><strong>Admin Notes:</strong> {$admin_notes}</p>
                
                <p style='color: #059669; font-weight: bold;'>⏱️ Timeline: Complete process typically takes up to 5 months.</p>
                
                <p>You will receive detailed assessment instructions via email within 1-2 business days.</p>
                
                <p>Thank you for choosing Panadite Academy for your professional development!</p>
                
                <hr style='margin: 30px 0;'>
                <p style='font-size: 0.9em; color: #64748b;'>
                    <strong>Panadite Academy</strong><br>
                    Professional Development & SITA Accreditation<br>
                    This is an automated message. Please do not reply directly to this email.
                </p>
                ";
                
                $email_sent = queueEmail(
                    $request_data['email'],
                    $approval_email_subject,
                    $approval_email_body,
                    'accreditation_approval',
                    'high'
                );
                
                if ($email_sent) {
                    $message = 'Request approved successfully. Student has been notified via email.';
                    error_log("Accreditation approval email queued for request ID: $request_id, student: {$request_data['email']}");
                } else {
                    $message = 'Request approved successfully. Email notification failed to send.';
                    error_log("Failed to queue approval email for request ID: $request_id");
                }
                $messageType = 'success';
                
                // Create SITA assessment for this request
                $assessment_title = "SITA Assessment - {$request_data['course_title']}";
                $assessment_description = "This is your SITA accreditation assessment for {$request_data['course_title']}. You must achieve a minimum score of 70% to proceed to the next stage.";
                
                // Insert SITA assessment
                $assessment_stmt = $conn->prepare("INSERT INTO sita_assessments (student_id, course_id, accreditation_request_id, title, description, status, created_date, due_date) VALUES (?, ?, ?, ?, ?, 'pending', NOW(), DATE_ADD(NOW(), INTERVAL 30 DAY))");
                $assessment_stmt->bind_param("iiiss", $request_data['student_id'], $request_data['course_id'], $request_id, $assessment_title, $assessment_description);
                
                if ($assessment_stmt->execute()) {
                    $assessment_id = $conn->insert_id;
                    
                    // Update request status to indicate assessment has been created
                    $status_stmt = $conn->prepare("UPDATE accreditation_requests SET status = 'assessment_created' WHERE request_id = ?");
                    $status_stmt->bind_param("i", $request_id);
                    $status_stmt->execute();
                    
                    // Add sample SITA questions (you can customize these based on the course)
                    $sample_questions = [
                        "Describe the key learning objectives you achieved in this course and how they apply to your professional development.",
                        "Explain how the skills gained from this course will benefit your career or organization.",
                        "Provide specific examples of how you have applied or plan to apply the course concepts in real-world scenarios.",
                        "Analyze the most challenging aspects of the course content and how you overcame these challenges.",
                        "Evaluate the effectiveness of the course materials and suggest areas for improvement."
                    ];
                    
                    foreach ($sample_questions as $index => $question) {
                        $question_stmt = $conn->prepare("INSERT INTO sita_assessment_questions (assessment_id, question_text, question_order, question_type, max_words) VALUES (?, ?, ?, 'essay', 500)");
                        $order = $index + 1;
                        $question_stmt->bind_param("isi", $assessment_id, $question, $order);
                        $question_stmt->execute();
                    }
                    
                    error_log("SITA assessment created successfully for request ID: $request_id, assessment ID: $assessment_id");
                } else {
                    error_log("Failed to create SITA assessment for request ID: $request_id");
                }
                
            } else {
                $message = 'Error approving request.';
                $messageType = 'error';
            }
            
        } elseif ($action === 'reject') {
            $rejection_reason = $_POST['rejection_reason'] ?? 'Request rejected by admin';
            $stmt = $conn->prepare("UPDATE accreditation_requests SET status = 'rejected', rejection_reason = ?, approved_by = ?, approved_date = NOW() WHERE request_id = ?");
            $stmt->bind_param("sii", $rejection_reason, $_SESSION['user_id'], $request_id);
            
            if ($stmt->execute()) {
                // Get request details for rejection email
                $req_stmt = $conn->prepare("SELECT ar.*, u.email, u.first_name, u.last_name, c.title as course_title FROM accreditation_requests ar 
                                           JOIN users u ON ar.student_id = u.user_id 
                                           JOIN courses c ON ar.course_id = c.course_id 
                                           WHERE ar.request_id = ?");
                $req_stmt->bind_param("i", $request_id);
                $req_stmt->execute();
                $request_data = $req_stmt->get_result()->fetch_assoc();
                
                // Send rejection email using existing email service
                require_once __DIR__ . '/../config/fast_email_service.php';
                
                $rejection_email_subject = "SITA Accreditation Request Update - {$request_data['course_title']}";
                $rejection_email_body = "
                <h2>📋 SITA Accreditation Request Status Update</h2>
                <p>Dear {$request_data['first_name']},</p>
                <p>Thank you for your interest in SITA accreditation for <strong>{$request_data['course_title']}</strong>.</p>
                
                <div style='background: #fef2f2; border: 2px solid #f87171; border-radius: 8px; padding: 15px; margin: 20px 0;'>
                    <h3 style='color: #dc2626; margin-top: 0;'>❌ Request Status: Not Approved</h3>
                    <p style='color: #1e293b; line-height: 1.6; margin-bottom: 10px;'>
                        After careful review, we are unable to approve your accreditation request at this time.
                    </p>
                    <p><strong>Reason:</strong> {$rejection_reason}</p>
                </div>
                
                <div style='background: #f0fdf4; border: 2px solid #22c55e; border-radius: 8px; padding: 15px; margin: 20px 0;'>
                    <h3 style='color: #16a34a; margin-top: 0;'>💡 What You Can Do Next:</h3>
                    <ul style='color: #1e293b; line-height: 1.6;'>
                        <li>Review the reason provided above and address any identified issues</li>
                        <li>Complete additional course requirements if needed</li>
                        <li>You may resubmit your accreditation request once requirements are met</li>
                        <li>Contact our support team if you have questions about the decision</li>
                    </ul>
                </div>
                
                <p><strong>Request ID:</strong> #" . str_pad($request_id, 6, '0', STR_PAD_LEFT) . "</p>
                
                <p>We encourage you to continue your learning journey with Panadite Academy. Our support team is available to help you understand the requirements for future accreditation requests.</p>
                
                <p>Thank you for choosing Panadite Academy for your professional development.</p>
                
                <hr style='margin: 30px 0;'>
                <p style='font-size: 0.9em; color: #64748b;'>
                    <strong>Panadite Academy</strong><br>
                    Professional Development & SITA Accreditation<br>
                    Support: support@panaditeacademy.com<br>
                    This is an automated message. Please do not reply directly to this email.
                </p>
                ";
                
                $email_sent = queueEmail(
                    $request_data['email'],
                    $rejection_email_subject,
                    $rejection_email_body,
                    'accreditation_rejection',
                    'normal'
                );
                
                if ($email_sent) {
                    $message = 'Request rejected successfully. Student has been notified via email.';
                    error_log("Accreditation rejection email queued for request ID: $request_id, student: {$request_data['email']}");
                } else {
                    $message = 'Request rejected successfully. Email notification failed to send.';
                    error_log("Failed to queue rejection email for request ID: $request_id");
                }
                $messageType = 'success';
            } else {
                $message = 'Error rejecting request.';
                $messageType = 'error';
            }
        } elseif ($action === 'delete') {
            // Delete accreditation request
            try {
                $conn->begin_transaction();
                
                // Get request details for cleanup
                $req_stmt = $conn->prepare("SELECT * FROM accreditation_requests WHERE request_id = ?");
                $req_stmt->bind_param("i", $request_id);
                $req_stmt->execute();
                $request_data = $req_stmt->get_result()->fetch_assoc();
                
                if (!$request_data) {
                    throw new Exception('Request not found');
                }
                
                // Delete associated files if any
                if (!empty($request_data['document_path'])) {
                    $file_path = $_SERVER['DOCUMENT_ROOT'] . '/' . ltrim($request_data['document_path'], '/');
                    if (file_exists($file_path)) {
                        unlink($file_path);
                    }
                }
                
                if (!empty($request_data['assignment_document_path'])) {
                    $assignment_files = json_decode($request_data['assignment_document_path'], true);
                    if (is_array($assignment_files)) {
                        foreach ($assignment_files as $file) {
                            $file_path = $_SERVER['DOCUMENT_ROOT'] . '/' . ltrim($file['file_path'], '/');
                            if (file_exists($file_path)) {
                                unlink($file_path);
                            }
                        }
                    }
                }
                
                // Delete related SITA assessments (if table exists)
                $delete_assessments = $conn->prepare("DELETE FROM sita_assessments WHERE accreditation_request_id = ?");
                if ($delete_assessments) {
                    $delete_assessments->bind_param("i", $request_id);
                    $delete_assessments->execute();
                } else {
                    // Table might not exist, log but continue
                    error_log("sita_assessments table might not exist: " . $conn->error);
                }
                
                // Delete related accreditation certificates (foreign key constraint)
                $delete_certificates = $conn->prepare("DELETE FROM accreditation_certificates WHERE request_id = ?");
                if ($delete_certificates) {
                    $delete_certificates->bind_param("i", $request_id);
                    $delete_certificates->execute();
                } else {
                    error_log("accreditation_certificates table prepare failed: " . $conn->error);
                }
                
                // Delete the accreditation request
                $delete_stmt = $conn->prepare("DELETE FROM accreditation_requests WHERE request_id = ?");
                if (!$delete_stmt) {
                    throw new Exception('Failed to prepare delete statement: ' . $conn->error);
                }
                $delete_stmt->bind_param("i", $request_id);
                
                if ($delete_stmt->execute()) {
                    $conn->commit();
                    $message = 'Accreditation request deleted successfully.';
                    $messageType = 'success';
                } else {
                    throw new Exception('Failed to delete request: ' . $conn->error);
                }
                
            } catch (Exception $e) {
                $conn->rollback();
                $message = 'Error deleting request: ' . $e->getMessage();
                $messageType = 'error';
            }
        }
    }
}

// Get all accreditation requests with user and course info
$requests_query = "SELECT ar.*, 
                          u.first_name, u.last_name, u.email, u.profile_picture,
                          c.title as course_title, c.course_id,
                          admin.first_name as admin_first_name, admin.last_name as admin_last_name
                   FROM accreditation_requests ar
                   JOIN users u ON ar.student_id = u.user_id
                   JOIN courses c ON ar.course_id = c.course_id
                   LEFT JOIN users admin ON ar.approved_by = admin.user_id
                   ORDER BY ar.request_date DESC";

$requests_result = $conn->query($requests_query);
$requests = [];
if ($requests_result) {
    while ($row = $requests_result->fetch_assoc()) {
        $requests[] = $row;
    }
}

// Get statistics
$stats_query = "SELECT 
                    COUNT(*) as total_requests,
                    SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending_requests,
                    SUM(CASE WHEN status = 'approved' THEN 1 ELSE 0 END) as approved_requests,
                    SUM(CASE WHEN status = 'rejected' THEN 1 ELSE 0 END) as rejected_requests,
                    SUM(CASE WHEN status = 'certificate_issued' THEN 1 ELSE 0 END) as completed_requests
                FROM accreditation_requests";

$stats_result = $conn->query($stats_query);
$stats = $stats_result->fetch_assoc();

function getStatusBadgeClass($status) {
    switch ($status) {
        case 'pending': return 'bg-warning';
        case 'approved': return 'bg-primary';
        case 'assessment_created': return 'bg-info';
        case 'assessment_completed': return 'bg-success';
        case 'documents_pending': return 'bg-secondary';
        case 'documents_submitted': return 'bg-info';
        case 'under_review': return 'bg-warning';
        case 'approved_final': return 'bg-success';
        case 'certificate_issued': return 'bg-success';
        case 'rejected': return 'bg-danger';
        default: return 'bg-secondary';
    }
}

function getStatusText($status) {
    switch ($status) {
        case 'pending': return 'Pending Review';
        case 'approved': return 'Approved';
        case 'assessment_created': return 'Assessment Created';
        case 'assessment_completed': return 'Assessment Completed';
        case 'documents_pending': return 'Documents Pending';
        case 'documents_submitted': return 'Documents Submitted';
        case 'under_review': return 'Under Review';
        case 'approved_final': return 'Final Approval';
        case 'certificate_issued': return 'Certificate Issued';
        case 'rejected': return 'Rejected';
        default: return ucfirst(str_replace('_', ' ', $status));
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Accreditation Requests - Admin Dashboard</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link rel="stylesheet" href="/assets/css/user-cards.css">
    <!-- DataTables CSS -->
    <link rel="stylesheet" type="text/css" href="https://cdn.datatables.net/1.11.5/css/dataTables.bootstrap5.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    
    <style>
        /* Custom styles for accreditation requests page */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }

        .stat-card {
            background: white;
            padding: 1.5rem;
            border-radius: 12px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.08);
            border: 1px solid #e2e8f0;
            transition: all 0.3s ease;
        }

        .stat-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.12);
        }

        .stat-icon {
            width: 50px;
            height: 50px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.5rem;
            margin-bottom: 1rem;
        }

        .stat-value {
            font-size: 2rem;
            font-weight: 700;
            margin-bottom: 0.25rem;
        }

        .stat-label {
            font-size: 0.9rem;
            font-weight: 500;
        }

        .action-buttons {
            display: flex;
            gap: 0.5rem;
        }

        .user-info {
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .user-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            object-fit: cover;
            border: 2px solid #e2e8f0;
        }

        .user-avatar.placeholder {
            background: #2563eb;
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 600;
        }

        .user-details h6 {
            margin: 0;
            font-weight: 600;
            color: #1e293b;
            font-size: 0.9rem;
        }

        .user-details small {
            color: #64748b;
            font-size: 0.8rem;
        }

        .badge {
            font-size: 0.75rem;
            padding: 0.4rem 0.8rem;
            border-radius: 20px;
            font-weight: 600;
        }

        .btn {
            border-radius: 8px;
            font-weight: 500;
            padding: 0.5rem 1rem;
        }

        .btn-sm {
            padding: 0.375rem 0.75rem;
            font-size: 0.8rem;
        }

        .modal-content {
            border-radius: 16px;
            border: none;
        }

        .modal-header {
            background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
            color: white;
            border: none;
            padding: 1.5rem;
        }

        .form-control, .form-select {
            border: 2px solid #e2e8f0;
            border-radius: 8px;
            padding: 0.75rem;
        }

        .form-control:focus, .form-select:focus {
            border-color: var(--panadite-primary);
            box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
        }

        @media (max-width: 768px) {
            .main-content {
                margin-left: 0;
            }
            
            .dashboard-content {
                padding: 1rem;
            }
            
            .stats-grid {
                grid-template-columns: 1fr;
            }
            
            .action-buttons {
                flex-direction: column;
            }
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <!-- Page Header -->
            <div class="page-header">
                <h1><i class="bx bx-award me-2"></i>SITA Accreditation Requests</h1>
            </div>

            <!-- Alert Messages -->
            <?php if (!empty($message)): ?>
                <div class="alert alert-<?= $messageType === 'success' ? 'success' : 'danger' ?> alert-dismissible fade show" role="alert">
                    <i class="bx <?= $messageType === 'success' ? 'bx-check-circle' : 'bx-error-circle' ?> me-2"></i>
                    <?= htmlspecialchars($message) ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            <?php endif; ?>

            <!-- Statistics Cards -->
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-icon" style="background: rgba(59, 130, 246, 0.1); color: #3b82f6;">
                        <i class="bx bx-file-find"></i>
                    </div>
                    <div class="stat-value"><?= $stats['total_requests'] ?></div>
                    <div class="stat-label">Total Requests</div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon" style="background: rgba(245, 158, 11, 0.1); color: #f59e0b;">
                        <i class="bx bx-time"></i>
                    </div>
                    <div class="stat-value"><?= $stats['pending_requests'] ?></div>
                    <div class="stat-label">Pending Review</div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon" style="background: rgba(16, 185, 129, 0.1); color: #10b981;">
                        <i class="bx bx-check"></i>
                    </div>
                    <div class="stat-value"><?= $stats['approved_requests'] ?></div>
                    <div class="stat-label">Approved</div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon" style="background: rgba(16, 185, 129, 0.1); color: #10b981;">
                        <i class="bx bx-certification"></i>
                    </div>
                    <div class="stat-value"><?= $stats['completed_requests'] ?></div>
                    <div class="stat-label">Certificates Issued</div>
                </div>
            </div>

            <!-- Requests Table -->
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0"><i class="bx bx-list-ul me-2"></i>All Accreditation Requests</h5>
                </div>
                <div class="card-body">
                    <?php if (count($requests) > 0): ?>
                        <div class="table-responsive">
                            <table id="requestsTable" class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Student</th>
                                        <th>Course</th>
                                        <th>Request Date</th>
                                        <th>Status</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($requests as $request): ?>
                                        <tr>
                                            <td>
                                                <div class="user-info">
                                                    <?php if (!empty($request['profile_picture'])): ?>
                                                        <img src="<?= htmlspecialchars($request['profile_picture']) ?>" alt="Avatar" class="user-avatar">
                                                    <?php else: ?>
                                                        <div class="user-avatar placeholder">
                                                            <?= strtoupper(substr($request['first_name'], 0, 1)) ?>
                                                        </div>
                                                    <?php endif; ?>
                                                    <div class="user-details">
                                                        <h6><?= htmlspecialchars($request['first_name'] . ' ' . $request['last_name']) ?></h6>
                                                        <small><?= htmlspecialchars($request['email']) ?></small>
                                                    </div>
                                                </div>
                                            </td>
                                            <td>
                                                <strong><?= htmlspecialchars($request['course_title']) ?></strong>
                                            </td>
                                            <td>
                                                <?= date('M d, Y', strtotime($request['request_date'])) ?><br>
                                                <small class="text-muted"><?= date('H:i', strtotime($request['request_date'])) ?></small>
                                            </td>
                                            <td>
                                                <span class="badge <?= getStatusBadgeClass($request['status']) ?>">
                                                    <?= getStatusText($request['status']) ?>
                                                </span>
                                            </td>
                                            <td>
                                                <div class="action-buttons">
                                                    <?php if ($request['status'] === 'pending'): ?>
                                                        <button class="btn btn-success btn-sm" onclick="approveRequest(<?= $request['request_id'] ?>, '<?= htmlspecialchars($request['first_name'] . ' ' . $request['last_name']) ?>')">
                                                            <i class="bx bx-check me-1"></i>Approve
                                                        </button>
                                                        <button class="btn btn-danger btn-sm" onclick="rejectRequest(<?= $request['request_id'] ?>, '<?= htmlspecialchars($request['first_name'] . ' ' . $request['last_name']) ?>')">
                                                            <i class="bx bx-x me-1"></i>Reject
                                                        </button>
                                                    <?php endif; ?>
                                                    <button class="btn btn-outline-primary btn-sm" onclick="viewRequest(<?= $request['request_id'] ?>)">
                                                        <i class="bx bx-show me-1"></i>View
                                                    </button>
                                                    <button class="btn btn-outline-danger btn-sm" onclick="deleteRequest(<?= $request['request_id'] ?>, '<?= htmlspecialchars($request['first_name'] . ' ' . $request['last_name']) ?>')">
                                                        <i class="bx bx-trash me-1"></i>Delete
                                                    </button>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php else: ?>
                        <div class="text-center py-5">
                            <i class="bx bx-inbox" style="font-size: 4rem; color: #cbd5e1;"></i>
                            <h4 class="text-muted mt-3">No Accreditation Requests</h4>
                            <p class="text-muted">No students have submitted accreditation requests yet.</p>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </section>

    <!-- Approve & Upload Assignment Modal -->
    <div class="modal fade" id="approveModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title"><i class="bx bx-check-circle me-2"></i>Approve Request & Upload Assignment</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST" enctype="multipart/form-data" id="approveAssignmentForm">
                    <div class="modal-body">
                        <input type="hidden" name="action" value="approve_with_assignment">
                        <input type="hidden" name="request_id" id="approve_request_id">
                        
                        <div class="alert alert-success">
                            <i class="bx bx-info-circle me-2"></i>
                            <strong>Streamlined Approval:</strong> Approving will immediately create the SITA assignment for <strong id="approve_student_name"></strong>
                        </div>
                        
                        <!-- Assignment Instructions -->
                        <div class="mb-4">
                            <label for="assignment_instructions" class="form-label">
                                <i class="bx bx-edit me-1"></i><strong>Assignment Instructions</strong> <span class="text-danger">*</span>
                            </label>
                            <textarea class="form-control" id="assignment_instructions" name="assignment_instructions" rows="4" required 
                                placeholder="Enter detailed assignment instructions for the student...">Based on your completed course, please submit the following documents:

1. Updated CV/Resume
2. Copy of relevant qualifications/certificates  
3. Written assignment: Reflect on how this course will benefit your career (minimum 500 words)
4. Proof of identity (ID copy)

Please upload all documents in a single ZIP file for easy processing.</textarea>
                        </div>

                        <!-- File Upload Section -->
                        <div class="mb-4">
                            <label class="form-label">
                                <i class="bx bx-upload me-1"></i><strong>Assignment Documents</strong> <span class="text-danger">*</span>
                            </label>
                            <div class="upload-area border border-2 border-dashed rounded p-4 text-center" style="border-color: #dee2e6;">
                                <i class="bx bx-cloud-upload" style="font-size: 2rem; color: #6c757d;"></i>
                                <p class="mb-2"><strong>Upload Assignment Files</strong></p>
                                <p class="text-muted small mb-3">Drag & drop files here or click to browse</p>
                                
                                <input type="file" class="form-control" id="assignment_files" name="assignment_files[]" 
                                       multiple required accept=".pdf,.doc,.docx,.zip,.jpg,.jpeg,.png">
                                
                                <div class="mt-2">
                                    <small class="text-muted">
                                        <strong>Supported formats:</strong> PDF, DOC, DOCX, ZIP, JPG, PNG<br>
                                        <strong>Max size per file:</strong> 10MB | <strong>Max total files:</strong> 10
                                    </small>
                                </div>
                            </div>
                            
                            <!-- File Preview Area -->
                            <div id="file-preview" class="mt-3" style="display: none;">
                                <h6><i class="bx bx-list-ul me-1"></i>Selected Files:</h6>
                                <div id="file-list"></div>
                            </div>
                        </div>

                        <!-- Admin Notes -->
                        <div class="mb-3">
                            <label for="admin_notes" class="form-label">
                                <i class="bx bx-note me-1"></i>Admin Notes (Optional)
                            </label>
                            <textarea class="form-control" id="admin_notes" name="admin_notes" rows="2" 
                                placeholder="Add any additional notes for the student...">Your accreditation request has been approved! Please complete the assignment above within 14 days.</textarea>
                        </div>
                        
                        <div class="alert alert-warning">
                            <i class="bx bx-time me-2"></i>
                            <strong>Note:</strong> Student will receive email notification with assignment details and download links immediately after approval.
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                            <i class="bx bx-x me-1"></i>Cancel
                        </button>
                        <button type="submit" class="btn btn-success">
                            <i class="bx bx-check me-1"></i>Approve & Create Assignment
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Reject Request Modal -->
    <div class="modal fade" id="rejectModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header bg-danger">
                    <h5 class="modal-title"><i class="bx bx-x-circle me-2"></i>Reject Request</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <input type="hidden" name="action" value="reject">
                        <input type="hidden" name="request_id" id="reject_request_id">
                        
                        <p>Are you sure you want to reject the accreditation request for <strong id="reject_student_name"></strong>?</p>
                        
                        <div class="mb-3">
                            <label for="rejection_reason" class="form-label">Reason for Rejection <span class="text-danger">*</span></label>
                            <textarea class="form-control" id="rejection_reason" name="rejection_reason" rows="3" placeholder="Please provide a clear reason for rejection..." required></textarea>
                        </div>
                        
                        <div class="alert alert-warning">
                            <i class="bx bx-exclamation-triangle me-2"></i>
                            <strong>Note:</strong> The student will be notified of the rejection via email with your reason.
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-danger">
                            <i class="bx bx-x me-1"></i>Reject Request
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- View Request Details Modal -->
    <div class="modal fade" id="viewRequestModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title"><i class="bx bx-show me-2"></i>Request Details</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label fw-bold">Student Name</label>
                                <p id="view_student_name" class="form-control-plaintext">-</p>
                            </div>
                            <div class="mb-3">
                                <label class="form-label fw-bold">Email</label>
                                <p id="view_student_email" class="form-control-plaintext">-</p>
                            </div>
                            <div class="mb-3">
                                <label class="form-label fw-bold">Course</label>
                                <p id="view_course_title" class="form-control-plaintext">-</p>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label fw-bold">Request Date</label>
                                <p id="view_request_date" class="form-control-plaintext">-</p>
                            </div>
                            <div class="mb-3">
                                <label class="form-label fw-bold">Status</label>
                                <p id="view_status" class="form-control-plaintext">-</p>
                            </div>
                            <div class="mb-3">
                                <label class="form-label fw-bold">Payment Status</label>
                                <p id="view_payment_status" class="form-control-plaintext">-</p>
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label fw-bold">Payment Reference</label>
                        <p id="view_payment_reference" class="form-control-plaintext font-monospace">-</p>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label fw-bold">Student Motivation</label>
                        <div class="card bg-light">
                            <div class="card-body">
                                <p id="view_motivation" class="mb-0">-</p>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <div id="view_action_buttons" class="me-auto">
                        <!-- Action buttons will be populated by JavaScript -->
                    </div>
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <!-- DataTables -->
    <script src="https://cdn.datatables.net/1.11.5/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.11.5/js/dataTables.bootstrap5.min.js"></script>
    
    <script>
        // Initialize DataTables
        $(document).ready(function() {
            $('#requestsTable').DataTable({
                responsive: true,
                order: [[2, 'desc']], // Sort by request date descending
                pageLength: 25,
                language: {
                    search: "Search requests:",
                    lengthMenu: "Show _MENU_ requests per page",
                    info: "Showing _START_ to _END_ of _TOTAL_ requests"
                }
            });
        });

        // Enhanced approval modal with assignment upload
        function openEnhancedApproval(requestId, studentName) {
            document.getElementById('approve_request_id').value = requestId;
            document.getElementById('approve_student_name').textContent = studentName;
            
            // Reset form fields
            document.getElementById('assignment_instructions').value = `Based on your completed course, please submit the following documents:

1. Updated CV/Resume
2. Copy of relevant qualifications/certificates  
3. Written assignment: Reflect on how this course will benefit your career (minimum 500 words)
4. Proof of identity (ID copy)

Please upload all documents in a single ZIP file for easy processing.`;
            
            document.getElementById('admin_notes').value = `Your accreditation request has been approved! Please complete the assignment above within 14 days.`;
            
            // Reset file input
            document.getElementById('assignment_files').value = '';
            document.getElementById('file-preview').style.display = 'none';
            
            // Show the enhanced approval modal
            new bootstrap.Modal(document.getElementById('approveModal')).show();
        }

        // Legacy approval modal (for backward compatibility)
        function approveRequest(requestId, studentName) {
            // Redirect to enhanced approval
            openEnhancedApproval(requestId, studentName);
        }

        // Reject request modal
        function rejectRequest(requestId, studentName) {
            document.getElementById('reject_request_id').value = requestId;
            document.getElementById('reject_student_name').textContent = studentName;
            new bootstrap.Modal(document.getElementById('rejectModal')).show();
        }

        // Delete request with confirmation
        function deleteRequest(requestId, studentName) {
            if (confirm(`Are you sure you want to delete the accreditation request from ${studentName}?\n\nThis action cannot be undone and will:\n• Delete the request permanently\n• Remove any associated files\n• Delete related SITA assessments\n\nClick OK to confirm deletion.`)) {
                // Create form to submit delete action
                const form = document.createElement('form');
                form.method = 'POST';
                form.style.display = 'none';
                
                const actionInput = document.createElement('input');
                actionInput.name = 'action';
                actionInput.value = 'delete';
                form.appendChild(actionInput);
                
                const requestIdInput = document.createElement('input');
                requestIdInput.name = 'request_id';
                requestIdInput.value = requestId;
                form.appendChild(requestIdInput);
                
                document.body.appendChild(form);
                form.submit();
            }
        }

        // View request details (fully implemented)
        function viewRequest(requestId) {
            // Fetch request details via AJAX
            fetch(`api/get_accreditation_request.php?request_id=${requestId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        const request = data.request;
                        
                        // Populate view modal with request details
                        document.getElementById('view_student_name').textContent = `${request.first_name} ${request.last_name}`;
                        document.getElementById('view_student_email').textContent = request.email;
                        document.getElementById('view_course_title').textContent = request.course_title;
                        document.getElementById('view_request_date').textContent = new Date(request.request_date).toLocaleDateString();
                        document.getElementById('view_motivation').textContent = request.motivation || 'No motivation provided';
                        document.getElementById('view_status').innerHTML = `<span class="badge ${getStatusBadgeClass(request.status)}">${getStatusText(request.status)}</span>`;
                        document.getElementById('view_payment_status').textContent = request.payment_status || 'Not available';
                        document.getElementById('view_payment_reference').textContent = request.payment_reference || 'N/A';
                        
                        // Set up action buttons
                        const actionButtons = document.getElementById('view_action_buttons');
                        if (request.status === 'pending' || request.status === 'under_review') {
                            actionButtons.innerHTML = `
                                <button class="btn btn-success btn-lg" onclick="openEnhancedApproval(${request.request_id}, '${request.first_name} ${request.last_name}'); closeViewModal();">
                                    <i class="bx bx-check me-1"></i>Approve & Upload Assignment
                                </button>
                                <button class="btn btn-danger ms-2" onclick="rejectRequest(${request.request_id}, '${request.first_name} ${request.last_name}'); closeViewModal();">
                                    <i class="bx bx-x me-1"></i>Reject Request
                                </button>
                            `;
                        } else if (request.status === 'assignment_created') {
                            actionButtons.innerHTML = `
                                <div class="alert alert-success mb-0">
                                    <i class="bx bx-check-circle me-2"></i>
                                    ✅ <strong>Assignment Created!</strong> Student has been notified and can now download assignment files.
                                </div>
                            `;
                        } else {
                            actionButtons.innerHTML = `
                                <div class="alert alert-info mb-0">
                                    <i class="bx bx-info-circle me-2"></i>
                                    Request status: <strong>${getStatusText(request.status)}</strong>. No further action required.
                                </div>
                            `;
                        }
                        
                        // Show modal
                        new bootstrap.Modal(document.getElementById('viewRequestModal')).show();
                    } else {
                        alert('Failed to load request details: ' + (data.message || 'Unknown error'));
                    }
                })
                .catch(error => {
                    console.error('Error fetching request details:', error);
                    alert('Failed to load request details. Please try again.');
                });
        }
        
        function closeViewModal() {
            const modal = bootstrap.Modal.getInstance(document.getElementById('viewRequestModal'));
            if (modal) modal.hide();
        }
        
        function getStatusBadgeClass(status) {
            switch(status) {
                case 'pending': return 'bg-warning text-dark';
                case 'under_review': return 'bg-info';
                case 'approved': return 'bg-success';
                case 'rejected': return 'bg-danger';
                case 'assessment_ready': return 'bg-primary';
                case 'documents_submitted': return 'bg-info';
                case 'completed': return 'bg-success';
                default: return 'bg-secondary';
            }
        }
        
        function getStatusText(status) {
            switch(status) {
                case 'pending': return 'Pending Review';
                case 'under_review': return 'Under Review';
                case 'approved': return 'Approved';
                case 'rejected': return 'Rejected';
                case 'assessment_ready': return 'Assessment Ready';
                case 'documents_submitted': return 'Documents Submitted';
                case 'completed': return 'Completed';
                default: return 'Unknown';
            }
        }
        
        // File Upload Handling
        document.addEventListener('DOMContentLoaded', function() {
            const fileInput = document.getElementById('assignment_files');
            const uploadArea = document.querySelector('.upload-area');
            const filePreview = document.getElementById('file-preview');
            const fileList = document.getElementById('file-list');
            
            // Drag and drop functionality
            uploadArea.addEventListener('dragover', function(e) {
                e.preventDefault();
                uploadArea.style.borderColor = '#007bff';
                uploadArea.style.backgroundColor = '#f8f9ff';
            });
            
            uploadArea.addEventListener('dragleave', function(e) {
                e.preventDefault();
                uploadArea.style.borderColor = '#dee2e6';
                uploadArea.style.backgroundColor = '';
            });
            
            uploadArea.addEventListener('drop', function(e) {
                e.preventDefault();
                uploadArea.style.borderColor = '#dee2e6';
                uploadArea.style.backgroundColor = '';
                
                const files = e.dataTransfer.files;
                fileInput.files = files;
                displayFiles(files);
            });
            
            // Click to browse
            uploadArea.addEventListener('click', function() {
                fileInput.click();
            });
            
            // File selection handling
            fileInput.addEventListener('change', function(e) {
                displayFiles(e.target.files);
            });
            
            function displayFiles(files) {
                if (files.length === 0) {
                    filePreview.style.display = 'none';
                    return;
                }
                
                filePreview.style.display = 'block';
                fileList.innerHTML = '';
                
                let totalSize = 0;
                const maxFileSize = 10 * 1024 * 1024; // 10MB
                const maxFiles = 10;
                
                if (files.length > maxFiles) {
                    alert(`Maximum ${maxFiles} files allowed. Please select fewer files.`);
                    fileInput.value = '';
                    filePreview.style.display = 'none';
                    return;
                }
                
                for (let i = 0; i < files.length; i++) {
                    const file = files[i];
                    totalSize += file.size;
                    
                    if (file.size > maxFileSize) {
                        alert(`File "${file.name}" is too large. Maximum size is 10MB per file.`);
                        fileInput.value = '';
                        filePreview.style.display = 'none';
                        return;
                    }
                    
                    const fileItem = document.createElement('div');
                    fileItem.className = 'file-item d-flex align-items-center p-2 border rounded mb-2';
                    
                    const fileIcon = getFileIcon(file.name);
                    const fileSize = formatFileSize(file.size);
                    
                    fileItem.innerHTML = `
                        <i class="bx ${fileIcon} me-2" style="font-size: 1.2rem; color: #6c757d;"></i>
                        <div class="flex-grow-1">
                            <div class="fw-medium">${file.name}</div>
                            <small class="text-muted">${fileSize}</small>
                        </div>
                        <span class="badge bg-success">Ready</span>
                    `;
                    
                    fileList.appendChild(fileItem);
                }
                
                // Show total size
                const totalSizeElement = document.createElement('div');
                totalSizeElement.className = 'text-muted text-center mt-2';
                totalSizeElement.innerHTML = `<small><strong>Total:</strong> ${files.length} file(s), ${formatFileSize(totalSize)}</small>`;
                fileList.appendChild(totalSizeElement);
            }
            
            function getFileIcon(filename) {
                const extension = filename.split('.').pop().toLowerCase();
                switch(extension) {
                    case 'pdf': return 'bxs-file-pdf';
                    case 'doc':
                    case 'docx': return 'bxs-file-doc';
                    case 'zip':
                    case 'rar': return 'bxs-file-archive';
                    case 'jpg':
                    case 'jpeg':
                    case 'png':
                    case 'gif': return 'bxs-image';
                    default: return 'bxs-file';
                }
            }
            
            function formatFileSize(bytes) {
                if (bytes === 0) return '0 Bytes';
                const k = 1024;
                const sizes = ['Bytes', 'KB', 'MB', 'GB'];
                const i = Math.floor(Math.log(bytes) / Math.log(k));
                return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
            }
            
            // Form submission handling
            document.getElementById('approveAssignmentForm').addEventListener('submit', function(e) {
                const files = fileInput.files;
                const instructions = document.getElementById('assignment_instructions').value.trim();
                
                if (files.length === 0) {
                    e.preventDefault();
                    alert('Please select at least one assignment file to upload.');
                    return;
                }
                
                if (instructions === '') {
                    e.preventDefault();
                    alert('Please provide assignment instructions for the student.');
                    return;
                }
                
                // Show loading state
                const submitBtn = e.target.querySelector('button[type="submit"]');
                const originalText = submitBtn.innerHTML;
                submitBtn.innerHTML = '<i class="bx bx-loader-alt bx-spin me-1"></i>Processing...';
                submitBtn.disabled = true;
                
                // Re-enable after 30 seconds as fallback
                setTimeout(() => {
                    submitBtn.innerHTML = originalText;
                    submitBtn.disabled = false;
                }, 30000);
            });
        });
    </script>
</body>
</html>
