<?php
require_once '../auth/functions.php';

// Ensure user is logged in as an admin
requireRole('admin');

// Process form submission
$message = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Collect and sanitize input
    $title = trim($_POST['title']);
    $description = trim($_POST['description']);
    $teacher_id = $_POST['teacher_id'];
    $category_id = $_POST['category_id'];
    $price = floatval($_POST['price']);
    $is_published = isset($_POST['is_published']) ? 1 : 0;
    
    // Validate input
    if (empty($title)) {
        $message = '<div class="alert alert-danger">Course title is required.</div>';
    } else {
        // Handle thumbnail upload if provided
        $thumbnail = '';
        if (!empty($_FILES['thumbnail']['name'])) {
            $upload_dir = '../uploads/thumbnails/';
            
            // Create directory if it doesn't exist
            if (!file_exists($upload_dir)) {
                mkdir($upload_dir, 0777, true);
            }
            
            $file_ext = strtolower(pathinfo($_FILES['thumbnail']['name'], PATHINFO_EXTENSION));
            $file_name = uniqid() . '_' . time() . '.' . $file_ext;
            $target_file = $upload_dir . $file_name;
            
            // Check file type
            $allowed_types = array('jpg', 'jpeg', 'png', 'gif');
            if (!in_array($file_ext, $allowed_types)) {
                $message = '<div class="alert alert-danger">Only JPG, JPEG, PNG and GIF files are allowed for thumbnails.</div>';
            } 
            // Check file size (max 2MB)
            elseif ($_FILES['thumbnail']['size'] > 2097152) {
                $message = '<div class="alert alert-danger">File size should be less than 2MB.</div>';
            }
            // Upload file
            elseif (move_uploaded_file($_FILES['thumbnail']['tmp_name'], $target_file)) {
                $thumbnail = '/Panadite academy/uploads/thumbnails/' . $file_name;
            } else {
                $message = '<div class="alert alert-danger">Failed to upload thumbnail.</div>';
            }
        }
        
        // If no error message, proceed with database insertion
        if (empty($message)) {
            $stmt = $conn->prepare("INSERT INTO courses (teacher_id, category_id, title, description, thumbnail, price, is_published) VALUES (?, ?, ?, ?, ?, ?, ?)");
            $stmt->bind_param("iisssdi", $teacher_id, $category_id, $title, $description, $thumbnail, $price, $is_published);
            
            if ($stmt->execute()) {
                $course_id = $stmt->insert_id;
                $message = '<div class="alert alert-success">Course added successfully. <a href="edit_course.php?id=' . $course_id . '" class="alert-link">Edit Course</a> or <a href="videos.php?course_id=' . $course_id . '" class="alert-link">Add Videos</a></div>';
                
                // Reset form after successful submission
                $title = $description = '';
                $price = 0;
                $is_published = 0;
            } else {
                $message = '<div class="alert alert-danger">Error adding course: ' . $conn->error . '</div>';
            }
        }
    }
}

// Get all teachers for the form
$teachers = [];
$result = $conn->query("SELECT user_id, username, first_name, last_name FROM users WHERE user_role = 'teacher' ORDER BY first_name, last_name");
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $teachers[] = $row;
    }
}

// Get all categories for the form
$categories = [];
$result = $conn->query("SELECT category_id, category_name FROM categories ORDER BY category_name");
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $categories[] = $row;
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Add Course - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="/Panadite academy/assets/css/style.css">
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-lg-2 col-md-3 p-0 bg-dark sidebar">
                <div class="text-center p-4">
                    <a href="/Panadite academy/index.php">
                        <img src="/Panadite academy/assets/images/logo.jpeg" alt="Panadite Academy" class="img-fluid" style="max-height: 60px;">
                    </a>
                </div>
                <div class="pt-2 pb-4">
                    <ul class="nav flex-column">
                        <li class="nav-item">
                            <a class="nav-link text-white" href="dashboard.php">
                                <i class="fas fa-tachometer-alt"></i> Dashboard
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="users.php">
                                <i class="fas fa-users"></i> Users
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link active text-white" href="courses.php">
                                <i class="fas fa-graduation-cap"></i> Courses
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="videos.php">
                                <i class="fas fa-video"></i> Videos
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="categories.php">
                                <i class="fas fa-tags"></i> Categories
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="payments.php">
                                <i class="fas fa-money-bill"></i> Payments
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="payfast.php">
                                <i class="fas fa-credit-card"></i> PayFast Settings
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="/Panadite academy/auth/logout.php">
                                <i class="fas fa-sign-out-alt"></i> Logout
                            </a>
                        </li>
                    </ul>
                </div>
            </div>
            
            <!-- Main Content -->
            <div class="col-lg-10 col-md-9 ms-sm-auto px-4 py-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-4 border-bottom">
                    <h1 class="h2">Add New Course</h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <a href="courses.php" class="btn btn-sm btn-outline-secondary">
                            <i class="fas fa-arrow-left me-1"></i> Back to Courses
                        </a>
                    </div>
                </div>
                
                <?php echo $message; ?>
                
                <!-- Add Course Form -->
                <div class="card shadow-sm">
                    <div class="card-body">
                        <form action="add_course.php" method="post" enctype="multipart/form-data">
                            <div class="row mb-3">
                                <div class="col-md-8">
                                    <div class="mb-3">
                                        <label for="title" class="form-label">Course Title</label>
                                        <input type="text" class="form-control" id="title" name="title" value="<?php echo isset($title) ? htmlspecialchars($title) : ''; ?>" required>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="description" class="form-label">Description</label>
                                        <textarea class="form-control" id="description" name="description" rows="6"><?php echo isset($description) ? htmlspecialchars($description) : ''; ?></textarea>
                                    </div>
                                </div>
                                
                                <div class="col-md-4">
                                    <div class="mb-3">
                                        <label for="teacher_id" class="form-label">Teacher</label>
                                        <select class="form-select" id="teacher_id" name="teacher_id" required>
                                            <option value="">Select a Teacher</option>
                                            <?php foreach ($teachers as $teacher): ?>
                                                <option value="<?php echo $teacher['user_id']; ?>">
                                                    <?php echo htmlspecialchars($teacher['first_name'] . ' ' . $teacher['last_name'] . ' (' . $teacher['username'] . ')'); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                        <div class="form-text">Assign a teacher for this course</div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="category_id" class="form-label">Category</label>
                                        <select class="form-select" id="category_id" name="category_id" required>
                                            <option value="">Select a Category</option>
                                            <?php foreach ($categories as $category): ?>
                                                <option value="<?php echo $category['category_id']; ?>">
                                                    <?php echo htmlspecialchars($category['category_name']); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="price" class="form-label">Price (ZAR)</label>
                                        <div class="input-group">
                                            <span class="input-group-text">R</span>
                                            <input type="number" class="form-control" id="price" name="price" min="0" step="0.01" value="<?php echo isset($price) ? $price : '0.00'; ?>">
                                        </div>
                                        <div class="form-text">Set to 0.00 for free courses</div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="thumbnail" class="form-label">Course Thumbnail</label>
                                        <input type="file" class="form-control" id="thumbnail" name="thumbnail" accept="image/*">
                                        <div class="form-text">Recommended size: 1280x720px (16:9 ratio)</div>
                                    </div>
                                    
                                    <div class="mb-3 form-check">
                                        <input type="checkbox" class="form-check-input" id="is_published" name="is_published" <?php echo isset($is_published) && $is_published ? 'checked' : ''; ?>>
                                        <label class="form-check-label" for="is_published">Publish course immediately</label>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="border-top pt-3">
                                <button type="submit" class="btn btn-primary">Create Course</button>
                                <a href="courses.php" class="btn btn-outline-secondary ms-2">Cancel</a>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
