<?php
// Initialize session if not started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Check if user is logged in and is an admin
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Content-Type: application/json');
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

// Include database connection
include_once "../../config/db_connect.php";

// Set content type to JSON
header('Content-Type: application/json');

// Check if assignment_id is provided
if (!isset($_GET['assignment_id']) || empty($_GET['assignment_id'])) {
    echo json_encode(['success' => false, 'message' => 'Assignment ID is required']);
    exit;
}

$assignmentId = (int)$_GET['assignment_id'];

try {
    // Query to get full assignment details including teacher and student info
    $query = "SELECT tsa.assignment_id, tsa.teacher_id, tsa.student_id, 
              tsa.assigned_date, tsa.status, tsa.notes, tsa.assigned_by,
              t.first_name as teacher_first_name, t.last_name as teacher_last_name, 
              t.email as teacher_email,
              s.first_name as student_first_name, s.last_name as student_last_name,
              s.email as student_email,
              a.first_name as admin_first_name, a.last_name as admin_last_name
              FROM teacher_student_assignments tsa
              JOIN users t ON tsa.teacher_id = t.user_id
              JOIN users s ON tsa.student_id = s.user_id
              JOIN users a ON tsa.assigned_by = a.user_id
              LEFT JOIN teacher_profiles tp ON t.user_id = tp.teacher_id
              LEFT JOIN student_profiles sp ON s.user_id = sp.student_id
              WHERE tsa.assignment_id = :assignment_id";
    
    $stmt = $conn->prepare($query);
    $stmt->bindParam(':assignment_id', $assignmentId, PDO::PARAM_INT);
    $stmt->execute();
    
    // Check if assignment exists
    if ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $assignment = [
            'assignment_id' => $row['assignment_id'],
            'teacher_id' => $row['teacher_id'],
            'student_id' => $row['student_id'],
            'assigned_date' => $row['assigned_date'],
            'status' => $row['status'],
            'notes' => $row['notes'] ? htmlspecialchars($row['notes']) : null,
            'assigned_by' => $row['assigned_by'],
            'teacher_first_name' => htmlspecialchars($row['teacher_first_name']),
            'teacher_last_name' => htmlspecialchars($row['teacher_last_name']),
            'teacher_email' => htmlspecialchars($row['teacher_email']),
            'teacher_initials' => strtoupper(substr($row['teacher_first_name'], 0, 1) . substr($row['teacher_last_name'], 0, 1)),
            'student_first_name' => htmlspecialchars($row['student_first_name']),
            'student_last_name' => htmlspecialchars($row['student_last_name']),
            'student_email' => htmlspecialchars($row['student_email']),
            'student_initials' => strtoupper(substr($row['student_first_name'], 0, 1) . substr($row['student_last_name'], 0, 1)),
            'admin_first_name' => htmlspecialchars($row['admin_first_name']),
            'admin_last_name' => htmlspecialchars($row['admin_last_name']),
        ];
        
        // Return success response with assignment data
        echo json_encode([
            'success' => true,
            'data' => $assignment
        ]);
    } else {
        // Assignment not found
        echo json_encode([
            'success' => false,
            'message' => 'Assignment not found'
        ]);
    }
    
} catch (PDOException $e) {
    // Return error response
    echo json_encode([
        'success' => false,
        'message' => 'Database error: ' . $e->getMessage()
    ]);
}
