<?php
// Initialize session if not started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Check if user is logged in and is an admin
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Content-Type: application/json');
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

// Include database connection
require_once "../../config/db_connect.php";

// Set content type to JSON
header('Content-Type: application/json');

// Check if required data is provided
if (!isset($_POST['assignment_id']) || empty($_POST['assignment_id']) || 
    !isset($_POST['status']) || !in_array($_POST['status'], ['active', 'inactive'])) {
    echo json_encode(['success' => false, 'message' => 'Assignment ID and valid status are required']);
    exit;
}

$assignmentId = (int)$_POST['assignment_id'];
$status = $_POST['status'];

try {
    // Update the assignment status
    $query = "UPDATE teacher_student_assignments 
              SET status = :status 
              WHERE assignment_id = :assignment_id";
    
    $stmt = $conn->prepare($query);
    $stmt->bindParam(':assignment_id', $assignmentId, PDO::PARAM_INT);
    $stmt->bindParam(':status', $status, PDO::PARAM_STR);
    $stmt->execute();
    
    // Check if the update was successful (affected rows)
    if ($stmt->rowCount() > 0) {
        // Get updated assignment information
        $selectQuery = "SELECT tsa.teacher_id, tsa.student_id,
                        t.first_name as teacher_first_name, t.last_name as teacher_last_name,
                        s.first_name as student_first_name, s.last_name as student_last_name
                        FROM teacher_student_assignments tsa
                        JOIN users t ON tsa.teacher_id = t.user_id
                        JOIN users s ON tsa.student_id = s.user_id
                        WHERE tsa.assignment_id = :assignment_id";
        
        $selectStmt = $conn->prepare($selectQuery);
        $selectStmt->bindParam(':assignment_id', $assignmentId, PDO::PARAM_INT);
        $selectStmt->execute();
        $assignment = $selectStmt->fetch(PDO::FETCH_ASSOC);
        
        if ($assignment) {
            $teacherName = htmlspecialchars($assignment['teacher_first_name'] . ' ' . $assignment['teacher_last_name']);
            $studentName = htmlspecialchars($assignment['student_first_name'] . ' ' . $assignment['student_last_name']);
            
            $message = $status === 'active' 
                ? "Successfully activated assignment of $studentName to $teacherName" 
                : "Successfully deactivated assignment of $studentName to $teacherName";
            
            echo json_encode([
                'success' => true,
                'message' => $message,
                'data' => [
                    'assignment_id' => $assignmentId,
                    'status' => $status
                ]
            ]);
        } else {
            echo json_encode([
                'success' => true,
                'message' => "Assignment status updated successfully",
                'data' => [
                    'assignment_id' => $assignmentId,
                    'status' => $status
                ]
            ]);
        }
    } else {
        echo json_encode([
            'success' => false,
            'message' => 'Assignment not found or status already set to ' . $status
        ]);
    }
    
} catch (PDOException $e) {
    // Return error response
    echo json_encode([
        'success' => false,
        'message' => 'Database error: ' . $e->getMessage()
    ]);
}
