<?php
header('Content-Type: application/json');

require_once __DIR__ . '/../../auth/functions.php';
require_once __DIR__ . '/../../config/db_connect.php';

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Start output buffering to catch any unwanted output
ob_start();

try {
    // Check authentication and role
    if (!isLoggedIn()) {
        throw new Exception('Authentication required');
    }
    
    if (!hasRole('admin')) {
        throw new Exception('Admin access required');
    }
    
    // Get request ID from GET parameter
    if (!isset($_GET['request_id']) || !is_numeric($_GET['request_id'])) {
        throw new Exception('Valid request ID is required');
    }
    
    $request_id = intval($_GET['request_id']);
    
    // Fetch accreditation request details with all related information
    $stmt = $conn->prepare("
        SELECT 
            ar.*,
            u.first_name,
            u.last_name,
            u.email,
            u.profile_picture,
            c.title as course_title,
            c.course_id,
            admin_user.first_name as admin_first_name,
            admin_user.last_name as admin_last_name
        FROM accreditation_requests ar
        JOIN users u ON ar.student_id = u.user_id
        JOIN courses c ON ar.course_id = c.course_id
        LEFT JOIN users admin_user ON ar.approved_by = admin_user.user_id
        WHERE ar.request_id = ?
    ");
    
    if (!$stmt) {
        throw new Exception('Database prepare failed: ' . $conn->error);
    }
    
    $stmt->bind_param("i", $request_id);
    
    if (!$stmt->execute()) {
        throw new Exception('Database query failed: ' . $stmt->error);
    }
    
    $result = $stmt->get_result();
    $request = $result->fetch_assoc();
    
    if (!$request) {
        throw new Exception('Accreditation request not found');
    }
    
    // Format dates for better display
    if ($request['request_date']) {
        $request['formatted_request_date'] = date('M d, Y \a\t H:i', strtotime($request['request_date']));
    }
    
    if ($request['approved_date']) {
        $request['formatted_approved_date'] = date('M d, Y \a\t H:i', strtotime($request['approved_date']));
    }
    
    if ($request['payment_date']) {
        $request['formatted_payment_date'] = date('M d, Y \a\t H:i', strtotime($request['payment_date']));
    }
    
    // Add admin name if available
    if ($request['admin_first_name'] && $request['admin_last_name']) {
        $request['admin_name'] = $request['admin_first_name'] . ' ' . $request['admin_last_name'];
    }
    
    // Add progress information
    $request['progress_info'] = getProgressInfo($request['status']);
    
    // Clean the output buffer before sending JSON
    ob_end_clean();
    
    // Send success response
    echo json_encode([
        'success' => true,
        'request' => $request
    ]);
    
} catch (Exception $e) {
    // Clean the output buffer
    ob_end_clean();
    
    // Log error for debugging
    error_log('get_accreditation_request.php error: ' . $e->getMessage());
    
    // Send error response
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}

/**
 * Get progress information for a given status
 */
function getProgressInfo($status) {
    $steps = [
        'pending' => [
            'current_step' => 'Admin Review',
            'next_step' => 'Approval Decision',
            'progress_percent' => 25
        ],
        'under_review' => [
            'current_step' => 'Admin Review',
            'next_step' => 'Approval Decision', 
            'progress_percent' => 40
        ],
        'approved' => [
            'current_step' => 'SITA Assessment Creation',
            'next_step' => 'Assessment Completion',
            'progress_percent' => 60
        ],
        'assessment_ready' => [
            'current_step' => 'Student Assessment',
            'next_step' => 'Document Submission',
            'progress_percent' => 75
        ],
        'documents_submitted' => [
            'current_step' => 'Final Review',
            'next_step' => 'Certificate Issuance',
            'progress_percent' => 90
        ],
        'completed' => [
            'current_step' => 'Certificate Issued',
            'next_step' => 'Process Complete',
            'progress_percent' => 100
        ],
        'rejected' => [
            'current_step' => 'Request Rejected',
            'next_step' => 'Process Ended',
            'progress_percent' => 0
        ]
    ];
    
    return $steps[$status] ?? [
        'current_step' => 'Unknown Status',
        'next_step' => 'Please contact admin',
        'progress_percent' => 0
    ];
}
?>
