<?php
// Database connection handled by parent file

/**
 * Automated Certificate Generation System
 * Generates and emails certificates automatically when students complete courses
 */

class AutoCertificateGenerator {
    private $conn;
    
    public function __construct($connection) {
        $this->conn = $connection;
        $this->createCertificatesTable();
    }
    
    /**
     * Create certificates table if it doesn't exist
     */
    private function createCertificatesTable() {
        $sql = "
            CREATE TABLE IF NOT EXISTS certificates (
                certificate_id INT AUTO_INCREMENT PRIMARY KEY,
                course_id INT NOT NULL,
                student_id INT NOT NULL,
                certificate_number VARCHAR(50) UNIQUE NOT NULL,
                student_name VARCHAR(255) NOT NULL,
                student_email VARCHAR(255) NOT NULL,
                course_title VARCHAR(255) NOT NULL,
                issue_date DATE NOT NULL,
                completion_date DATE NOT NULL,
                status ENUM('active', 'revoked') DEFAULT 'active',
                email_sent BOOLEAN DEFAULT FALSE,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                UNIQUE KEY unique_student_course (course_id, student_id),
                INDEX idx_student (student_id),
                INDEX idx_course (course_id),
                INDEX idx_email_sent (email_sent)
            )
        ";
        $this->conn->query($sql);
    }
    
    /**
     * Generate certificate automatically when student completes course
     */
    public function generateCertificateOnCompletion($student_id, $course_id, $completion_date = null) {
        try {
            // Check if certificate already exists
            $checkStmt = $this->conn->prepare("SELECT certificate_id FROM certificates WHERE course_id = ? AND student_id = ?");
            $checkStmt->bind_param("ii", $course_id, $student_id);
            $checkStmt->execute();
            $existing = $checkStmt->get_result();
            
            if ($existing->num_rows > 0) {
                return ['success' => false, 'message' => 'Certificate already exists'];
            }
            
            // Get student details
            $studentStmt = $this->conn->prepare("SELECT first_name, last_name, email FROM users WHERE user_id = ? AND user_role = 'student'");
            $studentStmt->bind_param("i", $student_id);
            $studentStmt->execute();
            $studentResult = $studentStmt->get_result();
            $student = $studentResult->fetch_assoc();
            
            if (!$student) {
                return ['success' => false, 'message' => 'Student not found'];
            }
            
            // Get course details
            $courseStmt = $this->conn->prepare("SELECT title as course_title FROM courses WHERE course_id = ?");
            if (!$courseStmt) {
                return ['success' => false, 'message' => 'Database error: ' . $this->conn->error];
            }
            $courseStmt->bind_param("i", $course_id);
            $courseStmt->execute();
            $courseResult = $courseStmt->get_result();
            $course = $courseResult->fetch_assoc();
            
            if (!$course) {
                return ['success' => false, 'message' => 'Course not found'];
            }
            
            // Generate unique certificate number
            $certificate_number = 'PANADITE-' . date('Y') . '-' . sprintf('%04d', $course_id) . '-' . sprintf('%03d', $student_id);
            $student_name = $student['first_name'] . ' ' . $student['last_name'];
            $completion_date = $completion_date ?: date('Y-m-d');
            
            // Insert certificate record (using actual table structure)
            $certificate_description = "Certificate of completion for $student_name in " . $course['course_title'];
            $certificate_url = "/certificates/" . $certificate_number . ".pdf"; // Will be updated when PDF is generated
            
            $insertStmt = $this->conn->prepare("
                INSERT INTO certificates (course_id, student_id, certificate_number, title, description, issue_date, certificate_url) 
                VALUES (?, ?, ?, ?, ?, CURDATE(), ?)
            ");
            
            if (!$insertStmt) {
                return ['success' => false, 'message' => 'Certificate insert prepare failed: ' . $this->conn->error];
            }
            
            $insertStmt->bind_param("iissss", $course_id, $student_id, $certificate_number, $course['course_title'], $certificate_description, $certificate_url);
            
            if ($insertStmt->execute()) {
                $certificate_id = $this->conn->insert_id;
                
                // Generate certificate HTML
                $certificateHtml = $this->generateCertificateHtml($certificate_number, $student_name, $course['course_title'], date('F j, Y'));
                
                // Send email to student
                $emailSent = $this->sendCertificateEmail($student['email'], $student_name, $course['course_title'], $certificateHtml, $certificate_number);
                
                // Update email sent status
                if ($emailSent) {
                    $updateStmt = $this->conn->prepare("UPDATE certificates SET email_sent = TRUE WHERE certificate_id = ?");
                    $updateStmt->bind_param("i", $certificate_id);
                    $updateStmt->execute();
                }
                
                return [
                    'success' => true,
                    'certificate_id' => $certificate_id,
                    'certificate_number' => $certificate_number,
                    'email_sent' => $emailSent
                ];
            } else {
                return ['success' => false, 'message' => 'Failed to generate certificate'];
            }
            
        } catch (Exception $e) {
            error_log("Certificate generation error: " . $e->getMessage());
            return ['success' => false, 'message' => 'Certificate generation error: ' . $e->getMessage()];
        }
    }
    
    /**
     * Generate universal certificate HTML design with modern geometric style and Panadite branding
     */
    private function generateCertificateHtml($certificate_number, $student_name, $course_title, $issue_date) {
        return "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <title>Certificate of Completion - {$student_name}</title>
            <style>
                @import url('https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700;800&family=Playfair+Display:wght@400;500;600;700&display=swap');
                
                body {
                    margin: 0;
                    padding: 20px;
                    background: #f8fafc;
                    font-family: 'Poppins', sans-serif;
                }
                
                .certificate {
                    width: 900px;
                    height: 640px;
                    margin: 0 auto;
                    background: #ffffff;
                    position: relative;
                    box-shadow: 0 25px 80px rgba(0,0,0,0.12);
                    overflow: hidden;
                    border-radius: 8px;
                }
                
                /* Clean Geometric Background Elements */
                .geo-corner-tl {
                    position: absolute;
                    top: 0;
                    left: 0;
                    width: 150px;
                    height: 150px;
                    background: linear-gradient(135deg, #4169E1 0%, #5B7CFA 100%);
                    clip-path: polygon(0 0, 100% 0, 0 100%);
                    opacity: 0.9;
                    z-index: 1;
                }
                
                .geo-corner-tr {
                    position: absolute;
                    top: 0;
                    right: 0;
                    width: 120px;
                    height: 120px;
                    background: linear-gradient(225deg, #FFD700 0%, #F4C430 100%);
                    clip-path: polygon(100% 0, 100% 100%, 0 0);
                    opacity: 0.9;
                    z-index: 1;
                }
                
                .geo-corner-bl {
                    position: absolute;
                    bottom: 0;
                    left: 0;
                    width: 100px;
                    height: 100px;
                    background: linear-gradient(45deg, #20B2AA 0%, #17a2b8 100%);
                    clip-path: polygon(0 0, 0 100%, 100% 100%);
                    opacity: 0.85;
                    z-index: 1;
                }
                
                .geo-corner-br {
                    position: absolute;
                    bottom: 0;
                    right: 0;
                    width: 80px;
                    height: 80px;
                    background: linear-gradient(315deg, #4169E1 0%, #20B2AA 100%);
                    clip-path: polygon(100% 0, 100% 100%, 0 100%);
                    opacity: 0.8;
                    z-index: 1;
                }
                
                /* Inner border */
                .certificate-border {
                    position: absolute;
                    top: 25px;
                    left: 25px;
                    right: 25px;
                    bottom: 25px;
                    border: 3px solid #e9ecef;
                    z-index: 2;
                }
                
                /* Main content */
                .certificate-content {
                    position: relative;
                    z-index: 3;
                    padding: 60px;
                    height: 100%;
                    box-sizing: border-box;
                    display: flex;
                    flex-direction: column;
                    justify-content: space-between;
                }
                
                /* Clean Header with logo */
                .certificate-header {
                    text-align: center;
                    margin-bottom: 15px;
                    padding-top: 10px;
                }
                
                .panadite-logo {
                    width: 80px;
                    height: 80px;
                    margin: 0 auto 12px;
                    background: url('/assets/images/logo.jpeg') center/contain no-repeat;
                    border: 2px solid #E5E7EB;
                    border-radius: 50%;
                    filter: drop-shadow(0 2px 8px rgba(0,0,0,0.08));
                }
                
                .academy-name {
                    font-family: 'Poppins', sans-serif;
                    font-size: 1.1rem;
                    font-weight: 600;
                    color: #1F2937;
                    margin: 0;
                    letter-spacing: 2px;
                    text-transform: uppercase;
                }
                
                .academy-tagline {
                    font-family: 'Poppins', sans-serif;
                    font-size: 0.75rem;
                    color: #6B7280;
                    margin: 4px 0 0 0;
                    font-weight: 400;
                    letter-spacing: 0.5px;
                }
                
                /* Clean Certificate title */
                .certificate-title {
                    text-align: center;
                    margin: 25px 0 30px 0;
                }
                
                .title-main {
                    font-family: 'Poppins', sans-serif;
                    font-size: 2.8rem;
                    font-weight: 700;
                    color: #111827;
                    margin: 0;
                    letter-spacing: 3px;
                    text-transform: uppercase;
                }
                
                .title-sub {
                    font-family: 'Poppins', sans-serif;
                    font-size: 1rem;
                    color: #6B7280;
                    margin: 6px 0 0 0;
                    letter-spacing: 1px;
                    font-weight: 400;
                }
                
                /* Clean Certificate body */
                .certificate-body {
                    text-align: center;
                    margin: 30px 0;
                    flex-grow: 1;
                    display: flex;
                    flex-direction: column;
                    justify-content: center;
                    padding: 0 40px;
                }
                
                .certificate-text {
                    font-size: 0.95rem;
                    color: #4B5563;
                    margin: 8px 0;
                    font-weight: 400;
                    line-height: 1.5;
                    font-family: 'Poppins', sans-serif;
                }
                
                .student-name {
                    font-family: 'Poppins', sans-serif;
                    font-size: 2.2rem;
                    font-weight: 600;
                    color: #4169E1;
                    margin: 20px 0;
                    position: relative;
                    display: inline-block;
                }
                
                .student-name::after {
                    content: '';
                    position: absolute;
                    bottom: -8px;
                    left: 50%;
                    transform: translateX(-50%);
                    width: 60px;
                    height: 2px;
                    background: #FFD700;
                }
                
                .course-name {
                    font-family: 'Poppins', sans-serif;
                    font-size: 1.3rem;
                    color: #20B2AA;
                    font-weight: 500;
                    margin: 20px 0;
                    font-style: italic;
                }
                
                /* Clean Achievement badge */
                .achievement-badge {
                    position: absolute;
                    right: 60px;
                    bottom: 100px;
                    width: 70px;
                    height: 70px;
                    background: linear-gradient(135deg, #FFD700 0%, #F4C430 100%);
                    border-radius: 50%;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    box-shadow: 0 4px 15px rgba(255, 215, 0, 0.25);
                    border: 2px solid #fff;
                    z-index: 4;
                }
                
                .badge-content {
                    text-align: center;
                    color: white;
                }
                
                .badge-year {
                    font-size: 0.9rem;
                    font-weight: 600;
                    margin-bottom: 1px;
                }
                
                .badge-text {
                    font-size: 0.5rem;
                    font-weight: 500;
                    text-transform: uppercase;
                    letter-spacing: 0.5px;
                }
                
                /* Clean Footer */
                .certificate-footer {
                    display: flex;
                    justify-content: space-between;
                    align-items: end;
                    margin-top: 35px;
                    padding: 20px 0 0 0;
                    border-top: 1px solid #E5E7EB;
                }
                
                .signature-section {
                    text-align: center;
                    flex: 1;
                }
                
                .signature-line {
                    width: 100px;
                    height: 1px;
                    background: #D1D5DB;
                    margin: 0 auto 6px;
                }
                
                .signature-title {
                    font-size: 0.7rem;
                    color: #6B7280;
                    font-weight: 400;
                    line-height: 1.3;
                    font-family: 'Poppins', sans-serif;
                }
                
                .certificate-info {
                    text-align: center;
                    flex: 1;
                }
                
                .cert-number {
                    font-weight: 500;
                    color: #374151;
                    font-size: 0.8rem;
                    margin-bottom: 4px;
                    font-family: 'Poppins', sans-serif;
                }
                
                .issue-date {
                    color: #9CA3AF;
                    font-size: 0.7rem;
                    font-family: 'Poppins', sans-serif;
                }
                
                /* Subtle Watermark */
                .watermark {
                    position: absolute;
                    top: 50%;
                    left: 50%;
                    transform: translate(-50%, -50%) rotate(-45deg);
                    font-size: 6rem;
                    color: rgba(65, 105, 225, 0.02);
                    font-weight: 300;
                    z-index: 0;
                    font-family: 'Poppins', sans-serif;
                    letter-spacing: 6px;
                }
                
                /* Inner border for clean frame */
                .certificate-border {
                    position: absolute;
                    top: 20px;
                    left: 20px;
                    right: 20px;
                    bottom: 20px;
                    border: 1px solid #F3F4F6;
                    z-index: 2;
                    border-radius: 4px;
                }
                
                @media print {
                    body {
                        background: white;
                        padding: 0;
                    }
                    .certificate {
                        box-shadow: none;
                        page-break-inside: avoid;
                    }
                }
            </style>
        </head>
        <body>
            <div class='certificate'>
                <!-- Geometric corner elements -->
                <div class='geo-corner-tl'></div>
                <div class='geo-corner-tr'></div>
                <div class='geo-corner-bl'></div>
                <div class='geo-corner-br'></div>
                
                <!-- Watermark -->
                <div class='watermark'>PANADITE</div>
                
                <!-- Certificate border -->
                <div class='certificate-border'></div>
                
                <!-- Main content -->
                <div class='certificate-content'>
                    <!-- Header -->
                    <div class='certificate-header'>
                        <div class='panadite-logo'></div>
                        <h2 class='academy-name'>PANADITE ACADEMY</h2>
                        <p class='academy-tagline'>Get ahead of the curve</p>
                    </div>
                    
                    <!-- Title -->
                    <div class='certificate-title'>
                        <h1 class='title-main'>Certificate</h1>
                        <p class='title-sub'>of Completion</p>
                    </div>
                    
                    <!-- Body -->
                    <div class='certificate-body'>
                        <p class='certificate-text'>This certificate is presented to</p>
                        <div class='student-name'>{$student_name}</div>
                        <p class='certificate-text'>in recognition of successful completion of</p>
                        <div class='course-name'>{$course_title}</div>
                        <p class='certificate-text'>demonstrating dedication and achievement in professional development</p>
                    </div>
                    
                    <!-- Footer -->
                    <div class='certificate-footer'>
                        <div class='signature-section'>
                            <div class='signature-line'></div>
                            <p class='signature-title'>Director<br>Panadite Academy</p>
                        </div>
                        
                        <div class='certificate-info'>
                            <p class='cert-number'>Certificate No: {$certificate_number}</p>
                            <p class='issue-date'>Issued on {$issue_date}</p>
                        </div>
                        
                        <div class='signature-section'>
                            <div class='signature-line'></div>
                            <p class='signature-title'>Academic Registrar<br>Panadite Academy</p>
                        </div>
                    </div>
                </div>
                
                <!-- Achievement Badge -->
                <div class='achievement-badge'>
                    <div class='badge-content'>
                        <div class='badge-year'>" . date('Y') . "</div>
                        <div class='badge-text'>Certified</div>
                    </div>
                </div>
            </div>
        </body>
        </html>";
    }
    
    /**
     * Generate test certificate HTML using existing design
     */
    public function generateTestCertificateHtml($certificate_number, $student_name, $course_title, $issue_date) {
        return $this->generateCertificateHtml($certificate_number, $student_name, $course_title, $issue_date);
    }
    
    /**
     * Generate EMAIL-FRIENDLY certificate HTML (simplified for email clients)
     */
    public function generateEmailFriendlyCertificateHtml($certificate_number, $student_name, $course_title, $issue_date) {
        return "
        <div style='max-width: 600px; margin: 20px auto; background: #ffffff; border: 3px solid #4169E1; border-radius: 15px; padding: 40px; font-family: Arial, sans-serif; text-align: center;'>
            <!-- Header -->
            <div style='margin-bottom: 30px;'>
                <div style='width: 60px; height: 60px; background: linear-gradient(135deg, #4169E1, #20B2AA); border-radius: 50%; margin: 0 auto 15px; display: inline-block; line-height: 60px; color: white; font-weight: bold; font-size: 24px;'>PA</div>
                <h2 style='margin: 0; color: #1F2937; font-size: 18px; font-weight: bold; letter-spacing: 2px;'>PANADITE ACADEMY</h2>
                <p style='margin: 5px 0 0 0; color: #6B7280; font-size: 12px;'>Excellence in Education</p>
            </div>
            
            <!-- Certificate Title -->
            <div style='margin: 30px 0;'>
                <h1 style='margin: 0 0 10px 0; color: #111827; font-size: 32px; font-weight: bold;'>CERTIFICATE</h1>
                <p style='margin: 0; color: #4169E1; font-size: 14px; font-weight: bold; letter-spacing: 1px;'>OF COMPLETION</p>
            </div>
            
            <!-- Main Content -->
            <div style='margin: 40px 0; padding: 20px; background: #f8faff; border-radius: 10px;'>
                <p style='margin: 0 0 15px 0; color: #374151; font-size: 16px;'>This is to certify that</p>
                <h2 style='margin: 0 0 15px 0; color: #111827; font-size: 28px; font-weight: bold; text-transform: uppercase;'>{$student_name}</h2>
                <p style='margin: 0 0 15px 0; color: #374151; font-size: 16px;'>has successfully completed the course</p>
                <h3 style='margin: 0; color: #4169E1; font-size: 22px; font-weight: bold;'>{$course_title}</h3>
            </div>
            
            <!-- Footer Info -->
            <div style='margin-top: 40px; padding-top: 20px; border-top: 2px solid #e9ecef;'>
                <table style='width: 100%; margin: 0;'>
                    <tr>
                        <td style='text-align: left; padding: 5px;'>
                            <strong style='color: #4169E1;'>Certificate Number:</strong><br>
                            <span style='color: #6B7280; font-size: 14px;'>{$certificate_number}</span>
                        </td>
                        <td style='text-align: right; padding: 5px;'>
                            <strong style='color: #4169E1;'>Issue Date:</strong><br>
                            <span style='color: #6B7280; font-size: 14px;'>{$issue_date}</span>
                        </td>
                    </tr>
                </table>
            </div>
            
            <!-- Signature Area -->
            <div style='margin-top: 30px; text-align: center;'>
                <div style='border-bottom: 2px solid #4169E1; width: 200px; margin: 0 auto 5px;'></div>
                <p style='margin: 0; color: #6B7280; font-size: 12px; font-weight: bold;'>PANADITE ACADEMY</p>
                <p style='margin: 0; color: #9CA3AF; font-size: 11px;'>Digital Certificate</p>
            </div>
            
            <!-- Decorative Elements -->
            <div style='margin-top: 20px;'>
                <span style='color: #FFD700; font-size: 20px;'>⭐</span>
                <span style='color: #4169E1; font-size: 16px; margin: 0 10px;'>◆</span>
                <span style='color: #20B2AA; font-size: 18px;'>✦</span>
                <span style='color: #4169E1; font-size: 16px; margin: 0 10px;'>◆</span>
                <span style='color: #FFD700; font-size: 20px;'>⭐</span>
            </div>
        </div>";
    }
    
    /**
     * Send certificate email to student
     */
    private function sendCertificateEmail($email, $student_name, $course_title, $certificateHtml, $certificate_number) {
        try {
            $subject = "🎓 Congratulations! Your Certificate of Completion - {$course_title}";
            
            $message = "
            <div style='font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; background: #f8faff; padding: 20px; border-radius: 10px;'>
                <div style='text-align: center; margin-bottom: 30px;'>
                    <div style='width: 60px; height: 60px; background: linear-gradient(135deg, #4169E1, #20B2AA); border-radius: 50%; display: inline-flex; align-items: center; justify-content: center; color: white; font-size: 1.5rem; font-weight: bold; margin-bottom: 15px;'>PA</div>
                    <h1 style='color: #4169E1; margin: 0;'>Panadite Academy</h1>
                </div>
                
                <div style='background: white; padding: 30px; border-radius: 10px; box-shadow: 0 5px 15px rgba(0,0,0,0.1);'>
                    <h2 style='color: #20B2AA; margin-top: 0;'>🎉 Congratulations, {$student_name}!</h2>
                    
                    <p style='color: #333; font-size: 16px; line-height: 1.6;'>
                        We are thrilled to inform you that you have successfully completed the course <strong style='color: #4169E1;'>{$course_title}</strong>!
                    </p>
                    
                    <p style='color: #333; font-size: 16px; line-height: 1.6;'>
                        Your dedication and hard work have paid off. As a testament to your achievement, we have generated your official Certificate of Completion.
                    </p>
                    
                    <div style='background: #f0f8ff; padding: 20px; border-radius: 8px; margin: 20px 0; border-left: 4px solid #4169E1;'>
                        <h3 style='color: #4169E1; margin-top: 0;'>📜 Certificate Details</h3>
                        <p style='margin: 5px 0; color: #333;'><strong>Student:</strong> {$student_name}</p>
                        <p style='margin: 5px 0; color: #333;'><strong>Course:</strong> {$course_title}</p>
                        <p style='margin: 5px 0; color: #333;'><strong>Certificate Number:</strong> {$certificate_number}</p>
                        <p style='margin: 5px 0; color: #333;'><strong>Issue Date:</strong> " . date('F j, Y') . "</p>
                    </div>
                    
                    <p style='color: #333; font-size: 16px; line-height: 1.6;'>
                        Your certificate is attached to this email and can be printed for your records. This certificate serves as official recognition of your course completion and can be used for professional development purposes.
                    </p>
                    
                    <div style='text-align: center; margin: 30px 0;'>
                        <div style='display: inline-block; background: linear-gradient(135deg, #4169E1, #20B2AA); color: white; padding: 12px 24px; border-radius: 8px; text-decoration: none; font-weight: bold;'>
                            🏆 Certificate Attached
                        </div>
                    </div>
                    
                    <p style='color: #666; font-size: 14px; text-align: center; margin-top: 30px; padding-top: 20px; border-top: 1px solid #eee;'>
                        Thank you for choosing Panadite Academy for your learning journey.<br>
                        Keep learning, keep growing! 🌟
                    </p>
                </div>
                
                <div style='text-align: center; margin-top: 20px; color: #666; font-size: 12px;'>
                    © " . date('Y') . " Panadite Academy. All rights reserved.
                </div>
            </div>";
            
            $headers = [
                'MIME-Version: 1.0',
                'Content-type: text/html; charset=UTF-8',
                'From: Panadite Academy <certificates@panaditeacademy.com>',
                'Reply-To: support@panaditeacademy.com',
                'X-Mailer: PHP/' . phpversion()
            ];
            
            // For now, use basic mail() function
            // In production, consider using PHPMailer for better email delivery
            $success = mail($email, $subject, $message, implode("\r\n", $headers));
            
            // Log certificate email
            error_log("Certificate email sent to {$email} for course {$course_title}: " . ($success ? 'SUCCESS' : 'FAILED'));
            
            return $success;
            
        } catch (Exception $e) {
            error_log("Certificate email error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get all certificates for admin viewing
     */
    public function getAllCertificates($limit = 50) {
        try {
            // Ensure table exists first
            $this->createCertificatesTable();
            
            $query = "
                SELECT c.*, 
                       CASE WHEN c.email_sent = 1 THEN 'Sent' ELSE 'Pending' END as email_status
                FROM certificates c 
                ORDER BY c.created_at DESC 
                LIMIT ?
            ";
            
            $stmt = $this->conn->prepare($query);
            if (!$stmt) {
                error_log("Database prepare error: " . $this->conn->error);
                return [];
            }
            
            $stmt->bind_param("i", $limit);
            $stmt->execute();
            $result = $stmt->get_result();
            
            $certificates = [];
            while ($row = $result->fetch_assoc()) {
                $certificates[] = $row;
            }
            
            return $certificates;
            
        } catch (Exception $e) {
            error_log("Get certificates error: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get certificate statistics
     */
    public function getStatistics() {
        $stats = [
            'total_certificates' => 0,
            'emails_sent' => 0,
            'emails_pending' => 0,
            'this_month' => 0
        ];
        
        try {
            // Ensure table exists first
            $this->createCertificatesTable();
            
            // Total certificates
            $result = $this->conn->query("SELECT COUNT(*) as count FROM certificates");
            if ($result) {
                $stats['total_certificates'] = $result->fetch_assoc()['count'];
            }
            
            // Emails sent
            $result = $this->conn->query("SELECT COUNT(*) as count FROM certificates WHERE email_sent = 1");
            if ($result) {
                $stats['emails_sent'] = $result->fetch_assoc()['count'];
            }
            
            // Emails pending
            $result = $this->conn->query("SELECT COUNT(*) as count FROM certificates WHERE email_sent = 0");
            if ($result) {
                $stats['emails_pending'] = $result->fetch_assoc()['count'];
            }
            
            // This month
            $result = $this->conn->query("SELECT COUNT(*) as count FROM certificates WHERE MONTH(created_at) = MONTH(CURRENT_DATE()) AND YEAR(created_at) = YEAR(CURRENT_DATE())");
            if ($result) {
                $stats['this_month'] = $result->fetch_assoc()['count'];
            }
            
        } catch (Exception $e) {
            error_log("Get statistics error: " . $e->getMessage());
        }
        
        return $stats;
    }
}

// Function to be called when student completes a course
function generateCertificateOnCourseCompletion($student_id, $course_id, $completion_date = null) {
    global $conn;
    $generator = new AutoCertificateGenerator($conn);
    return $generator->generateCertificateOnCompletion($student_id, $course_id, $completion_date);
}
?>
