<?php
// Include database connection and required classes
require_once __DIR__ . '/../auth/functions.php';
require_once __DIR__ . '/../includes/classes/DatabaseBackup.php';
require_once __DIR__ . '/../includes/classes/BackupScheduler.php';

// Ensure user is logged in as an admin
requireRole('admin');

// Create backup directory if it doesn't exist
$backupDir = dirname(__DIR__) . '/backups';
if (!file_exists($backupDir)) {
    if (!mkdir($backupDir, 0755, true)) {
        die('Failed to create backup directory. Please check permissions.');
    }
}

// Ensure it's writable
if (!is_writable($backupDir)) {
    die('Backup directory is not writable. Please check permissions.');
}

// Initialize backup scheduler
$scheduler = new BackupScheduler();

// Create required tables if they don't exist
$scheduler->createHistoryTableIfNotExists();

// Get all database tables
$tables = array();
$result = $conn->query("SHOW TABLES");
while ($row = $result->fetch_row()) {
    $tables[] = $row[0];
}

// Process backup download if requested
if (isset($_GET['download']) && isset($_GET['id'])) {
    $id = intval($_GET['id']);
    $query = "SELECT file_path, filename FROM backup_history WHERE id = $id";
    $result = $conn->query($query);
    
    if ($result && $result->num_rows > 0) {
        $row = $result->fetch_assoc();
        $file_path = $row['file_path'];
        $filename = $row['filename'];
        
        // Log the file path for debugging
        error_log("Attempting to download file: {$file_path}");
        
        if (file_exists($file_path)) {
            // Log success
            error_log("File exists, sending for download: {$file_path}");
            
            // Set appropriate headers and send the file
            header('Content-Description: File Transfer');
            header('Content-Type: application/octet-stream');
            header('Content-Disposition: attachment; filename="' . $filename . '"');
            header('Expires: 0');
            header('Cache-Control: must-revalidate');
            header('Pragma: public');
            header('Content-Length: ' . filesize($file_path));
            
            // Clear any output buffers
            while (ob_get_level()) ob_end_clean();
            
            // Send the file
            if (readfile($file_path)) {
                exit;
            } else {
                error_log("Failed to read file: {$file_path}");
                $backup_message = '<div class="alert alert-danger">Error reading backup file.</div>';
            }
        } else {
            // Log the error for debugging
            error_log("Backup file does not exist at path: {$file_path}");
            
            // Check if the backups directory exists
            $backupsDir = dirname($file_path);
            if (!file_exists($backupsDir)) {
                error_log("Backups directory does not exist: {$backupsDir}");
            }
            
            $backup_message = '<div class="alert alert-danger">Backup file not found. Path: ' . htmlspecialchars($file_path) . '</div>';
        }
    } else {
        $backup_message = '<div class="alert alert-danger">Invalid backup ID.</div>';
    }
}

// Process delete backup if requested
if (isset($_GET['delete']) && isset($_GET['id'])) {
    // Validate the ID is numeric
    $id = filter_var($_GET['id'], FILTER_VALIDATE_INT);
    if ($id === false || $id <= 0) {
        $backup_message = '<div class="alert alert-danger">Invalid backup ID format.</div>';
    } else {
        // Log deletion attempt
        error_log("Attempting to delete backup with ID: $id");
        
        // Use prepared statement to prevent SQL injection
        $stmt = $conn->prepare("SELECT id, file_path, filename FROM backup_history WHERE id = ?");
        if ($stmt) {
            $stmt->bind_param("i", $id);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result && $result->num_rows > 0) {
                $row = $result->fetch_assoc();
                $file_path = $row['file_path'];
                $filename = $row['filename'];
                
                error_log("Found backup record: ID=$id, Path=$file_path, Name=$filename");
                
                // Delete file if exists
                $file_deleted = false;
                if (file_exists($file_path)) {
                    if (unlink($file_path)) {
                        error_log("Successfully deleted file: $file_path");
                        $file_deleted = true;
                    } else {
                        error_log("Failed to delete file: $file_path. Error: " . error_get_last()['message']);
                    }
                } else {
                    error_log("File not found for deletion: $file_path");
                }
                
                // Delete from history using prepared statement
                $delete_stmt = $conn->prepare("DELETE FROM backup_history WHERE id = ?");
                if ($delete_stmt) {
                    $delete_stmt->bind_param("i", $id);
                    if ($delete_stmt->execute()) {
                        error_log("Successfully deleted record from database: ID=$id");
                        $backup_message = '<div class="alert alert-success">Backup deleted successfully' . 
                                         ($file_deleted ? '.' : ', but the file was not found.') . '</div>';
                    } else {
                        error_log("Failed to delete record from database: ID=$id. Error: " . $delete_stmt->error);
                        $backup_message = '<div class="alert alert-warning">Failed to remove backup record from database.</div>';
                    }
                    $delete_stmt->close();
                } else {
                    error_log("Failed to prepare delete statement: " . $conn->error);
                    $backup_message = '<div class="alert alert-danger">Database error while deleting backup.</div>';
                }
            } else {
                error_log("No backup found with ID: $id");
                $backup_message = '<div class="alert alert-danger">Invalid backup ID. No matching backup found.</div>';
            }
            $stmt->close();
        } else {
            error_log("Failed to prepare select statement: " . $conn->error);
            $backup_message = '<div class="alert alert-danger">Database error while locating backup.</div>';
        }
    }
    
    // Redirect to prevent refresh issues and remove the delete parameters
    header("Location: backup.php?deleted=1");
    exit;
}

// Process form submission for manual backup
$backup_message = isset($backup_message) ? $backup_message : '';
if (isset($_POST['manual_backup'])) {
    $selected_tables = isset($_POST['tables']) ? $_POST['tables'] : array();
    $export_format = isset($_POST['export_format']) ? $_POST['export_format'] : 'sql';
    
    if (empty($selected_tables)) {
        $backup_message = '<div class="alert alert-danger">Please select at least one table to backup.</div>';
    } else {
        try {
            // Debug message about selected tables
            $tables_str = implode(', ', $selected_tables);
            $debug_message = "Creating backup of tables: $tables_str using format: $export_format";
            error_log($debug_message);
            
            // Create backup
            $backup = new DatabaseBackup($selected_tables, $export_format);
            $filePath = $backup->createBackup();
            
            // Verify the backup file exists
            if (!file_exists($filePath)) {
                throw new Exception("Backup file was not created at expected location: $filePath");
            }
            
            // Log success message
            error_log("Backup successfully created at: $filePath");
            
            // Log backup in history
            $scheduler->logBackupHistory($filePath);
            
            // Redirect to download
            $query = "SELECT id FROM backup_history ORDER BY id DESC LIMIT 1";
            $result = $conn->query($query);
            if ($result && $result->num_rows > 0) {
                $row = $result->fetch_assoc();
                header("Location: backup.php?download=1&id={$row['id']}");
                exit;
            } else {
                $backup_message = '<div class="alert alert-success">Backup created successfully! <a href="' . $filePath . '">Download</a></div>';
            }
        } catch (Exception $e) {
            // Log the detailed error
            error_log("Backup error: " . $e->getMessage());
            $backup_message = '<div class="alert alert-danger">Backup failed: ' . $e->getMessage() . '</div>';
        }
    }
}

// Save automatic backup settings
if (isset($_POST['save_settings'])) {
    $frequency = $_POST['frequency'];
    $format = $_POST['default_format'];
    $tables_to_backup = isset($_POST['auto_tables']) ? $_POST['auto_tables'] : array();
    $retention = intval($_POST['retention']);
    $interval_hours = isset($_POST['interval_hours']) ? intval($_POST['interval_hours']) : 24;
    
    // Save to database
    $scheduler->saveSettings([
        'frequency' => $frequency,
        'format' => $format,
        'tables' => $tables_to_backup,
        'retention' => $retention,
        'interval_hours' => $interval_hours,
        'enabled' => 1
    ]);
    
    // Redirect with settings_saved parameter instead of setting message directly
    header("Location: backup.php?settings_saved=1");
    exit;
}

// Check if automatic backup should run
if ($scheduler->shouldRunBackup()) {
    // Run in background - in a real environment this would be a cron job
    $scheduler->runAutomaticBackup();
}

// Get current settings
$current_settings = $scheduler->getSettings();

// Handle redirect parameters with different success messages
if (isset($_GET['deleted']) && $_GET['deleted'] == '1') {
    $backup_message = '<div class="alert alert-success">Backup deleted successfully.</div>';
} elseif (isset($_GET['settings_saved']) && $_GET['settings_saved'] == '1') {
    $backup_message = '<div class="alert alert-success">Backup settings saved successfully!</div>';
}

// Get backup history
$backup_history = $scheduler->getBackupHistory(10, 0);

// Function to calculate table size
function getTableSize($table) {
    return DatabaseBackup::getTableSize($table);
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Database Backup | Admin Dashboard</title>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css" rel="stylesheet">
    
    <!-- Boxicons CSS -->
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    
    <!-- Custom CSS -->
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link rel="stylesheet" href="/assets/css/user-cards.css">
    
    <!-- Google Fonts -->
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    
    <style>
        .backup-card {
            border-radius: 10px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.05);
            transition: all 0.3s ease;
        }
        
        .backup-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
        }
        
        .backup-icon {
            height: 60px;
            width: 60px;
            border-radius: 10px;
            background: rgba(67, 97, 238, 0.1);
            display: flex;
            align-items: center;
            justify-content: center;
            margin-bottom: 20px;
        }
        
        .backup-icon i {
            font-size: 30px;
            color: var(--primary-color);
        }
        
        .schedule-select {
            background-color: #f8f9fa;
            border-radius: 5px;
            padding: 10px;
            margin-bottom: 20px;
        }
        
        .table-checkbox {
            margin-right: 10px;
        }
        
        .format-option {
            display: flex;
            align-items: center;
            padding: 10px 15px;
            border-radius: 5px;
            margin-bottom: 10px;
            cursor: pointer;
            transition: all 0.2s ease;
        }
        
        .format-option:hover {
            background-color: #f8f9fa;
        }
        
        .format-option.active {
            background-color: rgba(67, 97, 238, 0.1);
            border-left: 3px solid var(--primary-color);
        }
        
        .format-icon {
            width: 40px;
            height: 40px;
            border-radius: 5px;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 15px;
        }
        
        .format-icon i {
            font-size: 24px;
        }
        
        .history-item {
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 10px;
            background-color: #f8f9fa;
            transition: all 0.2s ease;
        }
        
        .history-item:hover {
            background-color: #f1f3f5;
        }
        
        .progress-thin {
            height: 6px;
            border-radius: 3px;
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <div class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <div class="container-fluid">
                <!-- Page Header -->
                <div class="page-header">
                    <h1 class="animate-fade-in"><i class='bx bx-data me-2'></i>Database Backup</h1>
                </div>
                
                <?php if (!empty($backup_message)): ?>
                    <?= $backup_message ?>
                <?php endif; ?>
                
                <!-- Main Content Area -->
                <div class="row">
                    <!-- Left Column - Manual Backup -->
                    <div class="col-lg-8 mb-4">
                        <div class="card backup-card animate-fade-in">
                            <div class="card-body">
                                <h5 class="card-title mb-4">Manual Backup</h5>
                                
                                <form action="" method="POST">
                                    <div class="row mb-4">
                                        <div class="col-md-6">
                                            <div class="card h-100">
                                                <div class="card-header bg-light">
                                                    <h6 class="mb-0">Select Tables to Backup</h6>
                                                </div>
                                                <div class="card-body" style="max-height: 250px; overflow-y: auto;">
                                                    <div class="d-flex justify-content-between mb-3">
                                                        <div>
                                                            <input type="checkbox" id="select-all" class="form-check-input">
                                                            <label for="select-all" class="ms-2">Select All</label>
                                                        </div>
                                                        <span class="badge bg-primary"><?= count($tables) ?> Tables</span>
                                                    </div>
                                                    <hr>
                                                    <?php foreach ($tables as $table): ?>
                                                    <div class="form-check mb-2">
                                                        <input class="form-check-input table-checkbox" type="checkbox" name="tables[]" value="<?= $table ?>" id="table-<?= $table ?>">
                                                        <label class="form-check-label d-flex justify-content-between" for="table-<?= $table ?>">
                                                            <?= $table ?>
                                                            <span class="text-muted small">
                                                                <?php
                                                                // Get actual table size
                                                                $size = getTableSize($table);
                                                                echo $size;
                                                                ?>
                                                            </span>
                                                        </label>
                                                    </div>
                                                    <?php endforeach; ?>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <div class="col-md-6">
                                            <div class="card h-100">
                                                <div class="card-header bg-light">
                                                    <h6 class="mb-0">Export Format</h6>
                                                </div>
                                                <div class="card-body">
                                                    <div class="format-option active">
                                                        <div class="format-icon bg-light">
                                                            <i class='bx bx-data text-primary'></i>
                                                        </div>
                                                        <div class="flex-grow-1">
                                                            <div class="form-check">
                                                                <input class="form-check-input" type="radio" name="export_format" value="sql" id="format-sql" checked>
                                                                <label class="form-check-label" for="format-sql">
                                                                    <strong>SQL</strong>
                                                                    <p class="text-muted small mb-0">Complete database structure and data</p>
                                                                </label>
                                                            </div>
                                                        </div>
                                                    </div>
                                                    
                                                    <div class="format-option">
                                                        <div class="format-icon bg-success bg-opacity-10">
                                                            <i class='bx bx-file text-success'></i>
                                                        </div>
                                                        <div class="flex-grow-1">
                                                            <div class="form-check">
                                                                <input class="form-check-input" type="radio" name="export_format" value="excel" id="format-excel">
                                                                <label class="form-check-label" for="format-excel">
                                                                    <strong>Excel</strong>
                                                                    <p class="text-muted small mb-0">Data only in spreadsheet format</p>
                                                                </label>
                                                            </div>
                                                        </div>
                                                    </div>
                                                    
                                                    <div class="format-option">
                                                        <div class="format-icon bg-danger bg-opacity-10">
                                                            <i class='bx bxs-file-pdf text-danger'></i>
                                                        </div>
                                                        <div class="flex-grow-1">
                                                            <div class="form-check">
                                                                <input class="form-check-input" type="radio" name="export_format" value="pdf" id="format-pdf">
                                                                <label class="form-check-label" for="format-pdf">
                                                                    <strong>PDF</strong>
                                                                    <p class="text-muted small mb-0">Data in document format</p>
                                                                </label>
                                                            </div>
                                                        </div>
                                                    </div>
                                                    
                                                    <div class="format-option">
                                                        <div class="format-icon bg-primary bg-opacity-10">
                                                            <i class='bx bxs-file-doc text-primary'></i>
                                                        </div>
                                                        <div class="flex-grow-1">
                                                            <div class="form-check">
                                                                <input class="form-check-input" type="radio" name="export_format" value="csv" id="format-csv">
                                                                <label class="form-check-label" for="format-csv">
                                                                    <strong>CSV</strong>
                                                                    <p class="text-muted small mb-0">Data in comma-separated format</p>
                                                                </label>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="d-grid">
                                        <button type="submit" name="manual_backup" class="btn btn-primary">
                                            <i class='bx bx-download me-2'></i>Generate & Download Backup
                                        </button>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Right Column - Automatic Backup Settings -->
                    <div class="col-lg-4 mb-4">
                        <div class="card backup-card animate-fade-in" style="animation-delay: 0.2s;">
                            <div class="card-body">
                                <div class="d-flex align-items-center mb-4">
                                    <div class="backup-icon">
                                        <i class='bx bx-time'></i>
                                    </div>
                                    <div>
                                        <h5 class="card-title mb-0">Automatic Backup</h5>
                                        <p class="text-muted mb-0">Configure scheduled backups</p>
                                    </div>
                                </div>
                                
                                <form action="" method="POST">
                                    <div class="mb-3">
                                        <label class="form-label">Backup Frequency</label>
                                        <select class="form-select" name="frequency" id="backup-frequency">
                                            <option value="daily" <?= $current_settings['frequency'] == 'daily' ? 'selected' : '' ?>>Daily</option>
                                            <option value="weekly" <?= $current_settings['frequency'] == 'weekly' ? 'selected' : '' ?>>Weekly</option>
                                            <option value="biweekly" <?= $current_settings['frequency'] == 'biweekly' ? 'selected' : '' ?>>Bi-Weekly</option>
                                            <option value="monthly" <?= $current_settings['frequency'] == 'monthly' ? 'selected' : '' ?>>Monthly</option>
                                        </select>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="form-label d-flex justify-content-between">
                                            <span>Backup Interval</span>
                                            <span class="text-primary" id="interval-display"></span>
                                        </label>
                                        <div class="d-flex align-items-center">
                                            <input type="range" class="form-range flex-grow-1" id="interval-slider" name="interval_hours" min="1" max="720" step="1" value="<?= isset($current_settings['interval_hours']) ? $current_settings['interval_hours'] : 24 ?>">
                                        </div>
                                        <div class="mt-2">
                                            <div class="d-flex justify-content-between small text-muted">
                                                <span>Shorter</span>
                                                <span id="next-backup-time" class="text-primary fw-bold"></span>
                                                <span>Longer</span>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="form-label">Default Export Format</label>
                                        <select class="form-select" name="default_format">
                                            <option value="sql" <?= $current_settings['format'] == 'sql' ? 'selected' : '' ?>>SQL</option>
                                            <option value="excel" <?= $current_settings['format'] == 'excel' ? 'selected' : '' ?>>Excel</option>
                                            <option value="pdf" <?= $current_settings['format'] == 'pdf' ? 'selected' : '' ?>>PDF</option>
                                            <option value="csv" <?= $current_settings['format'] == 'csv' ? 'selected' : '' ?>>CSV</option>
                                        </select>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="form-label">Tables to Include</label>
                                        <select class="form-select" name="auto_tables[]" multiple size="5">
                                            <?php foreach ($tables as $table): ?>
                                                <option value="<?= $table ?>" <?= in_array($table, $current_settings['tables']) ? 'selected' : '' ?>><?= $table ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                        <div class="form-text">Hold Ctrl to select multiple tables</div>
                                    </div>
                                    
                                    <div class="mb-4">
                                        <label class="form-label">Retention Period (Days)</label>
                                        <input type="number" class="form-control" name="retention" value="<?= $current_settings['retention'] ?>" min="1" max="365">
                                        <div class="form-text">How long to keep automatic backups</div>
                                    </div>
                                    
                                    <div class="d-grid">
                                        <button type="submit" name="save_settings" class="btn btn-success">
                                            <i class='bx bx-save me-2'></i>Save Settings
                                        </button>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Backup History -->
                <div class="card backup-card animate-fade-in" style="animation-delay: 0.4s;">
                    <div class="card-body">
                        <h5 class="card-title mb-4">Backup History</h5>
                        
                        <div class="table-responsive">
                            <table class="table align-middle">
                                <thead class="table-light">
                                    <tr>
                                        <th>Filename</th>
                                        <th>Date</th>
                                        <th>Type</th>
                                        <th>Size</th>
                                        <th>Tables</th>
                                        <th>Status</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (empty($backup_history)): ?>
                                    <tr>
                                        <td colspan="7" class="text-center py-4">
                                            <i class='bx bx-data mb-3' style="font-size: 3rem; color: #adb5bd;"></i>
                                            <p class="text-muted">No backup history found. Create your first backup above.</p>
                                        </td>
                                    </tr>
                                    <?php else: ?>
                                    <?php foreach ($backup_history as $backup): ?>
                                    <tr>
                                        <td>
                                            <div class="d-flex align-items-center">
                                                <?php 
                                                // Set icon based on format
                                                $icon_class = 'bx bx-data text-primary'; // Default for SQL
                                                if ($backup['format'] == 'excel' || $backup['format'] == 'xlsx') {
                                                    $icon_class = 'bx bx-spreadsheet text-success';
                                                } elseif ($backup['format'] == 'pdf') {
                                                    $icon_class = 'bx bxs-file-pdf text-danger';
                                                } elseif ($backup['format'] == 'csv') {
                                                    $icon_class = 'bx bxs-file text-primary';
                                                }
                                                ?>
                                                <i class='<?= $icon_class ?> me-2'></i>
                                                <span><?= htmlspecialchars($backup['filename']) ?></span>
                                            </div>
                                        </td>
                                        <td><?= date('M j, Y', strtotime($backup['created_at'])) ?></td>
                                        <td>
                                            <?php if ($backup['type'] == 'auto'): ?>
                                            <span class="badge bg-success">Auto</span>
                                            <?php else: ?>
                                            <span class="badge bg-primary">Manual</span>
                                            <?php endif; ?>
                                        </td>
                                        <td><?= $backup['size'] ?></td>
                                        <td>
                                            <?php 
                                            $tables_array = explode(',', $backup['tables']);
                                            if ($backup['tables'] == 'all') {
                                                echo "All (" . count($tables) . ")";
                                            } else {
                                                $table_count = count($tables_array);
                                                if ($table_count > 2) {
                                                    echo htmlspecialchars($tables_array[0]) . ', ' . htmlspecialchars($tables_array[1]) . ' (+' . ($table_count - 2) . ' more)';
                                                } else {
                                                    echo htmlspecialchars($backup['tables']) . ' (' . $table_count . ')';
                                                }
                                            }
                                            ?>
                                        </td>
                                        <td><span class="badge bg-success"><?= ucfirst($backup['status']) ?></span></td>
                                        <td>
                                            <a href="backup.php?download=1&id=<?= $backup['id'] ?>" class="btn btn-sm btn-outline-primary" title="Download Backup"><i class='bx bx-download'></i></a>
                                            <a href="backup.php?delete=1&id=<?= $backup['id'] ?>" class="btn btn-sm btn-outline-danger" title="Delete Backup" onclick="return confirm('Are you sure you want to delete this backup?');"><i class='bx bx-trash'></i></a>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    
    <!-- Custom Scripts -->
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        // Select All Checkbox
        const selectAllCheckbox = document.getElementById('select-all');
        const tableCheckboxes = document.querySelectorAll('.table-checkbox');
        
        if (selectAllCheckbox) {
            selectAllCheckbox.addEventListener('change', function() {
                tableCheckboxes.forEach(checkbox => {
                    checkbox.checked = selectAllCheckbox.checked;
                });
            });
        }
        
        // Format options
        const formatOptions = document.querySelectorAll('.format-option');
        formatOptions.forEach(option => {
            option.addEventListener('click', function() {
                // Find the radio input inside this option
                const radio = this.querySelector('input[type="radio"]');
                radio.checked = true;
                
                // Remove active class from all options
                formatOptions.forEach(opt => opt.classList.remove('active'));
                
                // Add active class to this option
                this.classList.add('active');
            });
        });
        
        // Backup Interval Slider
        const intervalSlider = document.getElementById('interval-slider');
        const intervalDisplay = document.getElementById('interval-display');
        const nextBackupTime = document.getElementById('next-backup-time');
        const frequencySelect = document.getElementById('backup-frequency');
        
        if (intervalSlider) {
            // Function to format the interval
            function formatInterval(hours) {
                if (hours < 24) {
                    return hours + (hours === 1 ? ' hour' : ' hours');
                } else if (hours === 24) {
                    return '1 day';
                } else if (hours < 168) {
                    const days = Math.floor(hours / 24);
                    const remainingHours = hours % 24;
                    return days + (days === 1 ? ' day' : ' days') + 
                           (remainingHours > 0 ? ' ' + remainingHours + (remainingHours === 1 ? ' hour' : ' hours') : '');
                } else if (hours === 168) {
                    return '1 week';
                } else if (hours < 720) {
                    const weeks = Math.floor(hours / 168);
                    const remainingDays = Math.floor((hours % 168) / 24);
                    return weeks + (weeks === 1 ? ' week' : ' weeks') + 
                           (remainingDays > 0 ? ' ' + remainingDays + (remainingDays === 1 ? ' day' : ' days') : '');
                } else if (hours === 720) {
                    return '1 month';
                } else {
                    const months = Math.floor(hours / 720);
                    const remainingWeeks = Math.floor((hours % 720) / 168);
                    return months + (months === 1 ? ' month' : ' months') + 
                           (remainingWeeks > 0 ? ' ' + remainingWeeks + (remainingWeeks === 1 ? ' week' : ' weeks') : '');
                }
            }
            
            // Function to calculate and display next backup time
            function updateNextBackupTime() {
                const now = new Date();
                const hours = parseInt(intervalSlider.value);
                const next = new Date(now.getTime() + hours * 60 * 60 * 1000);
                
                // Format date for display
                const options = { weekday: 'short', month: 'short', day: 'numeric', hour: '2-digit', minute: '2-digit' };
                nextBackupTime.textContent = next.toLocaleDateString(undefined, options);
            }
            
            // Update display when slider changes
            function updateDisplay() {
                const hours = intervalSlider.value;
                intervalDisplay.textContent = formatInterval(hours);
                updateNextBackupTime();
            }
            
            // Set initial values based on frequency selection
            function setIntervalFromFrequency() {
                const frequency = frequencySelect.value;
                switch (frequency) {
                    case 'daily':
                        intervalSlider.value = 24;
                        break;
                    case 'weekly':
                        intervalSlider.value = 168; // 7 days * 24 hours
                        break;
                    case 'biweekly':
                        intervalSlider.value = 336; // 14 days * 24 hours
                        break;
                    case 'monthly':
                        intervalSlider.value = 720; // ~30 days * 24 hours
                        break;
                }
                updateDisplay();
            }
            
            // Initialize
            updateDisplay();
            
            // Add event listeners
            intervalSlider.addEventListener('input', updateDisplay);
            frequencySelect.addEventListener('change', setIntervalFromFrequency);
        }
    });
    </script>
</body>
</html>
