<?php
/**
 * Check Certificates Table Structure
 * Inspect the actual columns in the certificates table
 */

require_once '../config/db_connect.php';

echo "<h2>🔍 Certificates Table Structure Check</h2>";
echo "<div style='font-family: Arial, sans-serif; max-width: 800px; margin: 20px;'>";

try {
    // Check if certificates table exists
    $tableCheck = $conn->query("SHOW TABLES LIKE 'certificates'");
    
    if ($tableCheck->num_rows === 0) {
        echo "<div style='background: #f8d7da; border: 1px solid #f5c6cb; padding: 15px; border-radius: 5px; margin: 10px 0;'>";
        echo "<h3>❌ Certificates Table Missing</h3>";
        echo "<p>The certificates table does not exist. Need to create it first.</p>";
        echo "</div>";
        
        // Create certificates table
        $createTableSQL = "
            CREATE TABLE IF NOT EXISTS certificates (
                certificate_id INT PRIMARY KEY AUTO_INCREMENT,
                course_id INT NOT NULL,
                student_id INT NOT NULL,
                certificate_number VARCHAR(100) UNIQUE NOT NULL,
                student_name VARCHAR(200) NOT NULL,
                student_email VARCHAR(200) NOT NULL,
                course_title VARCHAR(300) NOT NULL,
                issue_date DATE NOT NULL,
                completion_date DATE NOT NULL,
                status ENUM('active', 'revoked') DEFAULT 'active',
                email_sent BOOLEAN DEFAULT FALSE,
                file_path VARCHAR(500),
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                INDEX idx_student_course (student_id, course_id),
                INDEX idx_certificate_number (certificate_number)
            ) ENGINE=InnoDB;
        ";
        
        if ($conn->query($createTableSQL)) {
            echo "<div style='background: #d4edda; border: 1px solid #c3e6cb; padding: 15px; border-radius: 5px; margin: 10px 0;'>";
            echo "<h3>✅ Certificates Table Created</h3>";
            echo "<p>Successfully created the certificates table with all required columns.</p>";
            echo "</div>";
        } else {
            throw new Exception("Failed to create certificates table: " . $conn->error);
        }
    }
    
    // Show table structure
    echo "<div style='background: #f8f9fa; border: 1px solid #dee2e6; padding: 15px; border-radius: 5px; margin: 10px 0;'>";
    echo "<h3>📊 Certificates Table Structure</h3>";
    
    $describeResult = $conn->query("DESCRIBE certificates");
    
    if ($describeResult) {
        echo "<table border='1' style='border-collapse: collapse; width: 100%; margin: 10px 0;'>";
        echo "<tr style='background: #e9ecef;'>";
        echo "<th style='padding: 8px;'>Field</th>";
        echo "<th style='padding: 8px;'>Type</th>";
        echo "<th style='padding: 8px;'>Null</th>";
        echo "<th style='padding: 8px;'>Key</th>";
        echo "<th style='padding: 8px;'>Default</th>";
        echo "<th style='padding: 8px;'>Extra</th>";
        echo "</tr>";
        
        while ($row = $describeResult->fetch_assoc()) {
            echo "<tr>";
            echo "<td style='padding: 8px; font-weight: bold;'>" . htmlspecialchars($row['Field']) . "</td>";
            echo "<td style='padding: 8px;'>" . htmlspecialchars($row['Type']) . "</td>";
            echo "<td style='padding: 8px;'>" . htmlspecialchars($row['Null']) . "</td>";
            echo "<td style='padding: 8px;'>" . htmlspecialchars($row['Key']) . "</td>";
            echo "<td style='padding: 8px;'>" . htmlspecialchars($row['Default'] ?? 'NULL') . "</td>";
            echo "<td style='padding: 8px;'>" . htmlspecialchars($row['Extra']) . "</td>";
            echo "</tr>";
        }
        echo "</table>";
    } else {
        throw new Exception("Failed to describe certificates table: " . $conn->error);
    }
    echo "</div>";
    
    // Show existing certificates if any
    $countResult = $conn->query("SELECT COUNT(*) as cert_count FROM certificates");
    $count = $countResult->fetch_assoc()['cert_count'];
    
    echo "<div style='background: #e2e3e5; border: 1px solid #c6c8ca; padding: 15px; border-radius: 5px; margin: 10px 0;'>";
    echo "<h3>📋 Existing Certificates</h3>";
    echo "<p><strong>Total Certificates:</strong> $count</p>";
    
    if ($count > 0) {
        $sampleResult = $conn->query("SELECT * FROM certificates ORDER BY certificate_id DESC LIMIT 3");
        echo "<p><strong>Recent certificates:</strong></p>";
        echo "<table border='1' style='border-collapse: collapse; width: 100%; margin: 10px 0;'>";
        echo "<tr style='background: #e9ecef;'>";
        echo "<th style='padding: 8px;'>ID</th>";
        echo "<th style='padding: 8px;'>Certificate Number</th>";
        echo "<th style='padding: 8px;'>Student</th>";
        echo "<th style='padding: 8px;'>Course</th>";
        echo "<th style='padding: 8px;'>Issue Date</th>";
        echo "</tr>";
        
        while ($cert = $sampleResult->fetch_assoc()) {
            echo "<tr>";
            echo "<td style='padding: 8px;'>" . htmlspecialchars($cert['certificate_id']) . "</td>";
            echo "<td style='padding: 8px;'>" . htmlspecialchars($cert['certificate_number']) . "</td>";
            echo "<td style='padding: 8px;'>" . htmlspecialchars($cert['student_name'] ?? 'N/A') . "</td>";
            echo "<td style='padding: 8px;'>" . htmlspecialchars($cert['course_title'] ?? 'N/A') . "</td>";
            echo "<td style='padding: 8px;'>" . htmlspecialchars($cert['issue_date']) . "</td>";
            echo "</tr>";
        }
        echo "</table>";
    }
    echo "</div>";
    
} catch (Exception $e) {
    echo "<div style='background: #f8d7da; border: 1px solid #f5c6cb; padding: 15px; border-radius: 5px; margin: 10px 0;'>";
    echo "<h3>❌ Error</h3>";
    echo "<p>" . htmlspecialchars($e->getMessage()) . "</p>";
    echo "</div>";
}

echo "</div>";
?>

<div style="margin: 20px;">
    <a href="../student/test_certificate_api.php" style="background: #007bff; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px;">
        Test Certificate Generation Again
    </a>
</div>
