<?php
require_once __DIR__ . '/../../includes/config.php';

// Include comprehensive SEO optimization
require_once __DIR__ . '/../../includes/seo_header.php';

// Comprehensive Admin Favicon - Same approach as Teacher/Student components
echo '
<!-- Admin Favicon - Comprehensive Fallback Options -->
<link rel="shortcut icon" href="/assets/images/favicon.ico" type="image/x-icon">
<link rel="icon" href="/assets/images/favicon.ico" type="image/x-icon">
<link rel="icon" href="/favicon.ico" type="image/x-icon">
<link rel="icon" href="/logo.jpeg" type="image/jpeg">
<link rel="icon" href="/assets/images/logo.jpeg" type="image/jpeg">

<!-- Apple Touch Icons -->
<link rel="apple-touch-icon" href="/logo.jpeg">
<link rel="apple-touch-icon" href="/assets/images/logo.jpeg">
';

// Smart Admin Notification System - Real-time data tracking
function getAdminNotifications() {
    global $conn;
    $notifications = [];
    $total_count = 0;
    
    // Check if database connection exists
    if (!$conn) {
        error_log("Database connection not available in notification system");
        return ['notifications' => [], 'total' => 0];
    }
    
    try {
        // 1. New Users (last 24 hours) - using safe query
        $query = "SELECT COUNT(*) as count FROM users WHERE created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)";
        $result = $conn->query($query);
        if ($result) {
            $new_users = $result->fetch_assoc()['count'];
            if ($new_users > 0) {
                $notifications[] = [
                    'icon' => 'bx-user-plus',
                    'text' => "$new_users new user" . ($new_users > 1 ? 's' : '') . " registered",
                    'link' => 'users.php',
                    'time' => 'Last 24h',
                    'type' => 'users'
                ];
                $total_count += $new_users;
            }
        }
        
        // 2. Pending Withdrawal Requests (most important for admin)
        $query = "SELECT COUNT(*) as count FROM withdrawal_requests WHERE status = 'pending'";
        $result = $conn->query($query);
        if ($result) {
            $pending_withdrawals = $result->fetch_assoc()['count'];
            if ($pending_withdrawals > 0) {
                $notifications[] = [
                    'icon' => 'bx-wallet',
                    'text' => "$pending_withdrawals withdrawal request" . ($pending_withdrawals > 1 ? 's' : '') . " pending",
                    'link' => 'withdrawal-requests.php',
                    'time' => 'Awaiting review',
                    'type' => 'withdrawals'
                ];
                $total_count += $pending_withdrawals;
            }
        }
        
        // 3. New Courses (last 7 days)
        $query = "SELECT COUNT(*) as count FROM courses WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)";
        $result = $conn->query($query);
        if ($result) {
            $new_courses = $result->fetch_assoc()['count'];
            if ($new_courses > 0) {
                $notifications[] = [
                    'icon' => 'bx-book-content',
                    'text' => "$new_courses new course" . ($new_courses > 1 ? 's' : '') . " added",
                    'link' => 'courses.php',
                    'time' => 'Last 7 days',
                    'type' => 'courses'
                ];
                $total_count += $new_courses;
            }
        }
        
        // 4. New Payments (last 24 hours)
        $query = "SELECT COUNT(*) as count FROM payments WHERE created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)";
        $result = $conn->query($query);
        if ($result) {
            $new_payments = $result->fetch_assoc()['count'];
            if ($new_payments > 0) {
                $notifications[] = [
                    'icon' => 'bx-money',
                    'text' => "$new_payments new payment" . ($new_payments > 1 ? 's' : '') . " received",
                    'link' => 'payments.php',
                    'time' => 'Last 24h',
                    'type' => 'payments'
                ];
                $total_count += $new_payments;
            }
        }
        
    } catch (Exception $e) {
        error_log("Notification system error: " . $e->getMessage());
    }
    
    return ['notifications' => $notifications, 'total' => $total_count];
}

$notification_data = getAdminNotifications();
$notifications = $notification_data['notifications'];
$total_notifications = $notification_data['total'];
?>

<style>
/* Smart Admin Notification System Styles - Embedded for navbar component */
.navbar-notifications {
    position: relative;
    margin-right: 20px;
}

.navbar-notifications i.bx-bell {
    cursor: pointer;
    padding: 8px;
    border-radius: 8px;
    transition: all 0.3s ease;
    position: relative;
}

.navbar-notifications i.bx-bell:hover {
    background: rgba(255, 107, 107, 0.1);
    transform: rotate(15deg);
}

/* Bell wiggle animation when notifications exist */
.navbar-notifications.has-notifications i.bx-bell {
    animation: bellWiggle 2s infinite;
}

@keyframes bellWiggle {
    0%, 50%, 100% { transform: rotate(0deg); }
    10% { transform: rotate(-10deg); }
    20% { transform: rotate(10deg); }
    30% { transform: rotate(-8deg); }
    40% { transform: rotate(8deg); }
}

.notification-badge {
    position: absolute;
    top: -8px;
    right: -8px;
    background: linear-gradient(135deg, #ff6b6b, #ee5a52);
    color: white;
    font-size: 0.75rem;
    font-weight: 700;
    min-width: 20px;
    height: 20px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    border: 2px solid white;
    box-shadow: 0 2px 8px rgba(255, 107, 107, 0.3);
    animation: pulse 2s infinite;
}

@keyframes pulse {
    0% { transform: scale(1); }
    50% { transform: scale(1.1); }
    100% { transform: scale(1); }
}

.notification-dropdown {
    min-width: 380px;
    max-width: 400px;
    border: none !important;
    border-radius: 16px;
    box-shadow: 0 10px 40px rgba(0, 0, 0, 0.15);
    padding: 0;
    margin-top: 10px;
    background: white;
}

.notification-dropdown .dropdown-header {
    background: rgba(255, 193, 7, 0.1) !important;
    color: #856404 !important;
    padding: 16px 20px;
    border-radius: 16px 16px 0 0;
    display: flex;
    align-items: center;
    justify-content: space-between;
    border-bottom: none;
    margin-bottom: 0;
    border: 1px solid rgba(255, 193, 7, 0.2);
}

.notification-dropdown .dropdown-header h6 {
    margin: 0 !important;
    font-weight: 600;
    font-size: 1rem;
    display: flex;
    align-items: center;
    gap: 8px;
    color: #856404 !important; /* Dark amber for good contrast */
}

.notification-item {
    padding: 12px 20px !important;
    border: none;
    display: flex;
    align-items: flex-start;
    gap: 12px;
    transition: all 0.2s ease;
    text-decoration: none;
    color: inherit;
    background: white;
}

.notification-item:hover {
    background: linear-gradient(90deg, rgba(67, 97, 238, 0.08), rgba(118, 75, 162, 0.08)) !important;
    color: inherit;
    text-decoration: none;
}

.notification-icon {
    width: 40px;
    height: 40px;
    background: linear-gradient(135deg, #4361ee, #3f37c9);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    flex-shrink: 0;
}

.notification-icon i {
    font-size: 1.2rem;
}

.notification-content {
    flex-grow: 1;
    min-width: 0;
}

.notification-content p {
    margin: 0 !important;
    font-weight: 500;
    color: #2d3436;
    font-size: 0.9rem;
    line-height: 1.4;
}

.notification-content small {
    color: #636e72;
    font-size: 0.8rem;
    font-weight: 400;
}

.dropdown-item-text.text-center {
    padding: 30px 20px;
    background: white;
}

.dropdown-item-text i.bx-check-circle {
    opacity: 0.6;
}

.dropdown-item-text p {
    color: #74b9ff !important;
    font-weight: 500;
}

.notification-item[href*="users.php"] .notification-icon {
    background: linear-gradient(135deg, #00b894, #00a085);
}

.notification-item[href*="withdrawal-requests.php"] .notification-icon {
    background: linear-gradient(135deg, #ff6b6b, #fd79a8);
}

.notification-item[href*="courses.php"] .notification-icon {
    background: linear-gradient(135deg, #6c5ce7, #a29bfe);
}

.notification-item[href*="payments.php"] .notification-icon {
    background: linear-gradient(135deg, #00b894, #55a3ff);
}

.notification-dropdown .dropdown-item.fw-bold {
    background: linear-gradient(135deg, #4361ee, #3f37c9) !important;
    color: white !important;
    margin: 8px 12px 12px 12px;
    border-radius: 8px;
    padding: 10px 15px;
    text-align: center;
    font-weight: 600;
    transition: all 0.2s ease;
    max-width: calc(100% - 24px);
    box-sizing: border-box;
}

.notification-dropdown .dropdown-item.fw-bold:hover {
    background: linear-gradient(135deg, #5a67d8, #6b46c1) !important;
    color: white !important;
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(102, 126, 234, 0.3);
}

.notification-dropdown .dropdown-divider {
    margin: 0;
    opacity: 0.1;
}

.badge.bg-danger {
    background: linear-gradient(135deg, #ff6b6b, #ee5a52) !important;
    font-weight: 600;
}

.badge.bg-success {
    background: linear-gradient(135deg, #00b894, #00a085) !important;
    font-weight: 600;
}

/* Simple Clean Profile Icon */
.simple-profile-icon {
    width: 40px;
    height: 40px;
    background: rgba(255, 193, 7, 0.1); /* Very light amber background */
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    color: #856404; /* Dark amber for icon */
    font-size: 1.4rem;
    transition: all 0.2s ease;
    border: 1px solid rgba(255, 193, 7, 0.2);
}

.simple-profile-icon:hover {
    background: rgba(255, 193, 7, 0.15);
    transform: scale(1.05);
}

@media (max-width: 768px) {
    .notification-dropdown {
        min-width: 320px;
        max-width: 95vw;
        left: -280px !important;
    }
    
    .notification-item {
        padding: 10px 16px !important;
    }
    
    .notification-icon {
        width: 35px;
        height: 35px;
    }
    
    .notification-content p {
        font-size: 0.85rem;
    }
    
    .profile-dropdown {
        min-width: 260px;
        left: -180px !important;
    }
    
    .admin-profile-header {
        padding: 16px;
    }
    
    .admin-avatar-large {
        width: 50px;
        height: 50px;
    }
    
    .admin-avatar-large .avatar-gradient {
        width: 50px;
        height: 50px;
        font-size: 1.7rem;
    }
}
</style>

<nav class="navbar">
    <div class="navbar-content">
        <i class='bx bx-menu sidebar-toggle-mobile'></i>
        <span class="page-title">Admin Dashboard</span>
    </div>
    
    <div class="navbar-right">
        <!-- Smart Admin Notifications -->
        <div class="navbar-notifications dropdown <?php echo $total_notifications > 0 ? 'has-notifications' : ''; ?>">
            <i class='bx bx-bell dropdown-toggle' data-bs-toggle="dropdown" style="color: <?php echo $total_notifications > 0 ? '#ff6b6b' : '#74b9ff'; ?>; font-size: 1.4rem; cursor: pointer;"></i>
            <?php if ($total_notifications > 0): ?>
                <span class="notification-badge"><?php echo $total_notifications; ?></span>
            <?php endif; ?>
            
            <ul class="dropdown-menu dropdown-menu-end notification-dropdown">
                <li class="dropdown-header">
                    <h6><i class='bx bx-bell'></i> Admin Notifications</h6>
                    <?php if ($total_notifications > 0): ?>
                        <span class="badge rounded-pill bg-danger"><?php echo $total_notifications; ?> New</span>
                    <?php else: ?>
                        <span class="badge rounded-pill bg-success">All Clear</span>
                    <?php endif; ?>
                </li>
                <li><hr class="dropdown-divider"></li>
                
                <?php if (empty($notifications)): ?>
                    <li>
                        <div class="dropdown-item-text text-center py-3">
                            <i class='bx bx-check-circle text-success' style="font-size: 2rem;"></i>
                            <p class="mb-0 mt-2 text-muted">No new notifications</p>
                            <small>All caught up!</small>
                        </div>
                    </li>
                <?php else: ?>
                    <?php foreach ($notifications as $notification): ?>
                        <li>
                            <a class="dropdown-item notification-item" href="<?php echo $notification['link']; ?>">
                                <div class="notification-icon">
                                    <i class='bx <?php echo $notification['icon']; ?>'></i>
                                </div>
                                <div class="notification-content">
                                    <p class="mb-1"><?php echo htmlspecialchars($notification['text']); ?></p>
                                    <small class="text-muted"><?php echo htmlspecialchars($notification['time']); ?></small>
                                </div>
                            </a>
                        </li>
                    <?php endforeach; ?>
                <?php endif; ?>
                
                <li><hr class="dropdown-divider"></li>
                <li><a class="dropdown-item text-center fw-bold" href="dashboard.php"><i class='bx bx-tachometer'></i> Go to Dashboard</a></li>
            </ul>
        </div>
        
        <!-- Clean Profile Icon -->
        <div class="navbar-profile">
            <div class="simple-profile-icon">
                <i class='bx bx-user-circle'></i>
            </div>
        </div>
    </div>
</nav>
