<?php
// Include the centralized admin layout
require_once __DIR__ . '/components/admin_layout.php';
require_once __DIR__ . '/../includes/connection.php';

// Check if admin is logged in
if (!isset($_SESSION['admin_id'])) {
    header("Location: ../auth/admin_login.php");
    exit();
}

$success_message = '';
$error_message = '';

// Handle form submission
if ($_POST) {
    try {
        $admin_percentage = (float)$_POST['admin_percentage'];
        $teacher_percentage = (float)$_POST['teacher_percentage'];
        
        // Validate percentages
        if ($admin_percentage < 0 || $admin_percentage > 100) {
            throw new Exception("Admin percentage must be between 0 and 100");
        }
        
        if (abs(($admin_percentage + $teacher_percentage) - 100) > 0.01) {
            throw new Exception("Percentages must add up to 100%");
        }
        
        // Update or insert revenue settings
        $stmt = $conn->prepare("
            INSERT INTO revenue_settings (setting_name, setting_value, updated_at) 
            VALUES ('admin_percentage', ?, NOW())
            ON DUPLICATE KEY UPDATE 
            setting_value = VALUES(setting_value), 
            updated_at = NOW()
        ");
        $stmt->bind_param("s", $admin_percentage);
        $stmt->execute();
        
        $stmt = $conn->prepare("
            INSERT INTO revenue_settings (setting_name, setting_value, updated_at) 
            VALUES ('teacher_percentage', ?, NOW())
            ON DUPLICATE KEY UPDATE 
            setting_value = VALUES(setting_value), 
            updated_at = NOW()
        ");
        $stmt->bind_param("s", $teacher_percentage);
        $stmt->execute();
        
        $success_message = "Revenue sharing settings updated successfully!";
        
    } catch (Exception $e) {
        $error_message = "Error: " . $e->getMessage();
    }
}

// Get current settings
$admin_percentage = 20; // Default
$teacher_percentage = 80; // Default

try {
    $stmt = $conn->prepare("SELECT setting_name, setting_value FROM revenue_settings WHERE setting_name IN ('admin_percentage', 'teacher_percentage')");
    $stmt->execute();
    $result = $stmt->get_result();
    
    while ($row = $result->fetch_assoc()) {
        if ($row['setting_name'] === 'admin_percentage') {
            $admin_percentage = (float)$row['setting_value'];
        } elseif ($row['setting_name'] === 'teacher_percentage') {
            $teacher_percentage = (float)$row['setting_value'];
        }
    }
} catch (Exception $e) {
    // Use defaults if table doesn't exist yet
}

// Get revenue statistics
$total_revenue = 0;
$admin_earned = 0;
$teacher_earned = 0;

try {
    $stmt = $conn->prepare("
        SELECT 
            SUM(amount) as total_revenue,
            SUM(admin_amount) as admin_earned,
            SUM(teacher_amount) as teacher_earned
        FROM course_purchases 
        WHERE status = 'completed'
    ");
    $stmt->execute();
    $result = $stmt->get_result();
    $stats = $result->fetch_assoc();
    
    $total_revenue = $stats['total_revenue'] ?? 0;
    $admin_earned = $stats['admin_earned'] ?? 0;
    $teacher_earned = $stats['teacher_earned'] ?? 0;
} catch (Exception $e) {
    // Tables might not exist yet
}

// Set current page for sidebar
$current_page = 'revenue-settings.php';
?>

<style>
    /* Modern Revenue Settings Design */
    * {
        font-family: 'Inter', 'Poppins', sans-serif !important;
    }

    :root {
        --primary-color: #4361ee;
        --secondary-color: #3f37c9;
        --success-color: #22c55e;
        --info-color: #06b6d4;
        --warning-color: #f59e0b;
        --danger-color: #ef4444;
        --light-color: #f8fafc;
        --dark-color: #1f2937;
    }

    body {
        background-color: var(--light-color) !important;
    }

    .revenue-container {
        padding: 0 30px !important;
    }
    
    /* Page Header */
    .page-header {
        background: white !important;
        border: 1px solid #f1f5f9 !important;
        border-radius: 12px !important;
        padding: 24px !important;
        margin-bottom: 32px !important;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08) !important;
    }
    
    .page-title {
        font-size: 24px !important;
        font-weight: 700 !important;
        color: #1f2937 !important;
        margin-bottom: 8px !important;
        display: flex !important;
        align-items: center !important;
    }

    .page-title i {
        color: var(--primary-color) !important;
        font-size: 28px !important;
        margin-right: 12px !important;
    }
    
    .page-description {
        font-size: 14px !important;
        color: #6b7280 !important;
        line-height: 1.6 !important;
    }
    
    /* Stats Cards */
    .stats-container {
        display: grid !important;
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)) !important;
        gap: 20px !important;
        margin-bottom: 32px !important;
    }
    
    .stat-card {
        background: white !important;
        border: 1px solid #f1f5f9 !important;
        border-radius: 12px !important;
        padding: 24px !important;
        text-align: center !important;
        transition: all 0.3s ease !important;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08) !important;
    }
    
    .stat-card:hover {
        transform: translateY(-2px) !important;
        box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15) !important;
    }
    
    .stat-icon {
        width: 56px !important;
        height: 56px !important;
        border-radius: 50% !important;
        display: flex !important;
        align-items: center !important;
        justify-content: center !important;
        margin: 0 auto 16px !important;
        font-size: 24px !important;
        color: white !important;
    }
    
    .stat-icon.total { background: var(--info-color) !important; }
    .stat-icon.admin { background: var(--primary-color) !important; }
    .stat-icon.teacher { background: var(--success-color) !important; }
    
    .stat-value {
        font-size: 32px !important;
        font-weight: 700 !important;
        color: #1f2937 !important;
        margin-bottom: 4px !important;
        line-height: 1 !important;
    }
    
    .stat-label {
        font-size: 14px !important;
        color: #6b7280 !important;
        font-weight: 500 !important;
    }
    
    /* Form Container */
    .form-container {
        background: white !important;
        border: 1px solid #f1f5f9 !important;
        border-radius: 12px !important;
        padding: 32px !important;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08) !important;
    }
    
    .form-title {
        font-size: 20px !important;
        font-weight: 600 !important;
        color: #1f2937 !important;
        margin-bottom: 24px !important;
        display: flex !important;
        align-items: center !important;
    }
    
    .form-title i {
        color: var(--primary-color) !important;
        margin-right: 8px !important;
    }
    
    /* Form Groups */
    .form-group {
        margin-bottom: 24px !important;
    }
    
    .form-label {
        font-size: 14px !important;
        font-weight: 500 !important;
        color: #1f2937 !important;
        margin-bottom: 8px !important;
        display: block !important;
    }
    
    .form-input {
        width: 100% !important;
        padding: 12px 16px !important;
        border: 1px solid #d1d5db !important;
        border-radius: 8px !important;
        font-size: 14px !important;
        transition: all 0.3s ease !important;
        background: white !important;
    }
    
    .form-input:focus {
        outline: none !important;
        border-color: var(--primary-color) !important;
        box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.1) !important;
    }
    
    /* Percentage Display */
    .percentage-display {
        display: grid !important;
        grid-template-columns: 1fr 1fr !important;
        gap: 20px !important;
        margin: 24px 0 !important;
        padding: 20px !important;
        background: #f8fafc !important;
        border-radius: 8px !important;
        border: 1px solid #f1f5f9 !important;
    }
    
    .percentage-item {
        text-align: center !important;
    }
    
    .percentage-value {
        font-size: 28px !important;
        font-weight: 700 !important;
        margin-bottom: 4px !important;
    }
    
    .percentage-value.admin { color: var(--primary-color) !important; }
    .percentage-value.teacher { color: var(--success-color) !important; }
    
    .percentage-label {
        font-size: 14px !important;
        color: #6b7280 !important;
        font-weight: 500 !important;
    }
    
    /* Buttons */
    .btn-primary {
        background: var(--primary-color) !important;
        color: white !important;
        padding: 12px 24px !important;
        border: none !important;
        border-radius: 8px !important;
        font-weight: 500 !important;
        font-size: 14px !important;
        cursor: pointer !important;
        transition: all 0.3s ease !important;
        display: inline-flex !important;
        align-items: center !important;
        gap: 8px !important;
    }
    
    .btn-primary:hover {
        background: var(--secondary-color) !important;
        transform: translateY(-1px) !important;
        box-shadow: 0 4px 12px rgba(67, 97, 238, 0.3) !important;
    }
    
    /* Alerts */
    .alert {
        padding: 12px 16px !important;
        border-radius: 8px !important;
        margin-bottom: 20px !important;
        font-size: 14px !important;
        display: flex !important;
        align-items: center !important;
        gap: 8px !important;
    }
    
    .alert-success {
        background: #f0fdf4 !important;
        color: #166534 !important;
        border: 1px solid #bbf7d0 !important;
    }
    
    .alert-danger {
        background: #fef2f2 !important;
        color: #dc2626 !important;
        border: 1px solid #fecaca !important;
    }
    
    /* Responsive */
    @media (max-width: 768px) {
        .revenue-container {
            padding: 0 20px !important;
        }
        
        .percentage-display {
            grid-template-columns: 1fr !important;
        }
    }
</style>

<div class="revenue-container">
    <!-- Page Header -->
    <div class="page-header">
        <h1 class="page-title">
            <i class='bx bx-pie-chart-alt-2'></i>
            Revenue Sharing Settings
        </h1>
        <p class="page-description">
            Configure how course revenue is split between the platform and teachers. This applies to all course purchases.
        </p>
    </div>

    <!-- Revenue Statistics -->
    <div class="stats-container">
        <div class="stat-card">
            <div class="stat-icon total">
                <i class='bx bx-money'></i>
            </div>
            <div class="stat-value">R<?= number_format($total_revenue, 2) ?></div>
            <div class="stat-label">Total Revenue</div>
        </div>
        
        <div class="stat-card">
            <div class="stat-icon admin">
                <i class='bx bx-building'></i>
            </div>
            <div class="stat-value">R<?= number_format($admin_earned, 2) ?></div>
            <div class="stat-label">Platform Earned</div>
        </div>
        
        <div class="stat-card">
            <div class="stat-icon teacher">
                <i class='bx bx-user-circle'></i>
            </div>
            <div class="stat-value">R<?= number_format($teacher_earned, 2) ?></div>
            <div class="stat-label">Teachers Earned</div>
        </div>
    </div>

    <!-- Settings Form -->
    <div class="form-container">
        <h2 class="form-title">
            <i class='bx bx-slider-alt'></i>
            Configure Revenue Split
        </h2>

        <?php if ($success_message): ?>
            <div class="alert alert-success">
                <i class='bx bx-check-circle'></i>
                <?= htmlspecialchars($success_message) ?>
            </div>
        <?php endif; ?>

        <?php if ($error_message): ?>
            <div class="alert alert-danger">
                <i class='bx bx-error-circle'></i>
                <?= htmlspecialchars($error_message) ?>
            </div>
        <?php endif; ?>

        <!-- Current Split Display -->
        <div class="percentage-display">
            <div class="percentage-item">
                <div class="percentage-value admin"><?= $admin_percentage ?>%</div>
                <div class="percentage-label">Platform Share</div>
            </div>
            <div class="percentage-item">
                <div class="percentage-value teacher"><?= $teacher_percentage ?>%</div>
                <div class="percentage-label">Teacher Share</div>
            </div>
        </div>

        <form method="POST" id="revenueForm">
            <div class="form-group">
                <label for="admin_percentage" class="form-label">
                    Platform Percentage (%)
                </label>
                <input 
                    type="number" 
                    id="admin_percentage" 
                    name="admin_percentage" 
                    class="form-input" 
                    value="<?= $admin_percentage ?>" 
                    min="0" 
                    max="100" 
                    step="0.1" 
                    required
                    oninput="updateTeacherPercentage()"
                >
                <small class="text-muted">The percentage the platform takes from each course purchase</small>
            </div>

            <div class="form-group">
                <label for="teacher_percentage" class="form-label">
                    Teacher Percentage (%)
                </label>
                <input 
                    type="number" 
                    id="teacher_percentage" 
                    name="teacher_percentage" 
                    class="form-input" 
                    value="<?= $teacher_percentage ?>" 
                    min="0" 
                    max="100" 
                    step="0.1" 
                    required
                    readonly
                >
                <small class="text-muted">Automatically calculated to total 100%</small>
            </div>

            <button type="submit" class="btn-primary">
                <i class='bx bx-save'></i>
                Update Revenue Settings
            </button>
        </form>
    </div>
</div>

<script>
function updateTeacherPercentage() {
    const adminPercentage = parseFloat(document.getElementById('admin_percentage').value) || 0;
    const teacherPercentage = 100 - adminPercentage;
    
    document.getElementById('teacher_percentage').value = teacherPercentage.toFixed(1);
    
    // Update display
    document.querySelector('.percentage-value.admin').textContent = adminPercentage.toFixed(1) + '%';
    document.querySelector('.percentage-value.teacher').textContent = teacherPercentage.toFixed(1) + '%';
}

// Form validation
document.getElementById('revenueForm').addEventListener('submit', function(e) {
    const adminPercentage = parseFloat(document.getElementById('admin_percentage').value);
    
    if (adminPercentage < 0 || adminPercentage > 100) {
        e.preventDefault();
        alert('Admin percentage must be between 0 and 100');
        return false;
    }
});
</script>

<?php
// Include the centralized admin layout end
require_once __DIR__ . '/components/admin_layout_end.php';
?>
