<?php
/**
 * Convert All Existing Courses to Quiz-Only System
 * Updates assignment_weight = 0%, quiz_weight = 100% for consistency
 */

require_once '../config/db_connect.php';

echo "<h2>🎯 Converting Courses to Quiz-Only System</h2>";
echo "<div style='font-family: Arial, sans-serif; max-width: 900px; margin: 20px;'>";

try {
    // Show current courses BEFORE conversion
    echo "<div style='background: #fff3cd; border: 1px solid #ffeaa7; padding: 15px; border-radius: 5px; margin: 10px 0;'>";
    echo "<h3>📊 BEFORE: Current Course Weights</h3>";
    
    $before_query = "SELECT course_id, title, assignment_weight, quiz_weight, course_passing_grade FROM courses ORDER BY course_id DESC";
    $before_result = $conn->query($before_query);
    
    if ($before_result && $before_result->num_rows > 0) {
        echo "<table border='1' style='border-collapse: collapse; width: 100%; margin: 10px 0;'>";
        echo "<tr style='background: #f8f9fa;'>";
        echo "<th style='padding: 8px;'>Course ID</th>";
        echo "<th style='padding: 8px;'>Title</th>";
        echo "<th style='padding: 8px;'>Assignment Weight</th>";
        echo "<th style='padding: 8px;'>Quiz Weight</th>";
        echo "<th style='padding: 8px;'>Passing Grade</th>";
        echo "</tr>";
        
        while ($row = $before_result->fetch_assoc()) {
            $assignment_color = ($row['assignment_weight'] > 0) ? 'color: red; font-weight: bold;' : 'color: green;';
            $quiz_color = ($row['quiz_weight'] < 100) ? 'color: red; font-weight: bold;' : 'color: green;';
            
            echo "<tr>";
            echo "<td style='padding: 8px;'>" . htmlspecialchars($row['course_id']) . "</td>";
            echo "<td style='padding: 8px;'>" . htmlspecialchars($row['title']) . "</td>";
            echo "<td style='padding: 8px; {$assignment_color}'>" . htmlspecialchars($row['assignment_weight']) . "%</td>";
            echo "<td style='padding: 8px; {$quiz_color}'>" . htmlspecialchars($row['quiz_weight']) . "%</td>";
            echo "<td style='padding: 8px;'>" . htmlspecialchars($row['course_passing_grade']) . "%</td>";
            echo "</tr>";
        }
        echo "</table>";
        echo "<p><strong>❌ Red = Needs fixing for quiz-only system</strong></p>";
    }
    echo "</div>";
    
    // Convert courses to quiz-only
    echo "<div style='background: #d4edda; border: 1px solid #c3e6cb; padding: 15px; border-radius: 5px; margin: 10px 0;'>";
    echo "<h3>🔧 Converting to Quiz-Only System</h3>";
    
    $convert_query = "
        UPDATE courses 
        SET 
            assignment_weight = 0.00,
            quiz_weight = 100.00
        WHERE assignment_weight != 0 OR quiz_weight != 100
    ";
    
    if ($conn->query($convert_query) === TRUE) {
        $affected_rows = $conn->affected_rows;
        echo "<p>✅ <strong>Successfully converted {$affected_rows} courses to quiz-only system!</strong></p>";
        echo "<ul>";
        echo "<li><strong>Assignment Weight:</strong> Set to 0.00%</li>";
        echo "<li><strong>Quiz Weight:</strong> Set to 100.00%</li>";
        echo "</ul>";
    } else {
        throw new Exception("Error converting courses: " . $conn->error);
    }
    echo "</div>";
    
    // Show courses AFTER conversion
    echo "<div style='background: #d1ecf1; border: 1px solid #bee5eb; padding: 15px; border-radius: 5px; margin: 10px 0;'>";
    echo "<h3>📊 AFTER: Updated Course Weights</h3>";
    
    $after_query = "SELECT course_id, title, assignment_weight, quiz_weight, course_passing_grade FROM courses ORDER BY course_id DESC";
    $after_result = $conn->query($after_query);
    
    if ($after_result && $after_result->num_rows > 0) {
        echo "<table border='1' style='border-collapse: collapse; width: 100%; margin: 10px 0;'>";
        echo "<tr style='background: #f8f9fa;'>";
        echo "<th style='padding: 8px;'>Course ID</th>";
        echo "<th style='padding: 8px;'>Title</th>";
        echo "<th style='padding: 8px;'>Assignment Weight</th>";
        echo "<th style='padding: 8px;'>Quiz Weight</th>";
        echo "<th style='padding: 8px;'>Passing Grade</th>";
        echo "<th style='padding: 8px;'>Status</th>";
        echo "</tr>";
        
        while ($row = $after_result->fetch_assoc()) {
            $is_correct = ($row['assignment_weight'] == 0 && $row['quiz_weight'] == 100);
            $status_color = $is_correct ? 'color: green; font-weight: bold;' : 'color: red; font-weight: bold;';
            $status_text = $is_correct ? '✅ Quiz-Only' : '❌ Needs Fix';
            
            echo "<tr>";
            echo "<td style='padding: 8px;'>" . htmlspecialchars($row['course_id']) . "</td>";
            echo "<td style='padding: 8px;'>" . htmlspecialchars($row['title']) . "</td>";
            echo "<td style='padding: 8px; color: green;'>" . htmlspecialchars($row['assignment_weight']) . "%</td>";
            echo "<td style='padding: 8px; color: green; font-weight: bold;'>" . htmlspecialchars($row['quiz_weight']) . "%</td>";
            echo "<td style='padding: 8px;'>" . htmlspecialchars($row['course_passing_grade']) . "%</td>";
            echo "<td style='padding: 8px; {$status_color}'>{$status_text}</td>";
            echo "</tr>";
        }
        echo "</table>";
    }
    echo "</div>";
    
    // Summary
    echo "<div style='background: #e8f5e8; border: 1px solid #28a745; padding: 15px; border-radius: 5px; margin: 10px 0;'>";
    echo "<h3>🏆 Quiz-Only System Summary</h3>";
    echo "<ul>";
    echo "<li><strong>✅ Database:</strong> All courses now have 0% assignment, 100% quiz weights</li>";
    echo "<li><strong>✅ UI:</strong> Create course form defaults to quiz-only</li>";
    echo "<li><strong>✅ Logic:</strong> Grade calculation ignores assignments completely</li>";
    echo "<li><strong>✅ Certificates:</strong> Based purely on quiz performance vs passing grade</li>";
    echo "</ul>";
    echo "<p><strong>🎯 Result:</strong> Complete quiz-only certificate system!</p>";
    echo "</div>";
    
} catch (Exception $e) {
    echo "<div style='background: #f8d7da; border: 1px solid #f5c6cb; padding: 15px; border-radius: 5px; margin: 10px 0;'>";
    echo "<h3>❌ Error</h3>";
    echo "<p>An error occurred: " . htmlspecialchars($e->getMessage()) . "</p>";
    echo "</div>";
}

echo "</div>";
?>
