<?php
require_once __DIR__ . '/../auth/functions.php';

// Ensure user is logged in as an admin
requireRole('admin');

// Get stats data from database
$statsData = [
    'users' => 0,
    'students' => 0,
    'teachers' => 0,
    'courses' => 0,
    'videos' => 0,
    'revenue' => 'R0.00'
];

// Get users count
$usersQuery = "SELECT 
    COUNT(*) as total_users,
    SUM(CASE WHEN user_role = 'student' THEN 1 ELSE 0 END) as students,
    SUM(CASE WHEN user_role = 'teacher' THEN 1 ELSE 0 END) as teachers
    FROM users";
$usersResult = $conn->query($usersQuery);
if ($usersResult && $row = $usersResult->fetch_assoc()) {
    $statsData['users'] = $row['total_users'];
    $statsData['students'] = $row['students'];
    $statsData['teachers'] = $row['teachers'];
}

// Get courses count
$coursesQuery = "SELECT COUNT(*) as total_courses FROM courses";
$coursesResult = $conn->query($coursesQuery);
if ($coursesResult && $row = $coursesResult->fetch_assoc()) {
    $statsData['courses'] = $row['total_courses'];
}

// Get videos count
$videosQuery = "SELECT COUNT(*) as total_videos FROM videos";
$videosResult = $conn->query($videosQuery);
if ($videosResult && $row = $videosResult->fetch_assoc()) {
    $statsData['videos'] = $row['total_videos'];
}

// Get revenue info from video_purchases, course_purchases, and referral commissions
$revenueQuery = "SELECT COALESCE(SUM(amount), 0) as total_revenue FROM (
    SELECT amount FROM video_purchases WHERE payment_status = 'completed'
    UNION ALL
    SELECT amount FROM course_purchases WHERE payment_status = 'completed'
    UNION ALL
    SELECT commission_amount as amount FROM teacher_referral_commissions WHERE status IN ('pending', 'approved', 'paid')
) AS all_payments";
$revenueResult = $conn->query($revenueQuery);
if ($revenueResult && $row = $revenueResult->fetch_assoc()) {
    $statsData['revenue'] = 'R' . number_format($row['total_revenue'], 2);
}

// Get recent users grouped by role
$recentUsers = [
    'admin' => [],
    'teacher' => [],
    'student' => []
];

// Get more details for better user display
$recentUsersQuery = "SELECT user_id, username, email, user_role, first_name, last_name, created_at, profile_picture FROM users ORDER BY created_at DESC LIMIT 15";
$recentUsersResult = $conn->query($recentUsersQuery);
if ($recentUsersResult) {
    while ($row = $recentUsersResult->fetch_assoc()) {
        $role = $row['user_role'];
        // Add user to appropriate role array (limit to 5 per role)
        if (!isset($recentUsers[$role])) {
            $recentUsers[$role] = [];
        }
        if (count($recentUsers[$role]) < 5) {
            $recentUsers[$role][] = $row;
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard - Panadite Academy</title>
    
    <!-- Favicon -->
    <link rel="shortcut icon" href="/assets/images/favicon.ico" type="image/x-icon">
    <link rel="icon" href="/assets/images/favicon.ico" type="image/x-icon">
    
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link rel="stylesheet" href="/assets/css/user-cards.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    
    <style>
        /* Page Content Management Internal Styling */
        .page-management-section {
            margin-bottom: 30px !important;
        }

        .page-edit-card {
            background: #fff !important;
            border-radius: 12px !important;
            padding: 25px !important;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05) !important;
            border: 1px solid #f1f1f1 !important;
            transition: all 0.3s ease !important;
            height: 100% !important;
            display: flex !important;
            flex-direction: column !important;
        }

        .page-edit-card:hover {
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1) !important;
            transform: translateY(-2px) !important;
        }

        .page-edit-header {
            display: flex !important;
            align-items: center !important;
            margin-bottom: 15px !important;
        }

        .page-edit-header h6 {
            margin: 0 !important;
            font-size: 16px !important;
            font-weight: 600 !important;
            color: #212529 !important;
        }

        .page-icon {
            width: 45px !important;
            height: 45px !important;
            border-radius: 10px !important;
            display: flex !important;
            align-items: center !important;
            justify-content: center !important;
            font-size: 20px !important;
            margin-right: 12px !important;
            color: #fff !important;
        }

        .home-icon {
            background: linear-gradient(135deg, #2563eb, #1d4ed8) !important;
        }

        .about-icon {
            background: linear-gradient(135deg, #059669, #047857) !important;
        }

        .legal-icon {
            background: linear-gradient(135deg, #dc2626, #b91c1c) !important;
        }

        .page-description {
            color: #adb5bd !important;
            font-size: 14px !important;
            line-height: 1.5 !important;
            margin-bottom: 20px !important;
            flex-grow: 1 !important;
        }

        .btn-edit-page {
            background: linear-gradient(135deg, #4361ee, #3f37c9) !important;
            color: #fff !important;
            border: none !important;
            border-radius: 8px !important;
            padding: 12px 20px !important;
            font-size: 14px !important;
            font-weight: 500 !important;
            text-decoration: none !important;
            transition: all 0.3s ease !important;
            display: inline-flex !important;
            align-items: center !important;
            justify-content: center !important;
            width: 100% !important;
        }

        .btn-edit-page:hover {
            background: linear-gradient(135deg, #3f37c9, #4361ee) !important;
            color: #fff !important;
            transform: translateY(-1px) !important;
            box-shadow: 0 4px 12px rgba(67, 97, 238, 0.3) !important;
            text-decoration: none !important;
        }

        .btn-edit-page.btn-sm {
            padding: 10px 16px !important;
            font-size: 13px !important;
        }

        .btn-group .btn-edit-page {
            border-radius: 0 !important;
        }

        .btn-group .btn-edit-page:first-child {
            border-top-left-radius: 8px !important;
            border-bottom-left-radius: 8px !important;
        }

        .btn-group .btn-edit-page:last-child {
            border-top-right-radius: 8px !important;
            border-bottom-right-radius: 8px !important;
        }

        .page-management-section .section-header {
            border-bottom: 2px solid #f1f1f1 !important;
            padding-bottom: 15px !important;
            margin-bottom: 25px !important;
        }

        .page-management-section .section-header h5 {
            color: #212529 !important;
            font-weight: 600 !important;
            display: flex !important;
            align-items: center !important;
        }

        .page-management-section .section-header h5 i {
            color: #4361ee !important;
            margin-right: 8px !important;
        }

        .page-management-section .section-header small {
            font-size: 13px !important;
            color: #adb5bd !important;
        }

        /* Responsive adjustments */
        @media (max-width: 768px) {
            .page-edit-card {
                padding: 20px !important;
            }
            
            .page-icon {
                width: 40px !important;
                height: 40px !important;
                font-size: 18px !important;
            }
            
            .btn-edit-page {
                padding: 10px 16px !important;
                font-size: 13px !important;
            }
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <!-- Page Header -->
            <div class="page-header">
                <h1>Dashboard Overview</h1>
                <div>
                    <a href="reports.php" class="btn btn-sm btn-outline-primary">
                        <i class='bx bx-bar-chart-alt'></i> Reports
                    </a>
                </div>
            </div>
            
            <!-- Stats Cards -->
            <div class="stats-cards">
                <div class="stat-card users animate-fade-in">
                    <div class="stat-card-header">
                        <div>
                            <p class="stat-label">Total Users</p>
                            <h2 class="stat-value"><?= $statsData['users'] ?></h2>
                            <p class="text-muted"><?= $statsData['students'] ?> students, <?= $statsData['teachers'] ?> teachers</p>
                        </div>
                        <div class="stat-card-icon">
                            <i class='bx bxs-user-detail'></i>
                        </div>
                    </div>
                    <a href="users.php" class="stat-link">View All Users <i class='bx bx-right-arrow-alt'></i></a>
                </div>
                
                <div class="stat-card courses animate-fade-in">
                    <div class="stat-card-header">
                        <div>
                            <p class="stat-label">Total Courses</p>
                            <h2 class="stat-value"><?= $statsData['courses'] ?></h2>
                            <p class="text-muted">Educational resources</p>
                        </div>
                        <div class="stat-card-icon">
                            <i class='bx bxs-graduation'></i>
                        </div>
                    </div>
                    <a href="courses.php" class="stat-link">Manage Courses <i class='bx bx-right-arrow-alt'></i></a>
                </div>
                
                <div class="stat-card videos animate-fade-in">
                    <div class="stat-card-header">
                        <div>
                            <p class="stat-label">Total Videos</p>
                            <h2 class="stat-value"><?= $statsData['videos'] ?></h2>
                            <p class="text-muted">Available for purchase</p>
                        </div>
                        <div class="stat-card-icon">
                            <i class='bx bxs-video'></i>
                        </div>
                    </div>
                    <a href="videos.php" class="stat-link">View All Videos <i class='bx bx-right-arrow-alt'></i></a>
                </div>
                
                <div class="stat-card revenue animate-fade-in">
                    <div class="stat-card-header">
                        <div>
                            <p class="stat-label">Total Revenue</p>
                            <h2 class="stat-value"><?= $statsData['revenue'] ?></h2>
                            <p class="text-muted">From all purchases</p>
                        </div>
                        <div class="stat-card-icon">
                            <i class='bx bx-money'></i>
                        </div>
                    </div>
                    <a href="payments.php" class="stat-link">View Transactions <i class='bx bx-right-arrow-alt'></i></a>
                </div>
            </div>
            
            <!-- Page Content Management Section -->
            <div class="page-management-section animate-fade-in" style="animation-delay: 0.3s;">
                <div class="section-header d-flex justify-content-between align-items-center mb-4">
                    <h5 class="mb-0"><i class='bx bx-edit me-2'></i>Page Content Management</h5>
                    <small class="text-muted">Edit content on main landing pages</small>
                </div>
                
                <div class="row g-3">
                    <div class="col-md-4">
                        <div class="page-edit-card">
                            <div class="page-edit-header">
                                <i class='bx bx-home-alt page-icon home-icon'></i>
                                <h6>Homepage</h6>
                            </div>
                            <p class="page-description">Edit hero section, featured courses, and main content</p>
                            <a href="../index.php" target="_blank" class="btn btn-edit-page">
                                <i class='bx bx-edit-alt me-1'></i>Edit Home Page
                            </a>
                        </div>
                    </div>
                    
                    <div class="col-md-4">
                        <div class="page-edit-card">
                            <div class="page-edit-header">
                                <i class='bx bx-info-circle page-icon about-icon'></i>
                                <h6>About Us</h6>
                            </div>
                            <p class="page-description">Edit company information and team details</p>
                            <a href="../about.php" target="_blank" class="btn btn-edit-page">
                                <i class='bx bx-edit-alt me-1'></i>Edit About Page
                            </a>
                        </div>
                    </div>
                    
                    <div class="col-md-4">
                        <div class="page-edit-card">
                            <div class="page-edit-header">
                                <i class='bx bx-shield-check page-icon legal-icon'></i>
                                <h6>Legal Pages</h6>
                            </div>
                            <p class="page-description">Edit terms of service and privacy policy</p>
                            <div class="btn-group w-100" role="group">
                                <a href="../terms-of-use.php" target="_blank" class="btn btn-edit-page btn-sm">
                                    <i class='bx bx-file-blank me-1'></i>Terms
                                </a>
                                <a href="../privacy-policy.php" target="_blank" class="btn btn-edit-page btn-sm">
                                    <i class='bx bx-lock-alt me-1'></i>Privacy
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Recent Users Section with Role Grouping -->
            <div class="users-section animate-fade-in" style="animation-delay: 0.5s;">
                <div class="section-header d-flex justify-content-between align-items-center mb-4">
                    <h5 class="mb-0">Recent Users</h5>
                    <a href="users.php" class="btn btn-sm btn-primary">View All Users</a>
                </div>
                
                <!-- Role tabs navigation -->
                <ul class="nav nav-tabs mb-4" id="userRoleTabs" role="tablist">
                    <li class="nav-item" role="presentation">
                        <button class="nav-link active" id="all-tab" data-bs-toggle="tab" data-bs-target="#all-users" type="button">
                            <i class='bx bx-group me-1'></i> All Users
                            <span class="badge rounded-pill bg-secondary ms-1"><?= count($recentUsers['admin']) + count($recentUsers['teacher']) + count($recentUsers['student']) ?></span>
                        </button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="admin-tab" data-bs-toggle="tab" data-bs-target="#admin-users" type="button">
                            <i class='bx bx-shield-quarter me-1'></i> Admins
                            <span class="badge rounded-pill bg-danger ms-1"><?= count($recentUsers['admin']) ?></span>
                        </button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="teacher-tab" data-bs-toggle="tab" data-bs-target="#teacher-users" type="button">
                            <i class='bx bx-chalkboard me-1'></i> Teachers
                            <span class="badge rounded-pill bg-success ms-1"><?= count($recentUsers['teacher']) ?></span>
                        </button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="student-tab" data-bs-toggle="tab" data-bs-target="#student-users" type="button">
                            <i class='bx bx-book-reader me-1'></i> Students
                            <span class="badge rounded-pill bg-primary ms-1"><?= count($recentUsers['student']) ?></span>
                        </button>
                    </li>
                </ul>
                
                <!-- Tab content -->
                <div class="tab-content" id="userRoleTabContent">
                    <!-- All Users Tab -->
                    <div class="tab-pane fade show active" id="all-users" role="tabpanel">
                        <div class="row">
                            <!-- Admins -->
                            <?php if (!empty($recentUsers['admin'])): ?>
                                <?php foreach($recentUsers['admin'] as $admin): ?>
                                    <div class="col-md-6 col-lg-4 mb-4">
                                        <div class="card user-card admin-card">
                                            <div class="card-body p-0">
                                                <div class="user-card-header d-flex align-items-center p-3">
                                                    <div class="user-avatar admin-avatar">
                                                        <?php if (!empty($admin['profile_picture'])): ?>
                                                            <img src="../<?= htmlspecialchars($admin['profile_picture']) ?>" alt="Profile" class="img-fluid rounded-circle">
                                                        <?php else: ?>
                                                            <div class="avatar-text">
                                                                <?= strtoupper(substr($admin['first_name'] ?? $admin['username'], 0, 1)) ?>
                                                            </div>
                                                        <?php endif; ?>
                                                    </div>
                                                    <div class="user-info ms-3">
                                                        <h6 class="mb-0"><?= htmlspecialchars($admin['first_name'] . ' ' . $admin['last_name']) ?></h6>
                                                        <p class="text-muted mb-0 small"><?= htmlspecialchars($admin['username']) ?></p>
                                                        <span class="badge bg-danger">Administrator</span>
                                                    </div>
                                                </div>
                                                <div class="user-card-body p-3 border-top">
                                                    <div class="d-flex justify-content-between align-items-center mb-2">
                                                        <div class="user-email small"><i class='bx bx-envelope me-1'></i> <?= htmlspecialchars($admin['email']) ?></div>
                                                    </div>
                                                    <div class="d-flex justify-content-between align-items-center">
                                                        <div class="user-joined small text-muted">
                                                            <i class='bx bx-calendar me-1'></i> Joined <?= date('M d, Y', strtotime($admin['created_at'])) ?>
                                                        </div>
                                                        <div class="user-actions">
                                                            <a href="user_profile.php?id=<?= $admin['user_id'] ?>" class="btn btn-sm btn-outline-primary me-1"><i class='bx bx-show'></i></a>
                                                            <a href="edit_user.php?id=<?= $admin['user_id'] ?>" class="btn btn-sm btn-outline-info"><i class='bx bx-edit'></i></a>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            <?php endif; ?>
                            
                            <!-- Teachers -->
                            <?php if (!empty($recentUsers['teacher'])): ?>
                                <?php foreach($recentUsers['teacher'] as $teacher): ?>
                                    <div class="col-md-6 col-lg-4 mb-4">
                                        <div class="card user-card teacher-card">
                                            <div class="card-body p-0">
                                                <div class="user-card-header d-flex align-items-center p-3">
                                                    <div class="user-avatar teacher-avatar">
                                                        <?php if (!empty($teacher['profile_picture'])): ?>
                                                            <img src="../<?= htmlspecialchars($teacher['profile_picture']) ?>" alt="Profile" class="img-fluid rounded-circle">
                                                        <?php else: ?>
                                                            <div class="avatar-text">
                                                                <?= strtoupper(substr($teacher['first_name'] ?? $teacher['username'], 0, 1)) ?>
                                                            </div>
                                                        <?php endif; ?>
                                                    </div>
                                                    <div class="user-info ms-3">
                                                        <h6 class="mb-0"><?= htmlspecialchars($teacher['first_name'] . ' ' . $teacher['last_name']) ?></h6>
                                                        <p class="text-muted mb-0 small"><?= htmlspecialchars($teacher['username']) ?></p>
                                                        <span class="badge bg-success">Teacher</span>
                                                    </div>
                                                </div>
                                                <div class="user-card-body p-3 border-top">
                                                    <div class="d-flex justify-content-between align-items-center mb-2">
                                                        <div class="user-email small"><i class='bx bx-envelope me-1'></i> <?= htmlspecialchars($teacher['email']) ?></div>
                                                    </div>
                                                    <div class="d-flex justify-content-between align-items-center">
                                                        <div class="user-joined small text-muted">
                                                            <i class='bx bx-calendar me-1'></i> Joined <?= date('M d, Y', strtotime($teacher['created_at'])) ?>
                                                        </div>
                                                        <div class="user-actions">
                                                            <a href="teacher_content.php?id=<?= $teacher['user_id'] ?>" class="btn btn-sm btn-outline-success me-1" title="View Teacher Content"><i class='bx bx-video'></i></a>
                                                            <a href="user_profile.php?id=<?= $teacher['user_id'] ?>" class="btn btn-sm btn-outline-primary me-1"><i class='bx bx-show'></i></a>
                                                            <a href="edit_user.php?id=<?= $teacher['user_id'] ?>" class="btn btn-sm btn-outline-info"><i class='bx bx-edit'></i></a>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            <?php endif; ?>
                            
                            <!-- Students -->
                            <?php if (!empty($recentUsers['student'])): ?>
                                <?php foreach($recentUsers['student'] as $student): ?>
                                    <div class="col-md-6 col-lg-4 mb-4">
                                        <div class="card user-card student-card">
                                            <div class="card-body p-0">
                                                <div class="user-card-header d-flex align-items-center p-3">
                                                    <div class="user-avatar student-avatar">
                                                        <?php if (!empty($student['profile_picture'])): ?>
                                                            <img src="../<?= htmlspecialchars($student['profile_picture']) ?>" alt="Profile" class="img-fluid rounded-circle">
                                                        <?php else: ?>
                                                            <div class="avatar-text">
                                                                <?= strtoupper(substr($student['first_name'] ?? $student['username'], 0, 1)) ?>
                                                            </div>
                                                        <?php endif; ?>
                                                    </div>
                                                    <div class="user-info ms-3">
                                                        <h6 class="mb-0"><?= htmlspecialchars($student['first_name'] . ' ' . $student['last_name']) ?></h6>
                                                        <p class="text-muted mb-0 small"><?= htmlspecialchars($student['username']) ?></p>
                                                        <span class="badge bg-primary">Student</span>
                                                    </div>
                                                </div>
                                                <div class="user-card-body p-3 border-top">
                                                    <div class="d-flex justify-content-between align-items-center mb-2">
                                                        <div class="user-email small"><i class='bx bx-envelope me-1'></i> <?= htmlspecialchars($student['email']) ?></div>
                                                    </div>
                                                    <div class="d-flex justify-content-between align-items-center">
                                                        <div class="user-joined small text-muted">
                                                            <i class='bx bx-calendar me-1'></i> Joined <?= date('M d, Y', strtotime($student['created_at'])) ?>
                                                        </div>
                                                        <div class="user-actions">
                                                            <a href="student_courses.php?id=<?= $student['user_id'] ?>" class="btn btn-sm btn-outline-primary me-1" title="View Enrollments"><i class='bx bx-book'></i></a>
                                                            <a href="user_profile.php?id=<?= $student['user_id'] ?>" class="btn btn-sm btn-outline-primary me-1"><i class='bx bx-show'></i></a>
                                                            <a href="edit_user.php?id=<?= $student['user_id'] ?>" class="btn btn-sm btn-outline-info"><i class='bx bx-edit'></i></a>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            <?php endif; ?>
                            
                            <?php if (empty($recentUsers['admin']) && empty($recentUsers['teacher']) && empty($recentUsers['student'])): ?>
                                <div class="col-12">
                                    <div class="alert alert-info text-center">
                                        <i class='bx bx-info-circle me-2'></i> No users found
                                    </div>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <!-- Admins Tab -->
                    <div class="tab-pane fade" id="admin-users" role="tabpanel">
                        <div class="row">
                            <?php if (!empty($recentUsers['admin'])): ?>
                                <?php foreach($recentUsers['admin'] as $admin): ?>
                                    <div class="col-md-6 col-lg-4 mb-4">
                                        <div class="card user-card admin-card">
                                            <div class="card-body p-0">
                                                <div class="user-card-header d-flex align-items-center p-3">
                                                    <div class="user-avatar admin-avatar">
                                                        <?php if (!empty($admin['profile_picture'])): ?>
                                                            <img src="../<?= htmlspecialchars($admin['profile_picture']) ?>" alt="Profile" class="img-fluid rounded-circle">
                                                        <?php else: ?>
                                                            <div class="avatar-text">
                                                                <?= strtoupper(substr($admin['first_name'] ?? $admin['username'], 0, 1)) ?>
                                                            </div>
                                                        <?php endif; ?>
                                                    </div>
                                                    <div class="user-info ms-3">
                                                        <h6 class="mb-0"><?= htmlspecialchars($admin['first_name'] . ' ' . $admin['last_name']) ?></h6>
                                                        <p class="text-muted mb-0 small"><?= htmlspecialchars($admin['username']) ?></p>
                                                        <span class="badge bg-danger">Administrator</span>
                                                    </div>
                                                </div>
                                                <div class="user-card-body p-3 border-top">
                                                    <div class="d-flex justify-content-between align-items-center mb-2">
                                                        <div class="user-email small"><i class='bx bx-envelope me-1'></i> <?= htmlspecialchars($admin['email']) ?></div>
                                                    </div>
                                                    <div class="d-flex justify-content-between align-items-center">
                                                        <div class="user-joined small text-muted">
                                                            <i class='bx bx-calendar me-1'></i> Joined <?= date('M d, Y', strtotime($admin['created_at'])) ?>
                                                        </div>
                                                        <div class="user-actions">
                                                            <a href="user_profile.php?id=<?= $admin['user_id'] ?>" class="btn btn-sm btn-outline-primary me-1"><i class='bx bx-show'></i></a>
                                                            <a href="edit_user.php?id=<?= $admin['user_id'] ?>" class="btn btn-sm btn-outline-info"><i class='bx bx-edit'></i></a>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <div class="col-12">
                                    <div class="alert alert-info text-center">
                                        <i class='bx bx-info-circle me-2'></i> No admin users found
                                    </div>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <!-- Teachers Tab -->
                    <div class="tab-pane fade" id="teacher-users" role="tabpanel">
                        <div class="row">
                            <?php if (!empty($recentUsers['teacher'])): ?>
                                <?php foreach($recentUsers['teacher'] as $teacher): ?>
                                    <div class="col-md-6 col-lg-4 mb-4">
                                        <div class="card user-card teacher-card">
                                            <div class="card-body p-0">
                                                <div class="user-card-header d-flex align-items-center p-3">
                                                    <div class="user-avatar teacher-avatar">
                                                        <?php if (!empty($teacher['profile_picture'])): ?>
                                                            <img src="../<?= htmlspecialchars($teacher['profile_picture']) ?>" alt="Profile" class="img-fluid rounded-circle">
                                                        <?php else: ?>
                                                            <div class="avatar-text">
                                                                <?= strtoupper(substr($teacher['first_name'] ?? $teacher['username'], 0, 1)) ?>
                                                            </div>
                                                        <?php endif; ?>
                                                    </div>
                                                    <div class="user-info ms-3">
                                                        <h6 class="mb-0"><?= htmlspecialchars($teacher['first_name'] . ' ' . $teacher['last_name']) ?></h6>
                                                        <p class="text-muted mb-0 small"><?= htmlspecialchars($teacher['username']) ?></p>
                                                        <span class="badge bg-success">Teacher</span>
                                                    </div>
                                                </div>
                                                <div class="user-card-body p-3 border-top">
                                                    <div class="d-flex justify-content-between align-items-center mb-2">
                                                        <div class="user-email small"><i class='bx bx-envelope me-1'></i> <?= htmlspecialchars($teacher['email']) ?></div>
                                                    </div>
                                                    <div class="d-flex justify-content-between align-items-center">
                                                        <div class="user-joined small text-muted">
                                                            <i class='bx bx-calendar me-1'></i> Joined <?= date('M d, Y', strtotime($teacher['created_at'])) ?>
                                                        </div>
                                                        <div class="user-actions">
                                                            <a href="teacher_content.php?id=<?= $teacher['user_id'] ?>" class="btn btn-sm btn-outline-success me-1" title="View Teacher Content"><i class='bx bx-video'></i></a>
                                                            <a href="user_profile.php?id=<?= $teacher['user_id'] ?>" class="btn btn-sm btn-outline-primary me-1"><i class='bx bx-show'></i></a>
                                                            <a href="edit_user.php?id=<?= $teacher['user_id'] ?>" class="btn btn-sm btn-outline-info"><i class='bx bx-edit'></i></a>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <div class="col-12">
                                    <div class="alert alert-info text-center">
                                        <i class='bx bx-info-circle me-2'></i> No teacher users found
                                    </div>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <!-- Students Tab -->
                    <div class="tab-pane fade" id="student-users" role="tabpanel">
                        <div class="row">
                            <?php if (!empty($recentUsers['student'])): ?>
                                <?php foreach($recentUsers['student'] as $student): ?>
                                    <div class="col-md-6 col-lg-4 mb-4">
                                        <div class="card user-card student-card">
                                            <div class="card-body p-0">
                                                <div class="user-card-header d-flex align-items-center p-3">
                                                    <div class="user-avatar student-avatar">
                                                        <?php if (!empty($student['profile_picture'])): ?>
                                                            <img src="../<?= htmlspecialchars($student['profile_picture']) ?>" alt="Profile" class="img-fluid rounded-circle">
                                                        <?php else: ?>
                                                            <div class="avatar-text">
                                                                <?= strtoupper(substr($student['first_name'] ?? $student['username'], 0, 1)) ?>
                                                            </div>
                                                        <?php endif; ?>
                                                    </div>
                                                    <div class="user-info ms-3">
                                                        <h6 class="mb-0"><?= htmlspecialchars($student['first_name'] . ' ' . $student['last_name']) ?></h6>
                                                        <p class="text-muted mb-0 small"><?= htmlspecialchars($student['username']) ?></p>
                                                        <span class="badge bg-primary">Student</span>
                                                    </div>
                                                </div>
                                                <div class="user-card-body p-3 border-top">
                                                    <div class="d-flex justify-content-between align-items-center mb-2">
                                                        <div class="user-email small"><i class='bx bx-envelope me-1'></i> <?= htmlspecialchars($student['email']) ?></div>
                                                    </div>
                                                    <div class="d-flex justify-content-between align-items-center">
                                                        <div class="user-joined small text-muted">
                                                            <i class='bx bx-calendar me-1'></i> Joined <?= date('M d, Y', strtotime($student['created_at'])) ?>
                                                        </div>
                                                        <div class="user-actions">
                                                            <a href="student_courses.php?id=<?= $student['user_id'] ?>" class="btn btn-sm btn-outline-primary me-1" title="View Enrollments"><i class='bx bx-book'></i></a>
                                                            <a href="user_profile.php?id=<?= $student['user_id'] ?>" class="btn btn-sm btn-outline-primary me-1"><i class='bx bx-show'></i></a>
                                                            <a href="edit_user.php?id=<?= $student['user_id'] ?>" class="btn btn-sm btn-outline-info"><i class='bx bx-edit'></i></a>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <div class="col-12">
                                    <div class="alert alert-info text-center">
                                        <i class='bx bx-info-circle me-2'></i> No student users found
                                    </div>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
            <!-- Action Cards -->
            <div class="action-cards">
                <div class="action-card animate-fade-in" style="animation-delay: 0.6s;">
                    <div class="card-header">
                        <h5>Quick Actions</h5>
                    </div>
                    <div class="card-body">
                        <div class="list-group">
                            <a href="add_category.php" class="list-group-item">
                                <i class='bx bx-plus-circle text-primary'></i> Add New Category
                            </a>
                            <a href="payfast.php" class="list-group-item">
                                <i class='bx bx-credit-card text-success'></i> Configure PayFast Settings
                            </a>
                            <a href="reports.php" class="list-group-item">
                                <i class='bx bx-chart text-info'></i> View Sales Reports
                            </a>
                            <a href="backup.php" class="list-group-item">
                                <i class='bx bx-data text-warning'></i> Backup Database
                            </a>
                        </div>
                    </div>
                </div>
                
                <div class="action-card animate-fade-in" style="animation-delay: 0.7s;">
                    <div class="card-header">
                        <h5>System Information</h5>
                    </div>
                    <div class="card-body">
                        <div class="list-group">
                            <div class="list-group-item d-flex justify-content-between align-items-center">
                                PHP Version
                                <span class="badge bg-light text-dark"><?php echo phpversion(); ?></span>
                            </div>
                            <div class="list-group-item d-flex justify-content-between align-items-center">
                                MySQL Version
                                <span class="badge bg-light text-dark"><?php 
                                    $result = $conn->query('SELECT VERSION() as version');
                                    $row = $result->fetch_assoc();
                                    echo $row['version'] ?? 'Unknown';
                                ?></span>
                            </div>
                            <div class="list-group-item d-flex justify-content-between align-items-center">
                                Server Software
                                <span class="badge bg-light text-dark"><?php echo $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown'; ?></span>
                            </div>
                            <div class="list-group-item d-flex justify-content-between align-items-center">
                                Current Date/Time
                                <span class="badge bg-light text-dark"><?php echo date('Y-m-d H:i:s'); ?></span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        // Sidebar toggle functionality
        const sidebarToggle = document.querySelector('.sidebar-toggle');
        const sidebarToggleMobile = document.querySelector('.sidebar-toggle-mobile');
        const sidebar = document.querySelector('.sidebar');
        
        if (sidebarToggle) {
            sidebarToggle.addEventListener('click', () => {
                sidebar.classList.toggle('close');
            });
        }
        
        if (sidebarToggleMobile) {
            sidebarToggleMobile.addEventListener('click', () => {
                sidebar.classList.toggle('show');
            });
        }
        
        // Responsive sidebar behavior
        function checkScreenSize() {
            if (window.innerWidth < 768) {
                sidebar.classList.remove('close');
                sidebar.classList.remove('show');
            } else if (window.innerWidth < 992) {
                sidebar.classList.add('close');
                sidebar.classList.remove('show');
            }
        }
        
        // Initial check and listen for window resize
        checkScreenSize();
        window.addEventListener('resize', checkScreenSize);
        
        // Dropdown functionality (if not using Bootstrap's dropdown)
        document.querySelectorAll('.dropdown-toggle').forEach(item => {
            item.addEventListener('click', event => {
                event.preventDefault();
                const dropdownMenu = item.nextElementSibling;
                if (dropdownMenu && dropdownMenu.classList.contains('dropdown-menu')) {
                    dropdownMenu.classList.toggle('show');
                }
            });
        });
        
        // Close dropdowns when clicking outside
        document.addEventListener('click', event => {
            const isDropdownButton = event.target.classList.contains('dropdown-toggle');
            if (!isDropdownButton) {
                document.querySelectorAll('.dropdown-menu.show').forEach(dropdown => {
                    dropdown.classList.remove('show');
                });
            }
        });
    });
    </script>
</body>
</html>
