<?php
require_once __DIR__ . '/../auth/functions.php';

// Ensure user is logged in as an admin
requireRole('admin');

// Enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Purchase Flow Debug - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        .debug-container { padding: 30px; background: #f8fafc; min-height: 100vh; }
        .debug-card { background: white; border-radius: 12px; padding: 24px; margin-bottom: 24px; box-shadow: 0 4px 12px rgba(0,0,0,0.1); }
        .debug-title { color: #1f2937; font-weight: 600; margin-bottom: 20px; display: flex; align-items: center; gap: 12px; }
        .debug-title i { color: #4361ee; font-size: 24px; }
        .table-container { overflow-x: auto; max-height: 400px; border: 1px solid #e5e7eb; border-radius: 8px; }
        .table { margin: 0; font-size: 14px; }
        .table th { background: #f8fafc; color: #374151; font-weight: 600; position: sticky; top: 0; }
        .table td { vertical-align: middle; }
        .badge { font-size: 12px; }
        .info-box { background: #eff6ff; border: 1px solid #3b82f6; border-radius: 8px; padding: 16px; margin-bottom: 20px; }
        .success-box { background: #f0fdf4; border: 1px solid #22c55e; border-radius: 8px; padding: 16px; margin-bottom: 20px; }
        .warning-box { background: #fef3c7; border: 1px solid #f59e0b; border-radius: 8px; padding: 16px; margin-bottom: 20px; }
        .error-box { background: #fef2f2; border: 1px solid #ef4444; border-radius: 8px; padding: 16px; margin-bottom: 20px; }
        .stat-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 16px; margin-bottom: 24px; }
        .stat-card { background: white; border-radius: 8px; padding: 20px; text-align: center; border: 1px solid #e5e7eb; }
        .stat-value { font-size: 28px; font-weight: 700; color: #4361ee; }
        .stat-label { font-size: 14px; color: #6b7280; margin-top: 4px; }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>

        <div class="debug-container">
            <div class="debug-card">
                <h1 class="debug-title">
                    <i class='bx bx-bug'></i>
                    Purchase Flow Debug Analysis
                </h1>
                <p style="color: #6b7280; margin-bottom: 0;">
                    This debug tool analyzes the current purchase flow to understand how course purchases are recorded.
                    We'll examine enrollments, payments, and identify where to integrate revenue sharing.
                </p>
            </div>

            <?php
            // 1. Check what tables exist related to purchases/enrollments
            echo "<div class='debug-card'>";
            echo "<h3 class='debug-title'><i class='bx bx-data'></i>Database Tables Analysis</h3>";
            
            $purchase_related_tables = [];
            $all_tables = $conn->query("SHOW TABLES");
            if ($all_tables) {
                while ($table = $all_tables->fetch_array()) {
                    $table_name = $table[0];
                    if (stripos($table_name, 'enroll') !== false || 
                        stripos($table_name, 'purchase') !== false || 
                        stripos($table_name, 'payment') !== false ||
                        stripos($table_name, 'order') !== false) {
                        $purchase_related_tables[] = $table_name;
                    }
                }
            }
            
            if (!empty($purchase_related_tables)) {
                echo "<div class='success-box'><strong>✅ Found purchase-related tables:</strong><br>";
                foreach ($purchase_related_tables as $table) {
                    echo "<span class='badge bg-primary me-2'>$table</span>";
                }
                echo "</div>";
            } else {
                echo "<div class='warning-box'><strong>⚠️ No obvious purchase-related tables found.</strong></div>";
            }
            echo "</div>";

            // 2. Analyze enrollments table structure
            if (in_array('enrollments', $purchase_related_tables) || $conn->query("SHOW TABLES LIKE 'enrollments'")->num_rows > 0) {
                echo "<div class='debug-card'>";
                echo "<h3 class='debug-title'><i class='bx bx-table'></i>Enrollments Table Structure</h3>";
                
                $columns = $conn->query("DESCRIBE enrollments");
                if ($columns) {
                    echo "<div class='table-container'>";
                    echo "<table class='table table-striped'>";
                    echo "<thead><tr><th>Column</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th><th>Extra</th></tr></thead><tbody>";
                    while ($col = $columns->fetch_assoc()) {
                        echo "<tr>";
                        echo "<td><strong>{$col['Field']}</strong></td>";
                        echo "<td>{$col['Type']}</td>";
                        echo "<td>{$col['Null']}</td>";
                        echo "<td>{$col['Key']}</td>";
                        echo "<td>{$col['Default']}</td>";
                        echo "<td>{$col['Extra']}</td>";
                        echo "</tr>";
                    }
                    echo "</tbody></table></div>";
                } else {
                    echo "<div class='error-box'>❌ Could not read enrollments table structure</div>";
                }
                echo "</div>";
            }

            // 3. Get recent enrollments data
            echo "<div class='debug-card'>";
            echo "<h3 class='debug-title'><i class='bx bx-history'></i>Recent Enrollments (Last 10)</h3>";
            
            // Get sample users (fix role field name)
            $stmt = $conn->prepare("SELECT user_id, first_name, last_name, email FROM users WHERE user_role = 'student' LIMIT 10");
            $stmt->execute();
            $users = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            
            // Get sample courses (use simpler query that works)
            $courses_result = $conn->query("
                SELECT c.course_id, c.title, c.price, 
                       CONCAT(u.first_name, ' ', u.last_name) as teacher_name
                FROM courses c
                LEFT JOIN users u ON c.teacher_id = u.user_id 
                LIMIT 10
            ");
            $courses = $courses_result ? $courses_result->fetch_all(MYSQLI_ASSOC) : [];
            
            $recent_enrollments = $conn->query("
                SELECT e.enrollment_id, e.user_id, e.course_id, e.course_price, e.purchase_id, e.enrolled_date, e.progress,
                       u.first_name, u.last_name, u.email,
                       c.title as course_title, c.price as course_original_price
                FROM enrollments e
                LEFT JOIN users u ON e.user_id = u.user_id  
                LEFT JOIN courses c ON e.course_id = c.course_id
                ORDER BY e.enrolled_date DESC 
                LIMIT 10
            ");
            
            if ($recent_enrollments && $recent_enrollments->num_rows > 0) {
                echo "<div class='table-container'>";
                echo "<table class='table table-striped'>";
                echo "<thead><tr><th>ID</th><th>Student</th><th>Course</th><th>Teacher</th><th>Price</th><th>Date</th><th>Purchase ID</th></tr></thead><tbody>";
                while ($enrollment = $recent_enrollments->fetch_assoc()) {
                    echo "<tr>";
                    echo "<td>{$enrollment['enrollment_id']}</td>";
                    echo "<td>{$enrollment['first_name']} {$enrollment['last_name']}<br><small class='text-muted'>{$enrollment['email']}</small></td>";
                    echo "<td>{$enrollment['course_title']}</td>";
                    echo "<td>{$enrollment['teacher_name']}</td>";
                    echo "<td>R" . number_format($enrollment['course_price'] ?? 0, 2) . "</td>";
                    echo "<td>" . date('Y-m-d H:i', strtotime($enrollment['enrollment_date'])) . "</td>";
                    echo "<td>" . ($enrollment['purchase_id'] ?? '<span class="text-muted">N/A</span>') . "</td>";
                    echo "</tr>";
                }
                echo "</tbody></table></div>";
            } else {
                echo "<div class='info-box'><strong>ℹ️ No recent enrollments found.</strong></div>";
            }
            echo "</div>";

            // 4. Check for existing payment/transaction tables
            foreach ($purchase_related_tables as $table) {
                if ($table != 'enrollments') {
                    echo "<div class='debug-card'>";
                    echo "<h3 class='debug-title'><i class='bx bx-credit-card'></i>Table: $table</h3>";
                    
                    // Show structure
                    $structure = $conn->query("DESCRIBE $table");
                    if ($structure) {
                        echo "<h5>Structure:</h5>";
                        echo "<div class='table-container'>";
                        echo "<table class='table table-sm'>";
                        echo "<thead><tr><th>Column</th><th>Type</th><th>Key</th></tr></thead><tbody>";
                        while ($col = $structure->fetch_assoc()) {
                            echo "<tr><td><strong>{$col['Field']}</strong></td><td>{$col['Type']}</td><td>{$col['Key']}</td></tr>";
                        }
                        echo "</tbody></table></div>";
                    }
                    
                    // Show recent data
                    $recent_data = $conn->query("SELECT * FROM $table ORDER BY 1 DESC LIMIT 5");
                    if ($recent_data && $recent_data->num_rows > 0) {
                        echo "<h5>Recent Data (Last 5 records):</h5>";
                        echo "<div class='table-container'>";
                        echo "<table class='table table-sm'>";
                        
                        // Get column names
                        $fields = $recent_data->fetch_fields();
                        echo "<thead><tr>";
                        foreach ($fields as $field) {
                            echo "<th>{$field->name}</th>";
                        }
                        echo "</tr></thead><tbody>";
                        
                        // Show data
                        $recent_data->data_seek(0);
                        while ($row = $recent_data->fetch_assoc()) {
                            echo "<tr>";
                            foreach ($row as $value) {
                                echo "<td>" . htmlspecialchars($value ?? 'NULL') . "</td>";
                            }
                            echo "</tr>";
                        }
                        echo "</tbody></table></div>";
                    } else {
                        echo "<div class='info-box'>ℹ️ No data found in $table table</div>";
                    }
                    echo "</div>";
                }
            }

            // 5. Statistics
            echo "<div class='debug-card'>";
            echo "<h3 class='debug-title'><i class='bx bx-chart-line'></i>Purchase Statistics</h3>";
            
            $stats = [];
            
            // Total enrollments
            $total_enrollments = $conn->query("SELECT COUNT(*) as count FROM enrollments")->fetch_assoc()['count'] ?? 0;
            
            // Enrollments with price info
            $enrollments_with_price = $conn->query("SELECT COUNT(*) as count FROM enrollments WHERE course_price > 0")->fetch_assoc()['count'] ?? 0;
            
            // Enrollments with purchase_id
            $enrollments_with_purchase_id = $conn->query("SELECT COUNT(*) as count FROM enrollments WHERE purchase_id IS NOT NULL")->fetch_assoc()['count'] ?? 0;
            
            // Total revenue from enrollments
            $total_revenue = $conn->query("SELECT SUM(course_price) as total FROM enrollments WHERE course_price > 0")->fetch_assoc()['total'] ?? 0;
            
            echo "<div class='stat-grid'>";
            echo "<div class='stat-card'><div class='stat-value'>$total_enrollments</div><div class='stat-label'>Total Enrollments</div></div>";
            echo "<div class='stat-card'><div class='stat-value'>$enrollments_with_price</div><div class='stat-label'>Paid Enrollments</div></div>";
            echo "<div class='stat-card'><div class='stat-value'>$enrollments_with_purchase_id</div><div class='stat-label'>With Purchase ID</div></div>";
            echo "<div class='stat-card'><div class='stat-value'>R" . number_format($total_revenue, 2) . "</div><div class='stat-label'>Total Revenue</div></div>";
            echo "</div>";
            echo "</div>";

            // 6. Integration recommendations
            echo "<div class='debug-card'>";
            echo "<h3 class='debug-title'><i class='bx bx-lightbulb'></i>Integration Analysis & Recommendations</h3>";
            
            if ($enrollments_with_price > 0) {
                echo "<div class='success-box'>";
                echo "<strong>✅ Good News:</strong> The system is recording course purchases with price information!<br>";
                echo "Found $enrollments_with_price paid enrollments with a total revenue of R" . number_format($total_revenue, 2) . ".";
                echo "</div>";
                
                echo "<div class='info-box'>";
                echo "<strong>🔧 Revenue Sharing Integration Points:</strong><br>";
                echo "1. <strong>Enrollment Creation:</strong> Hook into the enrollment process to calculate revenue splits<br>";
                echo "2. <strong>Payment Processing:</strong> Integrate with the payment success callback<br>";
                echo "3. <strong>Data Structure:</strong> Use existing enrollment data to calculate teacher/admin shares<br>";
                echo "4. <strong>Retroactive Processing:</strong> We can process existing enrollments to populate revenue sharing tables";
                echo "</div>";
            } else {
                echo "<div class='warning-box'>";
                echo "<strong>⚠️ Issue Found:</strong> No paid enrollments detected. This could mean:<br>";
                echo "• Purchases are being recorded elsewhere<br>";
                echo "• Price information is not being captured<br>";
                echo "• The payment integration needs investigation";
                echo "</div>";
            }
            
            if ($enrollments_with_purchase_id == 0) {
                echo "<div class='warning-box'>";
                echo "<strong>⚠️ Missing Purchase IDs:</strong> Enrollments don't have purchase_id references.<br>";
                echo "This suggests the payment tracking system needs to be connected to enrollments.";
                echo "</div>";
            }
            echo "</div>";

            // 7. Next steps
            echo "<div class='debug-card'>";
            echo "<h3 class='debug-title'><i class='bx bx-check-circle'></i>Recommended Next Steps</h3>";
            
            echo "<div class='info-box'>";
            echo "<strong>📋 Action Plan:</strong><br>";
            echo "1. <strong>Identify Payment Success Hook:</strong> Find where successful payments trigger enrollment creation<br>";
            echo "2. <strong>Create Revenue Split Logic:</strong> Add revenue calculation to the payment success process<br>";
            echo "3. <strong>Update Enrollment Process:</strong> Modify enrollment creation to populate revenue sharing tables<br>";
            echo "4. <strong>Process Existing Data:</strong> Run a migration to populate revenue data for existing enrollments<br>";
            echo "5. <strong>Test Integration:</strong> Verify revenue splitting works with new purchases";
            echo "</div>";
            
            echo "<div style='text-align: center; margin-top: 24px;'>";
            echo "<a href='revenue-settings.php' class='btn btn-primary me-2'><i class='bx bx-cog'></i> Configure Revenue Settings</a>";
            echo "<a href='test-revenue-system.php' class='btn btn-outline-primary'><i class='bx bx-test-tube'></i> Test Revenue System</a>";
            echo "</div>";
            echo "</div>";
            ?>
        </div>
    </section>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
