<?php
require_once '../auth/functions.php';
require_once '../config/db_connect.php';
require_once '../config/fast_email_service.php';
require_once 'auto_certificate_generator.php';

// Check if user is admin
requireRole('admin');

$test_results = [];

// Handle progressive email tests
if ($_POST && isset($_POST['test_email'])) {
    $testEmail = filter_var($_POST['test_email'], FILTER_VALIDATE_EMAIL);
    $testType = $_POST['test_type'] ?? '1';
    
    if (!$testEmail) {
        $test_results[] = ['type' => 'error', 'message' => 'Invalid email address'];
    } else {
        
        // Test certificate data
        $certificateNumber = 'TEST-' . date('Y') . '-' . strtoupper(substr(uniqid(), -6));
        $studentName = 'John Doe';
        $courseTitle = 'Computer Science';
        $issueDate = date('F j, Y');
        
        $subject = '';
        $emailBody = '';
        $success = false;
        
        switch($testType) {
            case '1':
                // TEST 1: Exact same as working debug script
                $subject = "TEST 1 - Simple Debug Style - " . date('H:i:s');
                $emailBody = "<h2>🎓 Certificate Email Test - SUCCESS</h2>
                <p><strong>Student:</strong> {$studentName}</p>
                <p><strong>Course:</strong> {$courseTitle}</p>
                <p><strong>Certificate Number:</strong> {$certificateNumber}</p>
                <p><strong>Time sent:</strong> " . date('Y-m-d H:i:s') . "</p>";
                break;
                
            case '2':
                // TEST 2: Add basic certificate styling
                $subject = "TEST 2 - Basic Certificate Style - " . date('H:i:s');
                $emailBody = "
                <div style='font-family: Arial, sans-serif; max-width: 500px; margin: 0 auto; padding: 20px; border: 2px solid #007bff; border-radius: 10px;'>
                    <h1 style='text-align: center; color: #007bff;'>🎓 CERTIFICATE</h1>
                    <p style='text-align: center;'>This certifies that</p>
                    <h2 style='text-align: center; color: #333;'>{$studentName}</h2>
                    <p style='text-align: center;'>completed</p>
                    <h3 style='text-align: center; color: #007bff;'>{$courseTitle}</h3>
                    <p style='text-align: center; margin-top: 30px;'>Certificate: {$certificateNumber}</p>
                    <p style='text-align: center;'>Date: {$issueDate}</p>
                </div>";
                break;
                
            case '3':
                // TEST 3: Medium complexity
                $subject = "TEST 3 - Medium Certificate - " . date('H:i:s');
                $emailBody = "
                <div style='max-width: 600px; margin: 0 auto; background: #ffffff; border: 2px solid #4169E1; padding: 30px; font-family: Arial, sans-serif; text-align: center;'>
                    <div style='margin-bottom: 20px;'>
                        <h2 style='color: #4169E1; margin: 0;'>PANADITE ACADEMY</h2>
                        <p style='color: #666; margin: 5px 0;'>Excellence in Education</p>
                    </div>
                    <h1 style='color: #333; font-size: 28px;'>CERTIFICATE OF COMPLETION</h1>
                    <div style='margin: 30px 0; padding: 20px; background: #f8faff;'>
                        <p>This is to certify that</p>
                        <h2 style='color: #333; text-transform: uppercase;'>{$studentName}</h2>
                        <p>has completed</p>
                        <h3 style='color: #4169E1;'>{$courseTitle}</h3>
                    </div>
                    <p>Certificate: {$certificateNumber} | Date: {$issueDate}</p>
                </div>";
                break;
                
            case '4':
                // TEST 4: Our email-friendly certificate
                $certGenerator = new AutoCertificateGenerator($conn);
                $certificateHtml = $certGenerator->generateEmailFriendlyCertificateHtml($certificateNumber, $studentName, $courseTitle, $issueDate);
                
                $subject = "TEST 4 - Email-Friendly Certificate - " . date('H:i:s');
                $emailBody = "
                <div style='font-family: Arial, sans-serif;'>
                    <h2>🎓 Certificate Test</h2>
                    <p>Testing our email-friendly certificate design:</p>
                    {$certificateHtml}
                    <p>Time sent: " . date('Y-m-d H:i:s') . "</p>
                </div>";
                break;
                
            case '5':
                // TEST 5: Just the certificate HTML alone
                $certGenerator = new AutoCertificateGenerator($conn);
                $subject = "TEST 5 - Certificate HTML Only - " . date('H:i:s');
                $emailBody = $certGenerator->generateEmailFriendlyCertificateHtml($certificateNumber, $studentName, $courseTitle, $issueDate);
                break;
        }
        
        // Send the test email
        $success = sendEmailNow($testEmail, $subject, $emailBody);
        
        $test_results[] = [
            'type' => $success ? 'success' : 'error',
            'message' => "Test {$testType}: " . ($success ? 'SENT successfully' : 'FAILED to send'),
            'details' => "Subject: {$subject}"
        ];
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Progressive Certificate Email Debug - Panadite Academy</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        .debug-container {
            max-width: 800px;
            margin: 30px auto;
            background: white;
            border-radius: 15px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.2);
            overflow: hidden;
        }
        .header {
            background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
            color: white;
            text-align: center;
            padding: 30px;
        }
        .test-option {
            background: #f8f9fa;
            border: 2px solid #e9ecef;
            border-radius: 10px;
            padding: 15px;
            margin: 10px 0;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        .test-option:hover {
            border-color: #28a745;
            background: #e8f5e8;
        }
        .test-option.selected {
            border-color: #28a745;
            background: #d4edda;
        }
        .result-success {
            background: #d1edff;
            color: #0c5460;
            border-left: 4px solid #17a2b8;
            padding: 10px 15px;
            margin: 5px 0;
        }
        .result-error {
            background: #f8d7da;
            color: #721c24;
            border-left: 4px solid #dc3545;
            padding: 10px 15px;
            margin: 5px 0;
        }
    </style>
</head>
<body>
    <div class="debug-container">
        <div class="header">
            <i class="bx bx-search-alt bx-lg mb-2"></i>
            <h1>Progressive Certificate Email Debug</h1>
            <p class="mb-0">Test different email complexities to find what breaks</p>
        </div>
        
        <div class="p-4">
            <div class="alert alert-info">
                <i class="bx bx-info-circle me-2"></i>
                <strong>Goal:</strong> Test emails from simple to complex to find exactly what breaks certificate delivery.
            </div>
            
            <form method="POST">
                <div class="mb-3">
                    <label class="form-label fw-bold">
                        <i class="bx bx-envelope me-2"></i>Test Email Address
                    </label>
                    <input type="email" class="form-control" name="test_email" required 
                           value="<?= htmlspecialchars($_POST['test_email'] ?? '') ?>" 
                           placeholder="Enter your email address">
                </div>
                
                <div class="mb-4">
                    <label class="form-label fw-bold">Select Test Type:</label>
                    
                    <div class="test-option" onclick="selectTest('1')">
                        <input type="radio" name="test_type" value="1" id="test1" <?= ($_POST['test_type'] ?? '1') == '1' ? 'checked' : '' ?>>
                        <label for="test1" class="ms-2">
                            <strong>TEST 1: Simple Text</strong> - Exact same as working debug script
                        </label>
                    </div>
                    
                    <div class="test-option" onclick="selectTest('2')">
                        <input type="radio" name="test_type" value="2" id="test2" <?= ($_POST['test_type'] ?? '') == '2' ? 'checked' : '' ?>>
                        <label for="test2" class="ms-2">
                            <strong>TEST 2: Basic Certificate</strong> - Simple certificate with basic styling
                        </label>
                    </div>
                    
                    <div class="test-option" onclick="selectTest('3')">
                        <input type="radio" name="test_type" value="3" id="test3" <?= ($_POST['test_type'] ?? '') == '3' ? 'checked' : '' ?>>
                        <label for="test3" class="ms-2">
                            <strong>TEST 3: Medium Certificate</strong> - More styled certificate
                        </label>
                    </div>
                    
                    <div class="test-option" onclick="selectTest('4')">
                        <input type="radio" name="test_type" value="4" id="test4" <?= ($_POST['test_type'] ?? '') == '4' ? 'checked' : '' ?>>
                        <label for="test4" class="ms-2">
                            <strong>TEST 4: Email-Friendly Certificate</strong> - Our designed email certificate with wrapper
                        </label>
                    </div>
                    
                    <div class="test-option" onclick="selectTest('5')">
                        <input type="radio" name="test_type" value="5" id="test5" <?= ($_POST['test_type'] ?? '') == '5' ? 'checked' : '' ?>>
                        <label for="test5" class="ms-2">
                            <strong>TEST 5: Certificate HTML Only</strong> - Just the certificate without wrapper
                        </label>
                    </div>
                </div>
                
                <button type="submit" class="btn btn-success btn-lg w-100">
                    <i class="bx bx-send me-2"></i>Send Test Email
                </button>
            </form>
            
            <?php if (!empty($test_results)): ?>
            <div class="mt-4">
                <h5>Test Results:</h5>
                <?php foreach ($test_results as $result): ?>
                <div class="result-<?= $result['type'] ?>">
                    <strong><?= htmlspecialchars($result['message']) ?></strong>
                    <?php if (isset($result['details'])): ?>
                    <br><small><?= htmlspecialchars($result['details']) ?></small>
                    <?php endif; ?>
                </div>
                <?php endforeach; ?>
                
                <div class="alert alert-warning mt-3">
                    <i class="bx bx-time me-2"></i>
                    Check your email inbox (and spam folder) for test messages. If a test shows "SENT" but you don't receive it, that test level is too complex for email delivery.
                </div>
            </div>
            <?php endif; ?>
            
            <div class="text-center mt-4">
                <a href="debug_email_sending.php" class="btn btn-outline-primary me-2">
                    <i class="bx bx-bug me-2"></i>Full Email Debug
                </a>
                <a href="test_certificate_email.php" class="btn btn-outline-secondary">
                    <i class="bx bx-envelope me-2"></i>Certificate Test
                </a>
            </div>
        </div>
    </div>
    
    <script>
        function selectTest(testNum) {
            document.getElementById('test' + testNum).checked = true;
            
            // Update visual selection
            document.querySelectorAll('.test-option').forEach(option => {
                option.classList.remove('selected');
            });
            event.currentTarget.classList.add('selected');
        }
        
        // Set initial selection
        document.addEventListener('DOMContentLoaded', function() {
            const checked = document.querySelector('input[name="test_type"]:checked');
            if (checked) {
                checked.closest('.test-option').classList.add('selected');
            }
        });
    </script>
</body>
</html>
