<?php
session_start();
require_once '../includes/config.php';

// Prevent output buffering
if (ob_get_level()) {
    ob_end_clean();
}

header('Content-Type: text/html; charset=UTF-8');

echo "<h1>🔍 Quiz Attempts Diagnostic</h1>";
echo "<pre>";

try {
    $student_id = 15; // Kahindo Brian's student ID
    $course_id = 13;  // Computer science course ID
    
    echo "=== QUIZ ATTEMPTS ANALYSIS ===\n";
    echo "Student ID: $student_id\n";
    echo "Course ID: $course_id\n\n";
    
    // 1. Check quiz_attempts table structure
    echo "=== QUIZ_ATTEMPTS TABLE STRUCTURE ===\n";
    $structure = $conn->query("DESCRIBE quiz_attempts");
    if ($structure) {
        while ($column = $structure->fetch_assoc()) {
            echo "📋 {$column['Field']} ({$column['Type']})\n";
        }
    }
    
    // 2. Check all quiz attempts for this student
    echo "\n=== ALL QUIZ ATTEMPTS FOR STUDENT $student_id ===\n";
    $attempts_query = "
        SELECT 
            qa.*,
            q.title as quiz_title,
            q.course_id
        FROM quiz_attempts qa
        LEFT JOIN quizzes q ON qa.quiz_id = q.quiz_id
        WHERE qa.student_id = $student_id
        ORDER BY qa.attempt_id DESC
    ";
    
    $attempts_result = $conn->query($attempts_query);
    if ($attempts_result && $attempts_result->num_rows > 0) {
        echo "Found {$attempts_result->num_rows} attempts:\n";
        while ($attempt = $attempts_result->fetch_assoc()) {
            echo "📝 Attempt ID: {$attempt['attempt_id']} | Quiz: {$attempt['quiz_title']} | Score: {$attempt['score']} | Date: {$attempt['attempt_date']}\n";
        }
    } else {
        echo "❌ NO quiz attempts found for student $student_id\n";
    }
    
    // 3. Check actual quiz-course relationships for student's quizzes
    echo "\n=== QUIZ-COURSE RELATIONSHIP CHECK ===\n";
    $quiz_course_check = "
        SELECT 
            qa.attempt_id,
            qa.quiz_id,
            qa.student_id,
            qa.score,
            qa.end_time,
            qa.pass_status,
            q.title as quiz_title,
            q.course_id,
            q.passing_score,
            c.title as course_title
        FROM quiz_attempts qa
        INNER JOIN quizzes q ON qa.quiz_id = q.quiz_id
        LEFT JOIN courses c ON q.course_id = c.course_id
        WHERE qa.student_id = $student_id
        ORDER BY qa.attempt_id DESC
    ";
    
    $relationship_result = $conn->query($quiz_course_check);
    if ($relationship_result && $relationship_result->num_rows > 0) {
        echo "Quiz-Course relationships for your attempts:\n";
        while ($attempt = $relationship_result->fetch_assoc()) {
            $status = $attempt['pass_status'] ? 'PASSED' : ($attempt['score'] >= $attempt['passing_score'] ? 'PASSED' : 'FAILED');
            echo "🔗 Quiz: {$attempt['quiz_title']} | Course ID: {$attempt['course_id']} | Course: {$attempt['course_title']} | Score: {$attempt['score']}% | Status: $status\n";
        }
    } else {
        echo "❌ No quiz-course relationships found\n";
    }
    
    // 4. Check for Computer Science course specifically  
    echo "\n=== COMPUTER SCIENCE COURSE QUIZ ATTEMPTS ===\n";
    $cs_attempts = "
        SELECT 
            qa.attempt_id,
            qa.quiz_id,
            qa.student_id,
            qa.score,
            qa.end_time,
            qa.pass_status,
            q.title as quiz_title,
            q.passing_score,
            q.course_id
        FROM quiz_attempts qa
        INNER JOIN quizzes q ON qa.quiz_id = q.quiz_id
        WHERE q.course_id = $course_id AND qa.student_id = $student_id
        ORDER BY qa.attempt_id DESC
    ";
    
    $cs_result = $conn->query($cs_attempts);
    if ($cs_result && $cs_result->num_rows > 0) {
        echo "Found {$cs_result->num_rows} Computer Science quiz attempts:\n";
        while ($attempt = $cs_result->fetch_assoc()) {
            $status = $attempt['score'] >= $attempt['passing_score'] ? 'PASSED' : 'FAILED';
            echo "🎯 Quiz: {$attempt['quiz_title']} | Score: {$attempt['score']}% | Passing: {$attempt['passing_score']}% | Status: $status\n";
        }
    } else {
        echo "❌ NO Computer Science quiz attempts found\n";
    }
    
    // 5. Test the actual query used in getQuizPerformanceReport (FIXED COLUMNS)
    echo "\n=== TESTING FIXED QUERY FROM getQuizPerformanceReport ===\n";
    $test_query = "
        SELECT 
            q.quiz_id,
            q.title,
            q.passing_score,
            COALESCE(qa.score, 0) as score,
            qa.end_time as attempt_date,
            qa.attempt_id as attempt_number,
            qa.pass_status,
            CASE 
                WHEN qa.score IS NULL THEN 'not_attempted'
                WHEN qa.pass_status = 1 THEN 'passed'
                WHEN qa.score >= q.passing_score THEN 'passed' 
                ELSE 'failed' 
            END as status
        FROM quizzes q
        LEFT JOIN (
            SELECT 
                qa1.quiz_id,
                qa1.student_id, 
                qa1.score, 
                qa1.end_time, 
                qa1.attempt_id,
                qa1.pass_status
            FROM quiz_attempts qa1
            INNER JOIN (
                SELECT quiz_id, student_id, MAX(attempt_id) as max_attempt_id
                FROM quiz_attempts
                WHERE student_id = $student_id
                GROUP BY quiz_id, student_id
            ) qa2 ON qa1.quiz_id = qa2.quiz_id 
                AND qa1.student_id = qa2.student_id 
                AND qa1.attempt_id = qa2.max_attempt_id
        ) qa ON q.quiz_id = qa.quiz_id AND qa.student_id = $student_id
        WHERE q.course_id = $course_id
        ORDER BY q.quiz_id
    ";
    
    $test_result = $conn->query($test_query);
    if ($test_result) {
        echo "Query executed successfully:\n";
        while ($row = $test_result->fetch_assoc()) {
            echo "🔍 Quiz: {$row['title']} | Score: {$row['score']}% | Status: {$row['status']} | Date: {$row['attempt_date']}\n";
        }
    } else {
        echo "❌ Query failed: " . $conn->error . "\n";
        
        // Fallback: Try simpler query
        echo "\n=== TRYING SIMPLER QUERY ===\n";
        $simple_query = "
            SELECT 
                q.quiz_id,
                q.title,
                q.passing_score,
                qa.score,
                qa.attempt_date,
                qa.attempt_number
            FROM quizzes q
            LEFT JOIN quiz_attempts qa ON q.quiz_id = qa.quiz_id AND qa.student_id = $student_id
            WHERE q.course_id = $course_id
            ORDER BY q.quiz_id, qa.attempt_id DESC
        ";
        
        $simple_result = $conn->query($simple_query);
        if ($simple_result) {
            while ($row = $simple_result->fetch_assoc()) {
                echo "📋 Quiz: {$row['title']} | Score: {$row['score']}% | Date: {$row['attempt_date']}\n";
            }
        }
    }
    
    // 5. Check if ROW_NUMBER is supported
    echo "\n=== TESTING ROW_NUMBER SUPPORT ===\n";
    $version_result = $conn->query("SELECT VERSION() as version");
    if ($version_result) {
        $version = $version_result->fetch_assoc()['version'];
        echo "MySQL Version: $version\n";
        
        if (version_compare($version, '8.0', '<')) {
            echo "⚠️ ROW_NUMBER() might not be supported in this MySQL version\n";
        } else {
            echo "✅ ROW_NUMBER() should be supported\n";
        }
    }

} catch (Exception $e) {
    echo "❌ ERROR: " . $e->getMessage() . "\n";
}

echo "</pre>";
?>
