<?php
require_once __DIR__ . '/../auth/functions.php';
requireRole('admin');

require_once __DIR__ . '/../config/db_connect.php';

$request_id = intval($_GET['request_id'] ?? 0);

if (!$request_id) {
    header('HTTP/1.0 400 Bad Request');
    die('Invalid request ID');
}

// Get request and student details
$request_query = "SELECT ar.*, u.first_name, u.last_name, c.title as course_title 
                  FROM accreditation_requests ar 
                  JOIN users u ON ar.student_id = u.user_id 
                  JOIN courses c ON ar.course_id = c.course_id 
                  WHERE ar.request_id = ?";
$request_stmt = $conn->prepare($request_query);
$request_stmt->bind_param("i", $request_id);
$request_stmt->execute();
$request_data = $request_stmt->get_result()->fetch_assoc();

if (!$request_data) {
    header('HTTP/1.0 404 Not Found');
    die('Request not found');
}

// Get all documents for this request
$docs_query = "SELECT * FROM accreditation_documents WHERE request_id = ? ORDER BY document_type";
$docs_stmt = $conn->prepare($docs_query);
$docs_stmt->bind_param("i", $request_id);
$docs_stmt->execute();
$documents = $docs_stmt->get_result()->fetch_all(MYSQLI_ASSOC);

if (empty($documents)) {
    header('HTTP/1.0 404 Not Found');
    die('No documents found for this request');
}

// Create ZIP file
$zip = new ZipArchive();
$student_name = $request_data['first_name'] . '_' . $request_data['last_name'];
$safe_student_name = preg_replace('/[^a-zA-Z0-9_-]/', '', $student_name);
$zip_filename = 'SITA_Documents_' . $safe_student_name . '_Request_' . $request_id . '.zip';
$zip_path = sys_get_temp_dir() . '/' . $zip_filename;

$result = $zip->open($zip_path, ZipArchive::CREATE | ZipArchive::OVERWRITE);

if ($result !== TRUE) {
    header('HTTP/1.0 500 Internal Server Error');
    die('Cannot create ZIP file: ' . $result);
}

// Add student info file
$info_content = "SITA Accreditation Documents\n";
$info_content .= "============================\n\n";
$info_content .= "Student: " . $request_data['first_name'] . ' ' . $request_data['last_name'] . "\n";
$info_content .= "Course: " . $request_data['course_title'] . "\n";
$info_content .= "Request ID: " . $request_id . "\n";
$info_content .= "Request Date: " . date('M j, Y H:i', strtotime($request_data['request_date'])) . "\n";
$info_content .= "Documents Submitted: " . ($request_data['documents_submitted_date'] ? date('M j, Y H:i', strtotime($request_data['documents_submitted_date'])) : 'N/A') . "\n";
$info_content .= "Status: " . ucwords(str_replace('_', ' ', $request_data['status'])) . "\n\n";
$info_content .= "Documents Included:\n";
$info_content .= "==================\n";

$document_count = 0;
$missing_files = [];

foreach ($documents as $doc) {
    // Try multiple possible file paths - covering all server configurations
    $possible_paths = [
        $_SERVER['DOCUMENT_ROOT'] . $doc['file_path'],
        dirname(__DIR__) . $doc['file_path'],
        __DIR__ . '/..' . $doc['file_path'],
        '/home/twpurnyc/public_html' . $doc['file_path'],
        $_SERVER['DOCUMENT_ROOT'] . '/uploads/accreditation/request_' . $request_id . '/' . basename($doc['file_path']),
        dirname(__DIR__) . '/uploads/accreditation/request_' . $request_id . '/' . basename($doc['file_path'])
    ];
    
    $file_found = false;
    $actual_path = '';
    
    // Debug: log all attempted paths
    error_log("Looking for document: " . $doc['document_name']);
    error_log("Database file_path: " . $doc['file_path']);
    
    foreach ($possible_paths as $path) {
        error_log("Trying path: " . $path);
        if (file_exists($path)) {
            $file_found = true;
            $actual_path = $path;
            error_log("Found file at: " . $path);
            break;
        }
    }
    
    if ($file_found) {
        // Create safe filename for ZIP
        $doc_type = ucwords(str_replace('_', ' ', $doc['document_type']));
        $extension = pathinfo($doc['document_name'], PATHINFO_EXTENSION);
        $safe_filename = $doc_type . '_' . $safe_student_name . '.' . $extension;
        
        // Add file to ZIP
        $zip->addFile($actual_path, $safe_filename);
        
        // Add to info
        $info_content .= "- " . $doc_type . ": " . $doc['document_name'] . "\n";
        $info_content .= "  Size: " . round($doc['file_size']/1024/1024, 2) . "MB\n";
        $info_content .= "  Uploaded: " . date('M j, Y H:i', strtotime($doc['upload_date'])) . "\n";
        $info_content .= "  Status: " . ucwords($doc['verification_status']) . "\n\n";
        
        $document_count++;
    } else {
        $missing_files[] = $doc['document_name'];
        error_log("Missing file for request $request_id: " . $doc['file_path']);
    }
}

$info_content .= "\nTotal Documents: " . $document_count . "/" . count($documents) . "\n";

if (!empty($missing_files)) {
    $info_content .= "\nMissing Files:\n";
    foreach ($missing_files as $missing) {
        $info_content .= "- " . $missing . "\n";
    }
}

$info_content .= "\nGenerated: " . date('Y-m-d H:i:s') . " by Panadite Academy Admin\n";

// Add info file to ZIP
$zip->addFromString('DOCUMENT_INFO.txt', $info_content);

// Close ZIP file
$zip->close();

if ($document_count === 0) {
    unlink($zip_path);
    
    // Show debug information instead of just error
    echo "<h3>Debug Information</h3>";
    echo "<p>Documents in database: " . count($documents) . "</p>";
    echo "<p>Documents found on server: " . $document_count . "</p>";
    echo "<hr>";
    
    echo "<h4>Database Records:</h4>";
    foreach ($documents as $doc) {
        echo "<div style='border:1px solid #ccc; padding:10px; margin:10px;'>";
        echo "<strong>" . htmlspecialchars($doc['document_name']) . "</strong><br>";
        echo "Type: " . htmlspecialchars($doc['document_type']) . "<br>";
        echo "Database Path: " . htmlspecialchars($doc['file_path']) . "<br>";
        echo "Size: " . round($doc['file_size']/1024/1024, 2) . "MB<br>";
        echo "</div>";
    }
    
    echo "<h4>Server Environment:</h4>";
    echo "DOCUMENT_ROOT: " . $_SERVER['DOCUMENT_ROOT'] . "<br>";
    echo "Current Directory: " . __DIR__ . "<br>";
    echo "Parent Directory: " . dirname(__DIR__) . "<br>";
    
    die();
}

// Set headers for download
header('Content-Type: application/zip');
header('Content-Disposition: attachment; filename="' . $zip_filename . '"');
header('Content-Length: ' . filesize($zip_path));
header('Cache-Control: no-cache, must-revalidate');
header('Pragma: no-cache');
header('Expires: 0');

// Output ZIP file
readfile($zip_path);

// Clean up temporary file
unlink($zip_path);
exit();
?>
