<?php
/**
 * Admin Email Inbox - Panadite Academy
 * Receive and manage incoming emails
 */

require_once __DIR__ . '/../auth/functions.php';
require_once '../config/email_inbox_service.php';

// Ensure user is logged in as an admin
requireRole('admin');

// Handle AJAX requests
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    switch ($_POST['action']) {
        case 'delete_email':
            $emailId = intval($_POST['email_id']);
            $result = deleteEmail($emailId);
            echo json_encode($result);
            exit();
            
        case 'mark_read':
            $emailId = intval($_POST['email_id']);
            $read = $_POST['read'] === 'true';
            $result = markEmailRead($emailId, $read);
            echo json_encode($result);
            exit();
            
        case 'get_email_details':
            $emailId = intval($_POST['email_id']);
            $result = getEmailDetails($emailId);
            echo json_encode($result);
            exit();
            
        case 'get_stats':
            $statsResult = getInboxStats();
            echo json_encode($statsResult);
            exit();
    }
}

// Get search and pagination parameters
$search = isset($_GET['search']) ? trim($_GET['search']) : '';
$status = isset($_GET['status']) ? trim($_GET['status']) : '';
$dateFrom = isset($_GET['date_from']) ? trim($_GET['date_from']) : '';
$dateTo = isset($_GET['date_to']) ? trim($_GET['date_to']) : '';
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$perPage = 25;
$offset = ($page - 1) * $perPage;

// Create filters array
$filters = [
    'search' => $search,
    'status' => $status,
    'date_from' => $dateFrom,
    'date_to' => $dateTo
];

// Get emails with search and filters
$emailResult = getEmailList($perPage, $offset, $filters);
$emails = $emailResult['success'] ? $emailResult['emails'] : [];
$totalEmails = $emailResult['success'] ? $emailResult['total'] : 0;
$totalPages = ceil($totalEmails / $perPage);

// Get inbox stats
$statsResult = getInboxStats();
$stats = $statsResult['success'] ? $statsResult['stats'] : ['total' => 0, 'unread' => 0, 'recent' => 0];

$currentUser = getCurrentUser();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Email Inbox - Panadite Academy Admin</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link rel="stylesheet" href="/assets/css/user-cards.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        .email-row { cursor: pointer; transition: background-color 0.2s; }
        .email-row:hover { background-color: #f8f9fa; }
        .email-row.unread { font-weight: bold; background-color: #fff3cd; }
        .email-actions { display: none; }
        .email-row:hover .email-actions { display: block; }
        .email-preview { max-height: 500px; overflow-y: auto; }
        .stats-card { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; }
        .compose-btn { position: fixed; bottom: 30px; right: 30px; z-index: 999; }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
        
        <div class="container-fluid">
            <!-- Page Header -->
            <div class="d-flex justify-content-between align-items-center mb-4">
                <div>
                    <h2><i class="bx bx-envelope"></i> Email Inbox</h2>
                    <p class="text-muted">Manage incoming emails</p>
                </div>
                <div class="d-flex gap-2">
                    <button class="btn btn-outline-primary" onclick="refreshInbox()">
                        <i class="bx bx-refresh"></i> Refresh
                    </button>
                    <a href="email_compose.php" class="btn btn-primary">
                        <i class="bx bx-plus"></i> Compose
                    </a>
                </div>
            </div>

            <!-- Search and Filters -->
            <div class="card mb-4">
                <div class="card-body">
                    <form method="GET" action="email_inbox.php" id="searchForm">
                        <div class="row g-3">
                            <div class="col-md-6">
                                <div class="input-group">
                                    <input type="text" 
                                           class="form-control" 
                                           name="search" 
                                           id="searchInput"
                                           value="<?= htmlspecialchars($search) ?>"
                                           placeholder="Search emails by subject, sender, or content...">
                                </div>
                            </div>
                            <div class="col-md-2">
                                <button class="btn btn-primary w-100" type="submit">
                                    <i class="bx bx-search"></i> Search
                                </button>
                            </div>
                            <div class="col-md-2">
                                <button class="btn btn-info w-100" type="button" onclick="toggleFilters()">
                                    <i class="bx bx-filter"></i> Filters
                                </button>
                            </div>
                            <div class="col-md-2">
                                <?php if (!empty($search)): ?>
                                <a href="email_inbox.php" class="btn btn-outline-secondary w-100">
                                    <i class="bx bx-x"></i> Clear
                                </a>
                                <?php else: ?>
                                <button class="btn btn-outline-primary w-100" type="button" onclick="refreshInbox()">
                                    <i class="bx bx-refresh"></i> Refresh
                                </button>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <!-- Advanced Filters (Initially Hidden) -->
                        <div id="advancedFilters" class="mt-3" style="display: none;">
                            <div class="row g-3">
                                <div class="col-md-3">
                                    <label class="form-label">Status</label>
                                    <select class="form-select" name="status" id="statusFilter">
                                        <option value="">All Status</option>
                                        <option value="unread" <?= (isset($_GET['status']) && $_GET['status'] === 'unread') ? 'selected' : '' ?>>Unread Only</option>
                                        <option value="read" <?= (isset($_GET['status']) && $_GET['status'] === 'read') ? 'selected' : '' ?>>Read Only</option>
                                    </select>
                                </div>
                                <div class="col-md-3">
                                    <label class="form-label">From Date</label>
                                    <input type="date" class="form-control" name="date_from" id="dateFrom" 
                                           value="<?= isset($_GET['date_from']) ? htmlspecialchars($_GET['date_from']) : '' ?>">
                                </div>
                                <div class="col-md-3">
                                    <label class="form-label">To Date</label>
                                    <input type="date" class="form-control" name="date_to" id="dateTo"
                                           value="<?= isset($_GET['date_to']) ? htmlspecialchars($_GET['date_to']) : '' ?>">
                                </div>
                                <div class="col-md-3">
                                    <label class="form-label">&nbsp;</label>
                                    <div class="d-grid">
                                        <button class="btn btn-primary" type="submit">Apply Filters</button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Stats Cards -->
            <div class="row mb-4">
                <div class="col-md-4">
                    <div class="card stats-card">
                        <div class="card-body text-center">
                            <i class="bx bx-envelope fs-1"></i>
                            <h3><?php echo $stats['total']; ?></h3>
                            <p class="mb-0">Total Emails</p>
                        </div>
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="card bg-warning text-white">
                        <div class="card-body text-center">
                            <i class="bx bx-envelope-open fs-1"></i>
                            <h3><?php echo $stats['unread']; ?></h3>
                            <p class="mb-0">Unread Emails</p>
                        </div>
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="card bg-success text-white">
                        <div class="card-body text-center">
                            <i class="bx bx-time fs-1"></i>
                            <h3><?php echo $stats['recent']; ?></h3>
                            <p class="mb-0">Recent Emails</p>
                        </div>
                    </div>
                </div>
            </div>

            <?php if (!$emailResult['success']): ?>
            <div class="alert alert-danger">
                <i class="bx bx-error"></i> Failed to connect to email server: <?php echo htmlspecialchars($emailResult['error']); ?>
                <br><small>Please check your email configuration in <a href="email_settings.php">Email Settings</a></small>
            </div>
            <?php endif; ?>

            <!-- Email List -->
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">
                        <i class="bx bx-list-ul"></i> 
                        <?php 
                        $hasActiveFilters = !empty($search) || !empty($status) || !empty($dateFrom) || !empty($dateTo);
                        if ($hasActiveFilters): 
                        ?>
                            <?php if (!empty($search)): ?>
                                Search Results for "<?= htmlspecialchars($search) ?>"
                            <?php else: ?>
                                Filtered Results
                            <?php endif; ?>
                            (<?= $totalEmails ?> emails)
                            <?php if (!empty($status)): ?>
                                <small class="badge bg-info ms-2"><?= ucfirst($status) ?></small>
                            <?php endif; ?>
                            <?php if (!empty($dateFrom) || !empty($dateTo)): ?>
                                <small class="badge bg-secondary ms-1">Date Range</small>
                            <?php endif; ?>
                        <?php else: ?>
                            Inbox (<?= $totalEmails ?> emails)
                        <?php endif; ?>
                    </h5>
                    <?php if ($totalPages > 1): ?>
                    <div class="d-flex align-items-center gap-2">
                        <small class="text-muted">
                            Page <?= $page ?> of <?= $totalPages ?>
                            (<?= ($offset + 1) ?>-<?= min($offset + $perPage, $totalEmails) ?> of <?= $totalEmails ?>)
                        </small>
                    </div>
                    <?php endif; ?>
                </div>
                <div class="card-body p-0">
                    <?php if (empty($emails)): ?>
                    <div class="text-center p-5">
                        <i class="bx bx-envelope-open fs-1 text-muted"></i>
                        <h4 class="text-muted">No emails found</h4>
                        <p class="text-muted">Your inbox is empty or there was an error connecting to the email server.</p>
                    </div>
                    <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-hover mb-0">
                            <thead class="table-light">
                                <tr>
                                    <th width="5%"><input type="checkbox" id="selectAll"></th>
                                    <th width="25%">From</th>
                                    <th width="40%">Subject</th>
                                    <th width="15%">Date</th>
                                    <th width="10%">Size</th>
                                    <th width="5%">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($emails as $email): ?>
                                <tr class="email-row <?php echo !$email['seen'] ? 'unread' : ''; ?>" 
                                    data-email-id="<?php echo $email['id']; ?>"
                                    onclick="viewEmail(<?php echo $email['id']; ?>)">
                                    <td onclick="event.stopPropagation()">
                                        <input type="checkbox" class="email-checkbox" value="<?php echo $email['id']; ?>">
                                    </td>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <?php if (!$email['seen']): ?>
                                            <i class="bx bx-circle text-primary me-2"></i>
                                            <?php endif; ?>
                                            <div>
                                                <div class="fw-bold"><?php echo htmlspecialchars($email['from_name'] ?: $email['from']); ?></div>
                                                <small class="text-muted"><?php echo htmlspecialchars($email['from']); ?></small>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="fw-bold"><?php echo htmlspecialchars($email['subject']); ?></div>
                                    </td>
                                    <td>
                                        <small><?php echo date('M j, Y H:i', strtotime($email['date'])); ?></small>
                                    </td>
                                    <td>
                                        <small class="text-muted"><?php echo $email['size']; ?></small>
                                    </td>
                                    <td onclick="event.stopPropagation()">
                                        <div class="email-actions">
                                            <div class="btn-group btn-group-sm">
                                                <button class="btn btn-outline-primary btn-sm" 
                                                        onclick="markAsRead(<?php echo $email['id']; ?>, <?php echo $email['seen'] ? 'false' : 'true'; ?>)"
                                                        title="<?php echo $email['seen'] ? 'Mark as unread' : 'Mark as read'; ?>">
                                                    <i class="bx <?php echo $email['seen'] ? 'bx-envelope-open' : 'bx-envelope'; ?>"></i>
                                                </button>
                                                <button class="btn btn-outline-danger btn-sm" 
                                                        onclick="deleteEmail(<?php echo $email['id']; ?>)"
                                                        title="Delete">
                                                    <i class="bx bx-trash"></i>
                                                </button>
                                            </div>
                                        </div>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <!-- Enhanced Pagination -->
                    <?php if ($totalPages > 1): ?>
                    <div class="card-footer">
                        <div class="d-flex justify-content-between align-items-center">
                            <div class="d-flex align-items-center gap-2">
                                <small class="text-muted">
                                    Showing <?= ($offset + 1) ?>-<?= min($offset + $perPage, $totalEmails) ?> of <?= $totalEmails ?> emails
                                </small>
                            </div>
                            
                            <nav aria-label="Email pagination">
                                <ul class="pagination pagination-sm mb-0">
                                    <?php 
                                    // Build query string for pagination links
                                    $queryParams = [];
                                    if (!empty($search)) $queryParams['search'] = $search;
                                    if (!empty($status)) $queryParams['status'] = $status;
                                    if (!empty($dateFrom)) $queryParams['date_from'] = $dateFrom;
                                    if (!empty($dateTo)) $queryParams['date_to'] = $dateTo;
                                    
                                    function buildPaginationUrl($pageNum, $params) {
                                        $params['page'] = $pageNum;
                                        return 'email_inbox.php?' . http_build_query($params);
                                    }
                                    ?>
                                    
                                    <!-- Previous Button -->
                                    <li class="page-item <?= ($page <= 1) ? 'disabled' : '' ?>">
                                        <a class="page-link" 
                                           href="<?= ($page > 1) ? buildPaginationUrl($page - 1, $queryParams) : '#' ?>"
                                           <?= ($page <= 1) ? 'tabindex="-1" aria-disabled="true"' : '' ?>>
                                            <i class="bx bx-chevron-left"></i> Previous
                                        </a>
                                    </li>
                                    
                                    <!-- First Page -->
                                    <?php if ($page > 3): ?>
                                    <li class="page-item">
                                        <a class="page-link" href="<?= buildPaginationUrl(1, $queryParams) ?>">1</a>
                                    </li>
                                    <?php if ($page > 4): ?>
                                    <li class="page-item disabled">
                                        <span class="page-link">...</span>
                                    </li>
                                    <?php endif; ?>
                                    <?php endif; ?>
                                    
                                    <!-- Page Numbers -->
                                    <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                                    <li class="page-item <?= ($i == $page) ? 'active' : '' ?>">
                                        <a class="page-link" href="<?= buildPaginationUrl($i, $queryParams) ?>">
                                            <?= $i ?>
                                        </a>
                                    </li>
                                    <?php endfor; ?>
                                    
                                    <!-- Last Page -->
                                    <?php if ($page < $totalPages - 2): ?>
                                    <?php if ($page < $totalPages - 3): ?>
                                    <li class="page-item disabled">
                                        <span class="page-link">...</span>
                                    </li>
                                    <?php endif; ?>
                                    <li class="page-item">
                                        <a class="page-link" href="<?= buildPaginationUrl($totalPages, $queryParams) ?>"><?= $totalPages ?></a>
                                    </li>
                                    <?php endif; ?>
                                    
                                    <!-- Next Button -->
                                    <li class="page-item <?= ($page >= $totalPages) ? 'disabled' : '' ?>">
                                        <a class="page-link" 
                                           href="<?= ($page < $totalPages) ? buildPaginationUrl($page + 1, $queryParams) : '#' ?>"
                                           <?= ($page >= $totalPages) ? 'tabindex="-1" aria-disabled="true"' : '' ?>>
                                            Next <i class="bx bx-chevron-right"></i>
                                        </a>
                                    </li>
                                </ul>
                            </nav>
                            
                            <!-- Quick Page Jump -->
                            <div class="d-flex align-items-center gap-2">
                                <small class="text-muted">Go to:</small>
                                <input type="number" 
                                       class="form-control form-control-sm" 
                                       style="width: 60px;" 
                                       min="1" 
                                       max="<?= $totalPages ?>" 
                                       value="<?= $page ?>"
                                       onchange="goToPage(this.value)">
                            </div>
                        </div>
                    </div>
                    <?php endif; ?>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        
        <!-- Floating Compose Button -->
        <a href="email_compose.php" class="btn btn-primary btn-lg rounded-circle compose-btn" title="Compose Email">
            <i class="bx bx-plus fs-4"></i>
        </a>
        
        </div>
    </section>

    <!-- Email Preview Modal -->
    <div class="modal fade" id="emailModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="emailSubject">Email Preview</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div id="emailContent">
                        <div class="text-center p-4">
                            <div class="spinner-border" role="status">
                                <span class="visually-hidden">Loading...</span>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="button" class="btn btn-primary" id="replyBtn">Reply</button>
                    <button type="button" class="btn btn-outline-danger" id="deleteBtn">Delete</button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        let currentEmailId = null;

        function viewEmail(emailId) {
            currentEmailId = emailId;
            const modal = new bootstrap.Modal(document.getElementById('emailModal'));
            
            // Show loading
            document.getElementById('emailContent').innerHTML = `
                <div class="text-center p-4">
                    <div class="spinner-border" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                </div>
            `;
            
            modal.show();
            
            // Fetch email details
            fetch('email_inbox.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: `action=get_email_details&email_id=${emailId}`
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    const email = data.email;
                    document.getElementById('emailSubject').textContent = email.subject;
                    
                    const content = `
                        <div class="email-preview">
                            <div class="border-bottom pb-3 mb-3">
                                <h6><strong>From:</strong> ${email.from_name ? email.from_name + ' &lt;' + email.from + '&gt;' : email.from}</h6>
                                <h6><strong>To:</strong> ${email.to}</h6>
                                <h6><strong>Date:</strong> ${new Date(email.date).toLocaleString()}</h6>
                                <h6><strong>Size:</strong> ${email.size}</h6>
                            </div>
                            <div class="email-body">
                                ${email.html_body || email.body.replace(/\n/g, '<br>')}
                            </div>
                        </div>
                    `;
                    
                    document.getElementById('emailContent').innerHTML = content;
                    
                    // Mark as read
                    markEmailRowAsRead(emailId);
                } else {
                    document.getElementById('emailContent').innerHTML = `
                        <div class="alert alert-danger">
                            <i class="bx bx-error"></i> Failed to load email: ${data.error}
                        </div>
                    `;
                }
            })
            .catch(error => {
                document.getElementById('emailContent').innerHTML = `
                    <div class="alert alert-danger">
                        <i class="bx bx-error"></i> Error loading email: ${error.message}
                    </div>
                `;
            });
        }

        function markAsRead(emailId, read, buttonElement = null) {
            // Find the button element - either passed as parameter or find by email ID
            let button = buttonElement;
            if (!button) {
                // Find the mark as read button for this specific email
                const row = document.querySelector(`[data-email-id="${emailId}"]`);
                if (row) {
                    button = row.querySelector('.btn-outline-primary');
                }
            }
            
            let originalHTML = '';
            if (button) {
                originalHTML = button.innerHTML;
                button.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i>';
                button.disabled = true;
            }
            
            fetch('email_inbox.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: `action=mark_read&email_id=${emailId}&read=${read}`
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Update UI without full page reload
                    const row = document.querySelector(`[data-email-id="${emailId}"]`);
                    if (row) {
                        if (read === 'true' || read === true) {
                            row.classList.remove('unread');
                            const indicator = row.querySelector('.bx-circle');
                            if (indicator) indicator.remove();
                            
                            if (button) {
                                button.innerHTML = '<i class="bx bx-envelope-open"></i>';
                                button.title = 'Mark as unread';
                                button.setAttribute('onclick', `markAsRead(${emailId}, false)`);
                            }
                        } else {
                            row.classList.add('unread');
                            const fromCell = row.querySelector('td:nth-child(2) .d-flex');
                            if (fromCell && !fromCell.querySelector('.bx-circle')) {
                                fromCell.insertAdjacentHTML('afterbegin', '<i class="bx bx-circle text-primary me-2"></i>');
                            }
                            
                            if (button) {
                                button.innerHTML = '<i class="bx bx-envelope"></i>';
                                button.title = 'Mark as read';
                                button.setAttribute('onclick', `markAsRead(${emailId}, true)`);
                            }
                        }
                    }
                    
                    // Update stats without reload
                    updateInboxStats();
                } else {
                    console.error('Mark as read failed:', data);
                    alert('Failed to update email status: ' + (data.error || 'Unknown error'));
                    
                    // Restore button state on error
                    if (button && originalHTML) {
                        button.innerHTML = originalHTML;
                    }
                }
                
                // Reset button
                if (button) {
                    button.disabled = false;
                }
            })
            .catch(error => {
                console.error('Mark as read error:', error);
                if (button && originalHTML) {
                    button.innerHTML = originalHTML;
                    button.disabled = false;
                }
                alert('Network error while updating email status');
            });
        }

        function deleteEmail(emailId) {
            if (confirm('Are you sure you want to delete this email?')) {
                fetch('email_inbox.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: `action=delete_email&email_id=${emailId}`
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        location.reload();
                    } else {
                        alert('Failed to delete email: ' + data.error);
                    }
                });
            }
        }

        function markEmailRowAsRead(emailId) {
            const row = document.querySelector(`[data-email-id="${emailId}"]`);
            if (row) {
                row.classList.remove('unread');
                const indicator = row.querySelector('.bx-circle');
                if (indicator) {
                    indicator.remove();
                }
            }
        }

        function refreshInbox() {
            location.reload();
        }

        // Toggle advanced filters
        function toggleFilters() {
            const filters = document.getElementById('advancedFilters');
            if (filters.style.display === 'none') {
                filters.style.display = 'block';
            } else {
                filters.style.display = 'none';
            }
        }

        // Quick page navigation - preserves all current filters
        function goToPage(page) {
            const urlParams = new URLSearchParams(window.location.search);
            urlParams.set('page', page);
            window.location.href = `email_inbox.php?${urlParams.toString()}`;
        }

        // Update inbox stats without page reload
        function updateInboxStats() {
            fetch('email_inbox.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: 'action=get_stats'
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Update stats cards if they exist
                    const statsCards = document.querySelectorAll('.stats-card h3, .bg-warning h3, .bg-success h3');
                    if (data.stats) {
                        if (statsCards[0]) statsCards[0].textContent = data.stats.total;
                        if (statsCards[1]) statsCards[1].textContent = data.stats.unread;
                        if (statsCards[2]) statsCards[2].textContent = data.stats.recent;
                    }
                }
            })
            .catch(error => {
                console.error('Failed to update stats:', error);
            });
        }

        // Enhanced search with Enter key support and form validation
        document.addEventListener('DOMContentLoaded', function() {
            const searchInput = document.querySelector('input[name="search"]');
            const searchForm = document.getElementById('searchForm');
            
            if (searchInput) {
                searchInput.addEventListener('keypress', function(e) {
                    if (e.key === 'Enter') {
                        e.preventDefault();
                        searchForm.submit();
                    }
                });
            }
            
            // Auto-focus search input if there's a search term
            if (searchInput && searchInput.value) {
                searchInput.focus();
            }
            
            // Show filters section if any filters are active
            const urlParams = new URLSearchParams(window.location.search);
            if (urlParams.get('status') || urlParams.get('date_from') || urlParams.get('date_to')) {
                const filtersSection = document.getElementById('advancedFilters');
                if (filtersSection) {
                    filtersSection.style.display = 'block';
                }
            }
        });

        // Improved search form submission with loading state
        function submitSearch() {
            const form = document.getElementById('searchForm');
            const submitBtn = form.querySelector('button[type="submit"]');
            const originalText = submitBtn.innerHTML;
            
            submitBtn.innerHTML = '<i class="bx bx-loader-alt bx-spin"></i> Searching...';
            submitBtn.disabled = true;
            
            // Small delay to show the loading state
            setTimeout(() => {
                form.submit();
            }, 100);
        }

        // Clear all filters function
        function clearAllFilters() {
            window.location.href = 'email_inbox.php';
        }

        // Reply button functionality
        document.getElementById('replyBtn').addEventListener('click', function() {
            if (currentEmailId) {
                window.location.href = `email_compose.php?reply=${currentEmailId}`;
            }
        });

        // Delete button in modal
        document.getElementById('deleteBtn').addEventListener('click', function() {
            if (currentEmailId) {
                deleteEmail(currentEmailId);
                bootstrap.Modal.getInstance(document.getElementById('emailModal')).hide();
            }
        });

        // Select all checkbox
        document.getElementById('selectAll').addEventListener('change', function() {
            const checkboxes = document.querySelectorAll('.email-checkbox');
            checkboxes.forEach(cb => cb.checked = this.checked);
        });
    </script>
</body>
</html>
