<?php
/**
 * Email System Monitor & Auto-Processor
 * Real-time email queue monitoring with automatic background processing
 */

require_once __DIR__ . '/../auth/functions.php';
require_once '../config/fast_email_service.php';

// Ensure user is logged in as an admin
requireRole('admin');

// Handle AJAX processing request
if (isset($_POST['process_queue'])) {
    header('Content-Type: application/json');
    
    try {
        $processed = processEmailQueue(15);
        echo json_encode([
            'success' => true,
            'processed' => $processed,
            'timestamp' => date('H:i:s')
        ]);
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ]);
    }
    exit;
}

// Get queue statistics
function getQueueStats() {
    global $pdo;
    
    $stats = [
        'queued' => 0,
        'sending' => 0,
        'sent_today' => 0,
        'failed' => 0,
        'total_sent' => 0
    ];
    
    if ($pdo) {
        try {
            // Check if email_logs table exists first
            $tableCheck = $pdo->query("SHOW TABLES LIKE 'email_logs'");
            if ($tableCheck->rowCount() == 0) {
                // Create the table if it doesn't exist (this will happen automatically when first email is logged)
                return $stats;
            }
            
            // Get status counts from email_logs using correct column names
            $stmt = $pdo->query("SELECT status, COUNT(*) as count FROM email_logs GROUP BY status");
            while ($row = $stmt->fetch()) {
                if ($row['status'] == 'sent' || $row['status'] == 'delivered') {
                    $stats['total_sent'] += (int)$row['count'];
                } elseif ($row['status'] == 'failed' || $row['status'] == 'bounced') {
                    $stats['failed'] += (int)$row['count'];
                }
            }
            
            // Get emails sent today (use correct timestamp column)
            $stmt = $pdo->query("SELECT COUNT(*) as count FROM email_logs WHERE (status = 'sent' OR status = 'delivered') AND DATE(COALESCE(delivery_timestamp, created_at)) = CURDATE()");
            $stats['sent_today'] = (int)$stmt->fetchColumn();
            
            // Check email_queue for actual queued emails
            $queueCheck = $pdo->query("SHOW TABLES LIKE 'email_queue'");
            if ($queueCheck->rowCount() > 0) {
                $stmt = $pdo->query("SELECT COUNT(*) as count FROM email_queue WHERE status = 'queued'");
                $stats['queued'] = (int)$stmt->fetchColumn();
                
                // Check for sending status
                $stmt = $pdo->query("SELECT COUNT(*) as count FROM email_queue WHERE status = 'sending'");
                $stats['sending'] = (int)$stmt->fetchColumn();
            }
            
            // If no queue data, check for recent email activity as sending indicator
            if ($stats['sending'] == 0) {
                $stmt = $pdo->query("SELECT COUNT(*) as count FROM email_logs WHERE created_at > DATE_SUB(NOW(), INTERVAL 2 MINUTE)");
                $recentActivity = (int)$stmt->fetchColumn();
                $stats['sending'] = $recentActivity > 0 ? min($recentActivity, 2) : 0;
            }
            
        } catch (Exception $e) {
            error_log('Error getting queue stats: ' . $e->getMessage());
        }
    }
    
    return $stats;
}

$stats = getQueueStats();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Email Monitor - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    
    <style>
        .stat-card { transition: transform 0.3s ease; }
        .stat-card:hover { transform: translateY(-2px); }
        .stat-number { font-size: 2.5rem; font-weight: bold; }
        .auto-processor { 
            background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
            color: white;
            border-radius: 10px;
            padding: 20px;
        }
        .queue-status { padding: 10px; border-radius: 8px; margin: 10px 0; }
        .status-active { background-color: #d4edda; border: 1px solid #c3e6cb; }
        .status-idle { background-color: #fff3cd; border: 1px solid #ffeaa7; }
        .processing-log { height: 200px; overflow-y: auto; background: #f8f9fa; padding: 10px; border-radius: 5px; }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h1 class="page-title">
                    <i class='bx bx-mail-send'></i>
                    Email System Monitor
                </h1>
                <div>
                    <a href="email_settings.php" class="btn btn-outline-primary">
                        <i class='bx bx-cog'></i> Settings
                    </a>
                </div>
            </div>

            <!-- Stats Cards -->
            <div class="row mb-4">
                <div class="col-md-3">
                    <div class="card stat-card">
                        <div class="card-body text-center">
                            <i class='bx bx-time text-warning mb-2' style="font-size: 2rem;"></i>
                            <div class="stat-number text-warning"><?php echo number_format($stats['queued']); ?></div>
                            <h6 class="text-muted">Queued</h6>
                        </div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="card stat-card">
                        <div class="card-body text-center">
                            <i class='bx bx-paper-plane text-primary mb-2' style="font-size: 2rem;"></i>
                            <div class="stat-number text-primary"><?php echo number_format($stats['sending']); ?></div>
                            <h6 class="text-muted">Sending</h6>
                        </div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="card stat-card">
                        <div class="card-body text-center">
                            <i class='bx bx-check-circle text-success mb-2' style="font-size: 2rem;"></i>
                            <div class="stat-number text-success"><?php echo number_format($stats['sent_today']); ?></div>
                            <h6 class="text-muted">Sent Today</h6>
                        </div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="card stat-card">
                        <div class="card-body text-center">
                            <i class='bx bx-error text-danger mb-2' style="font-size: 2rem;"></i>
                            <div class="stat-number text-danger"><?php echo number_format($stats['failed']); ?></div>
                            <h6 class="text-muted">Failed</h6>
                        </div>
                    </div>
                </div>
            </div>

                <div class="row">
                    <!-- Auto Processor -->
                    <div class="col-md-6">
                        <div class="auto-processor">
                            <h4><i class='bx bx-bot'></i> Auto Email Processor</h4>
                            <p class="mb-3">Automatically processes queued emails in the background without slowing down your application.</p>
                            
                            <div class="queue-status status-idle" id="processorStatus">
                                <i class="fas fa-pause-circle me-2"></i>Status: <span id="statusText">Idle</span>
                                <div class="float-end">
                                    <small id="lastProcessed">Never processed</small>
                                </div>
                            </div>

                            <div class="d-flex gap-2">
                                <button class="btn btn-light" onclick="toggleAutoProcessor()" id="toggleBtn">
                                    <i class='bx bx-play'></i> Start Auto Processing
                                </button>
                                <button class="btn btn-outline-light" onclick="processNow()">
                                    <i class='bx bx-zap'></i> Process Now
                                </button>
                            </div>
                        </div>
                    </div>

                    <!-- Processing Log -->
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0"><i class='bx bx-list-ul'></i> Processing Log</h5>
                            </div>
                            <div class="card-body">
                                <div class="processing-log" id="processingLog">
                                    <div class="text-muted text-center">
                                        <i class="fas fa-info-circle me-2"></i>Processing log will appear here
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Recent Emails -->
                <div class="row mt-4">
                    <div class="col-12">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0"><i class='bx bx-history'></i> Recent Email Activity</h5>
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-hover">
                                        <thead>
                                            <tr>
                                                <th>Recipient</th>
                                                <th>Subject</th>
                                                <th>Status</th>
                                                <th>Time</th>
                                            </tr>
                                        </thead>
                                        <tbody id="recentEmails">
                                            <?php
                                            if ($pdo) {
                                                try {
                                                    // Check if email_logs table exists
                                                    $tableCheck = $pdo->query("SHOW TABLES LIKE 'email_logs'");
                                                    if ($tableCheck->rowCount() > 0) {
                                                        $stmt = $pdo->query("
                                                            SELECT to_email, subject, status, 
                                                                   COALESCE(delivery_timestamp, created_at) as email_time 
                                                            FROM email_logs 
                                                            ORDER BY id DESC 
                                                            LIMIT 10
                                                        ");
                                                        
                                                        $hasData = false;
                                                        while ($row = $stmt->fetch()) {
                                                            $hasData = true;
                                                            $statusClass = [
                                                                'sent' => 'success',
                                                                'failed' => 'danger'
                                                            ];
                                                            
                                                            $class = $statusClass[$row['status']] ?? 'secondary';
                                                            
                                                            echo '<tr>';
                                                            echo '<td>' . htmlspecialchars($row['to_email']) . '</td>';
                                                            echo '<td>' . htmlspecialchars(substr($row['subject'], 0, 50)) . (strlen($row['subject']) > 50 ? '...' : '') . '</td>';
                                                            echo '<td><span class="badge bg-' . $class . '">' . ucfirst($row['status']) . '</span></td>';
                                                            echo '<td>' . date('M j, H:i', strtotime($row['email_time'])) . '</td>';
                                                            echo '</tr>';
                                                        }
                                                        
                                                        if (!$hasData) {
                                                            echo '<tr><td colspan="4" class="text-muted text-center">No email activity yet</td></tr>';
                                                        }
                                                    } else {
                                                        echo '<tr><td colspan="4" class="text-muted text-center">Email logging table not created yet</td></tr>';
                                                    }
                                                } catch (Exception $e) {
                                                    echo '<tr><td colspan="4" class="text-muted text-center">Error loading email data: ' . htmlspecialchars($e->getMessage()) . '</td></tr>';
                                                }
                                            } else {
                                                echo '<tr><td colspan="4" class="text-muted text-center">Database connection not available</td></tr>';
                                            }
                                            ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
        </div>
    </section>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="/assets/js/admin-dashboard.js"></script>
    <script>
        let autoProcessing = false;
        let processingInterval = null;

        function addToLog(message, type = 'info') {
            const log = document.getElementById('processingLog');
            const time = new Date().toLocaleTimeString();
            const colors = {
                info: '#007bff',
                success: '#28a745',
                warning: '#ffc107',
                error: '#dc3545'
            };
            
            const entry = document.createElement('div');
            entry.innerHTML = `<small style="color: ${colors[type]}"><strong>${time}:</strong> ${message}</small>`;
            log.appendChild(entry);
            log.scrollTop = log.scrollHeight;
        }

        function processNow() {
            const btn = event.target;
            const originalText = btn.innerHTML;
            btn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Processing...';
            btn.disabled = true;

            fetch('../config/process_emails.php', {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: 'ajax_process=true'
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    addToLog(`✅ Processed ${data.processed} emails`, 'success');
                    updateStats();
                } else {
                    addToLog(`❌ Error: ${data.error}`, 'error');
                }
            })
            .catch(error => {
                addToLog(`❌ Network error: ${error}`, 'error');
            })
            .finally(() => {
                btn.innerHTML = originalText;
                btn.disabled = false;
            });
        }

        function toggleAutoProcessor() {
            const btn = document.getElementById('toggleBtn');
            const status = document.getElementById('processorStatus');
            const statusText = document.getElementById('statusText');

            if (!autoProcessing) {
                // Start auto processing
                autoProcessing = true;
                processingInterval = setInterval(processNow, 30000); // Every 30 seconds
                
                btn.innerHTML = '<i class="fas fa-pause me-2"></i>Stop Auto Processing';
                btn.className = 'btn btn-outline-light';
                status.className = 'queue-status status-active';
                statusText.textContent = 'Running (every 30 seconds)';
                
                addToLog('🚀 Auto processor started', 'success');
                
                // Process immediately
                processNow();
            } else {
                // Stop auto processing
                autoProcessing = false;
                if (processingInterval) {
                    clearInterval(processingInterval);
                }
                
                btn.innerHTML = '<i class="fas fa-play me-2"></i>Start Auto Processing';
                btn.className = 'btn btn-light';
                status.className = 'queue-status status-idle';
                statusText.textContent = 'Idle';
                
                addToLog('⏹️ Auto processor stopped', 'warning');
            }
        }

        function updateStats() {
            // Reload stats without refreshing page
            location.reload();
        }

        // Auto-start processor when page loads
        document.addEventListener('DOMContentLoaded', function() {
            addToLog('📧 Email monitor system ready', 'info');
            
            // Auto-start if there are queued emails
            <?php if ($stats['queued'] > 0): ?>
                setTimeout(() => {
                    toggleAutoProcessor();
                }, 2000);
            <?php endif; ?>
        });
    </script>
</body>
</html>
