<?php
/**
 * SMTP Email Configuration Admin Page
 * Panadite Academy - Admin Panel
 */

require_once __DIR__ . '/../auth/functions.php';
require_once __DIR__ . '/../config/functions.php';
require_once '../config/fast_email_service.php';

// Ensure user is logged in as an admin
requireRole('admin');

// Handle form submission for SMTP settings
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_smtp_settings'])) {
    try {
        // Create system_settings table if it doesn't exist
        $createTableSQL = "
        CREATE TABLE IF NOT EXISTS `system_settings` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `setting_key` varchar(100) NOT NULL,
            `setting_value` text,
            `setting_group` varchar(50) DEFAULT 'general',
            `description` varchar(255) DEFAULT NULL,
            `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
            `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            UNIQUE KEY `setting_key` (`setting_key`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
        
        if ($pdo) {
            $pdo->exec($createTableSQL);
        }
        
        // SMTP settings to save
        $smtpSettings = [
            'smtp_enabled' => isset($_POST['smtp_enabled']) ? '1' : '0',
            'smtp_host' => filterInput($_POST['smtp_host'] ?? ''),
            'smtp_port' => filterInput($_POST['smtp_port'] ?? '587'),
            'smtp_encryption' => filterInput($_POST['smtp_encryption'] ?? 'tls'),
            'smtp_username' => filterInput($_POST['smtp_username'] ?? ''),
            'smtp_password' => $_POST['smtp_password'] ?? '', // Don't filter password
            'from_email' => filterInput($_POST['from_email'] ?? ''),
            'from_name' => filterInput($_POST['from_name'] ?? 'Panadite Academy'),
            'smtp_auth' => isset($_POST['smtp_auth']) ? '1' : '0'
        ];
        
        // Save each setting
        foreach ($smtpSettings as $key => $value) {
            $sql = "INSERT INTO system_settings (setting_key, setting_value, setting_group, description) 
                    VALUES (?, ?, 'email', ?) 
                    ON DUPLICATE KEY UPDATE setting_value = ?, updated_at = CURRENT_TIMESTAMP";
            
            $descriptions = [
                'smtp_enabled' => 'Enable SMTP Email System',
                'smtp_host' => 'SMTP Server Host',
                'smtp_port' => 'SMTP Port Number',
                'smtp_encryption' => 'SMTP Encryption Method',
                'smtp_username' => 'SMTP Username',
                'smtp_password' => 'SMTP Password',
                'from_email' => 'Default From Email Address',
                'from_name' => 'Default From Name',
                'smtp_auth' => 'Enable SMTP Authentication'
            ];
            
            $description = $descriptions[$key] ?? $key;
            
            if ($pdo) {
                $stmt = $pdo->prepare($sql);
                $stmt->execute([$key, $value, $description, $value]);
            }
        }
        
        setFlashMessage('success', 'SMTP email settings saved successfully!');
        header('Location: email_settings.php');
        exit;
        
    } catch (Exception $e) {
        setFlashMessage('danger', 'Error saving settings: ' . $e->getMessage());
    }
}

// Handle test email
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['test_email'])) {
    $testEmail = filterInput($_POST['test_email_address'] ?? '');
    
    if (!empty($testEmail) && filter_var($testEmail, FILTER_VALIDATE_EMAIL)) {
        // Send test email using fast service
        $result = sendTestEmail($testEmail);
        
        if ($result) {
            setFlashMessage('success', '⚡ Test email sent instantly to ' . $testEmail . '! Check your inbox.');
        } else {
            setFlashMessage('danger', 'Failed to send test email. Please check your SMTP configuration.');
        }
    } else {
        setFlashMessage('warning', 'Please enter a valid email address for testing.');
    }
    
    header('Location: email_settings.php');
    exit;
}

// Get current SMTP settings
$currentSettings = [];
if ($pdo) {
    try {
        $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM system_settings WHERE setting_group = 'email'");
        $stmt->execute();
        $settings = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($settings as $setting) {
            $currentSettings[$setting['setting_key']] = $setting['setting_value'];
        }
    } catch (Exception $e) {
        // Table might not exist yet, that's okay
    }
}

// Default values
$defaults = [
    'smtp_enabled' => '0',
    'smtp_host' => '',
    'smtp_port' => '587',
    'smtp_encryption' => 'tls',
    'smtp_username' => '',
    'smtp_password' => '',
    'from_email' => '',
    'from_name' => 'Panadite Academy',
    'smtp_auth' => '1'
];

foreach ($defaults as $key => $default) {
    if (!isset($currentSettings[$key])) {
        $currentSettings[$key] = $default;
    }
}



$currentUser = getCurrentUser();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Email Settings - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h1 class="page-title">
                    <i class='bx bx-envelope'></i>
                    SMTP Email Configuration
                </h1>
                <div class="status-indicator">
                    <span class="badge <?php echo $currentSettings['smtp_enabled'] === '1' ? 'bg-success' : 'bg-danger'; ?>">
                        <?php echo $currentSettings['smtp_enabled'] === '1' ? '✅ Enabled' : '❌ Disabled'; ?>
                    </span>
                </div>
            </div>

            <?php echo displayFlashMessage(); ?>

            <div class="alert alert-info">
                <h5><i class='bx bx-info-circle'></i> SMTP Email System</h5>
                <p class="mb-0">Configure your SMTP email settings to enable automatic email notifications, password resets, and system communications. Make sure to test your configuration after saving.</p>
            </div>

                <div class="row">
                    <!-- SMTP Configuration Form -->
                    <div class="col-lg-8">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0"><i class="fas fa-server me-2"></i>SMTP Server Configuration</h5>
                            </div>
                            <div class="card-body">
                                <form method="POST" action="">
                                    <!-- Enable SMTP -->
                                    <div class="row mb-4">
                                        <div class="col-12">
                                            <div class="form-check form-switch">
                                                <input class="form-check-input" type="checkbox" id="smtp_enabled" 
                                                       name="smtp_enabled" <?php echo $currentSettings['smtp_enabled'] === '1' ? 'checked' : ''; ?>>
                                                <label class="form-check-label" for="smtp_enabled">
                                                    <strong>Enable SMTP Email System</strong>
                                                </label>
                                            </div>
                                            <small class="form-text text-muted">Turn on to use SMTP for sending emails</small>
                                        </div>
                                    </div>

                                    <div class="row mb-3">
                                        <div class="col-md-8">
                                            <label for="smtp_host" class="form-label">
                                                <i class="fas fa-server text-primary me-1"></i>SMTP Host
                                            </label>
                                            <input type="text" class="form-control" id="smtp_host" name="smtp_host" 
                                                   value="<?php echo htmlspecialchars($currentSettings['smtp_host']); ?>"
                                                   placeholder="smtp.gmail.com" required>
                                            <small class="form-text text-muted">Your email provider's SMTP server</small>
                                        </div>
                                        <div class="col-md-4">
                                            <label for="smtp_port" class="form-label">
                                                <i class="fas fa-plug text-primary me-1"></i>Port
                                            </label>
                                            <input type="number" class="form-control" id="smtp_port" name="smtp_port" 
                                                   value="<?php echo htmlspecialchars($currentSettings['smtp_port']); ?>"
                                                   placeholder="587" required>
                                        </div>
                                    </div>

                                    <div class="row mb-3">
                                        <div class="col-md-6">
                                            <label for="smtp_encryption" class="form-label">
                                                <i class="fas fa-shield-alt text-primary me-1"></i>Encryption
                                            </label>
                                            <select class="form-select" id="smtp_encryption" name="smtp_encryption" required>
                                                <option value="tls" <?php echo $currentSettings['smtp_encryption'] === 'tls' ? 'selected' : ''; ?>>TLS (Recommended)</option>
                                                <option value="ssl" <?php echo $currentSettings['smtp_encryption'] === 'ssl' ? 'selected' : ''; ?>>SSL</option>
                                                <option value="" <?php echo $currentSettings['smtp_encryption'] === '' ? 'selected' : ''; ?>>None</option>
                                            </select>
                                        </div>
                                        <div class="col-md-6">
                                            <label class="form-label">
                                                <i class="fas fa-key text-primary me-1"></i>Authentication
                                            </label>
                                            <div class="form-check form-switch mt-2">
                                                <input class="form-check-input" type="checkbox" id="smtp_auth" 
                                                       name="smtp_auth" <?php echo $currentSettings['smtp_auth'] === '1' ? 'checked' : ''; ?>>
                                                <label class="form-check-label" for="smtp_auth">
                                                    Require Authentication
                                                </label>
                                            </div>
                                        </div>
                                    </div>

                                    <div class="row mb-3">
                                        <div class="col-md-6">
                                            <label for="smtp_username" class="form-label">
                                                <i class="fas fa-user text-primary me-1"></i>Username
                                            </label>
                                            <input type="text" class="form-control" id="smtp_username" name="smtp_username" 
                                                   value="<?php echo htmlspecialchars($currentSettings['smtp_username']); ?>"
                                                   placeholder="your-email@domain.com">
                                        </div>
                                        <div class="col-md-6">
                                            <label for="smtp_password" class="form-label">
                                                <i class="fas fa-lock text-primary me-1"></i>Password
                                            </label>
                                            <input type="password" class="form-control" id="smtp_password" name="smtp_password" 
                                                   value="<?php echo htmlspecialchars($currentSettings['smtp_password']); ?>"
                                                   placeholder="Enter password">
                                        </div>
                                    </div>

                                    <div class="row mb-4">
                                        <div class="col-md-6">
                                            <label for="from_email" class="form-label">
                                                <i class="fas fa-envelope text-primary me-1"></i>From Email
                                            </label>
                                            <input type="email" class="form-control" id="from_email" name="from_email" 
                                                   value="<?php echo htmlspecialchars($currentSettings['from_email']); ?>"
                                                   placeholder="noreply@panaditeacademy.com" required>
                                        </div>
                                        <div class="col-md-6">
                                            <label for="from_name" class="form-label">
                                                <i class="fas fa-signature text-primary me-1"></i>From Name
                                            </label>
                                            <input type="text" class="form-control" id="from_name" name="from_name" 
                                                   value="<?php echo htmlspecialchars($currentSettings['from_name']); ?>"
                                                   placeholder="Panadite Academy" required>
                                        </div>
                                    </div>

                                    <div class="text-end">
                                        <button type="submit" name="save_smtp_settings" class="btn btn-primary btn-lg">
                                            <i class="fas fa-save me-2"></i>Save Configuration
                                        </button>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>

                    <!-- Test Email Section -->
                    <div class="col-lg-4">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0"><i class="fas fa-paper-plane me-2"></i>Test Email</h5>
                            </div>
                            <div class="card-body">
                                <?php if ($currentSettings['smtp_enabled'] === '1'): ?>
                                    <form method="POST" action="">
                                        <div class="mb-3">
                                            <label for="test_email_address" class="form-label">Test Email Address</label>
                                            <input type="email" class="form-control" id="test_email_address" 
                                                   name="test_email_address" placeholder="test@example.com" required>
                                        </div>
                                        <button type="submit" name="test_email" class="btn btn-success w-100">
                                            <i class="fas fa-paper-plane me-2"></i>Send Test Email
                                        </button>
                                    </form>
                                    
                                    <div class="mt-3 p-3 bg-light rounded">
                                        <small class="text-muted">
                                            <i class="fas fa-info-circle me-1"></i>
                                            This will send a test email to verify your SMTP configuration is working correctly.
                                        </small>
                                    </div>
                                <?php else: ?>
                                    <div class="alert alert-warning">
                                        <i class="fas fa-exclamation-triangle me-2"></i>
                                        Please enable SMTP first, then save the configuration to test email functionality.
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>

                        <!-- Quick Setup Guide -->
                        <div class="card mt-4">
                            <div class="card-header">
                                <h6 class="mb-0"><i class="fas fa-question-circle me-2"></i>Quick Setup Guide</h6>
                            </div>
                            <div class="card-body">
                                <div class="small">
                                    <h6 class="text-primary">📧 Gmail SMTP:</h6>
                                    <ul class="list-unstyled ps-3">
                                        <li>Host: smtp.gmail.com</li>
                                        <li>Port: 587</li>
                                        <li>Encryption: TLS</li>
                                        <li>Use App Password</li>
                                    </ul>
                                    
                                    <h6 class="text-primary">📧 Outlook/Hotmail:</h6>
                                    <ul class="list-unstyled ps-3">
                                        <li>Host: smtp-mail.outlook.com</li>
                                        <li>Port: 587</li>
                                        <li>Encryption: TLS</li>
                                    </ul>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
        </div>
    </section>

    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="/assets/js/admin-dashboard.js"></script>
    
    <script>
        // Auto-hide success messages after 5 seconds
        setTimeout(function() {
            const alerts = document.querySelectorAll('.alert-success');
            alerts.forEach(alert => {
                if (alert.querySelector('.btn-close')) {
                    alert.querySelector('.btn-close').click();
                }
            });
        }, 5000);
    </script>
</body>
</html>
