<?php
session_start();
require_once '../config/db_connect.php';
require_once '../config/functions.php';

// Ensure only admin or elevated users can access this
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_role']) || 
    ($_SESSION['user_role'] !== 'admin' && $_SESSION['user_role'] !== 'teacher')) {
    header("Location: ../index.php");
    exit;
}

// Handle delete request
if (isset($_POST['delete_course']) && isset($_POST['course_id'])) {
    $course_id = (int)$_POST['course_id'];
    
    try {
        // Begin transaction
        $conn->begin_transaction();
        
        // Delete related records first
        // Enrollments
        $conn->query("DELETE FROM enrollments WHERE course_id = $course_id");
        
        // Videos
        $conn->query("DELETE FROM videos WHERE course_id = $course_id");
        
        // Quizzes and submissions
        $quiz_ids_result = $conn->query("SELECT quiz_id FROM quizzes WHERE course_id = $course_id");
        if ($quiz_ids_result) {
            while ($quiz = $quiz_ids_result->fetch_assoc()) {
                $quiz_id = $quiz['quiz_id'];
                $conn->query("DELETE FROM quiz_submissions WHERE quiz_id = $quiz_id");
            }
        }
        $conn->query("DELETE FROM quizzes WHERE course_id = $course_id");
        
        // Finally delete the course
        $conn->query("DELETE FROM courses WHERE course_id = $course_id");
        
        // Commit transaction
        $conn->commit();
        
        $success_message = "Course and all related data successfully deleted.";
    } catch (Exception $e) {
        // Rollback in case of error
        $conn->rollback();
        $error_message = "Error deleting course: " . $e->getMessage();
    }
}

// Get all courses
// Initialize courses array to prevent undefined variable error
$courses = [];

// First, check if the categories table exists and what its column name is
$categoryField = 'name';
$categoryJoin = '';

// Check if categories table exists
$tablesResult = $conn->query("SHOW TABLES LIKE 'categories'");
if ($tablesResult && $tablesResult->num_rows > 0) {
    // Check columns in categories table
    $columnsResult = $conn->query("SHOW COLUMNS FROM categories");
    $hasNameColumn = false;
    if ($columnsResult) {
        while ($column = $columnsResult->fetch_assoc()) {
            if ($column['Field'] == 'name' || $column['Field'] == 'category_name' || $column['Field'] == 'title') {
                $categoryField = $column['Field'];
                $hasNameColumn = true;
                break;
            }
        }
    }
    
    if ($hasNameColumn) {
        $categoryJoin = "LEFT JOIN categories cat ON c.category_id = cat.category_id";
    }
}

// Check if enrollments table exists
$enrollmentsExists = false;
$enrollmentsQuery = '';
$tablesResult = $conn->query("SHOW TABLES LIKE 'enrollments'");
if ($tablesResult && $tablesResult->num_rows > 0) {
    $enrollmentsExists = true;
    $enrollmentsQuery = "(SELECT COUNT(*) FROM enrollments WHERE course_id = c.course_id) as student_count";
} else {
    $enrollmentsQuery = "0 as student_count";
}

try {
    // Build the query based on the database structure
    $query = "SELECT c.*, u.first_name, u.last_name";
    
    // Add category name if available
    if (!empty($categoryJoin)) {
        $query .= ", cat.{$categoryField} as category_name";
    } else {
        $query .= ", NULL as category_name";
    }
    
    // Add enrollment count
    $query .= ", {$enrollmentsQuery}
               FROM courses c
               LEFT JOIN users u ON c.teacher_id = u.user_id ";
    
    // Add category join if available
    if (!empty($categoryJoin)) {
        $query .= $categoryJoin;
    }
    
    $query .= " ORDER BY c.created_at DESC";
    
    $result = $conn->query($query);
    
    if ($result && $result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $courses[] = $row;
        }
    }
} catch (Exception $e) {
    $error_message = "Error fetching courses: " . $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Course Management - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary: #4e73df;
            --secondary: #858796;
            --success: #1cc88a;
            --info: #36b9cc;
            --warning: #f6c23e;
            --danger: #e74a3b;
            --light: #f8f9fc;
            --dark: #5a5c69;
        }
        
        body {
            background-color: var(--light);
            font-family: 'Nunito', -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
            padding: 20px;
        }
        
        .card {
            border: none;
            box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.15);
            margin-bottom: 1.5rem;
        }
        
        .card-header {
            background-color: white;
            border-bottom: 1px solid #e3e6f0;
            padding: 1.25rem;
        }
        
        .table th {
            font-weight: 600;
            color: var(--dark);
            background-color: var(--light);
        }
        
        .table-hover tbody tr:hover {
            background-color: rgba(78, 115, 223, 0.05);
        }
        
        .badge {
            font-weight: 600;
            padding: 0.35em 0.65em;
        }
        
        .course-thumbnail {
            width: 60px;
            height: 60px;
            object-fit: cover;
            border-radius: 5px;
        }
        
        .btn-circle {
            border-radius: 100%;
            width: 2.5rem;
            height: 2.5rem;
            font-size: 1rem;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 0;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="d-sm-flex align-items-center justify-content-between mb-4">
            <h1 class="h3 mb-0 text-gray-800">Course Management</h1>
            <a href="../index.php" class="btn btn-sm btn-primary shadow-sm">
                <i class="fas fa-home fa-sm text-white-50"></i> Back to Home
            </a>
        </div>
        
        <?php if (isset($error_message)): ?>
            <div class="alert alert-danger"><?php echo $error_message; ?></div>
        <?php endif; ?>
        
        <?php if (isset($success_message)): ?>
            <div class="alert alert-success"><?php echo $success_message; ?></div>
        <?php endif; ?>
        
        <div class="card shadow">
            <div class="card-header py-3 d-flex align-items-center justify-content-between">
                <h6 class="m-0 font-weight-bold text-primary">All Courses</h6>
                <span class="badge bg-primary"><?php echo count($courses); ?> Courses</span>
            </div>
            <div class="card-body">
                <?php if (empty($courses)): ?>
                    <div class="alert alert-info">No courses found in the database.</div>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Thumbnail</th>
                                    <th>Title</th>
                                    <th>Instructor</th>
                                    <th>Category</th>
                                    <th>Students</th>
                                    <th>Price</th>
                                    <th>Status</th>
                                    <th>Created</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($courses as $course): ?>
                                    <tr>
                                        <td><?php echo $course['course_id']; ?></td>
                                        <td>
                                            <?php if (!empty($course['thumbnail'])): ?>
                                                <img src="<?php echo htmlspecialchars($course['thumbnail']); ?>" 
                                                     class="course-thumbnail" alt="Thumbnail" 
                                                     onerror="this.src='../assets/images/placeholder.png';">
                                            <?php else: ?>
                                                <div class="course-thumbnail bg-light d-flex align-items-center justify-content-center">
                                                    <i class="fas fa-book text-primary"></i>
                                                </div>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <strong><?php echo htmlspecialchars($course['title']); ?></strong>
                                            <?php if (!empty($course['short_description'])): ?>
                                                <div class="small text-muted text-truncate" style="max-width: 200px;">
                                                    <?php echo htmlspecialchars($course['short_description']); ?>
                                                </div>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php 
                                            if (!empty($course['first_name']) && !empty($course['last_name'])) {
                                                echo htmlspecialchars($course['first_name'] . ' ' . $course['last_name']);
                                            } else {
                                                echo '<span class="text-muted">Unknown</span>';
                                            }
                                            ?>
                                        </td>
                                        <td>
                                            <?php echo !empty($course['category_name']) ? 
                                                htmlspecialchars($course['category_name']) : 
                                                '<span class="text-muted">Uncategorized</span>'; ?>
                                        </td>
                                        <td>
                                            <span class="badge bg-info">
                                                <?php echo number_format($course['student_count'] ?? 0); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php if (isset($course['price']) && $course['price'] == 0): ?>
                                                <span class="badge bg-success">Free</span>
                                            <?php else: ?>
                                                <span class="badge bg-primary">
                                                    R<?php echo number_format($course['price'] ?? 0, 2); ?>
                                                </span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if (isset($course['is_published']) && $course['is_published']): ?>
                                                <span class="badge bg-success">Published</span>
                                            <?php else: ?>
                                                <span class="badge bg-secondary">Draft</span>
                                            <?php endif; ?>
                                            
                                            <?php if (isset($course['is_approved']) && $course['is_approved'] == 0): ?>
                                                <span class="badge bg-warning">Pending Approval</span>
                                            <?php elseif (isset($course['is_approved']) && $course['is_approved'] == 1): ?>
                                                <span class="badge bg-info">Approved</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php 
                                            if (!empty($course['created_at'])) {
                                                echo date('M d, Y', strtotime($course['created_at']));
                                            } else {
                                                echo '<span class="text-muted">Unknown</span>';
                                            }
                                            ?>
                                        </td>
                                        <td>
                                            <div class="d-flex">
                                                <a href="../course.php?id=<?php echo $course['course_id']; ?>" 
                                                   class="btn btn-primary btn-sm me-1" title="View Course">
                                                    <i class="fas fa-eye"></i>
                                                </a>
                                                
                                                <form method="post" onsubmit="return confirm('Are you sure you want to delete this course? This will also delete all related videos, quizzes, and enrollments. This action cannot be undone.');">
                                                    <input type="hidden" name="course_id" value="<?php echo $course['course_id']; ?>">
                                                    <button type="submit" name="delete_course" class="btn btn-danger btn-sm" title="Delete Course">
                                                        <i class="fas fa-trash-alt"></i>
                                                    </button>
                                                </form>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
