<?php
require_once __DIR__ . '/../auth/functions.php';

// Ensure user is logged in as an admin
requireRole('admin');

// Handle form submissions
$message = '';
$messageType = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'add':
                $company_name = $_POST['company_name'];
                $website_url = $_POST['website_url'];
                $description = $_POST['description'];
                $display_order = $_POST['display_order'];
                
                // Handle file upload
                if (isset($_FILES['logo']) && $_FILES['logo']['error'] === 0) {
                    $uploadDir = '../assets/images/partnerships/';
                    
                    // Create directory if it doesn't exist
                    if (!is_dir($uploadDir)) {
                        mkdir($uploadDir, 0755, true);
                    }
                    
                    $fileExtension = strtolower(pathinfo($_FILES['logo']['name'], PATHINFO_EXTENSION));
                    $allowedTypes = ['jpg', 'jpeg', 'png', 'svg', 'gif'];
                    
                    if (in_array($fileExtension, $allowedTypes)) {
                        $fileName = uniqid('partnership_') . '.' . $fileExtension;
                        $filePath = $uploadDir . $fileName;
                        
                        if (move_uploaded_file($_FILES['logo']['tmp_name'], $filePath)) {
                            $logoPath = 'assets/images/partnerships/' . $fileName;
                            
                            $stmt = $conn->prepare("INSERT INTO partnerships (company_name, logo_path, website_url, description, display_order) VALUES (?, ?, ?, ?, ?)");
                            $stmt->bind_param("ssssi", $company_name, $logoPath, $website_url, $description, $display_order);
                            
                            if ($stmt->execute()) {
                                $message = "Partnership added successfully!";
                                $messageType = "success";
                            } else {
                                $message = "Error adding partnership to database.";
                                $messageType = "danger";
                            }
                        } else {
                            $message = "Error uploading logo file.";
                            $messageType = "danger";
                        }
                    } else {
                        $message = "Invalid file type. Please upload JPG, PNG, SVG, or GIF files only.";
                        $messageType = "danger";
                    }
                } else {
                    $message = "Please select a logo file.";
                    $messageType = "danger";
                }
                break;
                
            case 'edit':
                $id = $_POST['partnership_id'];
                $company_name = $_POST['company_name'];
                $website_url = $_POST['website_url'];
                $description = $_POST['description'];
                $display_order = $_POST['display_order'];
                $is_active = isset($_POST['is_active']) ? 1 : 0;
                
                // Handle logo update
                $logoPath = $_POST['existing_logo'];
                if (isset($_FILES['logo']) && $_FILES['logo']['error'] === 0) {
                    $uploadDir = '../assets/images/partnerships/';
                    $fileExtension = strtolower(pathinfo($_FILES['logo']['name'], PATHINFO_EXTENSION));
                    $allowedTypes = ['jpg', 'jpeg', 'png', 'svg', 'gif'];
                    
                    if (in_array($fileExtension, $allowedTypes)) {
                        // Delete old file
                        if (file_exists('../' . $logoPath)) {
                            unlink('../' . $logoPath);
                        }
                        
                        $fileName = uniqid('partnership_') . '.' . $fileExtension;
                        $filePath = $uploadDir . $fileName;
                        
                        if (move_uploaded_file($_FILES['logo']['tmp_name'], $filePath)) {
                            $logoPath = 'assets/images/partnerships/' . $fileName;
                        }
                    }
                }
                
                $stmt = $conn->prepare("UPDATE partnerships SET company_name = ?, logo_path = ?, website_url = ?, description = ?, display_order = ?, is_active = ? WHERE id = ?");
                $stmt->bind_param("ssssiii", $company_name, $logoPath, $website_url, $description, $display_order, $is_active, $id);
                
                if ($stmt->execute()) {
                    $message = "Partnership updated successfully!";
                    $messageType = "success";
                } else {
                    $message = "Error updating partnership.";
                    $messageType = "danger";
                }
                break;
                
            case 'delete':
                $id = $_POST['partnership_id'];
                
                // Get logo path to delete file
                $stmt = $conn->prepare("SELECT logo_path FROM partnerships WHERE id = ?");
                $stmt->bind_param("i", $id);
                $stmt->execute();
                $result = $stmt->get_result();
                
                if ($result && $row = $result->fetch_assoc()) {
                    $logoPath = '../' . $row['logo_path'];
                    if (file_exists($logoPath)) {
                        unlink($logoPath);
                    }
                    
                    $stmt = $conn->prepare("DELETE FROM partnerships WHERE id = ?");
                    $stmt->bind_param("i", $id);
                    
                    if ($stmt->execute()) {
                        $message = "Partnership deleted successfully!";
                        $messageType = "success";
                    } else {
                        $message = "Error deleting partnership.";
                        $messageType = "danger";
                    }
                }
                break;
        }
    }
}

// Get all partnerships
$partnershipsQuery = "SELECT * FROM partnerships ORDER BY display_order ASC, company_name ASC";
$partnershipsResult = $conn->query($partnershipsQuery);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Partnerships Management - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    
    <style>
        .partnership-logo {
            max-height: 60px;
            max-width: 120px;
            object-fit: contain;
            border-radius: 8px;
            border: 1px solid #e0e0e0;
            padding: 8px;
            background: white;
        }
        
        .card-hover {
            transition: all 0.3s ease;
        }
        
        .card-hover:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 20px rgba(0,0,0,0.1);
        }
        
        .btn-action {
            padding: 6px 12px;
            font-size: 12px;
            border-radius: 6px;
            margin: 2px;
        }
        
        .order-badge {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            font-size: 12px;
            padding: 4px 8px;
            border-radius: 12px;
        }
        
        .status-active {
            background: linear-gradient(135deg, #11998e 0%, #38ef7d 100%);
        }
        
        .status-inactive {
            background: linear-gradient(135deg, #fc4a1a 0%, #f7b733 100%);
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <!-- Page Header -->
            <div class="page-header">
                <h1>Partnership Management</h1>
                <div>
                    <button class="btn btn-sm btn-primary" data-bs-toggle="modal" data-bs-target="#addPartnershipModal">
                        <i class='bx bx-plus'></i> Add Partnership
                    </button>
                </div>
            </div>

            <!-- Alert Messages -->
            <?php if (!empty($message)): ?>
            <div class="alert alert-<?php echo $messageType; ?> alert-dismissible fade show" role="alert">
                <i class="bx bx-check me-1"></i> <?php echo htmlspecialchars($message); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
            <?php endif; ?>

            <!-- Partnerships Grid -->
            <?php if ($partnershipsResult && $partnershipsResult->num_rows > 0): ?>
            <div class="partnerships-grid" style="display: grid; grid-template-columns: repeat(auto-fill, minmax(280px, 1fr)); gap: 20px; margin-top: 20px;">
                <?php while ($partnership = $partnershipsResult->fetch_assoc()): ?>
                <div class="partnership-card" style="background: white; border-radius: 12px; padding: 20px; box-shadow: 0 4px 12px rgba(0,0,0,0.1); transition: transform 0.3s ease;"
                     onmouseover="this.style.transform='translateY(-2px)'" onmouseout="this.style.transform='translateY(0)'">
                    
                    <!-- Logo -->
                    <div class="text-center mb-3">
                        <img src="../<?php echo htmlspecialchars($partnership['logo_path']); ?>" 
                             alt="<?php echo htmlspecialchars($partnership['company_name']); ?>" 
                             class="partnership-logo"
                             onerror="this.src='../assets/images/placeholder-logo.png'">
                    </div>
                    
                    <!-- Company Info -->
                    <div class="text-center mb-3">
                        <h6 style="font-weight: 600; margin-bottom: 5px;"><?php echo htmlspecialchars($partnership['company_name']); ?></h6>
                        <?php if (!empty($partnership['website_url'])): ?>
                        <a href="<?php echo htmlspecialchars($partnership['website_url']); ?>" 
                           target="_blank" style="color: #4361ee; text-decoration: none; font-size: 12px;">
                            <i class="bx bx-link-external"></i> Visit Website
                        </a>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Status & Order -->
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
                        <span style="background: #4361ee; color: white; padding: 4px 8px; border-radius: 12px; font-size: 11px;">
                            Order: <?php echo $partnership['display_order']; ?>
                        </span>
                        <span style="background: <?php echo $partnership['is_active'] ? '#28a745' : '#dc3545'; ?>; color: white; padding: 4px 8px; border-radius: 12px; font-size: 11px;">
                            <?php echo $partnership['is_active'] ? 'Active' : 'Inactive'; ?>
                        </span>
                    </div>
                    
                    <!-- Description -->
                    <?php if (!empty($partnership['description'])): ?>
                    <p style="color: #6c757d; font-size: 11px; margin-bottom: 15px; line-height: 1.4;">
                        <?php echo htmlspecialchars(substr($partnership['description'], 0, 80)); ?>
                        <?php echo strlen($partnership['description']) > 80 ? '...' : ''; ?>
                    </p>
                    <?php endif; ?>
                    
                    <!-- Action Buttons -->
                    <div style="display: flex; gap: 8px;">
                        <button class="btn btn-outline-primary btn-sm" 
                                style="flex: 1; font-size: 11px; padding: 6px 12px;"
                                data-bs-toggle="modal" 
                                data-bs-target="#editPartnershipModal"
                                data-id="<?php echo $partnership['id']; ?>"
                                data-name="<?php echo htmlspecialchars($partnership['company_name']); ?>"
                                data-url="<?php echo htmlspecialchars($partnership['website_url']); ?>"
                                data-description="<?php echo htmlspecialchars($partnership['description']); ?>"
                                data-order="<?php echo $partnership['display_order']; ?>"
                                data-active="<?php echo $partnership['is_active']; ?>"
                                data-logo="<?php echo htmlspecialchars($partnership['logo_path']); ?>">
                            <i class="bx bx-edit"></i> Edit
                        </button>
                        <button class="btn btn-outline-danger btn-sm" 
                                style="font-size: 11px; padding: 6px 12px;"
                                data-bs-toggle="modal" 
                                data-bs-target="#deletePartnershipModal"
                                data-id="<?php echo $partnership['id']; ?>"
                                data-name="<?php echo htmlspecialchars($partnership['company_name']); ?>">
                            <i class="bx bx-trash"></i>
                        </button>
                    </div>
                </div>
                <?php endwhile; ?>
            </div>
            <?php else: ?>
            <div class="empty-state" style="text-align: center; padding: 60px 20px; background: white; border-radius: 12px; box-shadow: 0 4px 12px rgba(0,0,0,0.1);">
                <i class="bx bx-buildings" style="font-size: 4rem; color: #adb5bd; margin-bottom: 20px;"></i>
                <h5 style="color: #6c757d; margin-bottom: 15px;">No Partnerships Found</h5>
                <p style="color: #adb5bd; margin-bottom: 20px;">Start by adding your first partnership to showcase your company collaborations.</p>
                <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addPartnershipModal">
                    <i class="bx bx-plus"></i> Add First Partnership
                </button>
            </div>
            <?php endif; ?>
        </div>
    </section>

    <!-- Add Partnership Modal -->
    <div class="modal fade" id="addPartnershipModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content border-0">
                <div class="modal-header bg-primary text-white">
                    <h5 class="modal-title"><i class="bx bx-plus me-2"></i>Add New Partnership</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST" enctype="multipart/form-data">
                    <input type="hidden" name="action" value="add">
                    <div class="modal-body p-4">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="company_name" class="form-label fw-bold">Company Name *</label>
                                    <input type="text" class="form-control" id="company_name" name="company_name" required>
                                </div>
                                <div class="mb-3">
                                    <label for="website_url" class="form-label fw-bold">Website URL</label>
                                    <input type="url" class="form-control" id="website_url" name="website_url" placeholder="https://www.company.com">
                                </div>
                                <div class="mb-3">
                                    <label for="display_order" class="form-label fw-bold">Display Order</label>
                                    <input type="number" class="form-control" id="display_order" name="display_order" value="0" min="0">
                                    <small class="text-muted">Lower numbers appear first</small>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="logo" class="form-label fw-bold">Company Logo *</label>
                                    <input type="file" class="form-control" id="logo" name="logo" accept="image/*" required>
                                    <small class="text-muted">Recommended: PNG, SVG, or JPG format</small>
                                </div>
                                <div class="mb-3">
                                    <label for="description" class="form-label fw-bold">Description</label>
                                    <textarea class="form-control" id="description" name="description" rows="4" placeholder="Brief description about the company..."></textarea>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">
                            <i class="bx bx-save me-1"></i> Add Partnership
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Edit Partnership Modal -->
    <div class="modal fade" id="editPartnershipModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content border-0">
                <div class="modal-header bg-warning text-white">
                    <h5 class="modal-title"><i class="bx bx-edit me-2"></i>Edit Partnership</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST" enctype="multipart/form-data">
                    <input type="hidden" name="action" value="edit">
                    <input type="hidden" name="partnership_id" id="edit_partnership_id">
                    <input type="hidden" name="existing_logo" id="edit_existing_logo">
                    <div class="modal-body p-4">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="edit_company_name" class="form-label fw-bold">Company Name *</label>
                                    <input type="text" class="form-control" id="edit_company_name" name="company_name" required>
                                </div>
                                <div class="mb-3">
                                    <label for="edit_website_url" class="form-label fw-bold">Website URL</label>
                                    <input type="url" class="form-control" id="edit_website_url" name="website_url">
                                </div>
                                <div class="mb-3">
                                    <label for="edit_display_order" class="form-label fw-bold">Display Order</label>
                                    <input type="number" class="form-control" id="edit_display_order" name="display_order" min="0">
                                </div>
                                <div class="mb-3">
                                    <div class="form-check">
                                        <input type="checkbox" class="form-check-input" id="edit_is_active" name="is_active">
                                        <label class="form-check-label" for="edit_is_active">Active</label>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="edit_logo" class="form-label fw-bold">Company Logo</label>
                                    <input type="file" class="form-control" id="edit_logo" name="logo" accept="image/*">
                                    <small class="text-muted">Leave empty to keep current logo</small>
                                </div>
                                <div class="mb-3">
                                    <label for="edit_description" class="form-label fw-bold">Description</label>
                                    <textarea class="form-control" id="edit_description" name="description" rows="4"></textarea>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-warning">
                            <i class="bx bx-save me-1"></i> Update Partnership
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Delete Partnership Modal -->
    <div class="modal fade" id="deletePartnershipModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content border-0">
                <div class="modal-header bg-danger text-white">
                    <h5 class="modal-title"><i class="bx bx-trash me-2"></i>Delete Partnership</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="partnership_id" id="delete_partnership_id">
                    <div class="modal-body p-4 text-center">
                        <i class="bx bx-error-circle display-1 text-danger mb-3"></i>
                        <h5 class="mb-3">Are you sure you want to delete this partnership?</h5>
                        <p class="text-muted">Company: <strong id="delete_company_name"></strong></p>
                        <p class="text-danger">This action cannot be undone.</p>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-danger">
                            <i class="bx bx-trash me-1"></i> Delete Partnership
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="/assets/js/admin-dashboard.js"></script>
    <script>
        // Edit Partnership Modal
        document.getElementById('editPartnershipModal').addEventListener('show.bs.modal', function (event) {
            const button = event.relatedTarget;
            document.getElementById('edit_partnership_id').value = button.getAttribute('data-id');
            document.getElementById('edit_company_name').value = button.getAttribute('data-name');
            document.getElementById('edit_website_url').value = button.getAttribute('data-url');
            document.getElementById('edit_description').value = button.getAttribute('data-description');
            document.getElementById('edit_display_order').value = button.getAttribute('data-order');
            document.getElementById('edit_existing_logo').value = button.getAttribute('data-logo');
            document.getElementById('edit_is_active').checked = button.getAttribute('data-active') === '1';
        });

        // Delete Partnership Modal
        document.getElementById('deletePartnershipModal').addEventListener('show.bs.modal', function (event) {
            const button = event.relatedTarget;
            document.getElementById('delete_partnership_id').value = button.getAttribute('data-id');
            document.getElementById('delete_company_name').textContent = button.getAttribute('data-name');
        });
    </script>
</body>
</html>
