<?php
/**
 * PDF Certificate Generator for Panadite Academy
 * Converts beautiful HTML certificate to downloadable PDF
 */

require_once '../config/db_connect.php';

class PdfCertificateGenerator {
    
    private $conn;
    
    public function __construct($connection) {
        $this->conn = $connection;
    }
    
    /**
     * Generate certificate HTML optimized for PDF (can use complex CSS)
     */
    private function generatePdfCertificateHtml($certificate_number, $student_name, $course_title, $issue_date) {
        return "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <style>
                @page {
                    margin: 0;
                    size: A4 landscape;
                }
                
                body {
                    margin: 0;
                    padding: 0;
                    font-family: 'DejaVu Sans', Arial, sans-serif;
                    background: #ffffff;
                }
                
                .certificate {
                    width: 297mm;
                    height: 210mm;
                    position: relative;
                    background: linear-gradient(135deg, #f8faff 0%, #ffffff 100%);
                    overflow: hidden;
                    padding: 30px;
                    box-sizing: border-box;
                }
                
                /* Geometric corner decorations */
                .geo-corner-tl {
                    position: absolute;
                    top: 0;
                    left: 0;
                    width: 100px;
                    height: 100px;
                    background: linear-gradient(135deg, #4169E1 0%, #5B7CFA 100%);
                    clip-path: polygon(0 0, 100% 0, 0 100%);
                    opacity: 0.9;
                }
                
                .geo-corner-tr {
                    position: absolute;
                    top: 0;
                    right: 0;
                    width: 80px;
                    height: 80px;
                    background: linear-gradient(225deg, #FFD700 0%, #F4C430 100%);
                    clip-path: polygon(100% 0, 100% 100%, 0 0);
                    opacity: 0.9;
                }
                
                .geo-corner-bl {
                    position: absolute;
                    bottom: 0;
                    left: 0;
                    width: 70px;
                    height: 70px;
                    background: linear-gradient(45deg, #20B2AA 0%, #17a2b8 100%);
                    clip-path: polygon(0 0, 0 100%, 100% 100%);
                    opacity: 0.85;
                }
                
                .geo-corner-br {
                    position: absolute;
                    bottom: 0;
                    right: 0;
                    width: 60px;
                    height: 60px;
                    background: linear-gradient(315deg, #4169E1 0%, #20B2AA 100%);
                    clip-path: polygon(100% 0, 100% 100%, 0 100%);
                    opacity: 0.8;
                }
                
                /* Certificate border */
                .certificate-border {
                    position: absolute;
                    top: 20px;
                    left: 20px;
                    right: 20px;
                    bottom: 20px;
                    border: 3px solid #e9ecef;
                    border-radius: 8px;
                }
                
                /* Main content */
                .certificate-content {
                    position: relative;
                    z-index: 10;
                    text-align: center;
                    padding: 40px;
                    height: 100%;
                    display: flex;
                    flex-direction: column;
                    justify-content: space-between;
                }
                
                /* Header */
                .certificate-header {
                    margin-bottom: 20px;
                }
                
                .panadite-logo {
                    width: 80px;
                    height: 80px;
                    margin: 0 auto 15px;
                    background: linear-gradient(135deg, #4169E1, #20B2AA);
                    border-radius: 50%;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    color: white;
                    font-size: 32px;
                    font-weight: bold;
                    border: 3px solid #E5E7EB;
                }
                
                .academy-name {
                    font-size: 18px;
                    font-weight: 700;
                    color: #1F2937;
                    margin: 0;
                    letter-spacing: 3px;
                    text-transform: uppercase;
                }
                
                .academy-tagline {
                    font-size: 12px;
                    color: #6B7280;
                    margin: 8px 0 0 0;
                    font-weight: 400;
                    letter-spacing: 1px;
                }
                
                /* Certificate title */
                .certificate-title {
                    margin: 30px 0;
                }
                
                .title-main {
                    font-size: 48px;
                    font-weight: 800;
                    color: #111827;
                    margin: 0 0 10px 0;
                    letter-spacing: 2px;
                    text-transform: uppercase;
                }
                
                .title-sub {
                    font-size: 16px;
                    color: #4169E1;
                    font-weight: 600;
                    letter-spacing: 2px;
                    text-transform: uppercase;
                    margin: 0;
                }
                
                /* Main content */
                .certificate-main {
                    flex: 1;
                    display: flex;
                    flex-direction: column;
                    justify-content: center;
                    margin: 40px 0;
                }
                
                .certificate-text {
                    font-size: 18px;
                    color: #374151;
                    margin: 0 0 20px 0;
                    font-weight: 400;
                }
                
                .student-name {
                    font-size: 42px;
                    font-weight: 700;
                    color: #111827;
                    margin: 20px 0;
                    text-transform: capitalize;
                    border-bottom: 3px solid #4169E1;
                    padding-bottom: 10px;
                    display: inline-block;
                    min-width: 400px;
                }
                
                .course-title {
                    font-size: 28px;
                    font-weight: 600;
                    color: #4169E1;
                    margin: 30px 0 20px 0;
                    text-transform: uppercase;
                    letter-spacing: 1px;
                }
                
                /* Footer */
                .certificate-footer {
                    display: flex;
                    justify-content: space-between;
                    align-items: flex-end;
                    margin-top: 40px;
                    padding-top: 20px;
                    border-top: 2px solid #e9ecef;
                }
                
                .footer-left, .footer-right {
                    text-align: center;
                    flex: 1;
                }
                
                .footer-label {
                    font-size: 12px;
                    color: #6B7280;
                    font-weight: 600;
                    text-transform: uppercase;
                    letter-spacing: 1px;
                    margin-bottom: 5px;
                }
                
                .footer-value {
                    font-size: 14px;
                    color: #111827;
                    font-weight: 500;
                }
                
                .signature-line {
                    width: 200px;
                    height: 2px;
                    background: #4169E1;
                    margin: 0 auto 8px;
                }
                
                .signature-title {
                    font-size: 12px;
                    color: #6B7280;
                    font-weight: 600;
                    text-transform: uppercase;
                }
                
                /* Watermark */
                .watermark {
                    position: absolute;
                    top: 50%;
                    left: 50%;
                    transform: translate(-50%, -50%) rotate(-45deg);
                    font-size: 120px;
                    color: rgba(65, 105, 225, 0.05);
                    font-weight: 900;
                    z-index: 1;
                    pointer-events: none;
                    text-transform: uppercase;
                    letter-spacing: 20px;
                }
                
                /* Decorative elements */
                .decorative-line {
                    width: 80px;
                    height: 3px;
                    background: linear-gradient(90deg, #4169E1, #20B2AA, #FFD700);
                    margin: 20px auto;
                    border-radius: 2px;
                }
            </style>
        </head>
        <body>
            <div class='certificate'>
                <!-- Geometric corners -->
                <div class='geo-corner-tl'></div>
                <div class='geo-corner-tr'></div>
                <div class='geo-corner-bl'></div>
                <div class='geo-corner-br'></div>
                
                <!-- Certificate border -->
                <div class='certificate-border'></div>
                
                <!-- Watermark -->
                <div class='watermark'>PANADITE</div>
                
                <!-- Main content -->
                <div class='certificate-content'>
                    <!-- Header -->
                    <div class='certificate-header'>
                        <div class='panadite-logo'>PA</div>
                        <h2 class='academy-name'>Panadite Academy</h2>
                        <p class='academy-tagline'>Excellence in Education</p>
                    </div>
                    
                    <!-- Certificate title -->
                    <div class='certificate-title'>
                        <h1 class='title-main'>Certificate</h1>
                        <p class='title-sub'>of Completion</p>
                        <div class='decorative-line'></div>
                    </div>
                    
                    <!-- Main content -->
                    <div class='certificate-main'>
                        <p class='certificate-text'>This is to certify that</p>
                        <div class='student-name'>{$student_name}</div>
                        <p class='certificate-text'>has successfully completed the course</p>
                        <div class='course-title'>{$course_title}</div>
                        <p class='certificate-text'>demonstrating proficiency and dedication to learning excellence</p>
                    </div>
                    
                    <!-- Footer -->
                    <div class='certificate-footer'>
                        <div class='footer-left'>
                            <div class='footer-label'>Certificate Number</div>
                            <div class='footer-value'>{$certificate_number}</div>
                        </div>
                        
                        <div class='footer-center'>
                            <div class='signature-line'></div>
                            <div class='signature-title'>Panadite Academy</div>
                            <div style='font-size: 10px; color: #9CA3AF; margin-top: 5px;'>Digital Certificate</div>
                        </div>
                        
                        <div class='footer-right'>
                            <div class='footer-label'>Issue Date</div>
                            <div class='footer-value'>{$issue_date}</div>
                        </div>
                    </div>
                </div>
            </div>
        </body>
        </html>";
    }
    
    /**
     * Generate PDF certificate using HTML to PDF conversion
     * We'll use a simple HTML to PDF approach that works without external libraries
     */
    public function generatePdfCertificate($certificate_number, $student_name, $course_title, $issue_date) {
        // Get the certificate HTML
        $certificateHtml = $this->generatePdfCertificateHtml($certificate_number, $student_name, $course_title, $issue_date);
        
        // For now, we'll save the HTML and provide instructions for PDF conversion
        // In production, you could use mPDF, TCPDF, or wkhtmltopdf
        
        $certificatesDir = '../certificates/';
        if (!file_exists($certificatesDir)) {
            mkdir($certificatesDir, 0755, true);
        }
        
        $filename = "certificate_{$certificate_number}.html";
        $filepath = $certificatesDir . $filename;
        
        file_put_contents($filepath, $certificateHtml);
        
        return [
            'success' => true,
            'html_path' => $filepath,
            'filename' => $filename,
            'download_url' => "/certificates/{$filename}",
            'certificate_number' => $certificate_number
        ];
    }
    
    /**
     * Create PDF using basic HTML to PDF conversion (browser print to PDF)
     * This creates an HTML file optimized for PDF printing
     */
    public function createPrintableCertificate($certificate_number, $student_name, $course_title, $issue_date) {
        $result = $this->generatePdfCertificate($certificate_number, $student_name, $course_title, $issue_date);
        
        if ($result['success']) {
            // Try to save certificate record to database (optional - won't break if it fails)
            try {
                $this->saveCertificateRecord($certificate_number, $student_name, $course_title, $issue_date, $result['html_path']);
            } catch (Exception $e) {
                error_log("Certificate database save failed (non-critical): " . $e->getMessage());
                // Continue anyway - certificate file was created successfully
            }
        }
        
        return $result;
    }
    
    /**
     * Create certificate without database dependencies (simplified version)
     */
    public function createSimpleCertificate($certificate_number, $student_name, $course_title, $issue_date) {
        // Generate certificate HTML
        $certificateHtml = $this->generatePdfCertificateHtml($certificate_number, $student_name, $course_title, $issue_date);
        
        // Create certificates directory
        $certificatesDir = '../certificates/';
        if (!file_exists($certificatesDir)) {
            if (!mkdir($certificatesDir, 0755, true)) {
                error_log("Failed to create certificates directory");
                return ['success' => false, 'error' => 'Could not create certificates directory'];
            }
        }
        
        // Save HTML file
        $filename = "certificate_{$certificate_number}.html";
        $filepath = $certificatesDir . $filename;
        
        if (file_put_contents($filepath, $certificateHtml) === false) {
            error_log("Failed to save certificate file");
            return ['success' => false, 'error' => 'Could not save certificate file'];
        }
        
        return [
            'success' => true,
            'html_path' => $filepath,
            'filename' => $filename,
            'download_url' => "/certificates/{$filename}",
            'certificate_number' => $certificate_number
        ];
    }
    
    /**
     * Save certificate record to database
     */
    private function saveCertificateRecord($certificate_number, $student_name, $course_title, $issue_date, $file_path) {
        try {
            // First ensure the certificates table exists
            $this->createCertificatesTable();
            
            // Use simpler SQL that works across MySQL versions
            $sql = "INSERT INTO certificates 
                    (certificate_number, student_name, course_title, issue_date, file_path, created_at) 
                    VALUES (?, ?, ?, ?, ?, NOW())";
            
            $stmt = $this->conn->prepare($sql);
            
            if ($stmt === false) {
                error_log("Database prepare failed: " . $this->conn->error);
                return false;
            }
            
            $stmt->bind_param("sssss", $certificate_number, $student_name, $course_title, $issue_date, $file_path);
            $result = $stmt->execute();
            
            if (!$result) {
                error_log("Database execute failed: " . $stmt->error);
                return false;
            }
            
            $stmt->close();
            return true;
            
        } catch (Exception $e) {
            error_log("Error saving certificate record: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Create certificates table if it doesn't exist
     */
    private function createCertificatesTable() {
        $sql = "
        CREATE TABLE IF NOT EXISTS `certificates` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `certificate_number` varchar(50) NOT NULL,
            `student_name` varchar(255) NOT NULL,
            `course_title` varchar(255) NOT NULL,
            `issue_date` varchar(50) NOT NULL,
            `file_path` varchar(500) DEFAULT NULL,
            `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            KEY `certificate_number` (`certificate_number`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
        
        try {
            $this->conn->query($sql);
        } catch (Exception $e) {
            error_log("Error creating certificates table: " . $e->getMessage());
        }
    }
}

// Create certificates table if it doesn't exist
function createCertificatesTable($conn) {
    $sql = "
    CREATE TABLE IF NOT EXISTS `certificates` (
        `id` int(11) NOT NULL AUTO_INCREMENT,
        `certificate_number` varchar(50) NOT NULL UNIQUE,
        `student_name` varchar(255) NOT NULL,
        `course_title` varchar(255) NOT NULL,
        `issue_date` varchar(50) NOT NULL,
        `file_path` varchar(500) DEFAULT NULL,
        `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
        `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (`id`),
        KEY `certificate_number` (`certificate_number`),
        KEY `student_name` (`student_name`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
    
    $conn->query($sql);
}

// Initialize certificates table
if (isset($conn)) {
    createCertificatesTable($conn);
}
?>
