<?php
session_start();
require_once '../config/db_connect.php';
require_once '../includes/functions.php';

// Check if user is logged in and is an admin
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'admin') {
    // Redirect to login page
    header('Location: ../auth/login.php');
    exit;
}

// Process approval/rejection actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'], $_POST['course_id'])) {
        $course_id = (int)$_POST['course_id'];
        $action = $_POST['action'];
        
        if ($action === 'approve') {
            // Approve the course
            $updateQuery = "UPDATE courses SET is_approved = 1 WHERE course_id = ?";
            $stmt = $conn->prepare($updateQuery);
            $stmt->bind_param("i", $course_id);
            
            if ($stmt->execute()) {
                $_SESSION['success_message'] = "Course has been approved successfully.";
            } else {
                $_SESSION['error_message'] = "Failed to approve course: " . $conn->error;
            }
            $stmt->close();
        } 
        elseif ($action === 'reject') {
            // Reject the course - set is_approved to 0
            $updateQuery = "UPDATE courses SET is_approved = 0 WHERE course_id = ?";
            $stmt = $conn->prepare($updateQuery);
            $stmt->bind_param("i", $course_id);
            
            if ($stmt->execute()) {
                $_SESSION['success_message'] = "Course has been rejected.";
            } else {
                $_SESSION['error_message'] = "Failed to reject course: " . $conn->error;
            }
            $stmt->close();
        }
        elseif ($action === 'delete') {
            // Delete the course
            $deleteQuery = "DELETE FROM courses WHERE course_id = ?";
            $stmt = $conn->prepare($deleteQuery);
            $stmt->bind_param("i", $course_id);
            
            if ($stmt->execute()) {
                $_SESSION['success_message'] = "Course has been deleted.";
            } else {
                $_SESSION['error_message'] = "Failed to delete course: " . $conn->error;
            }
            $stmt->close();
        }
        
        // Redirect back to the same page to prevent form resubmission
        header('Location: pending-courses.php');
        exit;
    }
}

// Get pending and recently approved/rejected courses
$pendingCourses = [];
$recentlyApprovedCourses = [];

try {
    // Fetch pending courses (is_approved = NULL)
    $pendingQuery = "
        SELECT c.*, u.first_name, u.last_name, cat.name as category_name
        FROM courses c
        JOIN users u ON c.teacher_id = u.user_id
        LEFT JOIN categories cat ON c.category_id = cat.category_id
        WHERE c.is_approved IS NULL
        ORDER BY c.created_at DESC
    ";
    $pendingCourses = fetchAll($pendingQuery);
    
    // Fetch recently approved/rejected courses
    $recentQuery = "
        SELECT c.*, u.first_name, u.last_name, cat.name as category_name
        FROM courses c
        JOIN users u ON c.teacher_id = u.user_id
        LEFT JOIN categories cat ON c.category_id = cat.category_id
        WHERE c.is_approved IS NOT NULL
        ORDER BY c.updated_at DESC
        LIMIT 10
    ";
    $recentCourses = fetchAll($recentQuery);
} catch (Exception $e) {
    $_SESSION['error_message'] = "Database error: " . $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Pending Courses - Admin Dashboard</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="../assets/css/admin.css">
</head>
<body>
    <?php include 'includes/admin-navbar.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/admin-sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4 py-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Course Approval Management</h1>
                </div>
                
                <?php if (isset($_SESSION['success_message'])): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <?php echo $_SESSION['success_message']; unset($_SESSION['success_message']); ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>
                
                <?php if (isset($_SESSION['error_message'])): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <?php echo $_SESSION['error_message']; unset($_SESSION['error_message']); ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>
                
                <div class="card mb-4">
                    <div class="card-header bg-primary text-white">
                        <h5 class="mb-0"><i class="fas fa-hourglass-half me-2"></i> Pending Courses</h5>
                    </div>
                    <div class="card-body">
                        <?php if (empty($pendingCourses)): ?>
                            <p class="text-muted">No pending courses require approval.</p>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>Title</th>
                                            <th>Category</th>
                                            <th>Instructor</th>
                                            <th>Price</th>
                                            <th>Submitted</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($pendingCourses as $course): ?>
                                            <tr>
                                                <td>
                                                    <div class="d-flex align-items-center">
                                                        <?php if (!empty($course['thumbnail'])): ?>
                                                            <img src="<?php echo htmlspecialchars($course['thumbnail']); ?>" 
                                                                class="me-2 rounded" width="40" height="40" 
                                                                alt="<?php echo htmlspecialchars($course['title']); ?>"
                                                                onerror="this.src='../assets/images/placeholder.png';">
                                                        <?php else: ?>
                                                            <div class="bg-light rounded me-2 p-2">
                                                                <i class="fas fa-book text-primary"></i>
                                                            </div>
                                                        <?php endif; ?>
                                                        <?php echo htmlspecialchars($course['title']); ?>
                                                    </div>
                                                </td>
                                                <td><?php echo htmlspecialchars($course['category_name'] ?? 'Uncategorized'); ?></td>
                                                <td><?php echo htmlspecialchars($course['first_name'] . ' ' . $course['last_name']); ?></td>
                                                <td>
                                                    <?php if ($course['price'] == 0): ?>
                                                        <span class="badge bg-success">Free</span>
                                                    <?php else: ?>
                                                        R<?php echo number_format($course['price'], 2); ?>
                                                    <?php endif; ?>
                                                </td>
                                                <td><?php echo date('M d, Y', strtotime($course['created_at'] ?? 'now')); ?></td>
                                                <td>
                                                    <div class="btn-group">
                                                        <a href="../course.php?id=<?php echo $course['course_id']; ?>&preview=true" 
                                                           class="btn btn-sm btn-info" target="_blank">
                                                            <i class="fas fa-eye"></i> Preview
                                                        </a>
                                                        <form method="post" class="d-inline" onsubmit="return confirm('Are you sure you want to approve this course?');">
                                                            <input type="hidden" name="course_id" value="<?php echo $course['course_id']; ?>">
                                                            <input type="hidden" name="action" value="approve">
                                                            <button type="submit" class="btn btn-sm btn-success">
                                                                <i class="fas fa-check"></i> Approve
                                                            </button>
                                                        </form>
                                                        <form method="post" class="d-inline" onsubmit="return confirm('Are you sure you want to reject this course?');">
                                                            <input type="hidden" name="course_id" value="<?php echo $course['course_id']; ?>">
                                                            <input type="hidden" name="action" value="reject">
                                                            <button type="submit" class="btn btn-sm btn-warning">
                                                                <i class="fas fa-times"></i> Reject
                                                            </button>
                                                        </form>
                                                        <form method="post" class="d-inline" onsubmit="return confirm('Are you sure you want to delete this course? This action cannot be undone.');">
                                                            <input type="hidden" name="course_id" value="<?php echo $course['course_id']; ?>">
                                                            <input type="hidden" name="action" value="delete">
                                                            <button type="submit" class="btn btn-sm btn-danger">
                                                                <i class="fas fa-trash"></i> Delete
                                                            </button>
                                                        </form>
                                                    </div>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <div class="card">
                    <div class="card-header bg-secondary text-white">
                        <h5 class="mb-0"><i class="fas fa-history me-2"></i> Recent Course Actions</h5>
                    </div>
                    <div class="card-body">
                        <?php if (empty($recentCourses)): ?>
                            <p class="text-muted">No recent course actions found.</p>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>Title</th>
                                            <th>Category</th>
                                            <th>Instructor</th>
                                            <th>Status</th>
                                            <th>Last Updated</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($recentCourses as $course): ?>
                                            <tr>
                                                <td>
                                                    <div class="d-flex align-items-center">
                                                        <?php if (!empty($course['thumbnail'])): ?>
                                                            <img src="<?php echo htmlspecialchars($course['thumbnail']); ?>" 
                                                                class="me-2 rounded" width="40" height="40" 
                                                                alt="<?php echo htmlspecialchars($course['title']); ?>"
                                                                onerror="this.src='../assets/images/placeholder.png';">
                                                        <?php else: ?>
                                                            <div class="bg-light rounded me-2 p-2">
                                                                <i class="fas fa-book text-primary"></i>
                                                            </div>
                                                        <?php endif; ?>
                                                        <?php echo htmlspecialchars($course['title']); ?>
                                                    </div>
                                                </td>
                                                <td><?php echo htmlspecialchars($course['category_name'] ?? 'Uncategorized'); ?></td>
                                                <td><?php echo htmlspecialchars($course['first_name'] . ' ' . $course['last_name']); ?></td>
                                                <td>
                                                    <?php if ($course['is_approved'] == 1): ?>
                                                        <span class="badge bg-success">Approved</span>
                                                    <?php else: ?>
                                                        <span class="badge bg-warning">Rejected</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td><?php echo date('M d, Y', strtotime($course['updated_at'] ?? 'now')); ?></td>
                                                <td>
                                                    <div class="btn-group">
                                                        <a href="../course.php?id=<?php echo $course['course_id']; ?>" 
                                                           class="btn btn-sm btn-info" target="_blank">
                                                            <i class="fas fa-eye"></i> View
                                                        </a>
                                                        <?php if ($course['is_approved'] == 0): ?>
                                                            <form method="post" class="d-inline">
                                                                <input type="hidden" name="course_id" value="<?php echo $course['course_id']; ?>">
                                                                <input type="hidden" name="action" value="approve">
                                                                <button type="submit" class="btn btn-sm btn-success">
                                                                    <i class="fas fa-check"></i> Approve
                                                                </button>
                                                            </form>
                                                        <?php else: ?>
                                                            <form method="post" class="d-inline">
                                                                <input type="hidden" name="course_id" value="<?php echo $course['course_id']; ?>">
                                                                <input type="hidden" name="action" value="reject">
                                                                <button type="submit" class="btn btn-sm btn-warning">
                                                                    <i class="fas fa-times"></i> Reject
                                                                </button>
                                                            </form>
                                                        <?php endif; ?>
                                                    </div>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </main>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Auto-hide alerts after 5 seconds
        window.addEventListener('DOMContentLoaded', (event) => {
            const alerts = document.querySelectorAll('.alert');
            alerts.forEach(alert => {
                setTimeout(() => {
                    const bsAlert = new bootstrap.Alert(alert);
                    bsAlert.close();
                }, 5000);
            });
        });
    </script>
</body>
</html>
