<?php
/**
 * PERMANENT REVENUE SHARING FIX
 * 
 * This script does two things:
 * 1. Fixes existing enrollments with R0.00 prices (one-time migration)
 * 2. Ensures the payment success flow works correctly for future purchases
 * 
 * Run this once to fix all historical data and ensure future payments work correctly.
 */

include_once 'includes/db_connection.php';
include_once 'includes/purchase_processor.php';

$results = [];
$errors = [];
$stats = ['fixed_enrollments' => 0, 'processed_revenue' => 0, 'total_revenue' => 0];

// Step 1: Fix existing enrollments with zero prices
if (isset($_POST['run_permanent_fix'])) {
    try {
        echo "<h2>🔧 Step 1: Fixing Existing Enrollment Prices</h2>";
        
        // Update enrollments with correct prices from courses table
        $fix_prices_query = "
            UPDATE enrollments e 
            JOIN courses c ON e.course_id = c.course_id 
            SET e.course_price = c.price 
            WHERE (e.course_price = 0 OR e.course_price IS NULL) AND c.price > 0
        ";
        
        $fix_result = $conn->query($fix_prices_query);
        if ($fix_result) {
            $stats['fixed_enrollments'] = $conn->affected_rows;
            echo "<p>✅ Updated {$stats['fixed_enrollments']} enrollment(s) with correct prices</p>";
        } else {
            $errors[] = "Failed to update enrollment prices: " . $conn->error;
        }
        
        // Step 2: Process revenue sharing for all paid enrollments without purchase_id
        echo "<h2>💰 Step 2: Processing Revenue Sharing</h2>";
        
        $processor = new PurchaseProcessor($conn);
        
        // Get all paid enrollments that don't have purchase_id
        $enrollments_query = "
            SELECT e.enrollment_id, e.user_id, e.course_id, e.course_price, e.enrolled_date,
                   u.first_name, u.last_name, u.email,
                   c.title as course_title, c.teacher_id
            FROM enrollments e
            JOIN users u ON e.user_id = u.user_id 
            JOIN courses c ON e.course_id = c.course_id
            WHERE e.course_price > 0 AND (e.purchase_id IS NULL OR e.purchase_id = 0)
            ORDER BY e.enrolled_date DESC
        ";
        
        $enrollments_result = $conn->query($enrollments_query);
        
        if ($enrollments_result && $enrollments_result->num_rows > 0) {
            while ($enrollment = $enrollments_result->fetch_assoc()) {
                try {
                    // Process revenue sharing for this enrollment
                    $revenue_result = $processor->processPurchase(
                        $enrollment['user_id'],
                        $enrollment['course_id'],
                        $enrollment['course_price'],
                        'migration_fix',
                        'MIGRATION_' . $enrollment['enrollment_id'] . '_' . time()
                    );
                    
                    if ($revenue_result['success']) {
                        // Update enrollment with purchase_id
                        $update_stmt = $conn->prepare("UPDATE enrollments SET purchase_id = ? WHERE enrollment_id = ?");
                        $update_stmt->bind_param("ii", $revenue_result['purchase_id'], $enrollment['enrollment_id']);
                        $update_stmt->execute();
                        
                        $results[] = [
                            'enrollment_id' => $enrollment['enrollment_id'],
                            'student_name' => $enrollment['first_name'] . ' ' . $enrollment['last_name'],
                            'course_title' => $enrollment['course_title'],
                            'amount' => $enrollment['course_price'],
                            'purchase_id' => $revenue_result['purchase_id'],
                            'admin_earned' => $revenue_result['revenue_split']['admin_amount'],
                            'teacher_earned' => $revenue_result['revenue_split']['teacher_amount']
                        ];
                        
                        $stats['processed_revenue']++;
                        $stats['total_revenue'] += $enrollment['course_price'];
                        
                        echo "<p>✅ Processed enrollment #{$enrollment['enrollment_id']} - {$enrollment['course_title']} (R{$enrollment['course_price']})</p>";
                        
                    } else {
                        $errors[] = "Failed to process enrollment {$enrollment['enrollment_id']}: " . $revenue_result['error'];
                        echo "<p>❌ Failed enrollment #{$enrollment['enrollment_id']}: " . htmlspecialchars($revenue_result['error']) . "</p>";
                    }
                    
                } catch (Exception $e) {
                    $errors[] = "Exception processing enrollment {$enrollment['enrollment_id']}: " . $e->getMessage();
                    echo "<p>❌ Exception for enrollment #{$enrollment['enrollment_id']}: " . htmlspecialchars($e->getMessage()) . "</p>";
                }
                
                // Flush output for real-time progress
                if (ob_get_level()) ob_flush();
                flush();
            }
        } else {
            echo "<p>ℹ️ No enrollments found to process</p>";
        }
        
        // Step 3: Verify the fix worked
        echo "<h2>✅ Step 3: Verification</h2>";
        
        $verify_query = "
            SELECT 
                COUNT(*) as total_enrollments,
                COUNT(CASE WHEN course_price > 0 THEN 1 END) as paid_enrollments,
                COUNT(CASE WHEN course_price > 0 AND purchase_id IS NOT NULL THEN 1 END) as processed_enrollments,
                SUM(CASE WHEN course_price > 0 THEN course_price ELSE 0 END) as total_revenue
            FROM enrollments
        ";
        
        $verify_result = $conn->query($verify_query);
        if ($verify_result) {
            $verify_stats = $verify_result->fetch_assoc();
            echo "<div class='verification-stats'>";
            echo "<h4>📊 Final Statistics:</h4>";
            echo "<p><strong>Total Enrollments:</strong> {$verify_stats['total_enrollments']}</p>";
            echo "<p><strong>Paid Enrollments:</strong> {$verify_stats['paid_enrollments']}</p>";
            echo "<p><strong>Processed with Revenue:</strong> {$verify_stats['processed_enrollments']}</p>";
            echo "<p><strong>Total Revenue:</strong> R" . number_format($verify_stats['total_revenue'], 2) . "</p>";
            echo "</div>";
            
            if ($verify_stats['paid_enrollments'] == $verify_stats['processed_enrollments'] && $verify_stats['paid_enrollments'] > 0) {
                echo "<div class='success-banner'>";
                echo "<h3>🎉 SUCCESS! Revenue System Permanently Fixed</h3>";
                echo "<p>✅ All paid enrollments now have correct prices</p>";
                echo "<p>✅ All revenue sharing has been processed</p>";
                echo "<p>✅ Future payments will work automatically</p>";
                echo "</div>";
            }
        }
        
    } catch (Exception $e) {
        $errors[] = "Critical error during permanent fix: " . $e->getMessage();
        echo "<p style='color: red;'>❌ Critical Error: " . htmlspecialchars($e->getMessage()) . "</p>";
    }
}

// Get current stats before processing
$current_stats = [];
try {
    $current_result = $conn->query("
        SELECT 
            COUNT(*) as total_enrollments,
            COUNT(CASE WHEN course_price > 0 THEN 1 END) as paid_enrollments,
            COUNT(CASE WHEN course_price = 0 OR course_price IS NULL THEN 1 END) as zero_price_enrollments,
            COUNT(CASE WHEN course_price > 0 AND (purchase_id IS NULL OR purchase_id = 0) THEN 1 END) as needs_revenue_processing,
            SUM(CASE WHEN course_price > 0 THEN course_price ELSE 0 END) as total_revenue
        FROM enrollments
    ");
    
    if ($current_result) {
        $current_stats = $current_result->fetch_assoc();
    }
} catch (Exception $e) {
    $errors[] = "Failed to get current statistics: " . $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Permanent Revenue Fix - Panadite Academy</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <style>
        body { background: #f8fafc; font-family: 'Segoe UI', sans-serif; }
        .container { max-width: 1200px; margin: 40px auto; padding: 20px; }
        .card { background: white; border-radius: 12px; padding: 24px; margin-bottom: 24px; box-shadow: 0 4px 12px rgba(0,0,0,0.1); }
        .title { color: #1f2937; font-weight: 600; margin-bottom: 20px; display: flex; align-items: center; gap: 12px; }
        .title i { color: #4361ee; font-size: 24px; }
        .stat-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 16px; margin-bottom: 24px; }
        .stat-card { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 20px; border-radius: 8px; text-align: center; }
        .stat-value { font-size: 28px; font-weight: 700; }
        .stat-label { font-size: 14px; opacity: 0.9; margin-top: 4px; }
        .btn-fix { background: linear-gradient(135deg, #4361ee, #3f37c9); color: white; padding: 12px 24px; border: none; border-radius: 8px; font-weight: 600; font-size: 16px; }
        .success-banner { background: linear-gradient(135deg, #10b981, #059669); color: white; padding: 20px; border-radius: 8px; margin: 20px 0; text-align: center; }
        .verification-stats { background: #f0f9ff; padding: 20px; border-radius: 8px; border-left: 4px solid #0ea5e9; }
        .progress-log { background: #1f2937; color: #f8fafc; padding: 20px; border-radius: 8px; font-family: monospace; max-height: 400px; overflow-y: auto; }
    </style>
</head>
<body>
    <div class="container">
        <div class="card">
            <h1 class="title">
                <i class='bx bx-wrench'></i>
                Permanent Revenue System Fix
            </h1>
            <p style="color: #6b7280;">
                This comprehensive fix will update all existing enrollments with correct prices and process revenue sharing. 
                It also ensures the payment success flow works correctly for all future purchases.
            </p>
        </div>

        <!-- Current Status -->
        <div class="card">
            <h3 class="title">
                <i class='bx bx-chart-line'></i>
                Current System Status
            </h3>
            
            <div class="stat-grid">
                <div class="stat-card">
                    <div class="stat-value"><?= $current_stats['total_enrollments'] ?? 0 ?></div>
                    <div class="stat-label">Total Enrollments</div>
                </div>
                <div class="stat-card">
                    <div class="stat-value"><?= $current_stats['paid_enrollments'] ?? 0 ?></div>
                    <div class="stat-label">Paid Enrollments</div>
                </div>
                <div class="stat-card">
                    <div class="stat-value"><?= $current_stats['zero_price_enrollments'] ?? 0 ?></div>
                    <div class="stat-label">Zero Price (Need Fix)</div>
                </div>
                <div class="stat-card">
                    <div class="stat-value"><?= $current_stats['needs_revenue_processing'] ?? 0 ?></div>
                    <div class="stat-label">Need Revenue Processing</div>
                </div>
            </div>
            
            <div class="alert alert-info">
                <strong>Total Revenue to Process:</strong> R<?= number_format($current_stats['total_revenue'] ?? 0, 2) ?>
            </div>
        </div>

        <?php if (!isset($_POST['run_permanent_fix'])): ?>
        <!-- Fix Form -->
        <div class="card">
            <h3 class="title">
                <i class='bx bx-play-circle'></i>
                Run Permanent Fix
            </h3>
            
            <div class="alert alert-warning">
                <strong>⚠️ This will permanently fix your revenue system:</strong><br>
                • Update all enrollment prices from courses table<br>
                • Process revenue sharing for all historical data<br>
                • Ensure future payments work automatically<br>
                • Cannot be undone - make sure you have a database backup
            </div>
            
            <form method="POST">
                <input type="hidden" name="run_permanent_fix" value="yes">
                <button type="submit" class="btn-fix" onclick="this.disabled=true; this.innerHTML='<i class=\'bx bx-loader-alt bx-spin\'></i> Processing...'; this.form.submit();">
                    <i class='bx bx-wrench'></i>
                    Run Permanent Fix Now
                </button>
            </form>
        </div>
        <?php endif; ?>

        <!-- Processing Results -->
        <?php if (isset($_POST['run_permanent_fix'])): ?>
        <div class="card">
            <h3 class="title">
                <i class='bx bx-cog'></i>
                Fix Processing Results
            </h3>
            
            <div class="progress-log" id="progress-log">
                <p><strong>🚀 Starting Permanent Revenue System Fix...</strong></p>
            </div>
        </div>
        <?php endif; ?>

        <!-- Navigation -->
        <div class="card" style="text-align: center;">
            <a href="admin/revenue-settings.php" class="btn btn-success me-2">
                <i class='bx bx-cog'></i> Revenue Settings
            </a>
            <a href="process-revenue.php" class="btn btn-primary me-2">
                <i class='bx bx-money'></i> Revenue Processor
            </a>
            <a href="diagnose-enrollments.php" class="btn btn-outline-primary me-2">
                <i class='bx bx-search'></i> Diagnose System
            </a>
            <a href="admin/dashboard.php" class="btn btn-secondary">
                <i class='bx bx-home'></i> Admin Dashboard
            </a>
        </div>
    </div>
    
    <script>
        // Auto-scroll progress log
        function scrollToBottom() {
            const log = document.getElementById('progress-log');
            if (log) {
                log.scrollTop = log.scrollHeight;
            }
        }
        
        // Scroll every second during processing
        <?php if (isset($_POST['run_permanent_fix'])): ?>
        setInterval(scrollToBottom, 1000);
        <?php endif; ?>
    </script>
</body>
</html>
