<?php
require_once __DIR__ . '/../auth/functions.php';
require_once __DIR__ . '/../includes/db_connection.php';
require_once __DIR__ . '/../includes/purchase_processor.php';

// Ensure user is logged in as an admin
requireRole('admin');

$processed = 0;
$errors = [];
$results = [];

// Process existing enrollments that have course prices but no purchase_id
if (isset($_POST['process_enrollments'])) {
    try {
        $processor = new PurchaseProcessor($conn);
        
        // Get enrollments with course prices but no purchase_id
        $query = "
            SELECT e.enrollment_id, e.user_id, e.course_id, e.course_price, e.enrolled_date,
                   c.title as course_title,
                   CONCAT(u.first_name, ' ', u.last_name) as student_name
            FROM enrollments e
            JOIN courses c ON e.course_id = c.course_id  
            JOIN users u ON e.user_id = u.user_id
            WHERE e.course_price > 0 AND e.purchase_id IS NULL
            ORDER BY e.enrolled_date DESC
        ";
        
        $enrollments = $conn->query($query);
        
        if ($enrollments && $enrollments->num_rows > 0) {
            while ($enrollment = $enrollments->fetch_assoc()) {
                try {
                    // Process revenue sharing for this enrollment
                    $revenue_result = $processor->processPurchase(
                        $enrollment['user_id'],
                        $enrollment['course_id'], 
                        $enrollment['course_price'],
                        'retroactive',
                        'RETRO_' . $enrollment['enrollment_id'] . '_' . time()
                    );
                    
                    if ($revenue_result['success']) {
                        // Update enrollment with purchase_id
                        $stmt = $conn->prepare("UPDATE enrollments SET purchase_id = ? WHERE enrollment_id = ?");
                        $stmt->bind_param("ii", $revenue_result['purchase_id'], $enrollment['enrollment_id']);
                        $stmt->execute();
                        
                        $results[] = [
                            'success' => true,
                            'enrollment_id' => $enrollment['enrollment_id'],
                            'student' => $enrollment['student_name'],
                            'course' => $enrollment['course_title'],
                            'amount' => $enrollment['course_price'],
                            'purchase_id' => $revenue_result['purchase_id'],
                            'admin_earned' => $revenue_result['revenue_split']['admin_amount'],
                            'teacher_earned' => $revenue_result['revenue_split']['teacher_amount']
                        ];
                        $processed++;
                    } else {
                        $errors[] = "Enrollment {$enrollment['enrollment_id']}: " . $revenue_result['error'];
                    }
                } catch (Exception $e) {
                    $errors[] = "Enrollment {$enrollment['enrollment_id']}: " . $e->getMessage();
                }
            }
        }
    } catch (Exception $e) {
        $errors[] = "Processing failed: " . $e->getMessage();
    }
}

// Get current stats
$total_enrollments = $conn->query("SELECT COUNT(*) as count FROM enrollments WHERE course_price > 0")->fetch_assoc()['count'] ?? 0;
$processed_enrollments = $conn->query("SELECT COUNT(*) as count FROM enrollments WHERE course_price > 0 AND purchase_id IS NOT NULL")->fetch_assoc()['count'] ?? 0;
$pending_enrollments = $total_enrollments - $processed_enrollments;
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Process Existing Enrollments - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        .process-container { padding: 30px; background: #f8fafc; min-height: 100vh; }
        .process-card { background: white; border-radius: 12px; padding: 24px; margin-bottom: 24px; box-shadow: 0 4px 12px rgba(0,0,0,0.1); }
        .process-title { color: #1f2937; font-weight: 600; margin-bottom: 20px; display: flex; align-items: center; gap: 12px; }
        .process-title i { color: #4361ee; font-size: 24px; }
        .stat-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 16px; margin-bottom: 24px; }
        .stat-card { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 20px; border-radius: 8px; text-align: center; }
        .stat-value { font-size: 28px; font-weight: 700; }
        .stat-label { font-size: 14px; opacity: 0.9; margin-top: 4px; }
        .btn-process { background: linear-gradient(135deg, #4361ee, #3f37c9); color: white; padding: 12px 24px; border: none; border-radius: 8px; font-weight: 600; }
        .result-item { padding: 12px; margin-bottom: 8px; border-radius: 6px; border-left: 4px solid; }
        .result-success { background: #f0fdf4; color: #166534; border-left-color: #22c55e; }
        .result-error { background: #fef2f2; color: #dc2626; border-left-color: #ef4444; }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>

        <div class="process-container">
            <div class="process-card">
                <h1 class="process-title">
                    <i class='bx bx-refresh'></i>
                    Process Existing Enrollments
                </h1>
                <p style="color: #6b7280; margin-bottom: 0;">
                    This tool processes existing paid enrollments and applies revenue sharing retroactively.
                    It will create purchase records and distribute earnings between admin and teachers.
                </p>
            </div>

            <!-- Stats -->
            <div class="process-card">
                <h3 class="process-title">
                    <i class='bx bx-chart-line'></i>
                    Current Status
                </h3>
                
                <div class="stat-grid">
                    <div class="stat-card">
                        <div class="stat-value"><?= $total_enrollments ?></div>
                        <div class="stat-label">Total Paid Enrollments</div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-value"><?= $processed_enrollments ?></div>
                        <div class="stat-label">Already Processed</div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-value"><?= $pending_enrollments ?></div>
                        <div class="stat-label">Pending Processing</div>
                    </div>
                </div>
            </div>

            <!-- Processing Form -->
            <?php if ($pending_enrollments > 0): ?>
            <div class="process-card">
                <h3 class="process-title">
                    <i class='bx bx-play-circle'></i>
                    Process Enrollments
                </h3>
                
                <form method="POST">
                    <div class="alert alert-info">
                        <strong>⚠️ Important:</strong> This will process <?= $pending_enrollments ?> enrollment(s) and create revenue sharing records.
                        Make sure the revenue settings are configured correctly before proceeding.
                    </div>
                    
                    <button type="submit" name="process_enrollments" class="btn-process">
                        <i class='bx bx-cog'></i>
                        Process <?= $pending_enrollments ?> Enrollment(s)
                    </button>
                </form>
            </div>
            <?php endif; ?>

            <!-- Results -->
            <?php if (!empty($results)): ?>
            <div class="process-card">
                <h3 class="process-title">
                    <i class='bx bx-check-circle'></i>
                    Processing Results
                </h3>
                
                <div class="alert alert-success">
                    <strong>✅ Success!</strong> Processed <?= $processed ?> enrollment(s) successfully.
                </div>
                
                <?php foreach ($results as $result): ?>
                <div class="result-item result-success">
                    <strong>Enrollment #<?= $result['enrollment_id'] ?></strong><br>
                    Student: <?= htmlspecialchars($result['student']) ?><br>
                    Course: <?= htmlspecialchars($result['course']) ?><br>
                    Amount: R<?= number_format($result['amount'], 2) ?><br>
                    Admin Earned: R<?= number_format($result['admin_earned'], 2) ?><br>
                    Teacher Earned: R<?= number_format($result['teacher_earned'], 2) ?><br>
                    Purchase ID: #<?= $result['purchase_id'] ?>
                </div>
                <?php endforeach; ?>
            </div>
            <?php endif; ?>

            <!-- Errors -->
            <?php if (!empty($errors)): ?>
            <div class="process-card">
                <h3 class="process-title">
                    <i class='bx bx-error-circle'></i>
                    Processing Errors
                </h3>
                
                <?php foreach ($errors as $error): ?>
                <div class="result-item result-error">
                    <?= htmlspecialchars($error) ?>
                </div>
                <?php endforeach; ?>
            </div>
            <?php endif; ?>

            <!-- Navigation -->
            <div class="process-card" style="text-align: center;">
                <a href="debug-purchases.php" class="btn btn-outline-primary me-2">
                    <i class='bx bx-bug'></i> Debug Purchases
                </a>
                <a href="revenue-settings.php" class="btn btn-outline-primary me-2">
                    <i class='bx bx-cog'></i> Revenue Settings
                </a>
                <a href="dashboard.php" class="btn btn-secondary">
                    <i class='bx bx-home'></i> Dashboard
                </a>
            </div>
        </div>
    </section>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
