<?php
require_once '../auth/functions.php';
require_once '../includes/config.php';
require_once '../includes/db_connection.php';

// Ensure user is logged in as an admin
requireRole('admin');

$userId = $_SESSION['user_id'];
$user = getCurrentUser();

// Initialize variables
$successMsg = '';
$errorMsg = '';

// Get summary statistics
$stats = [
    'total_users' => 0,
    'total_teachers' => 0,
    'total_students' => 0,
    'total_courses' => 0,
    'total_videos' => 0,
    'total_revenue' => 0
];

try {
    // Count total users
    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM users");
    $stmt->execute();
    $result = $stmt->get_result();
    if ($row = $result->fetch_assoc()) {
        $stats['total_users'] = $row['count'];
    }
    
    // Count teachers
    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM users WHERE user_role = 'teacher'");
    $stmt->execute();
    $result = $stmt->get_result();
    if ($row = $result->fetch_assoc()) {
        $stats['total_teachers'] = $row['count'];
    }
    
    // Count students
    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM users WHERE user_role = 'student'");
    $stmt->execute();
    $result = $stmt->get_result();
    if ($row = $result->fetch_assoc()) {
        $stats['total_students'] = $row['count'];
    }
    
    // Count courses
    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM courses");
    $stmt->execute();
    $result = $stmt->get_result();
    if ($row = $result->fetch_assoc()) {
        $stats['total_courses'] = $row['count'];
    }
    
    // Count videos
    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM videos");
    $stmt->execute();
    $result = $stmt->get_result();
    if ($row = $result->fetch_assoc()) {
        $stats['total_videos'] = $row['count'];
    }
    
    // Calculate total revenue with dynamic schema detection
    $hasPaymentDateColumn = false;
    $hasPaymentMethodColumn = false;
    
    // Check for payment_date column in video_purchases
    $checkColumnQuery = "SHOW COLUMNS FROM video_purchases LIKE 'payment_date'";
    $columnResult = $conn->query($checkColumnQuery);
    if ($columnResult && $columnResult->num_rows > 0) {
        $hasPaymentDateColumn = true;
    }
    
    // Get video purchase revenue
    $videoRevenueQuery = "SELECT SUM(amount) as total FROM video_purchases";
    $stmt = $conn->prepare($videoRevenueQuery);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($row = $result->fetch_assoc()) {
        $stats['total_revenue'] += ($row['total'] ? $row['total'] : 0);
    }
    
    // Get course purchase revenue
    $courseRevenueQuery = "SELECT SUM(amount) as total FROM course_purchases";
    $stmt = $conn->prepare($courseRevenueQuery);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($row = $result->fetch_assoc()) {
        $stats['total_revenue'] += ($row['total'] ? $row['total'] : 0);
    }
    
} catch (Exception $e) {
    $errorMsg = "Error generating reports: " . $e->getMessage();
}

// Get monthly sales data for the chart
$monthlyData = [];
$currentYear = date('Y');

try {
    // Initialize monthly data with zeros for all months
    for ($i = 1; $i <= 12; $i++) {
        $monthlyData[$i] = 0;
    }
    
    // Function to check if a column exists in a table
    function columnExists($conn, $table, $column) {
        $checkColumnQuery = "SHOW COLUMNS FROM $table LIKE '$column'";
        $columnResult = $conn->query($checkColumnQuery);
        return ($columnResult && $columnResult->num_rows > 0);
    }
    
    // Check which date columns exist in video_purchases
    $videoHasPaymentDate = columnExists($conn, 'video_purchases', 'payment_date');
    $videoHasCreatedAt = columnExists($conn, 'video_purchases', 'created_at');
    
    // Check which date columns exist in course_purchases
    $courseHasPaymentDate = columnExists($conn, 'course_purchases', 'payment_date');
    $courseHasCreatedAt = columnExists($conn, 'course_purchases', 'created_at');
    
    // Process video purchase data if we have either date column
    if ($videoHasPaymentDate || $videoHasCreatedAt) {
        $videoDateColumn = $videoHasPaymentDate ? 'payment_date' : ($videoHasCreatedAt ? 'created_at' : null);
        
        if ($videoDateColumn) {
            // Get video purchase data by month for current year
            $query = "SELECT MONTH($videoDateColumn) as month, SUM(amount) as total 
                      FROM video_purchases 
                      WHERE YEAR($videoDateColumn) = ? 
                      GROUP BY MONTH($videoDateColumn)";
            $stmt = $conn->prepare($query);
            $stmt->bind_param("i", $currentYear);
            $stmt->execute();
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $month = (int)$row['month'];
                $monthlyData[$month] += ($row['total'] ? $row['total'] : 0);
            }
        }
    }
    
    // Process course purchase data if we have either date column
    if ($courseHasPaymentDate || $courseHasCreatedAt) {
        $courseDateColumn = $courseHasPaymentDate ? 'payment_date' : ($courseHasCreatedAt ? 'created_at' : null);
        
        if ($courseDateColumn) {
            // Get course purchase data by month for current year
            $query = "SELECT MONTH($courseDateColumn) as month, SUM(amount) as total 
                      FROM course_purchases 
                      WHERE YEAR($courseDateColumn) = ? 
                      GROUP BY MONTH($courseDateColumn)";
            $stmt = $conn->prepare($query);
            $stmt->bind_param("i", $currentYear);
            $stmt->execute();
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $month = (int)$row['month'];
                $monthlyData[$month] += ($row['total'] ? $row['total'] : 0);
            }
        }
    }
    
    // If no date columns exist, we'll just show zeros for all months
    
} catch (Exception $e) {
    $errorMsg = "Error generating monthly data: " . $e->getMessage();
}

// Get top performing courses
$topCourses = [];

try {
    $query = "SELECT c.course_id, c.title, COUNT(cp.purchase_id) as purchases, SUM(cp.amount) as revenue 
              FROM courses c 
              LEFT JOIN course_purchases cp ON c.course_id = cp.course_id 
              GROUP BY c.course_id 
              ORDER BY revenue DESC 
              LIMIT 5";
    $stmt = $conn->prepare($query);
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) {
        $topCourses[] = $row;
    }
} catch (Exception $e) {
    $errorMsg = "Error retrieving top courses: " . $e->getMessage();
}

// Get top performing videos
$topVideos = [];

try {
    $query = "SELECT v.video_id, v.title, COUNT(vp.purchase_id) as purchases, SUM(vp.amount) as revenue 
              FROM videos v 
              LEFT JOIN video_purchases vp ON v.video_id = vp.video_id 
              GROUP BY v.video_id 
              ORDER BY revenue DESC 
              LIMIT 5";
    $stmt = $conn->prepare($query);
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) {
        $topVideos[] = $row;
    }
} catch (Exception $e) {
    $errorMsg = "Error retrieving top videos: " . $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reports & Analytics | Panadite Admin</title>
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Boxicons CSS -->
    <link href="https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css" rel="stylesheet">
    <style>
        /* Modern color variables */
        :root {
            --primary-color: #4e73df;
            --success-color: #1cc88a;
            --info-color: #36b9cc;
            --warning-color: #f6c23e;
            --danger-color: #e74a3b;
            --secondary-color: #858796;
            --light-color: #f8f9fc;
            --dark-color: #5a5c69;
        }
        
        /* Card hover effects */
        .hover-shadow {
            transition: all 0.3s ease;
        }
        .hover-shadow:hover {
            transform: translateY(-5px);
            box-shadow: 0 0.5rem 1.5rem rgba(0, 0, 0, 0.15) !important;
        }
        
        /* Icon wrapper styles */
        .icon-wrapper {
            width: 48px;
            height: 48px;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        /* Background subtle colors for icons */
        .bg-primary-subtle {
            background-color: rgba(78, 115, 223, 0.1);
        }
        .bg-success-subtle {
            background-color: rgba(28, 200, 138, 0.1);
        }
        .bg-info-subtle {
            background-color: rgba(54, 185, 204, 0.1);
        }
        .bg-warning-subtle {
            background-color: rgba(246, 194, 62, 0.1);
        }
        .bg-danger-subtle {
            background-color: rgba(231, 74, 59, 0.1);
        }
        
        /* Chart container styles */
        .chart-container canvas {
            max-width: 100%;
        }
        
        /* Rounded tables */
        .table-responsive {
            border-radius: 0.5rem;
            overflow: hidden;
        }
    </style>
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <!-- Chart.js for data visualization -->
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <div class="container-fluid p-4">
                <!-- Page Header -->
                <div class="d-sm-flex justify-content-between align-items-center mb-4">
                    <div class="page-header-wrapper">
                        <h2 class="mb-0 fw-bold">
                            <div class="d-flex align-items-center">
                                <div class="icon-wrapper bg-primary-subtle rounded-circle p-3 me-3">
                                    <i class="bx bx-line-chart fs-3 text-primary"></i>
                                </div>
                                Reports & Analytics
                            </div>
                        </h2>
                        <p class="text-muted mt-2">Track your academy's performance metrics and growth</p>
                    </div>
                    <div class="d-flex">
                        <button class="btn btn-sm btn-outline-primary rounded-pill px-3 me-2">
                            <i class="bx bx-export me-1"></i> Export Data
                        </button>
                        <button class="btn btn-sm btn-primary rounded-pill px-3">
                            <i class="bx bx-refresh me-1"></i> Refresh
                        </button>
                    </div>
                </div>
                
                <!-- Display success and error messages -->
                <?php if(!empty($successMsg)): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <i class="bx bx-check-circle me-2"></i>
                        <?php echo $successMsg; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>
                
                <?php if(!empty($errorMsg)): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <i class="bx bx-error-circle me-2"></i>
                        <?php echo $errorMsg; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>
                
                <!-- Key Stats Cards -->
                <div class="row g-4 mb-5">
                    <div class="col-md-6 col-lg-4">
                        <div class="card stat-card h-100 border-0 rounded-4 shadow-sm hover-shadow">
                            <div class="card-body d-flex p-4">
                                <div class="icon-wrapper bg-primary-subtle rounded-circle p-3 me-3 d-flex align-items-center justify-content-center" style="width: 60px; height: 60px; min-width: 60px;">
                                    <i class="bx bx-user fs-2 text-primary"></i>
                                </div>
                                <div class="d-flex flex-column justify-content-center">
                                    <h6 class="card-subtitle text-muted mb-1">Total Users</h6>
                                    <h3 class="card-text fw-bold mb-0"><?php echo number_format($stats['total_users']); ?></h3>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6 col-lg-4">
                        <div class="card stat-card h-100 border-0 rounded-4 shadow-sm hover-shadow">
                            <div class="card-body d-flex p-4">
                                <div class="icon-wrapper bg-success-subtle rounded-circle p-3 me-3 d-flex align-items-center justify-content-center" style="width: 60px; height: 60px; min-width: 60px;">
                                    <i class="bx bx-user-voice fs-2 text-success"></i>
                                </div>
                                <div class="d-flex flex-column justify-content-center">
                                    <h6 class="card-subtitle text-muted mb-1">Teachers</h6>
                                    <h3 class="card-text fw-bold mb-0"><?php echo number_format($stats['total_teachers']); ?></h3>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6 col-lg-4">
                        <div class="card stat-card h-100 border-0 rounded-4 shadow-sm hover-shadow">
                            <div class="card-body d-flex p-4">
                                <div class="icon-wrapper bg-info-subtle rounded-circle p-3 me-3 d-flex align-items-center justify-content-center" style="width: 60px; height: 60px; min-width: 60px;">
                                    <i class="bx bx-user-pin fs-2 text-info"></i>
                                </div>
                                <div class="d-flex flex-column justify-content-center">
                                    <h6 class="card-subtitle text-muted mb-1">Students</h6>
                                    <h3 class="card-text fw-bold mb-0"><?php echo number_format($stats['total_students']); ?></h3>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6 col-lg-4">
                        <div class="card stat-card h-100 border-0 rounded-4 shadow-sm hover-shadow">
                            <div class="card-body d-flex p-4">
                                <div class="icon-wrapper bg-warning-subtle rounded-circle p-3 me-3 d-flex align-items-center justify-content-center" style="width: 60px; height: 60px; min-width: 60px;">
                                    <i class="bx bx-book fs-2 text-warning"></i>
                                </div>
                                <div class="d-flex flex-column justify-content-center">
                                    <h6 class="card-subtitle text-muted mb-1">Courses</h6>
                                    <h3 class="card-text fw-bold mb-0"><?php echo number_format($stats['total_courses']); ?></h3>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6 col-lg-4">
                        <div class="card stat-card h-100 border-0 rounded-4 shadow-sm hover-shadow">
                            <div class="card-body d-flex p-4">
                                <div class="icon-wrapper bg-danger-subtle rounded-circle p-3 me-3 d-flex align-items-center justify-content-center" style="width: 60px; height: 60px; min-width: 60px;">
                                    <i class="bx bx-video fs-2 text-danger"></i>
                                </div>
                                <div class="d-flex flex-column justify-content-center">
                                    <h6 class="card-subtitle text-muted mb-1">Videos</h6>
                                    <h3 class="card-text fw-bold mb-0"><?php echo number_format($stats['total_videos']); ?></h3>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6 col-lg-4">
                        <div class="card stat-card h-100 border-0 rounded-4 shadow-sm hover-shadow">
                            <div class="card-body d-flex p-4">
                                <div class="icon-wrapper bg-success-subtle rounded-circle p-3 me-3 d-flex align-items-center justify-content-center" style="width: 60px; height: 60px; min-width: 60px;">
                                    <i class="bx bx-money fs-2 text-success"></i>
                                </div>
                                <div class="d-flex flex-column justify-content-center">
                                    <h6 class="card-subtitle text-muted mb-1">Revenue</h6>
                                    <h3 class="card-text fw-bold mb-0">R<?php echo number_format($stats['total_revenue'], 2); ?></h3>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Revenue Chart -->
                <div class="row mb-5">
                    <div class="col-12">
                        <div class="card border-0 rounded-4 shadow-sm hover-shadow">
                            <div class="card-header bg-white py-4 border-0">
                                <div class="d-flex justify-content-between align-items-center">
                                    <div class="d-flex align-items-center">
                                        <div class="icon-wrapper bg-primary-subtle rounded-circle p-3 me-3">
                                            <i class="bx bx-bar-chart-alt-2 fs-3 text-primary"></i>
                                        </div>
                                        <h5 class="mb-0 fw-semibold">Monthly Revenue (<?php echo date('Y'); ?>)</h5>
                                    </div>
                                    <div class="btn-group">
                                        <button type="button" class="btn btn-sm btn-outline-primary rounded-pill px-3">
                                            <i class="bx bx-calendar me-1"></i> This Year
                                        </button>
                                    </div>
                                </div>
                            </div>
                            <div class="card-body p-4">
                                <div class="chart-container" style="position: relative; height:350px;">
                                    <canvas id="revenueChart"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Top Performers -->
                <div class="row g-4 mb-5">
                    <!-- Top Courses -->
                    <div class="col-md-6">
                        <div class="card border-0 rounded-4 shadow-sm h-100 hover-shadow">
                            <div class="card-header bg-white py-4 border-0">
                                <div class="d-flex align-items-center">
                                    <div class="icon-wrapper bg-warning-subtle rounded-circle p-3 me-3">
                                        <i class="bx bx-book-open fs-3 text-warning"></i>
                                    </div>
                                    <h5 class="mb-0 fw-semibold">Top Performing Courses</h5>
                                </div>
                            </div>
                            <div class="card-body p-0">
                                <?php if(count($topCourses) > 0): ?>
                                    <div class="table-responsive">
                                        <table class="table table-hover align-middle mb-0">
                                            <thead class="table-light">
                                                <tr>
                                                    <th class="ps-4">#</th>
                                                    <th>Course</th>
                                                    <th>Sales</th>
                                                    <th class="text-end pe-4">Revenue</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach($topCourses as $index => $course): ?>
                                                    <tr>
                                                        <td class="ps-4">
                                                            <span class="badge rounded-pill <?php echo $index === 0 ? 'bg-warning' : ($index === 1 ? 'bg-secondary' : 'bg-light text-dark'); ?>">
                                                                <?php echo $index + 1; ?>
                                                            </span>
                                                        </td>
                                                        <td>
                                                            <div class="text-wrap" style="max-width: 250px;">
                                                                <?php echo htmlspecialchars($course['title']); ?>
                                                            </div>
                                                        </td>
                                                        <td>
                                                            <span class="badge bg-primary-subtle text-primary rounded-pill">
                                                                <?php echo $course['purchases'] ? $course['purchases'] : 0; ?>
                                                            </span>
                                                        </td>
                                                        <td class="text-end pe-4 fw-bold">
                                                            R<?php echo number_format($course['revenue'] ? $course['revenue'] : 0, 2); ?>
                                                        </td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                <?php else: ?>
                                    <div class="alert alert-info border-0 m-4 rounded-3">
                                        <div class="d-flex align-items-center">
                                            <i class="bx bx-info-circle fs-3 me-3"></i>
                                            <p class="mb-0">No course sales data available yet.</p>
                                        </div>
                                    </div>
                                <?php endif; ?>
                            </div>
                            <div class="card-footer bg-white border-0 text-end p-3">
                                <a href="courses.php" class="btn btn-sm btn-outline-primary rounded-pill px-3">
                                    <i class="bx bx-list-ul me-1"></i> View All Courses
                                </a>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Top Videos -->
                    <div class="col-md-6">
                        <div class="card border-0 rounded-4 shadow-sm h-100 hover-shadow">
                            <div class="card-header bg-white py-4 border-0">
                                <div class="d-flex align-items-center">
                                    <div class="icon-wrapper bg-danger-subtle rounded-circle p-3 me-3">
                                        <i class="bx bx-video-plus fs-3 text-danger"></i>
                                    </div>
                                    <h5 class="mb-0 fw-semibold">Top Performing Videos</h5>
                                </div>
                            </div>
                            <div class="card-body p-0">
                                <?php if(count($topVideos) > 0): ?>
                                    <div class="table-responsive">
                                        <table class="table table-hover align-middle mb-0">
                                            <thead class="table-light">
                                                <tr>
                                                    <th class="ps-4">#</th>
                                                    <th>Video</th>
                                                    <th>Sales</th>
                                                    <th class="text-end pe-4">Revenue</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach($topVideos as $index => $video): ?>
                                                    <tr>
                                                        <td class="ps-4">
                                                            <span class="badge rounded-pill <?php echo $index === 0 ? 'bg-danger' : ($index === 1 ? 'bg-secondary' : 'bg-light text-dark'); ?>">
                                                                <?php echo $index + 1; ?>
                                                            </span>
                                                        </td>
                                                        <td>
                                                            <div class="text-wrap" style="max-width: 250px;">
                                                                <?php echo htmlspecialchars($video['title']); ?>
                                                            </div>
                                                        </td>
                                                        <td>
                                                            <span class="badge bg-primary-subtle text-primary rounded-pill">
                                                                <?php echo $video['purchases'] ? $video['purchases'] : 0; ?>
                                                            </span>
                                                        </td>
                                                        <td class="text-end pe-4 fw-bold">
                                                            R<?php echo number_format($video['revenue'] ? $video['revenue'] : 0, 2); ?>
                                                        </td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                <?php else: ?>
                                    <div class="alert alert-info border-0 m-4 rounded-3">
                                        <div class="d-flex align-items-center">
                                            <i class="bx bx-info-circle fs-3 me-3"></i>
                                            <p class="mb-0">No video sales data available yet.</p>
                                        </div>
                                    </div>
                                <?php endif; ?>
                            </div>
                            <div class="card-footer bg-white border-0 text-end p-3">
                                <a href="videos.php" class="btn btn-sm btn-outline-primary rounded-pill px-3">
                                    <i class="bx bx-list-ul me-1"></i> View All Videos
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- User Statistics -->
                <div class="row">
                    <div class="col-12">
                        <div class="card shadow-sm">
                            <div class="card-header bg-white py-3">
                                <h5 class="mb-0">
                                    <i class="bx bx-line-chart me-2 text-primary"></i>
                                    User Growth
                                </h5>
                            </div>
                            <div class="card-body">
                                <p class="text-muted">This chart will display user registration growth over time. Coming soon!</p>
                                <div class="text-center text-muted py-4">
                                    <i class="bx bx-chart bx-lg"></i>
                                    <p>User growth visualization will be available in the next update.</p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        // Sidebar toggle functionality
        const sidebarToggle = document.querySelector('.sidebar-toggle');
        const sidebarToggleMobile = document.querySelector('.sidebar-toggle-mobile');
        const sidebar = document.querySelector('.sidebar');
        
        if (sidebarToggle) {
            sidebarToggle.addEventListener('click', () => {
                sidebar.classList.toggle('close');
            });
        }
        
        if (sidebarToggleMobile) {
            sidebarToggleMobile.addEventListener('click', () => {
                sidebar.classList.toggle('show');
            });
        }
        
        // Responsive sidebar behavior
        function checkScreenSize() {
            if (window.innerWidth < 768) {
                sidebar.classList.remove('close');
                sidebar.classList.remove('show');
            } else if (window.innerWidth < 992) {
                sidebar.classList.add('close');
                sidebar.classList.remove('show');
            }
        }
        
        // Initial check and listen for window resize
        checkScreenSize();
        window.addEventListener('resize', checkScreenSize);
        
        // Revenue Chart
        const ctx = document.getElementById('revenueChart').getContext('2d');
        const monthNames = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];
        const monthlyRevenue = [
            <?php echo implode(', ', array_values($monthlyData)); ?>
        ];
        
        new Chart(ctx, {
            type: 'bar',
            data: {
                labels: monthNames,
                datasets: [{
                    label: 'Revenue (R)',
                    data: monthlyRevenue,
                    backgroundColor: 'rgba(59, 130, 246, 0.5)',
                    borderColor: 'rgb(59, 130, 246)',
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                return 'R' + context.raw.toFixed(2);
                            }
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            callback: function(value) {
                                return 'R' + value;
                            }
                        }
                    }
                }
            }
        });
    });
    </script>
</body>
</html>
