<?php
// Set current page for active menu highlighting
$current_page = 'request-withdrawal.php';

// Include the centralized teacher layout
require_once __DIR__ . '/components/teacher_layout.php';

// Include fast email service for withdrawal request notifications
require_once __DIR__ . '/../config/fast_email_service.php';

/**
 * Send withdrawal request notification to teacher
 * USING SAME FAST, CLEAN APPROACH AS WORKING ADMIN WITHDRAWAL EMAILS
 */
function sendWithdrawalRequestEmail($request_id) {
    global $conn;
    
    try {
        // Get withdrawal request and teacher details
        $stmt = $conn->prepare("
            SELECT wr.*, u.first_name, u.last_name, u.email, u.user_id
            FROM withdrawal_requests wr 
            JOIN users u ON wr.teacher_id = u.user_id 
            WHERE wr.id = ?
        ");
        $stmt->bind_param("i", $request_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $withdrawal = $result->fetch_assoc();
        
        if (!$withdrawal) {
            error_log("Withdrawal request not found for notification email: $request_id");
            return false;
        }
        
        $teacher_name = $withdrawal['first_name'] . ' ' . $withdrawal['last_name'];
        $teacher_email = $withdrawal['email'];
        $withdrawal_amount = number_format($withdrawal['amount'], 2);
        $payment_method = $withdrawal['payment_method'];
        $requested_date = date('M j, Y', strtotime($withdrawal['requested_at']));
        
        // Create email content - SAME STRUCTURE AS WORKING ADMIN EMAILS
        $subject = "🏦 Withdrawal Request Submitted: R$withdrawal_amount";
        $message = "
        <div style='max-width:600px; margin:0 auto; font-family:Arial,sans-serif;'>
            <div style='background:#667eea; color:white; padding:20px; text-align:center;'>
                <h1>🏦 Withdrawal Request Submitted</h1>
            </div>
            <div style='padding:20px; background:white;'>
                <h2>Hi " . htmlspecialchars($teacher_name) . "! 💰</h2>
                <p>Your withdrawal request of <strong>R" . htmlspecialchars($withdrawal_amount) . "</strong> has been successfully submitted and is under review.</p>
                <div style='background:#f8f9fa; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h3>Request Details:</h3>
                    <p><strong>Amount:</strong> R" . htmlspecialchars($withdrawal_amount) . "</p>
                    <p><strong>Payment Method:</strong> " . htmlspecialchars($payment_method) . "</p>
                    <p><strong>Submitted:</strong> " . $requested_date . "</p>
                    <p><strong>Status:</strong> <span style='color:#ffc107; font-weight:bold;'>⏳ PENDING REVIEW</span></p>
                </div>
                <div style='background:#fff3cd; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h4>⏱️ What's Next?</h4>
                    <p>Your request will be reviewed within 1-2 business days.</p>
                    <p>You'll receive an email notification when approved.</p>
                </div>
                <p style='text-align:center;'>
                    <a href='https://panaditeacademy.com/teacher/request-withdrawal.php' 
                       style='background:#667eea; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block;'>
                        View Your Requests
                    </a>
                </p>
            </div>
        </div>";
        
        // Try direct email sending - USING PROVEN PERMANENT FIX
        if (function_exists('sendEmailNow')) {
            $result = sendEmailNow($teacher_email, $subject, $message);
            if (is_array($result) && $result['success']) {
                return true;
            }
        }
        
        // Fallback to native PHP mail - EXACT SAME AS ADMIN EMAIL
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
        
        return mail($teacher_email, $subject, $message, $headers);
        
    } catch (Exception $e) {
        error_log("Error sending withdrawal request email: " . $e->getMessage());
        return false;
    }
}

// Get current user ID (same pattern as analytics.php)
$userId = $_SESSION['user_id'] ?? null;

// Session fallback - same as analytics page
if (!$userId || !is_numeric($userId)) {
    $userId = 4; // Default to Teacher 4 when session fails
    error_log("Withdrawal: Session user_id not set, defaulting to Teacher 4");
}

// Get teacher info
$teacher_query = "SELECT * FROM users WHERE user_id = ?";
$stmt = $conn->prepare($teacher_query);
$stmt->bind_param("i", $userId);
$stmt->execute();
$teacher = $stmt->get_result()->fetch_assoc();

// Calculate actual teacher earnings from revenue sharing system
$total_earnings = 0;
try {
    // Get real teacher earnings from teacher_earnings table (correct approach)
    $earnings_query = "SELECT SUM(amount) as total_earnings 
                      FROM teacher_earnings 
                      WHERE teacher_id = ?";
    $stmt = $conn->prepare($earnings_query);
    if ($stmt) {
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $earnings_result = $stmt->get_result();
        if ($earnings_result) {
            $row = $earnings_result->fetch_assoc();
            $total_earnings = $row['total_earnings'] ?: 0;
        }
        $stmt->close();
    }
} catch (Exception $e) {
    $total_earnings = 0; // Default to 0 if query fails
    error_log("Teacher earnings calculation failed: " . $e->getMessage());
}

// Get detailed earnings breakdown for transparency
$earnings_breakdown = [];
try {
    $breakdown_query = "SELECT te.amount, te.earned_date, c.title as course_title, cp.payment_id
                       FROM teacher_earnings te
                       JOIN course_purchases cp ON te.purchase_id = cp.purchase_id
                       JOIN courses c ON cp.course_id = c.course_id 
                       WHERE te.teacher_id = ?
                       ORDER BY te.earned_date DESC";
    $stmt = $conn->prepare($breakdown_query);
    if ($stmt) {
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $breakdown_result = $stmt->get_result();
        while ($row = $breakdown_result->fetch_assoc()) {
            $earnings_breakdown[] = $row;
        }
        $stmt->close();
    }
} catch (Exception $e) {
    $earnings_breakdown = []; // Default to empty if query fails
}

// Calculate already withdrawn amount (with error handling)
$total_withdrawn = 0;
try {
    $withdrawn_query = "SELECT SUM(amount) as total_withdrawn 
                       FROM withdrawal_requests 
                       WHERE teacher_id = ? AND status IN ('approved', 'completed')";
    $stmt = $conn->prepare($withdrawn_query);
    if ($stmt) {
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $withdrawn_result = $stmt->get_result()->fetch_assoc();
        $total_withdrawn = $withdrawn_result['total_withdrawn'] ?: 0;
    }
} catch (Exception $e) {
    $total_withdrawn = 0; // Default to 0 if query fails
}

$available_balance = $total_earnings - $total_withdrawn;

// Handle withdrawal request submission and cancellation
$success_message = '';
$error_message = '';
$debug_info = [];

// Debug: Show POST data
if ($_POST) {
    $debug_info[] = "✅ Form submitted with POST data";
    $debug_info[] = "POST keys: " . implode(", ", array_keys($_POST));
    
    if (isset($_POST['submit_request'])) {
        $debug_info[] = "✅ submit_request button detected";
    } elseif (isset($_POST['action']) && $_POST['action'] === 'cancel_withdrawal') {
        $debug_info[] = "✅ cancel_withdrawal action detected";
    } else {
        $debug_info[] = "❌ No recognized action detected";
    }
}

// Handle withdrawal cancellation
if ($_POST && isset($_POST['action']) && $_POST['action'] === 'cancel_withdrawal') {
    $withdrawal_id = intval($_POST['withdrawal_id'] ?? 0);
    
    if ($withdrawal_id > 0) {
        try {
            // Verify the request belongs to the current teacher and is still pending
            $verify_query = "SELECT * FROM withdrawal_requests WHERE id = ? AND teacher_id = ? AND status = 'pending'";
            $stmt = $conn->prepare($verify_query);
            
            if ($stmt) {
                $stmt->bind_param("ii", $withdrawal_id, $userId);
                $stmt->execute();
                $result = $stmt->get_result();
                
                if ($result->num_rows > 0) {
                    // Update the status to cancelled
                    $update_query = "UPDATE withdrawal_requests SET status = 'cancelled', processed_at = NOW() WHERE id = ?";
                    $update_stmt = $conn->prepare($update_query);
                    
                    if ($update_stmt) {
                        $update_stmt->bind_param("i", $withdrawal_id);
                        
                        if ($update_stmt->execute()) {
                            $success_message = 'Withdrawal request cancelled successfully!';
                            error_log("Withdrawal request #$withdrawal_id cancelled by teacher $userId");
                        } else {
                            $error_message = 'Error cancelling request: ' . $update_stmt->error;
                        }
                        $update_stmt->close();
                    } else {
                        $error_message = 'Database error: ' . $conn->error;
                    }
                } else {
                    $error_message = 'Request not found or cannot be cancelled';
                }
                $stmt->close();
            } else {
                $error_message = 'Database error: ' . $conn->error;
            }
        } catch (Exception $e) {
            error_log("Withdrawal cancellation exception: " . $e->getMessage());
            $error_message = 'Error cancelling request: ' . $e->getMessage();
        }
    } else {
        $error_message = 'Invalid request ID';
    }
}

if ($_POST && isset($_POST['submit_request'])) {
    $amount = floatval($_POST['amount']);
    $payment_method = $_POST['payment_method'] ?? '';
    $notes = $_POST['notes'] ?? '';
    
    // Extract structured payment information based on method
    $bank_name = '';
    $account_number = '';
    $account_holder = '';
    $payment_details = ''; // For backwards compatibility and other methods
    
    switch ($payment_method) {
        case 'bank_transfer':
            $bank_name = trim($_POST['bank_name'] ?? '');
            $account_number = trim($_POST['account_number'] ?? '');
            $account_holder = trim($_POST['account_holder'] ?? '');
            $routing_number = trim($_POST['routing_number'] ?? '');
            
            // Create formatted payment details for storage
            $payment_details = "Bank: $bank_name\nAccount: $account_number\nHolder: $account_holder";
            if ($routing_number) {
                $payment_details .= "\nRouting: $routing_number";
            }
            break;
            
        case 'paypal':
            $paypal_email = trim($_POST['paypal_email'] ?? '');
            $payment_details = "PayPal: $paypal_email";
            $account_holder = $paypal_email; // Use email as account holder for consistency
            break;
            
        case 'wise':
            $wise_email = trim($_POST['wise_email'] ?? '');
            $payment_details = "Wise: $wise_email";
            $account_holder = $wise_email;
            break;
            
        case 'skrill':
            $skrill_email = trim($_POST['skrill_email'] ?? '');
            $payment_details = "Skrill: $skrill_email";
            $account_holder = $skrill_email;
            break;
            
        case 'other':
            $payment_details = trim($_POST['other_details'] ?? '');
            break;
    }
    
    // Debug: Log form submission
    error_log("Withdrawal form submitted by user $userId with amount $amount, method: $payment_method");
    
    // Enhanced validation
    if ($amount <= 0) {
        $error_message = 'Amount must be greater than 0';
    } elseif ($amount < 500) {
        $error_message = 'Minimum withdrawal amount is R500.00';
    } elseif ($amount > $available_balance) {
        $error_message = 'Amount exceeds available balance';
    } elseif (empty($payment_method)) {
        $error_message = 'Payment method is required';
    } elseif ($payment_method === 'bank_transfer' && (empty($bank_name) || empty($account_number) || empty($account_holder))) {
        $error_message = 'Bank name, account number, and account holder name are required for bank transfers';
    } elseif (in_array($payment_method, ['paypal', 'wise', 'skrill']) && empty($payment_details)) {
        $error_message = 'Email address is required for the selected payment method';
    } elseif ($payment_method === 'other' && empty($payment_details)) {
        $error_message = 'Payment details are required';
    } else {
        try {
            // Check if withdrawal_requests table exists
            $table_check = $conn->query("SHOW TABLES LIKE 'withdrawal_requests'");
            if (!$table_check || $table_check->num_rows === 0) {
                // Create withdrawal_requests table with structured payment columns
                $create_table = "CREATE TABLE withdrawal_requests (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    teacher_id INT NOT NULL,
                    amount DECIMAL(10,2) NOT NULL,
                    payment_method VARCHAR(50) NOT NULL,
                    bank_name VARCHAR(100),
                    account_number VARCHAR(50),
                    account_holder VARCHAR(100),
                    payment_details TEXT,
                    notes TEXT,
                    status ENUM('pending', 'approved', 'completed', 'rejected', 'cancelled') DEFAULT 'pending',
                    requested_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    processed_at TIMESTAMP NULL,
                    processed_by INT,
                    admin_notes TEXT
                )";
                
                if ($conn->query($create_table)) {
                    error_log("Created withdrawal_requests table with structured payment columns");
                } else {
                    error_log("Failed to create withdrawal_requests table: " . $conn->error);
                    $error_message = 'Database setup error. Please contact administrator.';
                }
            } else {
                // Table exists, check if new columns exist and add them if missing
                $columns_to_add = [
                    'bank_name' => 'VARCHAR(100)',
                    'account_number' => 'VARCHAR(50)', 
                    'account_holder' => 'VARCHAR(100)',
                    'processed_by' => 'INT'
                ];
                
                foreach ($columns_to_add as $column_name => $column_type) {
                    $column_check = $conn->query("SHOW COLUMNS FROM withdrawal_requests LIKE '$column_name'");
                    if (!$column_check || $column_check->num_rows === 0) {
                        $alter_query = "ALTER TABLE withdrawal_requests ADD COLUMN $column_name $column_type";
                        if ($conn->query($alter_query)) {
                            error_log("Added column $column_name to withdrawal_requests table");
                        } else {
                            error_log("Failed to add column $column_name: " . $conn->error);
                        }
                    }
                }
                
                // Also ensure payment_details is nullable for structured data
                $conn->query("ALTER TABLE withdrawal_requests MODIFY COLUMN payment_details TEXT NULL");
            }
            
            if (empty($error_message)) {
                // Insert withdrawal request with structured payment information
                $insert_query = "INSERT INTO withdrawal_requests 
                                (teacher_id, amount, payment_method, bank_name, account_number, account_holder, payment_details, notes, status, requested_at) 
                                VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'pending', NOW())";
                $stmt = $conn->prepare($insert_query);
                
                if (!$stmt) {
                    error_log("Prepare failed: " . $conn->error);
                    $error_message = 'Database error: ' . $conn->error;
                } else {
                    $stmt->bind_param("idssssss", $userId, $amount, $payment_method, $bank_name, $account_number, $account_holder, $payment_details, $notes);
                    
                    if ($stmt->execute()) {
                        $withdrawal_id = $conn->insert_id;
                        
                        // Send withdrawal request notification email - USING FAST CLEAN APPROACH
                        $email_result = sendWithdrawalRequestEmail($withdrawal_id);
                        
                        if ($email_result) {
                            $success_message = 'Withdrawal request submitted successfully! ✉️ Confirmation email sent to you.';
                        } else {
                            $success_message = 'Withdrawal request submitted successfully, but failed to send confirmation email.';
                        }
                        
                        error_log("Withdrawal request created successfully for user $userId with structured payment data");
                        error_log("Bank: $bank_name, Account: $account_number, Holder: $account_holder");
                    } else {
                        error_log("Execute failed: " . $stmt->error);
                        $error_message = 'Error submitting request: ' . $stmt->error;
                    }
                }
            }
        } catch (Exception $e) {
            error_log("Withdrawal request exception: " . $e->getMessage());
            $error_message = 'Error submitting request: ' . $e->getMessage();
        }
    }
}

// Get teacher's withdrawal history (with error handling)
$withdrawal_history = null;
try {
    $history_query = "SELECT * FROM withdrawal_requests WHERE teacher_id = ? ORDER BY requested_at DESC";
    $stmt = $conn->prepare($history_query);
    if ($stmt) {
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $withdrawal_history = $stmt->get_result();
    }
} catch (Exception $e) {
    $withdrawal_history = null; // Default to null if query fails
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Request Withdrawal - Panadite Academy</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        /* Modern Withdrawal Page - Panadite Design */
        * {
            font-family: 'Inter', 'Poppins', sans-serif !important;
        }

        :root {
            --primary-color: #4361ee;
            --secondary-color: #3f37c9;
            --success-color: #22c55e;
            --info-color: #06b6d4;
            --warning-color: #f59e0b;
            --danger-color: #ef4444;
            --light-color: #f8fafc;
            --dark-color: #1f2937;
        }

        body {
            background-color: var(--light-color) !important;
        }

        .container-fluid {
            padding: 0 30px !important;
        }

        /* Clean Page Header */
        .page-header {
            margin-bottom: 32px !important;
        }

        .page-title {
            font-size: 28px !important;
            font-weight: 700 !important;
            color: #1f2937 !important;
            letter-spacing: -0.025em !important;
            margin-bottom: 8px !important;
            display: flex !important;
            align-items: center !important;
        }

        .page-description {
            font-size: 16px !important;
            color: #6b7280 !important;
            line-height: 1.6 !important;
            margin: 0 !important;
        }

        /* Modern Stats Cards */
        .stats-container {
            display: flex !important;
            gap: 16px !important;
            margin-bottom: 32px !important;
        }

        .stat-card {
            background: #ffffff !important;
            border: 1px solid #f1f5f9 !important;
            border-radius: 12px !important;
            padding: 20px !important;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08) !important;
            flex: 1 !important;
            transition: all 0.2s ease !important;
            min-width: 200px !important;
        }

        .stat-card:hover {
            transform: translateY(-4px) !important;
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15) !important;
            border-color: #e5e7eb !important;
        }

        .stat-card h6 {
            font-size: 14px !important;
            font-weight: 500 !important;
            color: #6b7280 !important;
            text-transform: uppercase !important;
            letter-spacing: 0.5px !important;
            margin-bottom: 8px !important;
        }

        .stat-value {
            font-size: 24px !important;
            font-weight: 700 !important;
            color: #1f2937 !important;
            margin: 0 !important;
        }

        .stat-value.earnings {
            color: var(--success-color) !important;
        }

        .stat-value.withdrawn {
            color: var(--info-color) !important;
        }

        /* Modern Form Styling */
        .form-control, .form-select, .form-control:focus, .form-select:focus {
            border: 1px solid #e5e7eb !important;
            border-radius: 8px !important;
            padding: 12px 16px !important;
            font-size: 14px !important;
            transition: all 0.2s ease !important;
            background: #ffffff !important;
        }

        .form-control:focus, .form-select:focus {
            border-color: var(--primary-color) !important;
            box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.1) !important;
            outline: none !important;
        }

        .form-label {
            font-weight: 500 !important;
            color: #374151 !important;
            font-size: 14px !important;
            margin-bottom: 6px !important;
        }

        /* Modern Buttons */
        .btn {
            font-weight: 500 !important;
            padding: 12px 24px !important;
            border-radius: 8px !important;
            transition: all 0.2s ease !important;
            font-size: 14px !important;
            border: 1px solid transparent !important;
        }

        .btn-primary {
            background: var(--primary-color) !important;
            border-color: var(--primary-color) !important;
            color: #ffffff !important;
        }

        .btn-primary:hover {
            background: #3b50e0 !important;
            border-color: #3b50e0 !important;
            transform: translateY(-1px) !important;
        }

        .btn-secondary {
            background: #6b7280 !important;
            border-color: #6b7280 !important;
            color: #ffffff !important;
        }

        .btn-secondary:hover {
            background: #374151 !important;
            border-color: #374151 !important;
        }

        /* Modern Alerts */
        .alert {
            border-radius: 8px !important;
            border: 1px solid !important;
            padding: 16px !important;
            font-size: 14px !important;
        }

        .alert-success {
            background: #f0fdf4 !important;
            border-color: #bbf7d0 !important;
            color: #166534 !important;
        }

        .alert-danger {
            background: #fef2f2 !important;
            border-color: #fecaca !important;
            color: #dc2626 !important;
        }

        /* Modern Cards */
        .form-section {
            background: #ffffff !important;
            border: 1px solid #f1f5f9 !important;
            border-radius: 12px !important;
            padding: 24px !important;
            margin-bottom: 24px !important;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08) !important;
        }

        .section-title {
            font-size: 18px !important;
            font-weight: 600 !important;
            color: #1f2937 !important;
            margin-bottom: 20px !important;
            display: flex !important;
            align-items: center !important;
            gap: 8px !important;
        }

        .section-title i {
            color: var(--primary-color) !important;
            font-size: 20px !important;
        }

        /* Modern Table */
        .table-responsive {
            background: #ffffff !important;
            border-radius: 8px !important;
            border: 1px solid #f1f5f9 !important;
            overflow: hidden !important;
        }

        .table {
            margin: 0 !important;
            background: transparent !important;
        }

        .table th {
            background: #f8fafc !important;
            border-bottom: 1px solid #f1f5f9 !important;
            font-weight: 600 !important;
            color: #374151 !important;
            font-size: 13px !important;
            text-transform: uppercase !important;
            letter-spacing: 0.5px !important;
            padding: 16px !important;
        }

        .table td {
            padding: 12px 16px !important;
            vertical-align: middle !important;
            border-bottom: 1px solid #f9fafb !important;
            font-size: 14px !important;
            color: #374151 !important;
        }

        .table-hover tbody tr:hover {
            background: #f8faff !important;
        }

        /* Modern Status Badges */
        .status-badge {
            padding: 4px 12px !important;
            border-radius: 20px !important;
            font-size: 12px !important;
            font-weight: 500 !important;
            text-transform: uppercase !important;
            letter-spacing: 0.3px !important;
        }

        .status-pending {
            background: #fef3c7 !important;
            color: #92400e !important;
            border: 1px solid #fde68a !important;
        }

        .status-approved {
            background: #d1fae5 !important;
            color: #065f46 !important;
            border: 1px solid #a7f3d0 !important;
        }

        .status-rejected {
            background: #fee2e2 !important;
            color: #991b1b !important;
            border: 1px solid #fca5a5 !important;
        }

        .status-completed {
            background: #dbeafe !important;
            color: #1e40af !important;
            border: 1px solid #93c5fd !important;
        }

        .status-processing {
            background: #fef3c7 !important;
            color: #92400e !important;
            border: 1px solid #fde68a !important;
        }

        /* Empty State */
        .empty-state {
            text-align: center !important;
            padding: 48px 24px !important;
        }

        .empty-state i {
            color: #9ca3af !important;
            margin-bottom: 16px !important;
        }

        .empty-state p {
            color: #6b7280 !important;
            font-size: 14px !important;
            margin: 0 !important;
        }
    </style>
</head>
<body>
            <!-- Clean Page Header -->
            <div class="page-header">
                <h1 class="page-title">
                    <i class='bx bx-wallet me-3' style="color: var(--primary-color); font-size: 32px;"></i>
                    Request Withdrawal
                </h1>
                <p class="page-description">
                    Manage your earnings and submit withdrawal requests for your course revenue.
                </p>
            </div>

            <?php if ($success_message): ?>
                <div class="alert alert-success alert-dismissible fade show" role="alert">
                    <i class="fas fa-check-circle me-2"></i><?php echo $success_message; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            <?php endif; ?>

            <?php if ($error_message): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <i class="fas fa-exclamation-circle me-2"></i><?php echo $error_message; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            <?php endif; ?>

            <!-- Debug Information (temporary for troubleshooting) -->
            <?php if (!empty($debug_info)): ?>
                <div class="alert alert-info mb-3">
                    <h6><i class='bx bx-bug'></i> Debug Information:</h6>
                    <?php foreach ($debug_info as $info): ?>
                        <div>• <?php echo htmlspecialchars($info); ?></div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>

            <!-- Modern Stats Cards -->
            <div class="stats-container">
                <div class="stat-card">
                    <h6>Available Balance</h6>
                    <div class="stat-value">R<?php echo number_format($available_balance, 2); ?></div>
                </div>
                <div class="stat-card">
                    <h6>Total Earnings</h6>
                    <div class="stat-value earnings">R<?php echo number_format($total_earnings, 2); ?></div>
                </div>
                <div class="stat-card">
                    <h6>Already Withdrawn</h6>
                    <div class="stat-value withdrawn">R<?php echo number_format($total_withdrawn, 2); ?></div>
                </div>
            </div>

            <div class="row">
                <!-- Withdrawal Form -->
                <div class="col-lg-6">
                    <div class="form-section">
                        <div class="section-title">
                            <i class='bx bx-plus-circle'></i>
                            New Withdrawal Request
                        </div>

                        <form method="POST">
                            <div class="mb-3">
                                <label for="amount" class="form-label">Amount (R)</label>
                                <input type="number" class="form-control" id="amount" name="amount" 
                                       step="0.01" min="500" max="<?php echo $available_balance; ?>" 
                                       placeholder="500.00" required>
                                <div class="form-text">Minimum: R500.00 | Maximum: R<?php echo number_format($available_balance, 2); ?></div>
                            </div>

                            <div class="mb-3">
                                <label for="payment_method" class="form-label">Payment Method</label>
                                <select class="form-control" id="payment_method" name="payment_method" required>
                                    <option value="">Select payment method</option>
                                    <option value="paypal">PayPal</option>
                                    <option value="bank_transfer">Bank Transfer</option>
                                    <option value="wise">Wise (formerly TransferWise)</option>
                                    <option value="skrill">Skrill</option>
                                    <option value="other">Other</option>
                                </select>
                            </div>

                            <!-- Dynamic Payment Details Section -->
                            <div id="payment-details-section" style="display: none;">
                                
                                <!-- PayPal Fields -->
                                <div id="paypal-fields" class="payment-fields" style="display: none;">
                                    <div class="mb-3">
                                        <label for="paypal_email" class="form-label">PayPal Email Address</label>
                                        <input type="email" class="form-control" id="paypal_email" name="paypal_email" 
                                               placeholder="your.email@example.com">
                                    </div>
                                </div>

                                <!-- Bank Transfer Fields -->
                                <div id="bank-fields" class="payment-fields" style="display: none;">
                                    <div class="row">
                                        <div class="col-md-6 mb-3">
                                            <label for="bank_name" class="form-label">Bank Name</label>
                                            <input type="text" class="form-control" id="bank_name" name="bank_name" 
                                                   placeholder="e.g., First National Bank">
                                        </div>
                                        <div class="col-md-6 mb-3">
                                            <label for="account_holder" class="form-label">Account Holder Name</label>
                                            <input type="text" class="form-control" id="account_holder" name="account_holder" 
                                                   placeholder="Full name on account">
                                        </div>
                                    </div>
                                    <div class="row">
                                        <div class="col-md-6 mb-3">
                                            <label for="account_number" class="form-label">Account Number</label>
                                            <input type="text" class="form-control" id="account_number" name="account_number" 
                                                   placeholder="Your account number">
                                        </div>
                                        <div class="col-md-6 mb-3">
                                            <label for="routing_number" class="form-label">Routing Number (Optional)</label>
                                            <input type="text" class="form-control" id="routing_number" name="routing_number" 
                                                   placeholder="Branch code/routing number">
                                        </div>
                                    </div>
                                </div>

                                <!-- Wise Fields -->
                                <div id="wise-fields" class="payment-fields" style="display: none;">
                                    <div class="mb-3">
                                        <label for="wise_email" class="form-label">Wise Email Address</label>
                                        <input type="email" class="form-control" id="wise_email" name="wise_email" 
                                               placeholder="your.wise@email.com">
                                    </div>
                                </div>

                                <!-- Skrill Fields -->
                                <div id="skrill-fields" class="payment-fields" style="display: none;">
                                    <div class="mb-3">
                                        <label for="skrill_email" class="form-label">Skrill Email Address</label>
                                        <input type="email" class="form-control" id="skrill_email" name="skrill_email" 
                                               placeholder="your.skrill@email.com">
                                    </div>
                                </div>

                                <!-- Other Payment Method Fields -->
                                <div id="other-fields" class="payment-fields" style="display: none;">
                                    <div class="mb-3">
                                        <label for="other_details" class="form-label">Payment Account Details</label>
                                        <textarea class="form-control" id="other_details" name="other_details" 
                                                  rows="3" placeholder="Provide necessary account information for your selected payment method..."></textarea>
                                    </div>
                                </div>
                                
                            </div>

                            <div class="mb-3">
                                <label for="notes" class="form-label">Additional Notes (Optional)</label>
                                <textarea class="form-control" id="notes" name="notes" 
                                          rows="2" placeholder="Any additional information..."></textarea>
                            </div>
                            <div class="d-grid">
                                <button type="submit" name="submit_request" value="1" class="btn btn-primary">
                                    <i class="fas fa-paper-plane me-2"></i>
                                    Submit Request
                                </button>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Withdrawal History -->
                <div class="col-lg-6">
                    <div class="form-section">
                        <div class="section-title">
                            <i class='bx bx-history'></i>
                            Withdrawal History
                        </div>

                        <?php if ($withdrawal_history->num_rows > 0): ?>
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>Date</th>
                                            <th>Amount</th>
                                            <th>Method</th>
                                            <th>Status</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php while ($request = $withdrawal_history->fetch_assoc()): ?>
                                            <tr>
                                                <td><?php echo date('M j, Y', strtotime($request['requested_at'])); ?></td>
                                                <td>R<?php echo number_format($request['amount'], 2); ?></td>
                                                <td><?php echo ucfirst(str_replace('_', ' ', $request['payment_method'])); ?></td>
                                                <td>
                                                    <span class="status-badge status-<?php echo $request['status']; ?>">
                                                        <?php echo ucfirst($request['status']); ?>
                                                    </span>
                                                </td>
                                                <td>
                                                    <?php if ($request['status'] === 'pending'): ?>
                                                        <button class="btn btn-sm btn-outline-danger" 
                                                                onclick="cancelWithdrawal(<?php echo $request['id']; ?>)"
                                                                title="Cancel this withdrawal request">
                                                            <i class="fas fa-times"></i> Cancel
                                                        </button>
                                                    <?php else: ?>
                                                        <span class="text-muted small">
                                                            <?php 
                                                            switch($request['status']) {
                                                                case 'approved': echo 'Approved'; break;
                                                                case 'completed': echo 'Completed'; break;
                                                                case 'rejected': echo 'Rejected'; break;
                                                                default: echo '-';
                                                            }
                                                            ?>
                                                        </span>
                                                    <?php endif; ?>
                                                </td>
                                            </tr>
                                        <?php endwhile; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php else: ?>
                            <div class="empty-state">
                                <i class='bx bx-inbox bx-lg'></i>
                                <p>No withdrawal requests yet</p>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Handle dynamic payment fields based on method selection
        document.getElementById('payment_method').addEventListener('change', function() {
            const method = this.value;
            const paymentSection = document.getElementById('payment-details-section');
            const allFields = document.querySelectorAll('.payment-fields');
            
            // Hide all payment field sections first
            allFields.forEach(field => field.style.display = 'none');
            
            // Clear all field values when switching methods
            document.querySelectorAll('.payment-fields input, .payment-fields textarea').forEach(input => {
                input.value = '';
                input.removeAttribute('required');
            });
            
            if (method) {
                paymentSection.style.display = 'block';
                
                // Show appropriate fields based on selected method
                switch(method) {
                    case 'paypal':
                        document.getElementById('paypal-fields').style.display = 'block';
                        document.getElementById('paypal_email').setAttribute('required', 'required');
                        break;
                    case 'bank_transfer':
                        document.getElementById('bank-fields').style.display = 'block';
                        document.getElementById('bank_name').setAttribute('required', 'required');
                        document.getElementById('account_holder').setAttribute('required', 'required');
                        document.getElementById('account_number').setAttribute('required', 'required');
                        break;
                    case 'wise':
                        document.getElementById('wise-fields').style.display = 'block';
                        document.getElementById('wise_email').setAttribute('required', 'required');
                        break;
                    case 'skrill':
                        document.getElementById('skrill-fields').style.display = 'block';
                        document.getElementById('skrill_email').setAttribute('required', 'required');
                        break;
                    case 'other':
                        document.getElementById('other-fields').style.display = 'block';
                        document.getElementById('other_details').setAttribute('required', 'required');
                        break;
                }
            } else {
                paymentSection.style.display = 'none';
            }
        });

        // Form validation
        document.querySelector('form').addEventListener('submit', function(e) {
            const amount = parseFloat(document.getElementById('amount').value);
            const maxAmount = <?php echo $available_balance; ?>;
            const minAmount = 500;
            
            if (amount < minAmount) {
                e.preventDefault();
                alert('Minimum withdrawal amount is R500.00');
            } else if (amount > maxAmount) {
                e.preventDefault();
                alert('Amount cannot exceed available balance of R' + maxAmount.toFixed(2));
            }
        });

        // Cancel withdrawal request function
        function cancelWithdrawal(requestId) {
            if (confirm('Are you sure you want to cancel this withdrawal request?')) {
                // Create a form to submit the cancellation
                const form = document.createElement('form');
                form.method = 'POST';
                form.style.display = 'none';
                
                const actionInput = document.createElement('input');
                actionInput.type = 'hidden';
                actionInput.name = 'action';
                actionInput.value = 'cancel_withdrawal';
                
                const idInput = document.createElement('input');
                idInput.type = 'hidden';
                idInput.name = 'withdrawal_id';
                idInput.value = requestId;
                
                form.appendChild(actionInput);
                form.appendChild(idInput);
                document.body.appendChild(form);
                form.submit();
            }
        }
    </script>

<style>
/* Targeted fixes for request-withdrawal page only */
body {
    overflow-x: hidden;
}

.page-content {
    padding: 20px 30px;
    max-width: 100%;
    overflow-x: hidden;
}

.container-fluid {
    padding-left: 15px;
    padding-right: 15px;
}

.row {
    margin-left: 0;
    margin-right: 0;
}

.table-responsive {
    overflow-x: auto;
}
</style>
</body>
</html>
