<?php
// Database setup script for Revenue Sharing System
require_once __DIR__ . '/../includes/connection.php';

// Check if admin is logged in
session_start();
if (!isset($_SESSION['admin_id'])) {
    header("Location: ../auth/admin_login.php");
    exit();
}

$setup_results = [];
$errors = [];

try {
    // 1. Create revenue_settings table
    $sql_revenue_settings = "
    CREATE TABLE IF NOT EXISTS `revenue_settings` (
        `id` int(11) NOT NULL AUTO_INCREMENT,
        `setting_name` varchar(100) NOT NULL UNIQUE,
        `setting_value` varchar(255) NOT NULL,
        `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
        `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (`id`),
        UNIQUE KEY `setting_name` (`setting_name`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
    ";
    
    if ($conn->query($sql_revenue_settings)) {
        $setup_results[] = "✅ Revenue settings table created successfully";
        
        // Insert default settings
        $conn->query("
            INSERT IGNORE INTO revenue_settings (setting_name, setting_value) 
            VALUES 
            ('admin_percentage', '20'),
            ('teacher_percentage', '80')
        ");
        $setup_results[] = "✅ Default revenue split settings inserted (Admin: 20%, Teacher: 80%)";
    } else {
        $errors[] = "❌ Error creating revenue_settings table: " . $conn->error;
    }

    // 2. Create course_purchases table (if it doesn't exist)
    $sql_course_purchases = "
    CREATE TABLE IF NOT EXISTS `course_purchases` (
        `purchase_id` int(11) NOT NULL AUTO_INCREMENT,
        `user_id` int(11) NOT NULL,
        `course_id` int(11) NOT NULL,
        `teacher_id` int(11) NOT NULL,
        `amount` decimal(10,2) NOT NULL,
        `admin_amount` decimal(10,2) NOT NULL,
        `teacher_amount` decimal(10,2) NOT NULL,
        `admin_percentage` decimal(5,2) NOT NULL,
        `teacher_percentage` decimal(5,2) NOT NULL,
        `status` enum('pending','completed','failed','refunded') DEFAULT 'pending',
        `payment_method` varchar(50) DEFAULT NULL,
        `transaction_id` varchar(255) DEFAULT NULL,
        `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
        `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (`purchase_id`),
        KEY `user_id` (`user_id`),
        KEY `course_id` (`course_id`),
        KEY `teacher_id` (`teacher_id`),
        KEY `status` (`status`),
        KEY `created_at` (`created_at`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
    ";
    
    if ($conn->query($sql_course_purchases)) {
        $setup_results[] = "✅ Course purchases table created successfully";
    } else {
        $errors[] = "❌ Error creating course_purchases table: " . $conn->error;
    }

    // 3. Create teacher_earnings table
    $sql_teacher_earnings = "
    CREATE TABLE IF NOT EXISTS `teacher_earnings` (
        `earning_id` int(11) NOT NULL AUTO_INCREMENT,
        `teacher_id` int(11) NOT NULL,
        `purchase_id` int(11) NOT NULL,
        `course_id` int(11) NOT NULL,
        `amount` decimal(10,2) NOT NULL,
        `status` enum('pending','available','withdrawn') DEFAULT 'pending',
        `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
        `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (`earning_id`),
        KEY `teacher_id` (`teacher_id`),
        KEY `purchase_id` (`purchase_id`),
        KEY `course_id` (`course_id`),
        KEY `status` (`status`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
    ";
    
    if ($conn->query($sql_teacher_earnings)) {
        $setup_results[] = "✅ Teacher earnings table created successfully";
    } else {
        $errors[] = "❌ Error creating teacher_earnings table: " . $conn->error;
    }

    // 4. Create admin_earnings table
    $sql_admin_earnings = "
    CREATE TABLE IF NOT EXISTS `admin_earnings` (
        `earning_id` int(11) NOT NULL AUTO_INCREMENT,
        `purchase_id` int(11) NOT NULL,
        `course_id` int(11) NOT NULL,
        `teacher_id` int(11) NOT NULL,
        `amount` decimal(10,2) NOT NULL,
        `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (`earning_id`),
        KEY `purchase_id` (`purchase_id`),
        KEY `course_id` (`course_id`),
        KEY `teacher_id` (`teacher_id`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
    ";
    
    if ($conn->query($sql_admin_earnings)) {
        $setup_results[] = "✅ Admin earnings table created successfully";
    } else {
        $errors[] = "❌ Error creating admin_earnings table: " . $conn->error;
    }

    // 5. Check if enrollments table needs course_price column
    $check_enrollments = $conn->query("SHOW COLUMNS FROM enrollments LIKE 'course_price'");
    if ($check_enrollments && $check_enrollments->num_rows == 0) {
        $alter_enrollments = "
        ALTER TABLE enrollments 
        ADD COLUMN `course_price` decimal(10,2) DEFAULT 0.00 AFTER `course_id`,
        ADD COLUMN `purchase_id` int(11) DEFAULT NULL AFTER `course_price`,
        ADD KEY `purchase_id` (`purchase_id`)
        ";
        
        if ($conn->query($alter_enrollments)) {
            $setup_results[] = "✅ Enrollments table updated with course_price and purchase_id columns";
        } else {
            $errors[] = "❌ Error updating enrollments table: " . $conn->error;
        }
    } else {
        $setup_results[] = "ℹ️ Enrollments table already has required columns";
    }

    // 6. Create revenue tracking functions
    $sql_function = "
    CREATE OR REPLACE VIEW `revenue_summary` AS
    SELECT 
        COUNT(*) as total_purchases,
        SUM(amount) as total_revenue,
        SUM(admin_amount) as total_admin_earnings,
        SUM(teacher_amount) as total_teacher_earnings,
        AVG(admin_percentage) as avg_admin_percentage,
        AVG(teacher_percentage) as avg_teacher_percentage
    FROM course_purchases 
    WHERE status = 'completed'
    ";
    
    if ($conn->query($sql_function)) {
        $setup_results[] = "✅ Revenue summary view created successfully";
    } else {
        $errors[] = "❌ Error creating revenue summary view: " . $conn->error;
    }

    $setup_results[] = "🎉 Revenue sharing system setup completed successfully!";
    
} catch (Exception $e) {
    $errors[] = "❌ Fatal error during setup: " . $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Revenue System Setup - Panadite Academy</title>
    <link href='https://unpkg.com/boxicons@2.0.9/css/boxicons.min.css' rel='stylesheet'>
    <style>
        * {
            font-family: 'Inter', 'Poppins', sans-serif !important;
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            background: #f8fafc;
            padding: 40px 20px;
        }
        
        .container {
            max-width: 800px;
            margin: 0 auto;
            background: white;
            border-radius: 12px;
            padding: 40px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.1);
        }
        
        .header {
            text-align: center;
            margin-bottom: 40px;
        }
        
        .header h1 {
            color: #1f2937;
            font-size: 28px;
            margin-bottom: 8px;
        }
        
        .header p {
            color: #6b7280;
            font-size: 16px;
        }
        
        .result-item {
            padding: 12px 16px;
            margin-bottom: 8px;
            border-radius: 8px;
            font-size: 14px;
            border-left: 4px solid;
        }
        
        .success {
            background: #f0fdf4;
            color: #166534;
            border-left-color: #22c55e;
        }
        
        .error {
            background: #fef2f2;
            color: #dc2626;
            border-left-color: #ef4444;
        }
        
        .info {
            background: #eff6ff;
            color: #1d4ed8;
            border-left-color: #3b82f6;
        }
        
        .celebration {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            text-align: center;
            font-weight: 600;
            font-size: 16px;
        }
        
        .nav-links {
            margin-top: 40px;
            text-align: center;
        }
        
        .nav-links a {
            display: inline-block;
            margin: 0 10px;
            padding: 12px 24px;
            background: #4361ee;
            color: white;
            text-decoration: none;
            border-radius: 8px;
            transition: all 0.3s ease;
        }
        
        .nav-links a:hover {
            background: #3f37c9;
            transform: translateY(-2px);
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1><i class='bx bx-pie-chart-alt-2'></i> Revenue System Setup</h1>
            <p>Database tables and configuration for the revenue sharing system</p>
        </div>

        <div class="results">
            <?php foreach ($setup_results as $result): ?>
                <div class="result-item <?= strpos($result, '✅') === 0 ? 'success' : (strpos($result, 'ℹ️') === 0 ? 'info' : (strpos($result, '🎉') === 0 ? 'celebration' : 'info')) ?>">
                    <?= htmlspecialchars($result) ?>
                </div>
            <?php endforeach; ?>

            <?php foreach ($errors as $error): ?>
                <div class="result-item error">
                    <?= htmlspecialchars($error) ?>
                </div>
            <?php endforeach; ?>
        </div>

        <div class="nav-links">
            <a href="revenue-settings.php">
                <i class='bx bx-slider-alt'></i> Configure Revenue Settings
            </a>
            <a href="dashboard.php">
                <i class='bx bx-home'></i> Back to Dashboard
            </a>
        </div>
    </div>
</body>
</html>
