<?php
require_once '../auth/functions.php';
require_once '../config/db_connect.php';
require_once '../config/fast_email_service.php';
require_once 'pdf_certificate_generator.php';

// Check if user is admin
requireRole('admin');

$message = '';
$messageType = '';

// Handle form submission
if ($_POST && isset($_POST['send_certificate'])) {
    try {
        $recipientEmail = filter_var($_POST['recipient_email'], FILTER_VALIDATE_EMAIL);
        $studentName = htmlspecialchars($_POST['student_name'] ?: 'John Doe');
        $courseTitle = htmlspecialchars($_POST['course_title'] ?: 'Sample Course');
        
        if (!$recipientEmail) {
            throw new Exception('Please enter a valid email address');
        }
        
        // Generate certificate
        $certificateNumber = 'TEST-' . date('Y') . '-' . strtoupper(substr(uniqid(), -6));
        $issueDate = date('F j, Y');
        
        $pdfGenerator = new PdfCertificateGenerator($conn);
        
        // Try to generate certificate (with fallback)
        try {
            $pdfResult = $pdfGenerator->createPrintableCertificate($certificateNumber, $studentName, $courseTitle, $issueDate);
        } catch (Exception $e) {
            $pdfResult = $pdfGenerator->createSimpleCertificate($certificateNumber, $studentName, $courseTitle, $issueDate);
        }
        
        if (!$pdfResult['success']) {
            throw new Exception('Failed to generate certificate');
        }
        
        // Create download URL
        $certificateUrl = "https://panaditeacademy.com" . $pdfResult['download_url'];
        
        // Send ultra-simple email (same format that works)
        $subject = "Certificate Ready - {$courseTitle} - " . date('H:i:s');
        
        $emailBody = "<h2>🎓 Certificate Ready - {$studentName}</h2>
        <p><strong>Course:</strong> {$courseTitle}</p>
        <p><strong>Certificate Number:</strong> {$certificateNumber}</p>
        <p><strong>Download Link:</strong> <a href='{$certificateUrl}'>Click here to view your certificate</a></p>
        <p><strong>Time sent:</strong> " . date('Y-m-d H:i:s') . "</p>
        <p>Congratulations on completing your course!</p>";
        
        // Send email using working system
        $emailSent = sendEmailNow($recipientEmail, $subject, $emailBody);
        
        if ($emailSent) {
            $message = "✅ Certificate generated and email sent successfully!<br>";
            $message .= "<strong>Email sent to:</strong> {$recipientEmail}<br>";
            $message .= "<strong>Certificate Number:</strong> {$certificateNumber}<br>";
            $message .= "<strong>Download Link:</strong> <a href='{$certificateUrl}' target='_blank'>View Certificate</a><br>";
            $message .= "<strong>Sent at:</strong> " . date('Y-m-d H:i:s');
            $messageType = 'success';
        } else {
            throw new Exception('Failed to send email');
        }
        
    } catch (Exception $e) {
        $message = "❌ Error: " . $e->getMessage();
        $messageType = 'danger';
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Simple Certificate Test - Panadite Academy</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        .test-container {
            max-width: 600px;
            margin: 50px auto;
            background: white;
            border-radius: 15px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.2);
            overflow: hidden;
        }
        .header {
            background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
            color: white;
            text-align: center;
            padding: 30px;
        }
    </style>
</head>
<body>
    <div class="test-container">
        <div class="header">
            <i class="bx bx-mail-send bx-lg mb-2"></i>
            <h1>Simple Certificate Test</h1>
            <p class="mb-0">Direct certificate generation and email sending</p>
        </div>
        
        <div class="p-4">
            <?php if ($message): ?>
            <div class="alert alert-<?= $messageType ?> mb-4">
                <?= $message ?>
            </div>
            <?php endif; ?>
            
            <div class="alert alert-info mb-4">
                <i class="bx bx-info-circle me-2"></i>
                <strong>Simple Test:</strong> This bypasses the complex API and tests certificate generation directly.
            </div>
            
            <form method="POST">
                <div class="mb-3">
                    <label for="recipient_email" class="form-label fw-bold">
                        <i class="bx bx-envelope me-2"></i>Recipient Email
                    </label>
                    <input type="email" class="form-control" id="recipient_email" name="recipient_email" 
                           placeholder="Enter email address" required
                           value="<?= htmlspecialchars($_POST['recipient_email'] ?? '') ?>">
                </div>
                
                <div class="mb-3">
                    <label for="student_name" class="form-label fw-bold">
                        <i class="bx bx-user me-2"></i>Student Name
                    </label>
                    <input type="text" class="form-control" id="student_name" name="student_name" 
                           placeholder="Student name"
                           value="<?= htmlspecialchars($_POST['student_name'] ?? 'John Doe') ?>">
                </div>
                
                <div class="mb-3">
                    <label for="course_title" class="form-label fw-bold">
                        <i class="bx bx-book me-2"></i>Course Title
                    </label>
                    <input type="text" class="form-control" id="course_title" name="course_title" 
                           placeholder="Course title"
                           value="<?= htmlspecialchars($_POST['course_title'] ?? 'Computer Science') ?>">
                </div>
                
                <button type="submit" name="send_certificate" class="btn btn-success btn-lg w-100">
                    <i class="bx bx-send me-2"></i>Generate & Send Certificate
                </button>
            </form>
            
            <div class="text-center mt-4">
                <a href="certificate_templates.php" class="btn btn-outline-secondary">
                    <i class="bx bx-arrow-back me-2"></i>Back to Admin Panel
                </a>
            </div>
        </div>
    </div>
</body>
</html>
