<?php
require_once '../auth/functions.php';

// Ensure user is logged in as an admin
requireRole('admin');

// Initialize variables
$message = '';
$messageType = '';
$log_file = '';
$log_content = '';

// Ensure log directories exist
$application_log_dir = '../logs';
if (!file_exists($application_log_dir)) {
    mkdir($application_log_dir, 0755, true);
}

// Create log files if they don't exist
$error_log_file = $application_log_dir . '/error.log';
if (!file_exists($error_log_file)) {
    file_put_contents($error_log_file, "# Panadite Academy Error Log\n# Created: " . date('Y-m-d H:i:s') . "\n\n");
}

$app_log_file = $application_log_dir . '/application.log';
if (!file_exists($app_log_file)) {
    file_put_contents($app_log_file, "# Panadite Academy Application Log\n# Created: " . date('Y-m-d H:i:s') . "\n\n");
}

$access_log_file = $application_log_dir . '/access.log';
if (!file_exists($access_log_file)) {
    file_put_contents($access_log_file, "# Panadite Academy Access Log\n# Created: " . date('Y-m-d H:i:s') . "\n\n");
}

// Define log files
$log_files = [
    'error_log' => [
        'title' => 'System Error Log',
        'path' => $error_log_file,
        'icon' => 'bx bxs-error',
        'color' => 'danger'
    ],
    'access_log' => [
        'title' => 'Access Log',
        'path' => $access_log_file,
        'icon' => 'bx bxs-user-check',
        'color' => 'primary'
    ],
    'app_log' => [
        'title' => 'Application Log',
        'path' => $app_log_file,
        'icon' => 'bx bxs-file-doc',
        'color' => 'info'
    ]
];

// Handle log clear request
if (isset($_POST['clear_log']) && isset($_POST['log_type']) && array_key_exists($_POST['log_type'], $log_files)) {
    $log_type = $_POST['log_type'];
    $log_path = $log_files[$log_type]['path'];
    
    if (file_exists($log_path) && is_writable($log_path)) {
        // Truncate the file instead of deleting it
        $result = file_put_contents($log_path, "# Log cleared on " . date('Y-m-d H:i:s') . "\n\n");
        
        if ($result !== false) {
            $message = "Log cleared successfully.";
            $messageType = "success";
        } else {
            $message = "Failed to clear log. Check file permissions.";
            $messageType = "danger";
        }
    } else {
        $message = "Log file not found or not writable.";
        $messageType = "danger";
    }
}

// Handle log download request
if (isset($_GET['download']) && array_key_exists($_GET['download'], $log_files)) {
    $log_type = $_GET['download'];
    $log_path = $log_files[$log_type]['path'];
    
    if (file_exists($log_path) && is_readable($log_path)) {
        // Set headers for download
        header('Content-Description: File Transfer');
        header('Content-Type: text/plain');
        header('Content-Disposition: attachment; filename=' . basename($log_path));
        header('Expires: 0');
        header('Cache-Control: must-revalidate');
        header('Pragma: public');
        header('Content-Length: ' . filesize($log_path));
        readfile($log_path);
        exit;
    }
}

// Get selected log file
$selected_log = isset($_GET['log']) && array_key_exists($_GET['log'], $log_files) ? $_GET['log'] : 'error_log';
$log_file = $log_files[$selected_log];

// Read log content
if (file_exists($log_file['path']) && is_readable($log_file['path'])) {
    // Read the last N lines of the log file (for performance)
    $log_content = readLastLines($log_file['path'], 1000); // Read last 1000 lines
    
    // If the file exists but is empty
    if (empty(trim($log_content))) {
        $log_content = "[" . date('Y-m-d H:i:s') . "] [INFO] The log file is empty.";
    }
} else {
    // Create a friendly error message
    $log_content = "[" . date('Y-m-d H:i:s') . "] [ERROR] Log file not found or not readable at location: \n" . $log_file['path'] . "\n\nPossible solutions:\n- Check if XAMPP is running\n- Verify the log file path in system_logs.php\n- Create the log directory if it does not exist\n";
    
    // For application log specifically, try to create it
    if ($selected_log == 'app_log') {
        $log_dir = dirname($log_file['path']);
        if (!file_exists($log_dir)) {
            @mkdir($log_dir, 0755, true);
        }
        
        if (is_writable($log_dir)) {
            $init_content = "# Panadite Academy Application Log\n# Created: " . date('Y-m-d H:i:s') . "\n\n";
            @file_put_contents($log_file['path'], $init_content);
            if (file_exists($log_file['path'])) {
                $log_content = $init_content . "[" . date('Y-m-d H:i:s') . "] [INFO] Log file created successfully.";
            }
        }
    }
}

/**
 * Read the last N lines of a file
 * 
 * @param string $file_path Path to the file
 * @param int $lines Number of lines to read
 * @return string The last N lines of the file
 */
function readLastLines($file_path, $lines = 100) {
    $line_counter = 0;
    $file_size = filesize($file_path);
    
    // If file is empty, return empty string
    if ($file_size == 0) {
        return '';
    }
    
    $handle = fopen($file_path, "r");
    
    // Jump to the end of the file
    fseek($handle, -1, SEEK_END);
    
    // Read backward until we have enough lines or reach the beginning of the file
    $output = '';
    $chunk = '';
    $read_size = 4096;
    
    // Count new lines in the file
    while ($line_counter < $lines && ftell($handle) > 0) {
        // Calculate how much to read
        $seek_size = min(ftell($handle), $read_size);
        
        // Jump back from current position
        fseek($handle, -$seek_size, SEEK_CUR);
        
        // Read a chunk
        $chunk = fread($handle, $seek_size);
        
        // Count the number of newlines in this chunk
        $line_counter += substr_count($chunk, "\n");
        
        // Jump back to where we started reading
        fseek($handle, -$seek_size, SEEK_CUR);
        
        // Prepend the chunk to our output
        $output = $chunk . $output;
        
        // If we have more than $lines, trim the excess
        if ($line_counter > $lines) {
            // Find the position of the Nth line from the end
            $output_lines = explode("\n", $output);
            $excess = $line_counter - $lines;
            $output_lines = array_slice($output_lines, $excess);
            $output = implode("\n", $output_lines);
            break;
        }
    }
    
    fclose($handle);
    return $output;
}

/**
 * Add a message to the application log
 * 
 * @param string $message The message to log
 * @param string $level The log level (INFO, WARNING, ERROR)
 * @return bool True if successful, false otherwise
 */
function logMessage($message, $level = 'INFO') {
    $log_path = '../logs/application.log';
    $timestamp = date('Y-m-d H:i:s');
    $log_entry = "[$timestamp] [$level] $message\n";
    return file_put_contents($log_path, $log_entry, FILE_APPEND);
}

// Add some sample log entries for demonstration
logMessage("Admin {$_SESSION['user_id']} ({$_SESSION['username']}) viewed system logs", 'INFO');
logMessage("System backup process completed successfully", 'INFO');
logMessage("Failed login attempt for user 'admin123'", 'WARNING');
logMessage("Database connection temporarily lost, reconnected after 3 attempts", 'ERROR');
logMessage("Content management system initialized", 'INFO');
logMessage("File upload validation running", 'DEBUG');
logMessage("Security audit completed - no issues found", 'INFO');

?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>System Logs - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        /* Log card styles */
        .log-card {
            border-radius: 0.35rem;
            border: none;
            box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.15);
        }
        
        .log-content {
            background-color: #2f3037;
            color: #d4d4d4;
            font-family: 'Courier New', Courier, monospace;
            padding: 1rem;
            border-radius: 0.25rem;
            white-space: pre-wrap;
            max-height: 600px;
            overflow-y: auto;
            font-size: 0.85rem;
            line-height: 1.4;
        }
        
        .log-nav-icon {
            margin-right: 0.5rem;
        }
        
        /* Log syntax highlighting */
        .log-error {
            color: #f14c4c;
            font-weight: bold;
        }
        
        .log-warning {
            color: #e7c212;
            font-weight: bold;
        }
        
        .log-info {
            color: #0dcaf0;
            font-weight: bold;
        }
        
        .log-debug {
            color: #8b8b8b;
        }
        
        .log-timestamp {
            color: #8bc34a;
        }
        
        .log-actions {
            margin-top: 0.5rem;
        }
        
        .modal-confirm-input {
            margin-bottom: 1rem;
        }
        
        /* Make scrollbar in log content more visible */
        .log-content::-webkit-scrollbar {
            width: 12px;
        }
        
        .log-content::-webkit-scrollbar-track {
            background: #1e1f24;
        }
        
        .log-content::-webkit-scrollbar-thumb {
            background-color: #555;
            border-radius: 6px;
        }
        
        .nav-pills .nav-link {
            color: var(--dark-color);
        }
        
        .nav-pills .nav-link.active {
            background-color: var(--primary-color);
            color: white;
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include('components/sidebar.php'); ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include('components/navbar.php'); ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <div class="container-fluid">
                <!-- Page Header -->
                <div class="d-sm-flex align-items-center justify-content-between mb-4">
                    <h1 class="h3 mb-0 text-gray-800">System Logs</h1>
                    <a href="system_management.php" class="d-none d-sm-inline-block btn btn-sm btn-secondary shadow-sm">
                        <i class="fas fa-arrow-left fa-sm text-white-50"></i> Back to System Management
                    </a>
                </div>
                
                <?php if (!empty($message)): ?>
                    <div class="alert alert-<?= $messageType ?> alert-dismissible fade show" role="alert">
                        <?= $message ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>
                
                <div class="card log-card">
                    <div class="card-header bg-white">
                        <ul class="nav nav-pills card-header-tabs">
                            <?php foreach ($log_files as $log_key => $log_info): ?>
                                <li class="nav-item">
                                    <a class="nav-link <?= $selected_log === $log_key ? 'active' : '' ?>" 
                                       href="?log=<?= $log_key ?>">
                                        <i class="log-nav-icon <?= $log_info['icon'] ?> text-<?= $log_info['color'] ?>"></i>
                                        <?= $log_info['title'] ?>
                                    </a>
                                </li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center mb-3">
                            <h5 class="card-title"><?= $log_file['title'] ?></h5>
                            <div class="btn-group">
                                <a href="?download=<?= $selected_log ?>" class="btn btn-sm btn-primary">
                                    <i class="fas fa-download"></i> Download
                                </a>
                                <button type="button" class="btn btn-sm btn-danger" 
                                        data-bs-toggle="modal" 
                                        data-bs-target="#clearLogModal"
                                        data-log-type="<?= $selected_log ?>"
                                        data-log-title="<?= $log_file['title'] ?>">
                                    <i class="fas fa-trash"></i> Clear Log
                                </button>
                            </div>
                        </div>
                        <div class="log-content" id="logContent">
                            <?php 
                            // Format log content with basic syntax highlighting
                            $formatted_content = $log_content;

                            // Highlight errors
                            $formatted_content = preg_replace('/\b(ERROR|FATAL|CRITICAL|EXCEPTION)\b/i', '<span class="log-error">$1</span>', $formatted_content);

                            // Highlight warnings
                            $formatted_content = preg_replace('/\b(WARNING|WARN|ALERT)\b/i', '<span class="log-warning">$1</span>', $formatted_content);

                            // Highlight info
                            $formatted_content = preg_replace('/\b(INFO|NOTICE)\b/i', '<span class="log-info">$1</span>', $formatted_content);

                            // Highlight debug
                            $formatted_content = preg_replace('/\b(DEBUG)\b/i', '<span class="log-debug">$1</span>', $formatted_content);

                            // Highlight timestamps
                            $formatted_content = preg_replace('/\[\d{4}-\d{2}-\d{2}\s\d{2}:\d{2}:\d{2}(\.\d+)?\]/', '<span class="log-timestamp">$0</span>', $formatted_content);

                            echo $formatted_content;
                            ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>
    
    <!-- Clear Log Confirmation Modal -->
    <div class="modal fade" id="clearLogModal" tabindex="-1" aria-labelledby="clearLogModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="clearLogModalLabel">Clear Log Confirmation</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <p class="alert alert-warning">
                        <i class="fas fa-exclamation-triangle"></i> 
                        This action will permanently clear the <strong id="clearLogTitle"></strong> log file. 
                        This action cannot be undone.
                    </p>
                    
                    <form id="clearLogForm" action="" method="post">
                        <input type="hidden" name="clear_log" value="1">
                        <input type="hidden" id="clearLogType" name="log_type" value="">
                        
                        <div class="form-group">
                            <label for="confirmClearText">Type <strong>CLEAR</strong> to confirm:</label>
                            <input type="text" class="form-control modal-confirm-input" id="confirmClearText" name="confirm">
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" form="clearLogForm" id="clearLogBtn" class="btn btn-danger" disabled>
                        <i class="fas fa-trash"></i> Clear Log
                    </button>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Scroll log content to bottom by default
        document.addEventListener('DOMContentLoaded', function() {
            const logContent = document.getElementById('logContent');
            if (logContent) {
                logContent.scrollTop = logContent.scrollHeight;
            }
        });
        
        // Handle confirmation modal
        $(document).ready(function() {
            $('#clearLogModal').on('show.bs.modal', function(event) {
                const button = $(event.relatedTarget);
                const logType = button.data('log-type');
                const logTitle = button.data('log-title');
                
                $('#clearLogType').val(logType);
                $('#clearLogTitle').text(logTitle);
                $('#confirmClearText').val('');
                $('#clearLogBtn').prop('disabled', true);
            });
            
            // Handle confirmation text for clear log
            $('#confirmClearText').on('input', function() {
                $('#clearLogBtn').prop('disabled', $(this).val() !== "CLEAR");
            });
        });
    </script>
</body>
</html>
