<?php
require_once '../auth/functions.php';

// Ensure user is logged in as an admin
requireRole('admin');

// Process user deletion
$message = '';
$messageType = '';

// Handle permanent user deletion
if (isset($_POST['delete_user']) && isset($_POST['user_id']) && is_numeric($_POST['user_id'])) {
    $user_id = $_POST['user_id'];
    
    // Don't allow deleting admin (user_id = 1)
    if ($user_id == 1) {
        $message = "Cannot delete the main admin account.";
        $messageType = "danger";
    } else {
        try {
            // Begin transaction
            $conn->begin_transaction();
            
            // Get user info before deletion for file cleanup
            $stmt = $conn->prepare("SELECT username, profile_picture, user_role FROM users WHERE user_id = ?");
            $stmt->bind_param("i", $user_id);
            $stmt->execute();
            $user = $stmt->get_result()->fetch_assoc();
            
            if ($user) {
                // Delete profile picture if it exists
                if (!empty($user['profile_picture']) && file_exists("../" . $user['profile_picture'])) {
                    unlink("../" . $user['profile_picture']);
                }
                
                // Delete user-specific files based on role
                if ($user['user_role'] == 'teacher') {
                    // Delete teacher course materials
                    $materials_stmt = $conn->prepare("SELECT file_path FROM course_materials WHERE teacher_id = ?");
                    $materials_stmt->bind_param("i", $user_id);
                    $materials_stmt->execute();
                    $materials = $materials_stmt->get_result();
                    
                    while ($material = $materials->fetch_assoc()) {
                        if (!empty($material['file_path']) && file_exists("../" . $material['file_path'])) {
                            unlink("../" . $material['file_path']);
                        }
                    }
                    
                    // Delete course thumbnails
                    $course_stmt = $conn->prepare("SELECT thumbnail FROM courses WHERE teacher_id = ?");
                    $course_stmt->bind_param("i", $user_id);
                    $course_stmt->execute();
                    $courses = $course_stmt->get_result();
                    
                    while ($course = $courses->fetch_assoc()) {
                        if (!empty($course['thumbnail']) && file_exists("../" . $course['thumbnail'])) {
                            unlink("../" . $course['thumbnail']);
                        }
                    }
                    
                    // Delete video files and thumbnails
                    $video_stmt = $conn->prepare("SELECT video_url, thumbnail FROM videos WHERE teacher_id = ?");
                    $video_stmt->bind_param("i", $user_id);
                    $video_stmt->execute();
                    $videos = $video_stmt->get_result();
                    
                    while ($video = $videos->fetch_assoc()) {
                        if (!empty($video['video_url']) && file_exists("../" . $video['video_url'])) {
                            unlink("../" . $video['video_url']);
                        }
                        if (!empty($video['thumbnail']) && file_exists("../" . $video['thumbnail'])) {
                            unlink("../" . $video['thumbnail']);
                        }
                    }
                } else if ($user['user_role'] == 'student') {
                    // Delete student assignment submissions
                    $assign_stmt = $conn->prepare("SELECT file_url FROM assignment_submissions WHERE student_id = ?");
                    $assign_stmt->bind_param("i", $user_id);
                    $assign_stmt->execute();
                    $assignments = $assign_stmt->get_result();
                    
                    while ($assignment = $assignments->fetch_assoc()) {
                        if (!empty($assignment['file_url']) && file_exists("../" . $assignment['file_url'])) {
                            unlink("../" . $assignment['file_url']);
                        }
                    }
                }
                
                // Delete user data from database - foreign key constraints will handle related data
                $delete_stmt = $conn->prepare("DELETE FROM users WHERE user_id = ?");
                $delete_stmt->bind_param("i", $user_id);
                if ($delete_stmt->execute()) {
                    // Commit transaction
                    $conn->commit();
                    $message = "User " . htmlspecialchars($user['username']) . " and all associated data have been permanently deleted.";
                    $messageType = "success";
                } else {
                    throw new Exception("Failed to delete user from database.");
                }
            } else {
                throw new Exception("User not found.");
            }
        } catch (Exception $e) {
            // Roll back transaction
            $conn->rollback();
            $message = "Error: " . $e->getMessage();
            $messageType = "danger";
        }
    }
}

// Fetch users for display
$users = [];
$query = "SELECT u.user_id, u.username, u.email, u.first_name, u.last_name, u.user_role, u.created_at,
          (SELECT COUNT(*) FROM courses WHERE teacher_id = u.user_id) as course_count,
          (SELECT COUNT(*) FROM videos WHERE teacher_id = u.user_id) as video_count,
          (SELECT COUNT(*) FROM enrollments WHERE student_id = u.user_id) as enrollment_count
          FROM users u ORDER BY u.user_role, u.created_at DESC";
$result = $conn->query($query);

if ($result) {
    while ($row = $result->fetch_assoc()) {
        $users[] = $row;
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>User Management - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.datatables.net/1.13.4/css/dataTables.bootstrap5.min.css">
    <style>
        /* Custom styles for user management */
        .badge-role {
            font-size: 0.8rem;
            padding: 0.25rem 0.5rem;
        }
        
        .badge-admin {
            background-color: #4e73df;
            color: white;
        }
        
        .badge-teacher {
            background-color: #1cc88a;
            color: white;
        }
        
        .badge-student {
            background-color: #f6c23e;
            color: white;
        }
        
        .user-table th, .user-table td {
            vertical-align: middle;
        }
        
        .confirmation-modal .modal-body {
            padding: 2rem;
            text-align: center;
        }
        
        .confirmation-modal .icon {
            font-size: 4rem;
            color: #e74a3b;
            margin-bottom: 1rem;
        }
        
        /* DataTables custom styling */
        .dataTables_wrapper .dataTables_length,
        .dataTables_wrapper .dataTables_filter,
        .dataTables_wrapper .dataTables_info {
            padding: 1rem;
        }
        
        .dataTables_wrapper .dataTables_paginate {
            padding: 1rem;
        }
        
        .dataTables_wrapper .dataTables_paginate .paginate_button.current {
            background: var(--primary-color) !important;
            color: white !important;
            border: none;
        }
        
        .dataTables_wrapper .dataTables_paginate .paginate_button:hover {
            background: var(--primary-color) !important;
            color: white !important;
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include('components/sidebar.php'); ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include('components/navbar.php'); ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <div class="container-fluid">
                <!-- Page Header -->
                <div class="d-sm-flex align-items-center justify-content-between mb-4">
                    <h1 class="h3 mb-0 text-gray-800">User Management</h1>
                    <a href="system_management.php" class="d-none d-sm-inline-block btn btn-sm btn-secondary shadow-sm">
                        <i class="fas fa-arrow-left fa-sm text-white-50"></i> Back to System Management
                    </a>
                </div>
                
                <?php if (!empty($message)): ?>
                    <div class="alert alert-<?= $messageType ?> alert-dismissible fade show" role="alert">
                        <?= $message ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>
                
                <div class="card shadow mb-4">
                    <div class="card-header py-3">
                        <h6 class="m-0 font-weight-bold text-primary">User Accounts</h6>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover user-table" id="usersTable">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>Username</th>
                                        <th>Name</th>
                                        <th>Email</th>
                                        <th>Role</th>
                                        <th>Data</th>
                                        <th>Joined</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($users as $user): ?>
                                        <tr>
                                            <td><?= $user['user_id'] ?></td>
                                            <td><?= htmlspecialchars($user['username']) ?></td>
                                            <td><?= htmlspecialchars($user['first_name'] . ' ' . $user['last_name']) ?></td>
                                            <td><?= htmlspecialchars($user['email']) ?></td>
                                            <td>
                                                <?php if ($user['user_role'] == 'admin'): ?>
                                                    <span class="badge bg-primary badge-role badge-admin">Admin</span>
                                                <?php elseif ($user['user_role'] == 'teacher'): ?>
                                                    <span class="badge bg-success badge-role badge-teacher">Teacher</span>
                                                <?php else: ?>
                                                    <span class="badge bg-warning text-dark badge-role badge-student">Student</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <?php if ($user['user_role'] == 'teacher'): ?>
                                                    <small>
                                                        <i class="bx bxs-graduation"></i> <?= $user['course_count'] ?> courses<br>
                                                        <i class="bx bxs-video"></i> <?= $user['video_count'] ?> videos
                                                    </small>
                                                <?php elseif ($user['user_role'] == 'student'): ?>
                                                    <small>
                                                        <i class="bx bx-book-open"></i> <?= $user['enrollment_count'] ?> enrollments
                                                    </small>
                                                <?php else: ?>
                                                    <small>System Administrator</small>
                                                <?php endif; ?>
                                            </td>
                                            <td><?= date('M d, Y', strtotime($user['created_at'])) ?></td>
                                            <td>
                                                <?php if ($user['user_id'] != 1): ?>
                                                    <button type="button" class="btn btn-sm btn-danger" 
                                                            data-bs-toggle="modal" 
                                                            data-bs-target="#deleteUserModal" 
                                                            data-userid="<?= $user['user_id'] ?>"
                                                            data-username="<?= htmlspecialchars($user['username']) ?>"
                                                            data-userrole="<?= $user['user_role'] ?>">
                                                        <i class="fas fa-trash"></i> Delete
                                                    </button>
                                                <?php else: ?>
                                                    <span class="text-muted"><i class="fas fa-lock"></i> Protected</span>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>
    
    <!-- Delete User Confirmation Modal -->
    <div class="modal fade confirmation-modal" id="deleteUserModal" tabindex="-1" aria-labelledby="deleteUserModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-body">
                    <i class="icon bx bxs-trash"></i>
                    <h4 class="mb-4">Permanently Delete User</h4>
                    <p class="mb-4">Are you sure you want to permanently delete this user? This will also delete:</p>
                    <div id="userDeleteDetails" class="text-start mb-4">
                        <!-- Content will be populated dynamically -->
                    </div>
                    <p class="text-danger fw-bold mb-4">This action cannot be undone!</p>
                    <form action="" method="post">
                        <input type="hidden" name="user_id" id="deleteUserId" value="">
                        <div class="d-flex justify-content-center gap-3 mt-4">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                            <button type="submit" name="delete_user" class="btn btn-danger">Confirm Delete</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.4/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.4/js/dataTables.bootstrap5.min.js"></script>
    <script>
        $(document).ready(function() {
            // Initialize DataTables
            $('#usersTable').DataTable({
                "order": [[0, "asc"]], // Sort by ID by default
                "pageLength": 10,
                "language": {
                    "search": "Search users:",
                    "lengthMenu": "Show _MENU_ users per page",
                    "info": "Showing _START_ to _END_ of _TOTAL_ users",
                    "infoEmpty": "No users found",
                    "infoFiltered": "(filtered from _MAX_ total users)",
                    "zeroRecords": "No matching users found"
                }
            });
            
            // Handle delete user modal data
            $('#deleteUserModal').on('show.bs.modal', function (event) {
                const button = $(event.relatedTarget);
                const userId = button.data('userid');
                const username = button.data('username');
                const userRole = button.data('userrole');
                
                $('#deleteUserId').val(userId);
                
                let details = `<ul>
                    <li><strong>Username:</strong> ${username}</li>
                    <li><strong>Role:</strong> ${userRole}</li>
                </ul>`;
                
                if (userRole === 'teacher') {
                    details += `<p><i class="fas fa-exclamation-triangle text-warning"></i> All courses, videos, and course materials will be deleted.</p>`;
                } else if (userRole === 'student') {
                    details += `<p><i class="fas fa-exclamation-triangle text-warning"></i> All enrollments and submitted assignments will be deleted.</p>`;
                }
                
                $('#userDeleteDetails').html(details);
            });
        });
    </script>
</body>
</html>
