<?php
require_once '../auth/functions.php';
require_once '../includes/db_connect.php';

// Ensure user is logged in as an admin
requireRole('admin');

// Check if user ID is provided
if (!isset($_GET['id']) || empty($_GET['id'])) {
    header("Location: users.php?error=No user ID provided");
    exit;
}

$user_id = intval($_GET['id']);

// Get user data
$stmt = $conn->prepare("SELECT * FROM users WHERE user_id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    // No user found with this ID
    header("Location: users.php?error=User not found");
    exit;
}

$user = $result->fetch_assoc();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>User Profile | Panadite Admin</title>
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Boxicons CSS -->
    <link href="https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css" rel="stylesheet">
    <!-- Google Fonts - Poppins -->
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <!-- Core CSS files -->
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <style>
        :root {
            --primary-color: #4e73df;
            --primary-light: #6f86e0;
            --primary-dark: #3a56c5;
            --secondary-color: #858796;
            --success-color: #1cc88a;
            --warning-color: #f6c23e;
            --danger-color: #e74a3b;
            --light-color: #f8f9fc;
            --dark-color: #5a5c69;
            --card-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.15);
            --border-radius: 0.5rem;
        }
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Poppins', sans-serif;
            background-color: #f8f9fc;
            overflow-x: hidden;
        }
        
        /* Page Header Styles */
        .page-header {
            position: relative;
            margin-bottom: 2rem;
            padding: 0.5rem 0.5rem 1.25rem;
            border-bottom: 1px solid rgba(230, 234, 241, 0.7);
        }
        
        .page-header::after {
            content: '';
            position: absolute;
            bottom: -2px;
            left: 0;
            width: 60px;
            height: 4px;
            background: linear-gradient(90deg, #4e73df, #6f86e0);
            border-radius: 2px;
        }
        
        .page-title-container {
            display: flex;
            flex-direction: column;
        }
        
        .page-title {
            font-size: 1.65rem;
            font-weight: 600;
            color: #333;
            margin-bottom: 0.3rem;
            display: flex;
            align-items: center;
        }
        
        .title-icon {
            font-size: 1.75rem;
            margin-right: 0.75rem;
            color: #4e73df;
            background: rgba(78, 115, 223, 0.1);
            width: 40px;
            height: 40px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(78, 115, 223, 0.12);
        }
        
        .page-subtitle {
            font-size: 0.95rem;
            color: #6c757d;
            margin: 0;
            padding-left: 3.5rem;
        }
        
        .back-btn {
            transition: all 0.2s ease;
            border-radius: 8px;
            padding: 0.5rem 1rem;
        }
        
        .back-btn:hover {
            transform: translateX(-3px);
        }
        
        /* Profile Card Styles */
        .profile-card {
            background: white;
            border-radius: var(--border-radius);
            box-shadow: var(--card-shadow);
            transition: transform 0.3s ease, box-shadow 0.3s ease;
            overflow: hidden;
            margin-bottom: 1.5rem;
        }
        
        .profile-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 0.25rem 2.25rem 0 rgba(58, 59, 69, 0.18);
        }
        
        .profile-header {
            background: linear-gradient(120deg, #4e73df, #6f86e0);
            color: white;
            padding: 2rem;
            position: relative;
            overflow: hidden;
            text-align: center;
        }
        
        .profile-header::before {
            content: '';
            position: absolute;
            top: -20px;
            right: -20px;
            width: 140px;
            height: 140px;
            background: rgba(255, 255, 255, 0.1);
            border-radius: 50%;
        }
        
        .profile-header::after {
            content: '';
            position: absolute;
            bottom: -40px;
            left: -40px;
            width: 180px;
            height: 180px;
            background: rgba(255, 255, 255, 0.08);
            border-radius: 50%;
        }
        
        .profile-avatar {
            width: 120px;
            height: 120px;
            background-color: rgba(255, 255, 255, 0.2);
            border-radius: 50%;
            margin: 0 auto 1.5rem;
            display: flex;
            align-items: center;
            justify-content: center;
            position: relative;
            z-index: 2;
            border: 5px solid rgba(255, 255, 255, 0.15);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.15);
            overflow: hidden;
        }
        
        .profile-avatar i {
            font-size: 3.5rem;
            color: white;
        }
        
        .profile-info {
            padding: 2rem;
        }
        
        .profile-stats {
            display: flex;
            justify-content: space-around;
            border-top: 1px solid #eaecf4;
            padding-top: 1.5rem;
            margin-top: 1.5rem;
        }
        
        .stat-item {
            text-align: center;
        }
        
        .stat-value {
            font-size: 1.5rem;
            font-weight: 600;
            color: var(--dark-color);
        }
        
        .stat-label {
            font-size: 0.85rem;
            color: var(--secondary-color);
            margin-top: 0.25rem;
        }
        
        .info-item {
            margin-bottom: 1.25rem;
            display: flex;
            align-items: center;
        }
        
        .info-item-icon {
            width: 40px;
            height: 40px;
            background: rgba(78, 115, 223, 0.1);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 1rem;
            color: var(--primary-color);
            flex-shrink: 0;
        }
        
        .info-item-content {
            flex-grow: 1;
        }
        
        .info-item-label {
            color: var(--secondary-color);
            font-size: 0.85rem;
            margin-bottom: 0.25rem;
        }
        
        .info-item-value {
            color: var(--dark-color);
            font-weight: 500;
        }
        
        .profile-actions {
            display: flex;
            gap: 1rem;
            margin-top: 2rem;
            flex-wrap: wrap;
        }
        
        .profile-actions .btn {
            padding: 0.5rem 1.25rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
            border-radius: 6px;
            transition: all 0.3s ease;
            font-weight: 500;
        }
        
        .profile-actions .btn-primary {
            background: linear-gradient(to right, #4e73df, #6f86e0);
            border: none;
            box-shadow: 0 4px 10px rgba(78, 115, 223, 0.25);
        }
        
        .profile-actions .btn-primary:hover {
            background: linear-gradient(to right, #3a56c5, #5c75d6);
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(78, 115, 223, 0.35);
        }
        
        .profile-actions .btn-outline-secondary {
            border: 1px solid #d1d3e2;
            color: #6c757d;
        }
        
        .profile-actions .btn-outline-secondary:hover {
            background-color: #f8f9fc;
        }
        
        /* Tabs and Content Styles */
        .nav-tabs {
            border-bottom: none;
            padding: 0.5rem 1rem 0;
            gap: 0.5rem;
        }
        
        .custom-tab {
            border: none !important;
            padding: 0.75rem 1.25rem;
            color: #6c757d;
            font-weight: 500;
            border-radius: 8px !important;
            transition: all 0.2s ease;
        }
        
        .custom-tab:hover {
            color: var(--primary-color);
            background-color: rgba(78, 115, 223, 0.05);
        }
        
        .custom-tab.active {
            color: var(--primary-color) !important;
            background-color: rgba(78, 115, 223, 0.1) !important;
        }
        
        .card-body {
            padding: 1.5rem;
        }
        
        .section-title {
            display: flex;
            align-items: center;
            font-size: 1.15rem;
            color: var(--dark-color);
            margin-bottom: 1.25rem;
            font-weight: 600;
        }
        
        .detail-section {
            background-color: #f8f9fc;
            padding: 1.25rem;
            border-radius: 8px;
            margin-bottom: 1.5rem;
        }
        
        .detail-item {
            display: flex;
            flex-direction: column;
        }
        
        .detail-label {
            color: var(--secondary-color);
            font-size: 0.85rem;
            margin-bottom: 0.25rem;
        }
        
        .detail-value {
            font-weight: 500;
            color: var(--dark-color);
        }
        
        .role-badge {
            display: inline-block;
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.85rem;
            font-weight: 500;
        }
        
        .admin-role {
            background-color: rgba(78, 115, 223, 0.15);
            color: var(--primary-dark);
        }
        
        .teacher-role {
            background-color: rgba(28, 200, 138, 0.15);
            color: #18714d;
        }
        
        .student-role {
            background-color: rgba(246, 194, 62, 0.15);
            color: #9d7816;
        }
        
        /* Progress Styles */
        .progress-stats {
            display: flex;
            flex-direction: column;
            gap: 1rem;
        }
        
        .progress-item {
            width: 100%;
        }
        
        .progress-header {
            display: flex;
            justify-content: space-between;
            margin-bottom: 0.5rem;
            font-size: 0.9rem;
        }
        
        .progress {
            overflow: hidden;
            background-color: #e9ecef;
            border-radius: 10px;
        }
        
        .progress-bar {
            transition: width 1s ease;
            border-radius: 10px;
        }
        
        /* Activity Timeline */
        .activity-timeline {
            position: relative;
            padding: 0.25rem 0 0.25rem 1.5rem;
        }
        
        .activity-timeline::before {
            content: '';
            position: absolute;
            left: 19px;
            top: 0;
            height: 100%;
            width: 2px;
            background-color: #e9ecef;
        }
        
        .activity-item {
            position: relative;
            padding-bottom: 1.5rem;
            display: flex;
            align-items: flex-start;
        }
        
        .activity-icon {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 1rem;
            position: relative;
            z-index: 1;
            flex-shrink: 0;
        }
        
        .bg-light-primary {
            background-color: rgba(78, 115, 223, 0.15);
            color: var(--primary-color);
        }
        
        .activity-content {
            flex-grow: 1;
        }
        
        .activity-text {
            margin: 0;
            font-weight: 500;
            color: var(--dark-color);
        }
        
        .activity-time {
            margin: 0;
            color: var(--secondary-color);
            font-size: 0.85rem;
        }
        
        /* Empty States */
        .activity-empty, .courses-empty {
            text-align: center;
            padding: 2rem 0;
            color: var(--secondary-color);
        }
        
        .empty-state {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            padding: 3rem 1rem;
            background-color: #f8f9fc;
            border-radius: 8px;
        }
        
        .empty-icon {
            font-size: 3.5rem;
            color: #d1d3e2;
            margin-bottom: 1rem;
        }
        
        .empty-img {
            max-width: 100px;
            margin-bottom: 1rem;
            opacity: 0.6;
        }
        
        .courses-empty h5 {
            color: var(--dark-color);
            margin-bottom: 0.5rem;
        }
        
        .courses-empty p {
            color: var(--secondary-color);
        }
        
        /* Responsive adjustments */
        @media (max-width: 768px) {
            
            .profile-header {
                padding: 1.5rem 1rem;
            }
            
            .profile-avatar {
                width: 100px;
                height: 100px;
            }
            
            .profile-avatar i {
                font-size: 3rem;
            }
            
            .profile-actions {
                flex-direction: column;
            }
            
            .profile-actions .btn {
                width: 100%;
            }
            
            .page-header {
                padding: 0.5rem 0 0.75rem;
            }
            
            .page-title {
                font-size: 1.35rem;
            }
            
            .page-subtitle {
                padding-left: 3rem;
                font-size: 0.85rem;
            }
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <div class="container-fluid p-0">
                <!-- Page Header -->
                <div class="page-header mb-4">
                    <div class="d-flex align-items-center justify-content-between">
                        <div class="page-title-container">
                            <h1 class="h3 mb-0 page-title"><i class='bx bx-user-circle title-icon'></i> User Profile</h1>
                            <p class="page-subtitle">View complete user information and activity</p>
                        </div>
                        <a href="users.php" class="btn btn-outline-secondary btn-sm back-btn">
                            <i class='bx bx-arrow-back'></i> Back to Users
                        </a>
                    </div>
                </div>
                
                <!-- Profile Content -->
                <div class="row">
                    <!-- User Profile Card -->
                    <div class="col-lg-4 col-md-5">
                        <div class="profile-card">
                            <div class="profile-header">
                                <div class="profile-avatar">
                                    <i class='bx bx-user'></i>
                                </div>
                                <h4 class="mb-0"><?php echo htmlspecialchars($user['first_name'] . ' ' . $user['last_name']); ?></h4>
                                <p class="text-white-50 mb-0"><?php echo ucfirst(htmlspecialchars($user['user_role'])); ?></p>
                            </div>
                            <div class="profile-info">
                                <div class="info-item">
                                    <div class="info-item-icon">
                                        <i class='bx bx-envelope'></i>
                                    </div>
                                    <div class="info-item-content">
                                        <p class="info-item-label">Email</p>
                                        <p class="info-item-value"><?php echo htmlspecialchars($user['email']); ?></p>
                                    </div>
                                </div>
                                
                                <div class="info-item">
                                    <div class="info-item-icon">
                                        <i class='bx bx-calendar'></i>
                                    </div>
                                    <div class="info-item-content">
                                        <p class="info-item-label">Joined Date</p>
                                        <p class="info-item-value">
                                            <?php 
                                                echo isset($user['registration_date']) ? 
                                                    date('F d, Y', strtotime($user['registration_date'])) : 
                                                    'Not available'; 
                                            ?>
                                        </p>
                                    </div>
                                </div>
                                
                                <div class="info-item">
                                    <div class="info-item-icon">
                                        <i class='bx bx-id-card'></i>
                                    </div>
                                    <div class="info-item-content">
                                        <p class="info-item-label">User ID</p>
                                        <p class="info-item-value">#<?php echo htmlspecialchars($user['user_id']); ?></p>
                                    </div>
                                </div>
                                
                                <div class="profile-stats">
                                    <?php if($user['user_role'] == 'admin'): // Admin stats ?>
                                    <div class="stat-item">
                                        <p class="stat-value"><i class='bx bx-check text-primary'></i></p>
                                        <p class="stat-label">System Admin</p>
                                    </div>
                                    <div class="stat-item">
                                        <p class="stat-value"><i class='bx bx-shield text-primary'></i></p>
                                        <p class="stat-label">Full Access</p>
                                    </div>
                                    <div class="stat-item">
                                        <p class="stat-value"><i class='bx bx-crown text-warning'></i></p>
                                        <p class="stat-label">Administrator</p>
                                    </div>
                                    <?php elseif($user['user_role'] == 'teacher'): // Teacher stats ?>
                                    <div class="stat-item">
                                        <p class="stat-value">0</p>
                                        <p class="stat-label">Teaching Courses</p>
                                    </div>
                                    <div class="stat-item">
                                        <p class="stat-value">0</p>
                                        <p class="stat-label">Students</p>
                                    </div>
                                    <div class="stat-item">
                                        <p class="stat-value"><i class='bx bx-book-reader text-success'></i></p>
                                        <p class="stat-label">Teacher</p>
                                    </div>
                                    <?php else: // Student stats ?>
                                    <div class="stat-item">
                                        <p class="stat-value">0</p>
                                        <p class="stat-label">Courses</p>
                                    </div>
                                    <div class="stat-item">
                                        <p class="stat-value">0</p>
                                        <p class="stat-label">Certifications</p>
                                    </div>
                                    <div class="stat-item">
                                        <p class="stat-value">0</p>
                                        <p class="stat-label">Points</p>
                                    </div>
                                    <?php endif; ?>
                                </div>
                                
                                <div class="profile-actions">
                                    <a href="edit_user.php?id=<?php echo $user['user_id']; ?>" class="btn btn-primary">
                                        <i class='bx bx-edit'></i> Edit Profile
                                    </a>
                                    <button type="button" class="btn btn-outline-secondary">
                                        <i class='bx bx-message-square-detail'></i> Send Message
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- User Activity and Details -->
                    <div class="col-lg-8 col-md-7">
                        <!-- Activity Tabs -->
                        <div class="profile-card mb-4">
                            <div class="card-header bg-white p-0">
                                <ul class="nav nav-tabs" id="userTabs" role="tablist">
                                    <li class="nav-item" role="presentation">
                                        <button class="nav-link active custom-tab" id="overview-tab" data-bs-toggle="tab" data-bs-target="#overview" type="button" role="tab" aria-controls="overview" aria-selected="true">
                                            <i class='bx bx-grid-alt me-2'></i>Overview
                                        </button>
                                    </li>
                                    <li class="nav-item" role="presentation">
                                        <button class="nav-link custom-tab" id="activity-tab" data-bs-toggle="tab" data-bs-target="#activity" type="button" role="tab" aria-controls="activity" aria-selected="false">
                                            <i class='bx bx-history me-2'></i>Activity
                                        </button>
                                    </li>
                                    <li class="nav-item" role="presentation">
                                        <button class="nav-link custom-tab" id="courses-tab" data-bs-toggle="tab" data-bs-target="#courses" type="button" role="tab" aria-controls="courses" aria-selected="false">
                                            <i class='bx bx-book-open me-2'></i>Courses
                                        </button>
                                    </li>
                                </ul>
                            </div>
                            <div class="card-body">
                                <div class="tab-content" id="userTabContent">
                                    <!-- Overview Tab -->
                                    <div class="tab-pane fade show active" id="overview" role="tabpanel" aria-labelledby="overview-tab">
                                        <h5 class="section-title"><i class='bx bx-user-detail me-2'></i>User Details</h5>
                                        <div class="detail-section">
                                            <div class="row g-4">
                                                <div class="col-md-6">
                                                    <div class="detail-item">
                                                        <span class="detail-label">First Name</span>
                                                        <span class="detail-value"><?php echo htmlspecialchars($user['first_name']); ?></span>
                                                    </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="detail-item">
                                                        <span class="detail-label">Last Name</span>
                                                        <span class="detail-value"><?php echo htmlspecialchars($user['last_name']); ?></span>
                                                    </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="detail-item">
                                                        <span class="detail-label">Email</span>
                                                        <span class="detail-value"><?php echo htmlspecialchars($user['email']); ?></span>
                                                    </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="detail-item">
                                                        <span class="detail-label">Role</span>
                                                        <span class="detail-value role-badge <?php echo $user['user_role']; ?>-role">
                                                            <?php echo ucfirst(htmlspecialchars($user['user_role'])); ?>
                                                        </span>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <?php if($user['user_role'] == 'student'): // Only show progress for students ?>
                                        <h5 class="section-title mt-4"><i class='bx bx-bar-chart-alt-2 me-2'></i>Progress Overview</h5>
                                        <div class="detail-section">
                                            <div class="progress-stats">
                                                <div class="progress-item">
                                                    <div class="progress-header">
                                                        <span>Course Completion</span>
                                                        <span>0%</span>
                                                    </div>
                                                    <div class="progress" style="height: 8px;">
                                                        <div class="progress-bar bg-primary" role="progressbar" style="width: 0%" aria-valuenow="0" aria-valuemin="0" aria-valuemax="100"></div>
                                                    </div>
                                                </div>
                                                
                                                <div class="progress-item">
                                                    <div class="progress-header">
                                                        <span>Assignment Completion</span>
                                                        <span>0%</span>
                                                    </div>
                                                    <div class="progress" style="height: 8px;">
                                                        <div class="progress-bar bg-success" role="progressbar" style="width: 0%" aria-valuenow="0" aria-valuemin="0" aria-valuemax="100"></div>
                                                    </div>
                                                </div>
                                                
                                                <div class="progress-item">
                                                    <div class="progress-header">
                                                        <span>Certification Progress</span>
                                                        <span>0%</span>
                                                    </div>
                                                    <div class="progress" style="height: 8px;">
                                                        <div class="progress-bar bg-warning" role="progressbar" style="width: 0%" aria-valuenow="0" aria-valuemin="0" aria-valuemax="100"></div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        <?php else: // Show alternative info for admin/teacher ?>
                                        <h5 class="section-title mt-4"><i class='bx bx-shield-quarter me-2'></i>System Access</h5>
                                        <div class="detail-section">
                                            <div class="mb-2">
                                                <div class="detail-label">Access Level</div>
                                                <div class="detail-value">
                                                    <?php if($user['user_role'] == 'admin'): ?>
                                                    <span class="badge bg-primary">Full System Access</span>
                                                    <?php elseif($user['user_role'] == 'teacher'): ?>
                                                    <span class="badge bg-success">Course Management Access</span>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                            <div class="mt-3">
                                                <div class="detail-label">Last Login</div>
                                                <div class="detail-value">Not available</div>
                                            </div>
                                        </div>
                                        <?php endif; ?>
                                    </div>
                                    
                                    <!-- Activity Tab -->
                                    <div class="tab-pane fade" id="activity" role="tabpanel" aria-labelledby="activity-tab">
                                        <div class="activity-timeline">
                                            <?php if($user['user_role'] == 'admin'): // Admin activity ?>
                                            <!-- Admin activity sample -->
                                            <div class="activity-item">
                                                <div class="activity-icon bg-light-primary">
                                                    <i class='bx bx-user-check'></i>
                                                </div>
                                                <div class="activity-content">
                                                    <p class="activity-text">User account approved</p>
                                                    <p class="activity-time">2 hours ago</p>
                                                </div>
                                            </div>
                                            <div class="activity-item">
                                                <div class="activity-icon bg-light-primary">
                                                    <i class='bx bx-cog'></i>
                                                </div>
                                                <div class="activity-content">
                                                    <p class="activity-text">System settings updated</p>
                                                    <p class="activity-time">Yesterday</p>
                                                </div>
                                            </div>
                                            <?php elseif($user['user_role'] == 'teacher'): // Teacher activity ?>
                                            <!-- Teacher activity sample -->
                                            <div class="activity-item">
                                                <div class="activity-icon bg-light-success">
                                                    <i class='bx bx-book-content'></i>
                                                </div>
                                                <div class="activity-content">
                                                    <p class="activity-text">Course material updated</p>
                                                    <p class="activity-time">3 days ago</p>
                                                </div>
                                            </div>
                                            <?php else: // Empty state ?>
                                            <!-- Empty state -->
                                            
                                            <div class="activity-empty">
                                                <img src="../assets/img/empty-state.svg" alt="No activity" class="empty-img">
                                                <p>No recent activities</p>
                                            </div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    
                                    <!-- Courses Tab -->
                                    <div class="tab-pane fade" id="courses" role="tabpanel" aria-labelledby="courses-tab">
                                        <?php if($user['user_role'] == 'admin'): // Admin courses view ?>
                                        <div class="p-3">
                                            <div class="d-flex align-items-center justify-content-between mb-3">
                                                <h5 class="m-0">System Management</h5>
                                            </div>
                                            <div class="row">
                                                <div class="col-md-6 mb-3">
                                                    <div class="card h-100">
                                                        <div class="card-body">
                                                            <div class="d-flex align-items-center mb-2">
                                                                <i class='bx bx-cog fs-4 text-primary me-2'></i>
                                                                <h6 class="card-title mb-0">Course Management</h6>
                                                            </div>
                                                            <p class="card-text text-muted small">Full access to create, edit and delete any course.</p>
                                                        </div>
                                                    </div>
                                                </div>
                                                <div class="col-md-6 mb-3">
                                                    <div class="card h-100">
                                                        <div class="card-body">
                                                            <div class="d-flex align-items-center mb-2">
                                                                <i class='bx bx-user-circle fs-4 text-primary me-2'></i>
                                                                <h6 class="card-title mb-0">User Management</h6>
                                                            </div>
                                                            <p class="card-text text-muted small">Full access to manage all user accounts and permissions.</p>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        <?php elseif($user['user_role'] == 'teacher'): // Teacher courses view ?>
                                        <div class="p-3">
                                            <div class="d-flex align-items-center justify-content-between mb-3">
                                                <h5 class="m-0">Teaching Courses</h5>
                                            </div>
                                            <div class="courses-empty">
                                                <div class="empty-state">
                                                    <i class='bx bx-chalkboard empty-icon text-success'></i>
                                                    <h5>No teaching assignments yet</h5>
                                                    <p>This teacher has not been assigned to any courses yet.</p>
                                                </div>
                                            </div>
                                        </div>
                                        <?php else: // Student courses view ?>
                                        <div class="courses-empty">
                                            <div class="empty-state">
                                                <i class='bx bx-book-open empty-icon'></i>
                                                <h5>No courses enrolled yet</h5>
                                                <p>This user has not enrolled in any courses.</p>
                                            </div>
                                        </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Bootstrap Bundle with Popper -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Initialize tooltips and popovers
        document.addEventListener('DOMContentLoaded', function() {
            // Toggle sidebar functionality
            const sidebarToggle = document.querySelector('.sidebar-toggle');
            const sidebarToggleMobile = document.querySelector('.sidebar-toggle-mobile');
            const sidebar = document.querySelector('.sidebar');
            
            if (sidebarToggle) {
                sidebarToggle.addEventListener('click', function() {
                    document.body.classList.toggle('sidebar-toggled');
                    sidebar.classList.toggle('toggled');
                });
            }
            
            if (sidebarToggleMobile) {
                sidebarToggleMobile.addEventListener('click', function() {
                    document.body.classList.toggle('sidebar-toggled');
                    sidebar.classList.toggle('toggled');
                });
            }

            // Add animation to progress bars
            const progressBars = document.querySelectorAll('.progress-bar');
            setTimeout(() => {
                progressBars.forEach(bar => {
                    // The width is set in the HTML via inline style
                    // This creates an animation effect when the page loads
                    console.log('Animating progress bars');
                });
            }, 300);

            // Initialize Bootstrap tabs
            const triggerTabList = document.querySelectorAll('#userTabs button');
            triggerTabList.forEach(triggerEl => {
                triggerEl.addEventListener('click', event => {
                    event.preventDefault();
                    const tab = new bootstrap.Tab(triggerEl);
                    tab.show();
                });
            });

            // Add hover effect to profile card
            const profileCard = document.querySelector('.profile-card');
            if (profileCard) {
                profileCard.addEventListener('mouseenter', function() {
                    this.style.transform = 'translateY(-5px)';
                });
                profileCard.addEventListener('mouseleave', function() {
                    this.style.transform = '';
                });
            }

            console.log('User Profile Page Initialized');
            console.log('User ID:', <?php echo $user_id; ?>);
        });
    </script>
</body>
</html>
