<?php
require_once '../auth/functions.php';

// Ensure user is logged in as an admin
requireRole('admin');

$message = '';
$course_filter = isset($_GET['course_id']) ? $_GET['course_id'] : null;

// Delete video
if (isset($_GET['delete']) && is_numeric($_GET['delete'])) {
    $video_id = $_GET['delete'];
    
    // Check if the video has any purchases
    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM video_purchases WHERE video_id = ?");
    $stmt->bind_param("i", $video_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    
    if ($row['count'] > 0) {
        $message = '<div class="alert alert-danger">Cannot delete this video because it has been purchased by users.</div>';
    } else {
        // Get video information for file deletion
        $stmt = $conn->prepare("SELECT video_url, thumbnail FROM videos WHERE video_id = ?");
        $stmt->bind_param("i", $video_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $video = $result->fetch_assoc();
        
        // Delete the video
        $stmt = $conn->prepare("DELETE FROM videos WHERE video_id = ?");
        $stmt->bind_param("i", $video_id);
        
        if ($stmt->execute()) {
            $message = '<div class="alert alert-success">Video deleted successfully.</div>';
            
            // Delete video file and thumbnail if they exist
            if (!empty($video['video_url'])) {
                $video_file_path = '../' . ltrim($video['video_url'], '/Panadite academy/');
                if (file_exists($video_file_path)) {
                    unlink($video_file_path);
                }
            }
            
            if (!empty($video['thumbnail'])) {
                $thumbnail_path = '../' . ltrim($video['thumbnail'], '/Panadite academy/');
                if (file_exists($thumbnail_path)) {
                    unlink($thumbnail_path);
                }
            }
        } else {
            $message = '<div class="alert alert-danger">Error deleting video: ' . $conn->error . '</div>';
        }
    }
}

// Toggle video publication status
if (isset($_GET['toggle_publish']) && is_numeric($_GET['toggle_publish'])) {
    $video_id = $_GET['toggle_publish'];
    
    // Get current status
    $stmt = $conn->prepare("SELECT is_published FROM videos WHERE video_id = ?");
    $stmt->bind_param("i", $video_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    
    if ($result->num_rows > 0) {
        // Toggle status
        $new_status = $row['is_published'] ? 0 : 1;
        $status_text = $new_status ? "published" : "unpublished";
        
        $stmt = $conn->prepare("UPDATE videos SET is_published = ? WHERE video_id = ?");
        $stmt->bind_param("ii", $new_status, $video_id);
        
        if ($stmt->execute()) {
            $message = '<div class="alert alert-success">Video ' . $status_text . ' successfully.</div>';
        } else {
            $message = '<div class="alert alert-danger">Error updating video status: ' . $conn->error . '</div>';
        }
    }
}

// Get all courses for filtering
$courses = [];
$result = $conn->query("SELECT course_id, title FROM courses ORDER BY title");
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $courses[] = $row;
    }
}

// Build query based on filters
$query = "SELECT v.*, c.title as course_title, u.first_name, u.last_name 
          FROM videos v 
          LEFT JOIN courses c ON v.course_id = c.course_id 
          LEFT JOIN users u ON v.teacher_id = u.user_id";

$queryParams = [];
if ($course_filter) {
    $query .= " WHERE v.course_id = ?";
    $queryParams[] = $course_filter;
}

$query .= " ORDER BY v.created_at DESC";

// Prepare and execute the query
if (!empty($queryParams)) {
    $stmt = $conn->prepare($query);
    $stmt->bind_param(str_repeat("i", count($queryParams)), ...$queryParams);
    $stmt->execute();
    $result = $stmt->get_result();
} else {
    $result = $conn->query($query);
}

// Fetch all videos
$videos = [];
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $videos[] = $row;
    }
}

// Get course details if filtering by course
$courseDetail = null;
if ($course_filter) {
    $stmt = $conn->prepare("SELECT title FROM courses WHERE course_id = ?");
    $stmt->bind_param("i", $course_filter);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($result->num_rows > 0) {
        $courseDetail = $result->fetch_assoc();
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Videos Management - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        /* Override styles for videos page */
        body {
            overflow-x: hidden;
        }
        .main-content {
            padding-top: 60px !important; /* Exact navbar height */
            margin-top: -11rem !important;
        }
        .dashboard-content {
            padding: 0 15px !important;
            margin-top: -10px !important;
        }
        .page-header {
            margin-top: 0 !important;
            margin-bottom: 10px !important;
            padding-top: 0 !important;
        }
        h1.h2 {
            margin-top: 0 !important;
            margin-bottom: 5px !important;
        }
        .filter-container {
            margin-bottom: 10px !important;
            margin-top: 0 !important;
        }
        .video-grid {
            margin-top: 0 !important;
            padding: 0 !important;
        }
        .table-container {
            margin-top: 0 !important;
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    <div class="pt-2 pb-4">
        <ul class="nav flex-column">
            <li class="nav-item">
                <a class="nav-link text-white" href="dashboard.php">
                    <i class="fas fa-tachometer-alt"></i> Dashboard
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link text-white" href="users.php">
                    <i class="fas fa-users"></i> Users
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link text-white" href="courses.php">
                    <i class="fas fa-graduation-cap"></i> Courses
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link active text-white" href="videos.php">
                    <i class="fas fa-video"></i> Videos
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link text-white" href="categories.php">
                    <i class="fas fa-tags"></i> Categories
                </a>
            </li>
    </div>
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>
        <!-- Dashboard Content -->
        <div class="dashboard-content" style="padding-top: 0; margin-top: -15px;">
            <div class="page-header">
                <h1 class="h2">
                    <?php if ($courseDetail): ?>
                        Videos for: <?php echo htmlspecialchars($courseDetail['title']); ?>
                    <?php else: ?>
                        Video Management
                    <?php endif; ?>
                </h1>
                <div class="btn-toolbar mb-2 mb-md-0">
                    <?php if ($course_filter): ?>
                    <a href="add_video.php?course_id=<?php echo $course_filter; ?>" class="btn btn-sm btn-primary me-2">
                        <i class="fas fa-plus me-1"></i> Add Video
                    </a>
                    <a href="courses.php" class="btn btn-sm btn-outline-secondary">
                        <i class="fas fa-arrow-left me-1"></i> Back to Courses
                    </a>
                    <?php else: ?>
                    <a href="add_video.php" class="btn btn-sm btn-primary">
                        <i class="fas fa-plus me-1"></i> Add Video
                    </a>
                    <?php endif; ?>
                </div>
            </div>
            <?php echo $message; ?>
            <!-- Filter Form -->
            <?php if (!$course_filter): ?>
            <div class="filter-container animate-fade-in mb-4">
                <form action="videos.php" method="get" class="row g-3">
                    <div class="col-md-6">
                        <label for="course_filter" class="form-label">Filter by Course</label>
                        <select class="form-select" id="course_filter" name="course_id">
                            <option value="">All Courses</option>
                            <?php foreach ($courses as $course): ?>
                                <option value="<?php echo $course['course_id']; ?>" <?php echo ($course_filter == $course['course_id']) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($course['title']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-md-12 mt-4">
                        <button type="submit" class="btn btn-primary">
                            <i class='bx bx-filter me-1'></i> Apply Filters
                        </button>
                        <a href="videos.php" class="btn btn-outline-secondary">
                            <i class='bx bx-reset me-1'></i> Reset
                        </a>
                    </div>
                </form>
            </div>
            <?php endif; ?>
            <!-- Videos Section -->
            <div class="table-container animate-fade-in" style="--delay: 0.1s">
                <div class="table-header d-flex justify-content-between align-items-center">
                    <h5>
                        <?php if (count($videos) > 0): ?>
                            <?php echo count($videos); ?> Videos
                        <?php else: ?>
                            No Videos Found
                        <?php endif; ?>
                    </h5>
                    <div>
                        <a href="add_video.php<?php echo $course_filter ? '?course_id=' . $course_filter : ''; ?>" class="btn btn-sm btn-primary">
                            <i class='bx bx-plus me-1'></i> Add New Video
                        </a>
                    </div>
                </div>
                
                <!-- Video Grid Layout -->
                <?php if (count($videos) > 0): ?>
                <div class="video-grid">
                    <?php foreach ($videos as $video): ?>
                    <div class="video-card animate-fade-in" style="--delay: <?php echo 0.1 + 0.05 * $loop = isset($loop) ? $loop + 1 : 1; ?>s">
                        <div class="video-thumbnail">
                            <?php if (!empty($video['thumbnail'])): ?>
                                <?php 
                                // Ensure the thumbnail path is properly formatted
                                $thumbnail = $video['thumbnail'];
                                if (strpos($thumbnail, 'http') !== 0 && strpos($thumbnail, '/') !== 0) {
                                    $thumbnail = '/' . $thumbnail;
                                }
                                ?>
                                <img src="<?php echo htmlspecialchars($thumbnail); ?>" alt="<?php echo htmlspecialchars($video['title']); ?>" loading="lazy" onerror="this.onerror=null;this.parentNode.innerHTML='<div class=\'video-thumbnail-placeholder\'><i class=\'bx bx-video\'></i></div>';">
                            <?php else: ?>
                                <div class="video-thumbnail-placeholder">
                                    <i class='bx bx-video'></i>
                                </div>
                            <?php endif; ?>
                            
                            <?php if ($video['is_published']): ?>
                                <div class="video-badge published"><i class='bx bx-check'></i> Published</div>
                            <?php else: ?>
                                <div class="video-badge unpublished"><i class='bx bx-time'></i> Draft</div>
                            <?php endif; ?>
                            
                            <div class="video-duration">
                                <i class='bx bx-time'></i> <?php echo isset($video['duration']) ? htmlspecialchars($video['duration']) : '00:00'; ?>
                            </div>
                        </div>
                        
                        <div class="video-details">
                            <h5 class="video-title"><?php echo htmlspecialchars($video['title']); ?></h5>
                            
                            <div class="video-meta">
                                <div class="video-course">
                                    <i class='bx bx-book-open'></i> <?php echo htmlspecialchars($video['course_title']); ?>
                                </div>
                                <div class="video-instructor">
                                    <i class='bx bx-user'></i> <?php echo htmlspecialchars($video['first_name'] . ' ' . $video['last_name']); ?>
                                </div>
                                <div class="video-date">
                                    <i class='bx bx-calendar'></i> <?php echo date('M d, Y', strtotime($video['created_at'])); ?>
                                </div>
                            </div>
                            
                            <div class="video-actions">
                                <a href="edit_video.php?id=<?php echo $video['video_id']; ?>" class="btn btn-sm btn-light" title="Edit Video">
                                    <i class='bx bx-edit'></i>
                                </a>
                                <a href="videos.php?toggle_publish=<?php echo $video['video_id']; ?>" class="btn btn-sm <?php echo $video['is_published'] ? 'btn-warning' : 'btn-success'; ?>" title="<?php echo $video['is_published'] ? 'Unpublish' : 'Publish'; ?>">
                                    <i class='bx <?php echo $video['is_published'] ? 'bx-hide' : 'bx-show'; ?>'></i>
                                </a>
                                <a href="preview_video.php?id=<?php echo $video['video_id']; ?>" class="btn btn-sm btn-info" title="Preview Video">
                                    <i class='bx bx-play'></i>
                                </a>
                                <a href="#" class="btn btn-sm btn-danger delete-video" data-id="<?php echo $video['video_id']; ?>" title="Delete Video">
                                    <i class='bx bx-trash'></i>
                                </a>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php else: ?>
                <div class="alert alert-info animate-fade-in" style="--delay: 0.2s">
                    <i class='bx bx-info-circle me-2'></i> No videos found. <a href="add_video.php">Add your first video</a>.
                </div>
                <?php endif; ?>
            </div>
            
            <!-- Delete Confirmation Modal -->
            <div class="modal fade" id="deleteConfirmationModal" tabindex="-1" aria-hidden="true">
                <div class="modal-dialog modal-dialog-centered">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title">Confirm Deletion</h5>
                            <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                        </div>
                        <div class="modal-body">
                            <p>Are you sure you want to delete this video? This action cannot be undone.</p>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Cancel</button>
                            <a href="#" id="confirmDelete" class="btn btn-danger">Delete</a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <style>
    /* Video Grid Styling */
    .video-grid {
        display: grid;
        grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
        gap: 20px;
        padding: 5px 0;
    }
    
    .video-card {
        background-color: #fff;
        border-radius: 10px;
        overflow: hidden;
        box-shadow: 0 5px 15px rgba(0, 0, 0, 0.05);
        transition: all 0.3s ease;
    }
    
    .video-card:hover {
        transform: translateY(-5px);
        box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
    }
    
    .video-thumbnail {
        position: relative;
        height: 180px;
        background-color: #f8f9fa;
        overflow: hidden;
    }
    
    .video-thumbnail img {
        width: 100%;
        height: 100%;
        object-fit: cover;
        transition: transform 0.3s ease;
    }
    
    .video-card:hover .video-thumbnail img {
        transform: scale(1.05);
    }
    
    .video-thumbnail-placeholder {
        display: flex;
        align-items: center;
        justify-content: center;
        width: 100%;
        height: 100%;
        background-color: var(--primary-color);
        background-image: linear-gradient(135deg, rgba(67, 97, 238, 0.8) 0%, rgba(63, 55, 201, 0.8) 100%);
        color: white;
    }
    
    .video-thumbnail-placeholder i {
        font-size: 50px;
        opacity: 0.7;
    }
    
    .video-badge {
        position: absolute;
        top: 10px;
        right: 10px;
        padding: 5px 10px;
        border-radius: 20px;
        font-size: 12px;
        font-weight: 600;
        color: white;
        display: flex;
        align-items: center;
    }
    
    .video-badge.published {
        background-color: var(--success-color);
    }
    
    .video-badge.unpublished {
        background-color: var(--warning-color);
    }
    
    .video-badge i {
        margin-right: 5px;
        font-size: 14px;
    }
    
    .video-duration {
        position: absolute;
        bottom: 10px;
        right: 10px;
        background-color: rgba(0, 0, 0, 0.7);
        color: white;
        padding: 3px 8px;
        border-radius: 4px;
        font-size: 12px;
        display: flex;
        align-items: center;
    }
    
    .video-duration i {
        margin-right: 5px;
        font-size: 14px;
    }
    
    .video-details {
        padding: 15px;
    }
    
    .video-title {
        font-size: 16px;
        font-weight: 600;
        margin-bottom: 10px;
        color: var(--dark-color);
        display: -webkit-box;
        -webkit-line-clamp: 2;
        -webkit-box-orient: vertical;
        overflow: hidden;
        text-overflow: ellipsis;
        height: 42px;
    }
    
    .video-meta {
        display: grid;
        grid-template-columns: 1fr;
        gap: 8px;
        margin-bottom: 15px;
    }
    
    .video-meta > div {
        display: flex;
        align-items: center;
        color: var(--grey-color);
        font-size: 13px;
        white-space: nowrap;
        overflow: hidden;
        text-overflow: ellipsis;
    }
    
    .video-meta i {
        margin-right: 5px;
        font-size: 14px;
    }
    
    .video-actions {
        display: flex;
        justify-content: space-between;
        border-top: 1px solid #f1f1f1;
        padding-top: 15px;
    }
    
    .video-actions .btn {
        width: 32px;
        height: 32px;
        padding: 0;
        display: flex;
        align-items: center;
        justify-content: center;
        border-radius: 4px;
    }
    
    .video-actions .btn i {
        font-size: 16px;
    }
    
    @media (max-width: 768px) {
        .video-grid {
            grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
        }
    }
    
    @media (max-width: 576px) {
        .video-grid {
            grid-template-columns: 1fr;
        }
    }
    </style>
    
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        // Sidebar toggle functionality
        const sidebarToggle = document.querySelector('.sidebar-toggle');
        const sidebarToggleMobile = document.querySelector('.sidebar-toggle-mobile');
        const sidebar = document.querySelector('.sidebar');
        
        if (sidebarToggle) {
            sidebarToggle.addEventListener('click', () => {
                sidebar.classList.toggle('close');
            });
        }
        
        if (sidebarToggleMobile) {
            sidebarToggleMobile.addEventListener('click', () => {
                sidebar.classList.toggle('show');
            });
        }
        
        // Responsive sidebar behavior
        function checkScreenSize() {
            if (window.innerWidth < 768) {
                sidebar.classList.remove('close');
                sidebar.classList.remove('show');
            } else if (window.innerWidth < 992) {
                sidebar.classList.add('close');
                sidebar.classList.remove('show');
            }
        }
        
        // Initial check and listen for window resize
        checkScreenSize();
        window.addEventListener('resize', checkScreenSize);
        
        // Dropdown functionality
        document.querySelectorAll('.dropdown-toggle').forEach(item => {
            item.addEventListener('click', event => {
                event.preventDefault();
                const dropdownMenu = item.nextElementSibling;
                if (dropdownMenu && dropdownMenu.classList.contains('dropdown-menu')) {
                    dropdownMenu.classList.toggle('show');
                }
            });
        });
        
        // Close dropdowns when clicking outside
        document.addEventListener('click', event => {
            const isDropdownButton = event.target.classList.contains('dropdown-toggle');
            if (!isDropdownButton) {
                document.querySelectorAll('.dropdown-menu.show').forEach(dropdown => {
                    dropdown.classList.remove('show');
                });
            }
        });
        
        // Delete confirmation modal handling
        const deleteModal = new bootstrap.Modal(document.getElementById('deleteConfirmationModal'));
        const confirmDeleteBtn = document.getElementById('confirmDelete');
        
        document.querySelectorAll('.delete-video').forEach(btn => {
            btn.addEventListener('click', function(e) {
                e.preventDefault();
                const videoId = this.getAttribute('data-id');
                confirmDeleteBtn.href = `videos.php?delete=${videoId}`;
                deleteModal.show();
            });
        });
        
        // Video hover effects
        document.querySelectorAll('.video-card').forEach(card => {
            card.addEventListener('mouseenter', function() {
                this.querySelector('.video-actions').style.opacity = '1';
            });
            
            card.addEventListener('mouseleave', function() {
                this.querySelector('.video-actions').style.opacity = '0.7';
            });
        });
    });
    </script>
</body>
</html>
