<?php
require_once __DIR__ . '/../auth/functions.php';
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../config/fast_email_service.php';

// Ensure user is logged in as an admin
requireRole('admin');

/**
 * Send withdrawal approval email notification to teacher
 * USING SAME FAST, CLEAN APPROACH AS WORKING LIBRARY/COURSE EMAILS
 */
function sendWithdrawalApprovalEmail($request_id) {
    global $conn;
    
    try {
        // Get withdrawal request and teacher details
        $stmt = $conn->prepare("
            SELECT wr.*, u.first_name, u.last_name, u.email, u.user_id
            FROM withdrawal_requests wr 
            JOIN users u ON wr.teacher_id = u.user_id 
            WHERE wr.id = ?
        ");
        $stmt->bind_param("i", $request_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $withdrawal = $result->fetch_assoc();
        
        if (!$withdrawal) {
            error_log("Withdrawal request not found for approval email: $request_id");
            return false;
        }
        
        $teacher_name = $withdrawal['first_name'] . ' ' . $withdrawal['last_name'];
        $teacher_email = $withdrawal['email'];
        $withdrawal_amount = number_format($withdrawal['amount'], 2);
        $payment_method = $withdrawal['payment_method'];
        $requested_date = date('M j, Y', strtotime($withdrawal['requested_at']));
        
        // Create email content - SAME STRUCTURE AS WORKING COURSE/LIBRARY EMAILS
        $subject = "💰 Withdrawal Approved: R$withdrawal_amount Processed!";
        $message = "
        <div style='max-width:600px; margin:0 auto; font-family:Arial,sans-serif;'>
            <div style='background:#28a745; color:white; padding:20px; text-align:center;'>
                <h1>💰 Withdrawal Approved!</h1>
            </div>
            <div style='padding:20px; background:white;'>
                <h2>Great news, " . htmlspecialchars($teacher_name) . "! 🎉</h2>
                <p>Your withdrawal request of <strong>R" . htmlspecialchars($withdrawal_amount) . "</strong> has been approved and processed!</p>
                <div style='background:#f8f9fa; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h3>Withdrawal Details:</h3>
                    <p><strong>Amount:</strong> R" . htmlspecialchars($withdrawal_amount) . "</p>
                    <p><strong>Payment Method:</strong> " . htmlspecialchars($payment_method) . "</p>
                    <p><strong>Requested:</strong> " . $requested_date . "</p>
                    <p><strong>Status:</strong> <span style='color:#28a745; font-weight:bold;'>✅ APPROVED</span></p>
                </div>
                <div style='background:#e8f5e8; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h4>🏦 What's Next?</h4>
                    <p>Your funds have been processed and should reflect in your bank account within 1-3 business days.</p>
                    <p>You will receive a confirmation once the transfer is complete.</p>
                </div>
                <p style='text-align:center;'>
                    <a href='https://panaditeacademy.com/teacher/earnings.php' 
                       style='background:#667eea; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block;'>
                        View Your Earnings
                    </a>
                </p>
            </div>
        </div>";
        
        // Try direct email sending - EXACT SAME PATTERN AS WORKING COURSE/LIBRARY EMAIL
        if (function_exists('sendEmailNow')) {
            $result = sendEmailNow($teacher_email, $subject, $message);
            if ($result) {
                return true;
            }
        }
        
        // Fallback to native PHP mail - EXACT SAME AS COURSE/LIBRARY EMAIL
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
        
        return mail($teacher_email, $subject, $message, $headers);
        
    } catch (Exception $e) {
        error_log("Error sending withdrawal approval email: " . $e->getMessage());
        return false;
    }
}

// Handle status updates with balance deduction for fraud prevention
if ($_POST && isset($_POST['update_status'])) { 
    $request_id = $_POST['request_id'];
    $new_status = $_POST['status']; 
    $admin_notes = $_POST['admin_notes'] ?? '';
    
    try {
        // Start transaction for atomic operations
        $conn->begin_transaction();
        
        // Get withdrawal request details first
        $get_request_query = "SELECT * FROM withdrawal_requests WHERE id = ?";
        $stmt = $conn->prepare($get_request_query);
        if (!$stmt) {
            throw new Exception("Failed to prepare get request query: " . $conn->error);
        }
        
        $stmt->bind_param("i", $request_id);
        if (!$stmt->execute()) {
            throw new Exception("Failed to execute get request query: " . $stmt->error);
        }
        
        $withdrawal_request = $stmt->get_result()->fetch_assoc();
        $stmt->close();
        
        if (!$withdrawal_request) {
            throw new Exception("Withdrawal request not found");
        }
        
        $teacher_id = $withdrawal_request['teacher_id'];
        $withdrawal_amount = $withdrawal_request['amount'];
        $current_status = $withdrawal_request['status'];
        
        // FRAUD PREVENTION: Deduct balance when approving withdrawal
        if ($new_status === 'approved' && $current_status !== 'approved') {
            // Verify teacher has sufficient balance
            $balance_check_query = "SELECT SUM(amount) as total_earnings FROM teacher_earnings WHERE teacher_id = ?";
            $stmt = $conn->prepare($balance_check_query);
            if (!$stmt) {
                throw new Exception("Failed to prepare balance check query: " . $conn->error);
            }
            
            $stmt->bind_param("i", $teacher_id);
            if (!$stmt->execute()) {
                throw new Exception("Failed to execute balance check query: " . $stmt->error);
            }
            
            $balance_result = $stmt->get_result()->fetch_assoc();
            $stmt->close();
            
            $available_balance = $balance_result['total_earnings'] ?? 0;
            
            if ($available_balance < $withdrawal_amount) {
                throw new Exception("Insufficient balance. Teacher has R" . number_format($available_balance, 2) . " but trying to withdraw R" . number_format($withdrawal_amount, 2));
            }
            
            // Create deduction record in teacher_earnings (negative amount)
            $deduction_query = "INSERT INTO teacher_earnings (teacher_id, amount) VALUES (?, ?)";
            $stmt = $conn->prepare($deduction_query);
            
            if (!$stmt) {
                throw new Exception("Failed to prepare deduction query: " . $conn->error);
            }
            
            $negative_amount = -$withdrawal_amount; // Negative for deduction
            $stmt->bind_param("id", $teacher_id, $negative_amount);
            
            if (!$stmt->execute()) {
                throw new Exception("Failed to create deduction record: " . $stmt->error);
            }
            $stmt->close();
            
            error_log("WITHDRAWAL APPROVED: Deducted R$withdrawal_amount from teacher $teacher_id for request #$request_id");
        }
        
        // Update withdrawal request status
        $update_query = "UPDATE withdrawal_requests SET 
                        status = ?, 
                        admin_notes = ?, 
                        processed_by = ?, 
                        processed_at = NOW() 
                        WHERE id = ?";
        $stmt = $conn->prepare($update_query);
        if (!$stmt) {
            throw new Exception("Failed to prepare status update query: " . $conn->error);
        }
        
        $stmt->bind_param("ssii", $new_status, $admin_notes, $_SESSION['user_id'], $request_id);
        
        if (!$stmt->execute()) {
            throw new Exception("Failed to update withdrawal request: " . $stmt->error);
        }
        $stmt->close();
        
        // Commit transaction
        $conn->commit();
        
        // Send email notification for approved withdrawals - USING FAST CLEAN APPROACH
        if ($new_status === 'approved') {
            $email_result = sendWithdrawalApprovalEmail($request_id);
            
            if ($email_result) {
                $success_message = "Withdrawal request approved successfully! ✉️ Approval email sent to teacher. R" . number_format($withdrawal_amount, 2) . " deducted from teacher's balance.";
            } else {
                $success_message = "Withdrawal request approved successfully and R" . number_format($withdrawal_amount, 2) . " deducted from teacher's balance, but failed to send notification email to teacher.";
            }
        } else {
            $success_message = "Withdrawal request updated successfully.";
        }
        
    } catch (Exception $e) {
        // Rollback on error
        $conn->rollback();
        $error_message = "Error updating withdrawal: " . $e->getMessage();
        error_log("Withdrawal update error: " . $e->getMessage());
    }
}

// Get filter parameters
$filter_status = $_GET['status'] ?? '';
$filter_teacher_id = $_GET['teacher_id'] ?? '';
$filter_date_from = $_GET['date_from'] ?? '';
$filter_date_to = $_GET['date_to'] ?? '';

// Create withdrawal_requests table if it doesn't exist
$create_table_sql = "CREATE TABLE IF NOT EXISTS withdrawal_requests (
    id INT AUTO_INCREMENT PRIMARY KEY,
    teacher_id INT NOT NULL,
    amount DECIMAL(10,2) NOT NULL,
    payment_method VARCHAR(50) DEFAULT 'bank_transfer',
    bank_name VARCHAR(100),
    account_number VARCHAR(50),
    account_holder VARCHAR(100),
    status ENUM('pending', 'approved', 'completed', 'rejected') DEFAULT 'pending',
    admin_notes TEXT,
    processed_by INT,
    processed_at TIMESTAMP NULL,
    requested_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
)";
$conn->query($create_table_sql);

// Build query with filters
$query = "SELECT wr.*, u.username, u.email 
          FROM withdrawal_requests wr 
          LEFT JOIN users u ON wr.teacher_id = u.user_id 
          WHERE 1=1";

$params = [];
$types = "";

if ($filter_status) {
    $query .= " AND wr.status = ?";
    $params[] = $filter_status;
    $types .= "s";
}

if ($filter_teacher_id) {
    $query .= " AND wr.teacher_id = ?";
    $params[] = $filter_teacher_id;
    $types .= "i";
}

if ($filter_date_from) {
    $query .= " AND DATE(wr.requested_at) >= ?";
    $params[] = $filter_date_from;
    $types .= "s";
}

if ($filter_date_to) {
    $query .= " AND DATE(wr.requested_at) <= ?";
    $params[] = $filter_date_to;
    $types .= "s";
}

$query .= " ORDER BY wr.requested_at DESC";

// Execute query
$requests = null;
if ($stmt = $conn->prepare($query)) {
    if ($params) {
        $stmt->bind_param($types, ...$params);
    }
    $stmt->execute();
    $requests = $stmt->get_result();
}

// Get summary statistics
$stats_query = "SELECT 
                COUNT(*) as total_requests,
                COUNT(CASE WHEN status = 'pending' THEN 1 END) as pending,
                COUNT(CASE WHEN status = 'approved' THEN 1 END) as approved,
                COUNT(CASE WHEN status = 'completed' THEN 1 END) as completed,
                COUNT(CASE WHEN status = 'rejected' THEN 1 END) as rejected,
                COALESCE(SUM(CASE WHEN status IN ('approved', 'completed') THEN amount ELSE 0 END), 0) as total_approved
                FROM withdrawal_requests";
$stats_result = $conn->query($stats_query);
$stats = $stats_result ? $stats_result->fetch_assoc() : [
    'total_requests' => 0, 'pending' => 0, 'approved' => 0, 
    'completed' => 0, 'rejected' => 0, 'total_approved' => 0
];
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Withdrawal Requests - Panadite Academy Admin</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href='https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css' rel='stylesheet'>
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="/assets/css/admin-dashboard.css">
    <link rel="stylesheet" href="/assets/css/user-cards.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        :root {
            --primary-color: #4361ee;
            --secondary-color: #3f37c9;
            --success-color: #10b981;
            --warning-color: #f59e0b;
            --danger-color: #ef4444;
            --info-color: #3b82f6;
            --dark-color: #1f2937;
            --light-bg: #f8fafc;
            --white: #ffffff;
            --border-color: #e2e8f0;
            --text-muted: #64748b;
            --shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
            --shadow-lg: 0 10px 15px -3px rgba(0, 0, 0, 0.1);
        }

        * { font-family: 'Inter', sans-serif !important; }
        body { background: var(--light-bg) !important; color: var(--dark-color) !important; }

        .page-header {
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%) !important;
            color: white !important;
            padding: 2rem !important;
            border-radius: 16px !important;
            margin-bottom: 2rem !important;
            box-shadow: var(--shadow-lg) !important;
        }

        .page-title {
            font-size: 2rem !important;
            font-weight: 700 !important;
            margin-bottom: 0.5rem !important;
            display: flex !important;
            align-items: center !important;
            gap: 12px !important;
        }

        .stats-container {
            display: grid !important;
            grid-template-columns: repeat(auto-fit, minmax(180px, 1fr)) !important;
            gap: 20px !important;
            margin-bottom: 32px !important;
        }

        .modern-stat-card {
            background: var(--white) !important;
            border-radius: 16px !important;
            padding: 24px !important;
            box-shadow: var(--shadow) !important;
            border: 1px solid var(--border-color) !important;
            transition: all 0.3s ease !important;
            position: relative !important;
            overflow: hidden !important;
        }

        .modern-stat-card:hover { transform: translateY(-2px) !important; box-shadow: var(--shadow-lg) !important; }
        .modern-stat-card::before {
            content: '' !important; position: absolute !important; top: 0 !important; left: 0 !important; 
            right: 0 !important; height: 4px !important; background: var(--primary-color) !important;
        }
        .modern-stat-card.success::before { background: var(--success-color) !important; }
        .modern-stat-card.warning::before { background: var(--warning-color) !important; }
        .modern-stat-card.danger::before { background: var(--danger-color) !important; }
        .modern-stat-card.info::before { background: var(--info-color) !important; }

        .stat-header { display: flex !important; align-items: center !important; justify-content: space-between !important; margin-bottom: 16px !important; }
        .stat-icon {
            width: 48px !important; height: 48px !important; border-radius: 12px !important; display: flex !important;
            align-items: center !important; justify-content: center !important; font-size: 20px !important;
            color: white !important; background: var(--primary-color) !important;
        }
        .modern-stat-card.success .stat-icon { background: var(--success-color) !important; }
        .modern-stat-card.warning .stat-icon { background: var(--warning-color) !important; }
        .modern-stat-card.danger .stat-icon { background: var(--danger-color) !important; }
        .modern-stat-card.info .stat-icon { background: var(--info-color) !important; }

        .stat-number { font-size: 2rem !important; font-weight: 800 !important; color: var(--dark-color) !important; line-height: 1 !important; margin-bottom: 8px !important; }
        .stat-label { font-size: 0.875rem !important; color: var(--text-muted) !important; font-weight: 500 !important; margin: 0 !important; }

        .filter-section { background: var(--white) !important; padding: 24px !important; border-radius: 16px !important; margin-bottom: 24px !important; box-shadow: var(--shadow) !important; border: 1px solid var(--border-color) !important; }
        .filter-title { font-size: 1.125rem !important; font-weight: 600 !important; color: var(--dark-color) !important; margin-bottom: 16px !important; display: flex !important; align-items: center !important; gap: 8px !important; }

        .modern-table-container { background: var(--white) !important; border-radius: 16px !important; padding: 24px !important; box-shadow: var(--shadow) !important; border: 1px solid var(--border-color) !important; overflow: hidden !important; }
        .table-title { font-size: 1.25rem !important; font-weight: 600 !important; color: var(--dark-color) !important; margin-bottom: 20px !important; display: flex !important; align-items: center !important; justify-content: space-between !important; }
        .table-count { background: var(--primary-color) !important; color: white !important; padding: 4px 12px !important; border-radius: 20px !important; font-size: 0.875rem !important; font-weight: 500 !important; }

        .table { border: none !important; margin-bottom: 0 !important; }
        .table thead th { background: var(--light-bg) !important; border: none !important; color: var(--dark-color) !important; font-weight: 600 !important; font-size: 0.875rem !important; text-transform: uppercase !important; letter-spacing: 0.5px !important; padding: 16px !important; }
        .table tbody td { padding: 16px !important; vertical-align: middle !important; border-bottom: 1px solid var(--border-color) !important; font-size: 0.875rem !important; }
        .table tbody tr:hover { background: rgba(67, 97, 238, 0.02) !important; }

        .status-badge { padding: 6px 12px !important; border-radius: 20px !important; font-weight: 600 !important; font-size: 0.75rem !important; text-transform: uppercase !important; letter-spacing: 0.5px !important; border: 2px solid transparent !important; }
        .status-pending { background: rgba(245, 158, 11, 0.1) !important; color: var(--warning-color) !important; border-color: rgba(245, 158, 11, 0.2) !important; }
        .status-approved { background: rgba(16, 185, 129, 0.1) !important; color: var(--success-color) !important; border-color: rgba(16, 185, 129, 0.2) !important; }
        .status-completed { background: rgba(59, 130, 246, 0.1) !important; color: var(--info-color) !important; border-color: rgba(59, 130, 246, 0.2) !important; }
        .status-rejected { background: rgba(239, 68, 68, 0.1) !important; color: var(--danger-color) !important; border-color: rgba(239, 68, 68, 0.2) !important; }

        .btn-modern { padding: 8px 16px !important; border-radius: 8px !important; font-weight: 500 !important; font-size: 0.875rem !important; border: 2px solid transparent !important; transition: all 0.2s ease !important; }
        .btn-modern.btn-primary { background: var(--primary-color) !important; color: white !important; border-color: var(--primary-color) !important; }
        .btn-modern.btn-success { background: var(--success-color) !important; color: white !important; border-color: var(--success-color) !important; }
        .btn-modern.btn-danger { background: var(--danger-color) !important; color: white !important; border-color: var(--danger-color) !important; }
        .btn-modern:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(67, 97, 238, 0.3);
        }

        /* Modal Detail Cards */
        .detail-card {
            background: #ffffff;
            border: 1px solid #e5e8eb;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 15px;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08);
            height: 100%;
        }

        .detail-card h6 {
            font-weight: 600;
            margin-bottom: 15px;
            padding-bottom: 8px;
            border-bottom: 2px solid #f1f3f4;
        }

        .detail-label {
            font-weight: 500;
            color: #495057;
            display: inline-block;
            min-width: 100px;
            margin-right: 10px;
            background: none !important;
            border: none !important;
            padding: 0 !important;
        }

        .detail-value {
            font-weight: 600;
            color: #2d3436;
            background: none !important;
            border: none !important;
            padding: 0 !important;
            user-select: text;
        }

        .modal-content {
            border: none;
            border-radius: 16px;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.15);
        }

        .modal-header {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            color: white;
            border-radius: 16px 16px 0 0;
            padding: 20px 24px;
        }

        .modal-header .btn-close {
            filter: invert(1);
            opacity: 0.8;
        }

        .modal-header .btn-close:hover {
            opacity: 1;
        }

        .modal-body {
            padding: 24px;
        }

        .modal-footer {
            padding: 20px 24px;
            border-top: 1px solid #e5e8eb;
            background: #f8f9fa;
            border-radius: 0 0 16px 16px;
        }

        /* Status badge in modal */
        #modal-status-badge.status-badge {
            font-size: 0.875rem;
            padding: 6px 12px;
            border-radius: 20px;
            font-weight: 600;
        }

        .form-select, .form-control { border: 2px solid var(--border-color) !important; border-radius: 8px !important; padding: 12px 16px !important; font-size: 0.875rem !important; transition: all 0.2s ease !important; }
        .form-select:focus, .form-control:focus { border-color: var(--primary-color) !important; box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.1) !important; }

        .no-data-state { text-align: center !important; padding: 60px 20px !important; color: var(--text-muted) !important; }
        .no-data-icon { font-size: 3rem !important; color: var(--border-color) !important; margin-bottom: 16px !important; }
        .no-data-text { font-size: 1.125rem !important; font-weight: 500 !important; margin-bottom: 8px !important; }
        .no-data-subtext { font-size: 0.875rem !important; opacity: 0.7 !important; }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <?php include 'components/sidebar.php'; ?>
    
    <!-- Main Content -->
    <section class="main-content">
        <!-- Navbar -->
        <?php include 'components/navbar.php'; ?>
        
        <!-- Dashboard Content -->
        <div class="dashboard-content">
            <div class="container-fluid p-4">
                <!-- Modern Page Header -->
                <div class="page-header">
                    <h1 class="page-title">
                        <i class='bx bx-money-withdraw'></i>
                        Withdrawal Requests Management
                    </h1>
                    <p class="page-description">Manage teacher withdrawal requests and payouts</p>
                </div>

                <!-- Modern Statistics Cards -->
                <div class="stats-container">
                    <div class="modern-stat-card">
                        <div class="stat-header">
                            <div class="stat-icon"><i class='bx bx-list-ul'></i></div>
                        </div>
                        <div class="stat-number"><?php echo $stats['total_requests']; ?></div>
                        <div class="stat-label">Total Requests</div>
                    </div>

                    <div class="modern-stat-card warning">
                        <div class="stat-header">
                            <div class="stat-icon"><i class='bx bx-time-five'></i></div>
                        </div>
                        <div class="stat-number"><?php echo $stats['pending']; ?></div>
                        <div class="stat-label">Pending</div>
                    </div>

                    <div class="modern-stat-card success">
                        <div class="stat-header">
                            <div class="stat-icon"><i class='bx bx-check-circle'></i></div>
                        </div>
                        <div class="stat-number"><?php echo $stats['approved']; ?></div>
                        <div class="stat-label">Approved</div>
                    </div>

                    <div class="modern-stat-card info">
                        <div class="stat-header">
                            <div class="stat-icon"><i class='bx bx-check-double'></i></div>
                        </div>
                        <div class="stat-number"><?php echo $stats['completed']; ?></div>
                        <div class="stat-label">Completed</div>
                    </div>

                    <div class="modern-stat-card danger">
                        <div class="stat-header">
                            <div class="stat-icon"><i class='bx bx-x-circle'></i></div>
                        </div>
                        <div class="stat-number"><?php echo $stats['rejected']; ?></div>
                        <div class="stat-label">Rejected</div>
                    </div>

                    <div class="modern-stat-card success">
                        <div class="stat-header">
                            <div class="stat-icon"><i class='bx bx-money'></i></div>
                        </div>
                        <div class="stat-number">R<?php echo number_format($stats['total_approved'], 2); ?></div>
                        <div class="stat-label">Total Approved</div>
                    </div>
                </div>

                <!-- Success Message -->
                <?php if (isset($success_message)): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <i class="bx bx-check-circle"></i> <?= htmlspecialchars($success_message) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <!-- Error Message -->
                <?php if (isset($error_message)): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <i class="bx bx-error-circle"></i> <?= htmlspecialchars($error_message) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <!-- Modern Filters -->
                <div class="filter-section">
                    <div class="filter-title">
                        <i class='bx bx-filter-alt'></i>
                        Filter Requests
                    </div>
                    <form method="GET" class="row g-3">
                        <div class="col-md-3">
                            <label class="form-label">Status</label>
                            <select name="status" class="form-select">
                                <option value="">All Status</option>
                                <option value="pending" <?php echo ($filter_status == 'pending') ? 'selected' : ''; ?>>Pending</option>
                                <option value="approved" <?php echo ($filter_status == 'approved') ? 'selected' : ''; ?>>Approved</option>
                                <option value="completed" <?php echo ($filter_status == 'completed') ? 'selected' : ''; ?>>Completed</option>
                                <option value="rejected" <?php echo ($filter_status == 'rejected') ? 'selected' : ''; ?>>Rejected</option>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">Teacher</label>
                            <select name="teacher_id" class="form-select">
                                <option value="">All Teachers</option>
                                <?php
                                $teacher_query = "SELECT user_id, username FROM users WHERE role = 'teacher'";
                                $teacher_result = $conn->query($teacher_query);
                                if ($teacher_result) {
                                    while ($teacher = $teacher_result->fetch_assoc()) {
                                        $selected = ($filter_teacher_id == $teacher['user_id']) ? 'selected' : '';
                                        echo "<option value='{$teacher['user_id']}' {$selected}>{$teacher['username']}</option>";
                                    }
                                }
                                ?>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">From Date</label>
                            <input type="date" name="date_from" class="form-control" value="<?php echo $filter_date_from; ?>">
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">To Date</label>
                            <input type="date" name="date_to" class="form-control" value="<?php echo $filter_date_to; ?>">
                        </div>
                        <div class="col-md-2">
                            <label class="form-label">&nbsp;</label>
                            <div>
                                <button type="submit" class="btn btn-modern btn-primary w-100">
                                    <i class='bx bx-search'></i> Filter
                                </button>
                            </div>
                        </div>
                    </form>
                </div>

                <!-- Modern Table -->
                <div class="modern-table-container">
                    <div class="table-title">
                        <span>
                            <i class='bx bx-table'></i>
                            Withdrawal Requests
                        </span>
                        <span class="table-count"><?php echo $requests ? $requests->num_rows : 0; ?></span>
                    </div>

                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Teacher</th>
                                    <th>Amount</th>
                                    <th>Payment Method</th>
                                    <th>Status</th>
                                    <th>Requested</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if ($requests && $requests->num_rows > 0): ?>
                                    <?php while ($request = $requests->fetch_assoc()): ?>
                                    <tr>
                                        <td><strong>#<?php echo htmlspecialchars($request['id']); ?></strong></td>
                                        <td>
                                            <div class="fw-bold text-primary"><?php echo htmlspecialchars($request['username'] ?? 'Unknown'); ?></div>
                                            <div class="text-muted small"><?php echo htmlspecialchars($request['email'] ?? ''); ?></div>
                                        </td>
                                        <td>
                                            <span class="fw-bold text-success">R<?php echo number_format($request['amount'], 2); ?></span>
                                        </td>
                                        <td>
                                            <span class="badge bg-light text-dark">
                                                <?php echo ucfirst(str_replace('_', ' ', $request['payment_method'] ?? 'Bank Transfer')); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php 
                                            $display_status = $request['status'] ?? 'pending';
                                            if (empty($display_status)) $display_status = 'pending';
                                            ?>
                                            <span class="status-badge status-<?php echo $display_status; ?>">
                                                <?php echo ucfirst($display_status); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <div class="fw-bold"><?php echo date('M j, Y', strtotime($request['requested_at'])); ?></div>
                                            <div class="text-muted small"><?php echo date('g:i A', strtotime($request['requested_at'])); ?></div>
                                        </td>
                                        <td class="text-center">
                                            <div class="btn-group" role="group">
                                                <button class="btn btn-modern btn-primary btn-sm" 
                                                        onclick="viewRequest(<?php echo $request['id']; ?>)" 
                                                        title="View Details"
                                                        data-request-id="<?php echo htmlspecialchars($request['id']); ?>"
                                                        data-teacher-name="<?php echo htmlspecialchars($request['username'] ?? 'Unknown'); ?>"
                                                        data-teacher-email="<?php echo htmlspecialchars($request['email'] ?? ''); ?>"
                                                        data-amount="R<?php echo number_format($request['amount'], 2); ?>"
                                                        data-payment-method="<?php echo htmlspecialchars(ucfirst(str_replace('_', ' ', $request['payment_method'] ?? 'Bank Transfer'))); ?>"
                                                        data-status="<?php echo htmlspecialchars(ucfirst($request['status'])); ?>"
                                                        data-requested-date="<?php echo date('M j, Y', strtotime($request['requested_at'])); ?>"
                                                        data-requested-time="<?php echo date('g:i A', strtotime($request['requested_at'])); ?>"
                                                        data-bank-name="<?php echo htmlspecialchars($request['bank_name'] ?? 'Not specified'); ?>"
                                                        data-account-number="<?php echo htmlspecialchars($request['account_number'] ?? 'Not specified'); ?>"
                                                        data-account-holder="<?php echo htmlspecialchars($request['account_holder'] ?? 'Not specified'); ?>"
                                                        data-branch-code="<?php echo htmlspecialchars($request['branch_code'] ?? 'Not specified'); ?>"
                                                        data-account-type="<?php echo htmlspecialchars($request['account_type'] ?? 'Not specified'); ?>"
                                                        data-payment-details="<?php echo htmlspecialchars($request['payment_details'] ?? ''); ?>"
                                                        data-notes="<?php echo htmlspecialchars($request['notes'] ?? ''); ?>"
                                                        data-admin-notes="<?php echo htmlspecialchars($request['admin_notes'] ?? ''); ?>"
                                                        data-processed-at="<?php echo $request['processed_at'] ? date('M j, Y g:i A', strtotime($request['processed_at'])) : ''; ?>"
                                                        data-processed-by="<?php echo htmlspecialchars($request['processed_by'] ?? ''); ?>">
                                                    <i class='bx bx-eye'></i>
                                                </button>
                                                
                                                <?php 
                                                $status = $request['status'] ?? 'pending';
                                                if ($status == 'pending' || empty($status)): 
                                                ?>
                                                    <button class="btn btn-modern btn-success btn-sm" onclick="updateStatus(<?php echo $request['id']; ?>, 'approved')" title="Approve">
                                                        <i class='bx bx-check'></i>
                                                    </button>
                                                    <button class="btn btn-modern btn-danger btn-sm" onclick="updateStatus(<?php echo $request['id']; ?>, 'rejected')" title="Reject">
                                                        <i class='bx bx-x'></i>
                                                    </button>
                                                <?php 
                                                elseif ($status == 'approved'): 
                                                ?>
                                                    <button class="btn btn-modern btn-success btn-sm" onclick="updateStatus(<?php echo $request['id']; ?>, 'completed')" title="Mark as Completed">
                                                        <i class='bx bx-check-double'></i>
                                                    </button>
                                                <?php 
                                                elseif ($status == 'completed'): 
                                                ?>
                                                    <span class="text-success small">
                                                        <i class='bx bx-check-double'></i> Completed
                                                    </span>
                                                <?php 
                                                elseif ($status == 'rejected'): 
                                                ?>
                                                    <span class="text-danger small">
                                                        <i class='bx bx-x'></i> Rejected
                                                    </span>
                                                <?php 
                                                elseif ($status == 'cancelled'): 
                                                ?>
                                                    <span class="text-muted small">
                                                        <i class='bx bx-x-circle'></i> Cancelled
                                                    </span>
                                                <?php 
                                                else: 
                                                ?>
                                                    <!-- Fallback for any unknown status - still show approve/reject -->
                                                    <button class="btn btn-modern btn-success btn-sm" onclick="updateStatus(<?php echo $request['id']; ?>, 'approved')" title="Approve">
                                                        <i class='bx bx-check'></i>
                                                    </button>
                                                    <button class="btn btn-modern btn-danger btn-sm" onclick="updateStatus(<?php echo $request['id']; ?>, 'rejected')" title="Reject">
                                                        <i class='bx bx-x'></i>
                                                    </button>
                                                <?php 
                                                endif; 
                                                ?>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endwhile; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="7" class="no-data-state">
                                            <div class="no-data-icon">
                                                <i class='bx bx-inbox'></i>
                                            </div>
                                            <div class="no-data-text">No withdrawal requests found</div>
                                            <div class="no-data-subtext">
                                                <?php if ($filter_status || $filter_teacher_id || $filter_date_from || $filter_date_to): ?>
                                                    Try adjusting your filters or <a href="withdrawal-requests.php" class="text-primary">clear all filters</a>
                                                <?php else: ?>
                                                    Withdrawal requests will appear here once teachers submit them
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Withdrawal Request Details Modal -->
    <div class="modal fade" id="requestDetailsModal" tabindex="-1" aria-labelledby="requestDetailsModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="requestDetailsModalLabel">
                        <i class='bx bx-money-withdraw'></i>
                        Withdrawal Request Details
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="row">
                        <!-- Request Information -->
                        <div class="col-md-6">
                            <div class="detail-card">
                                <h6 class="text-primary mb-3">
                                    <i class='bx bx-info-circle'></i>
                                    Request Information
                                </h6>
                                <div class="mb-2">
                                    <span class="detail-label">Request ID:</span>
                                    <span class="detail-value" id="modal-request-id">#--</span>
                                </div>
                                <div class="mb-2">
                                    <span class="detail-label">Amount:</span>
                                    <span class="detail-value text-success" id="modal-amount">R0.00</span>
                                </div>
                                <div class="mb-2">
                                    <span class="detail-label">Status:</span>
                                    <span class="detail-value" id="modal-status-badge">--</span>
                                </div>
                                <div class="mb-2">
                                    <span class="detail-label">Requested:</span>
                                    <span class="detail-value" id="modal-requested-date">--</span>
                                </div>
                                <div class="mb-0">
                                    <span class="detail-label">Processing:</span>
                                    <span class="detail-value" id="modal-processed-info">Not processed yet</span>
                                </div>
                            </div>
                        </div>

                        <!-- Teacher Information -->
                        <div class="col-md-6">
                            <div class="detail-card">
                                <h6 class="text-success mb-3">
                                    <i class='bx bx-user'></i>
                                    Teacher Information
                                </h6>
                                <div class="mb-2">
                                    <span class="detail-label">Name:</span>
                                    <span class="detail-value" id="modal-teacher-name">--</span>
                                </div>
                                <div class="mb-2">
                                    <span class="detail-label">Email:</span>
                                    <span class="detail-value" id="modal-teacher-email">--</span>
                                </div>
                                <div class="mb-0">
                                    <span class="detail-label">Teacher ID:</span>
                                    <span class="detail-value" id="modal-teacher-id">--</span>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Payment Information with Bank Logo -->
                    <div class="row mt-3">
                        <div class="col-12">
                            <div class="detail-card">
                                <h6 class="text-warning mb-3">
                                    <i class='bx bx-credit-card'></i>
                                    Payment Information
                                </h6>
                                
                                <!-- Bank Info with Logo -->
                                <div class="row mb-3">
                                    <div class="col-md-8">
                                        <div class="d-flex align-items-center mb-2">
                                            <img id="modal-bank-logo" src="" alt="Bank Logo" style="height: 40px; width: auto; margin-right: 10px; border-radius: 4px; display: none;">
                                            <div>
                                                <div class="mb-1">
                                                    <span class="detail-label">Bank:</span>
                                                    <span class="detail-value fw-bold" id="modal-bank-name">--</span>
                                                </div>
                                                <div class="mb-0">
                                                    <span class="detail-label">Payment Method:</span>
                                                    <span class="detail-value" id="modal-payment-method">--</span>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Account Details -->
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="mb-2">
                                            <span class="detail-label">Account Holder:</span>
                                            <span class="detail-value fw-bold" id="modal-account-holder">--</span>
                                        </div>
                                        <div class="mb-2">
                                            <span class="detail-label">Account Number:</span>
                                            <span class="detail-value font-monospace" id="modal-account-number">--</span>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="mb-2">
                                            <span class="detail-label">Branch Code:</span>
                                            <span class="detail-value font-monospace" id="modal-branch-code">--</span>
                                        </div>
                                        <div class="mb-2">
                                            <span class="detail-label">Account Type:</span>
                                            <span class="detail-value" id="modal-account-type">--</span>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Payment Details (if any) -->
                                <div class="mt-2" id="payment-details-section" style="display: none;">
                                    <span class="detail-label">Payment Details:</span>
                                    <div class="detail-value mt-1 p-2 bg-light border-start border-primary border-3" id="modal-payment-details">--</div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Teacher Notes (if any) -->
                    <div class="row mt-3" id="teacher-notes-section" style="display: none;">
                        <div class="col-12">
                            <div class="detail-card">
                                <h6 class="text-secondary mb-3">
                                    <i class='bx bx-message-dots'></i>
                                    Teacher Notes
                                </h6>
                                <div class="detail-value p-2 bg-light border-start border-secondary border-3" id="modal-teacher-notes">--</div>
                            </div>
                        </div>
                    </div>

                    <!-- Processing Information -->
                    <div class="row mt-3" id="processing-info-section">
                        <div class="col-12">
                            <div class="detail-card">
                                <h6 class="text-info mb-3">
                                    <i class='bx bx-cog'></i>
                                    Processing Information
                                </h6>
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="mb-2">
                                            <span class="detail-label">Processed:</span>
                                            <span class="detail-value" id="modal-processed-info">Not processed yet</span>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="mb-2">
                                            <span class="detail-label">Processed By:</span>
                                            <span class="detail-value" id="modal-processed-by">--</span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Admin Notes (if any) -->
                    <div class="row mt-3" id="admin-notes-section" style="display: none;">
                        <div class="col-12">
                            <div class="detail-card">
                                <h6 class="text-danger mb-3">
                                    <i class='bx bx-note'></i>
                                    Admin Notes
                                </h6>
                                <div class="detail-value p-2 bg-light border-start border-danger border-3" id="modal-admin-notes">--</div>
                            </div>
                        </div>
                    </div> 
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-modern btn-secondary" data-bs-dismiss="modal">
                        <i class='bx bx-x'></i> Close
                    </button>
                    <div id="modal-action-buttons">
                        <!-- Action buttons will be populated here based on request status -->
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function viewRequest(id) {
            // Find the clicked button to get the real data attributes
            const button = event.target.closest('button[data-request-id="' + id + '"]');
            if (!button) {
                alert('Request data not found');
                return;
            }

            // Extract real data from the button's data attributes
            const requestData = {
                id: button.dataset.requestId,
                requestId: '#' + button.dataset.requestId,
                teacherName: button.dataset.teacherName,
                teacherEmail: button.dataset.teacherEmail,
                amount: button.dataset.amount,
                paymentMethod: button.dataset.paymentMethod,
                status: button.dataset.status,
                requestedDate: button.dataset.requestedDate,
                requestedTime: button.dataset.requestedTime,
                bankName: button.dataset.bankName,
                accountNumber: button.dataset.accountNumber,
                accountHolder: button.dataset.accountHolder,
                branchCode: button.dataset.branchCode,
                accountType: button.dataset.accountType,
                paymentDetails: button.dataset.paymentDetails,
                notes: button.dataset.notes,
                adminNotes: button.dataset.adminNotes,
                processedAt: button.dataset.processedAt,
                processedBy: button.dataset.processedBy
            };

            // Bank logo mapping
            const bankLogos = {
                'fnb': '../banks/FNB.svg',
                'first national bank': '../banks/FNB.svg',
                'nedbank': '../banks/Nedbank.svg',
                'absa': '../banks/absa.png',
                'african bank': '../banks/african-bank.png',
                'capitec': '../banks/capitec.svg',
                'capitec bank': '../banks/capitec.svg',
                'discovery bank': '../banks/discovery_bank.svg',
                'discovery': '../banks/discovery_bank.svg',
                'old mutual bank': '../banks/old_mutual_bank.svg',
                'old mutual': '../banks/old_mutual_bank.svg',
                'standard bank': '../banks/standard_bank.png',
                'standard': '../banks/standard_bank.png',
                'tymebank': '../banks/tymebank.png',
                'tyme bank': '../banks/tymebank.png'
            };

            // Populate modal with REAL data from database
            document.getElementById('modal-request-id').textContent = requestData.requestId;
            document.getElementById('modal-amount').textContent = requestData.amount;
            document.getElementById('modal-teacher-name').textContent = requestData.teacherName;
            document.getElementById('modal-teacher-email').textContent = requestData.teacherEmail;
            document.getElementById('modal-teacher-id').textContent = `#${requestData.id}`;
            document.getElementById('modal-payment-method').textContent = requestData.paymentMethod;
            document.getElementById('modal-requested-date').textContent = `${requestData.requestedDate} at ${requestData.requestedTime}`;
            
            // Set status badge with proper styling
            const statusBadge = document.getElementById('modal-status-badge');
            statusBadge.textContent = requestData.status;
            statusBadge.className = `status-badge status-${requestData.status.toLowerCase()}`;
            
            // Set REAL payment information from teacher's bank details
            document.getElementById('modal-bank-name').textContent = requestData.bankName;
            document.getElementById('modal-account-number').textContent = requestData.accountNumber;
            document.getElementById('modal-account-holder').textContent = requestData.accountHolder;
            document.getElementById('modal-branch-code').textContent = requestData.branchCode;
            document.getElementById('modal-account-type').textContent = requestData.accountType;
            
            // Handle bank logo
            const bankLogo = document.getElementById('modal-bank-logo');
            const bankNameLower = requestData.bankName.toLowerCase();
            let logoFound = false;
            
            for (const [bankKey, logoPath] of Object.entries(bankLogos)) {
                if (bankNameLower.includes(bankKey)) {
                    bankLogo.src = logoPath;
                    bankLogo.style.display = 'inline-block';
                    logoFound = true;
                    break;
                }
            }
            
            if (!logoFound) {
                bankLogo.style.display = 'none';
            }
            
            // Handle payment details section
            if (requestData.paymentDetails && requestData.paymentDetails.trim() !== '') {
                document.getElementById('modal-payment-details').textContent = requestData.paymentDetails;
                document.getElementById('payment-details-section').style.display = 'block';
            } else {
                document.getElementById('payment-details-section').style.display = 'none';
            }
            
            // Handle teacher notes section
            if (requestData.notes && requestData.notes.trim() !== '') {
                document.getElementById('modal-teacher-notes').textContent = requestData.notes;
                document.getElementById('teacher-notes-section').style.display = 'block';
            } else {
                document.getElementById('teacher-notes-section').style.display = 'none';
            }
            
            // Handle processing information
            if (requestData.processedAt && requestData.processedAt.trim() !== '') {
                document.getElementById('modal-processed-info').textContent = requestData.processedAt;
                document.getElementById('modal-processed-by').textContent = requestData.processedBy || 'Admin';
            } else {
                document.getElementById('modal-processed-info').textContent = 'Not processed yet';
                document.getElementById('modal-processed-by').textContent = '--';
            }
            
            // Show admin notes if they exist
            if (requestData.adminNotes && requestData.adminNotes.trim() !== '') {
                document.getElementById('admin-notes-section').style.display = 'block';
                document.getElementById('modal-admin-notes').textContent = requestData.adminNotes;
            } else {
                document.getElementById('admin-notes-section').style.display = 'none';
            }
            
            // Add action buttons based on status
            const actionButtonsContainer = document.getElementById('modal-action-buttons');
            actionButtonsContainer.innerHTML = '';
            
            if (requestData.status.toLowerCase() === 'pending') {
                actionButtonsContainer.innerHTML = `
                    <button type="button" class="btn btn-modern btn-success me-2" onclick="updateStatus(${id}, 'approved')">
                        <i class='bx bx-check'></i> Approve
                    </button>
                    <button type="button" class="btn btn-modern btn-danger" onclick="updateStatus(${id}, 'rejected')">
                        <i class='bx bx-x'></i> Reject
                    </button>
                `;
            } else if (requestData.status.toLowerCase() === 'approved') {
                actionButtonsContainer.innerHTML = `
                    <button type="button" class="btn btn-modern btn-success" onclick="updateStatus(${id}, 'completed')">
                        <i class='bx bx-check-double'></i> Mark as Completed
                    </button>
                `;
            }
            
            // Show the modal
            const modal = new bootstrap.Modal(document.getElementById('requestDetailsModal'));
            modal.show();
        }

        function updateStatus(id, status) {
            if (confirm('Are you sure you want to ' + status + ' this request?')) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.innerHTML = `
                    <input type="hidden" name="update_status" value="1">
                    <input type="hidden" name="request_id" value="${id}">
                    <input type="hidden" name="status" value="${status}">
                    <input type="hidden" name="admin_notes" value="Status updated by admin">
                `;
                document.body.appendChild(form);
                form.submit();
            }
        }
    </script>
</body>
</html>
