<?php
/**
 * Dynamic Main Navbar Search API
 * Returns real-time search results for the main navbar dropdown
 */

// Prevent any output before JSON
error_reporting(0);
ini_set('display_errors', 0);

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST');
header('Access-Control-Allow-Headers: Content-Type');

// Include database connection - using same path structure as student API
require_once '../includes/config.php';
require_once '../includes/id_encryption.php';

try {
    // Check database connection
    if (!isset($conn) || !$conn) {
        echo json_encode([
            'success' => false,
            'error' => 'Database connection failed'
        ]);
        exit;
    }
    
    $query = isset($_GET['q']) ? trim($_GET['q']) : '';
    $category = isset($_GET['category']) ? trim($_GET['category']) : '';
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 8;
    
    // Basic validation - allow search by query OR category
    if (empty($query) && empty($category)) {
        echo json_encode([
            'success' => false,
            'error' => 'Search query or category required'
        ]);
        exit;
    }
    
    // Build search conditions
    $searchConditions = [];
    $params = [];
    $types = '';
    
    if (!empty($query)) {
        $searchConditions[] = "(c.title LIKE ? OR c.description LIKE ? OR cat.category_name LIKE ? OR CONCAT(u.first_name, ' ', u.last_name) LIKE ?)";
        $searchParam = "%$query%";
        $params = array_merge($params, [$searchParam, $searchParam, $searchParam, $searchParam]);
        $types .= 'ssss';
    }
    
    // Add category filtering
    if (!empty($category)) {
        $searchConditions[] = "cat.category_name = ?";
        $params[] = $category;
        $types .= 's';
    }
    
    // If no search conditions, create a default one to show all published courses
    if (empty($searchConditions)) {
        $searchConditions[] = "1=1";
    }
    
    $whereClause = implode(' AND ', $searchConditions);
    
    // Fixed search query with correct column names - matching student API structure
    $sql = "
        SELECT 
            c.course_id,
            c.title,
            c.description,
            c.price,
            c.thumbnail,
            c.created_at,
            IFNULL(cat.category_name, 'Uncategorized') as category_name,
            IFNULL(CONCAT(u.first_name, ' ', u.last_name), 'Unknown') as teacher_name,
            u.profile_picture as teacher_avatar
            
        FROM courses c
        LEFT JOIN categories cat ON c.category_id = cat.category_id
        LEFT JOIN users u ON c.teacher_id = u.user_id
        WHERE c.is_published = 1 AND $whereClause
        ORDER BY c.title ASC
        LIMIT ?
    ";
    
    // Add limit parameter
    $params[] = $limit;
    $types .= 'i';
    
    // Prepare and execute
    $stmt = $conn->prepare($sql);
    if (!$stmt) {
        echo json_encode([
            'success' => false,
            'error' => 'Query preparation failed'
        ]);
        exit;
    }
    
    if (!empty($params)) {
        $stmt->bind_param($types, ...$params);
    }
    
    $stmt->execute();
    $result = $stmt->get_result();
    
    $courses = [];
    while ($row = $result->fetch_assoc()) {
        // Format price
        $price = $row['price'] == 0 ? 'Free' : 'R' . number_format((float)$row['price'], 2);
        
        // Handle thumbnail - use exact path from database (same as student API)
        $thumbnail = !empty($row['thumbnail']) ? $row['thumbnail'] : '../assets/images/default-thumbnail.jpg';
        
        // Teacher avatar
        $teacherAvatar = !empty($row['teacher_avatar']) ? '/uploads/avatars/' . $row['teacher_avatar'] : null;
        
        // Clean description
        $description = !empty($row['description']) ? substr(strip_tags($row['description']), 0, 80) . '...' : '';
        
        $courses[] = [
            'id' => $row['course_id'],
            'title' => htmlspecialchars($row['title']),
            'description' => htmlspecialchars($description),
            'url' => 'https://panaditeacademy.com/course.php?id=' . IdEncryption::encrypt($row['course_id']),
            'thumbnail' => $thumbnail,
            'price' => $price,
            'category' => htmlspecialchars($row['category_name']),
            'teacher_name' => htmlspecialchars($row['teacher_name'])
        ];
    }
    
    echo json_encode([
        'success' => true,
        'data' => [
            'courses' => $courses,
            'total_results' => count($courses),
            'query' => $query,
            'category' => $category
        ]
    ]);

} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'error' => 'Search temporarily unavailable'
    ]);
}

exit;
?>
