<?php
// Enable error reporting for debugging live server issues
ini_set('display_errors', 0); // Don't display errors to users
ini_set('log_errors', 1);
error_log("Application.php started - Live server debug");

session_start();
require_once 'config/db_connect.php';
require_once 'includes/cms_content_loader.php';

// Initialize variables
$errors = [];
$success = false;
$name = $email = $phone = $address = $education = $grade_level = $message = '';
$username = $password = $confirm_password = $dob = $gender = $id_number = '';
$parent_name = $parent_phone = $parent_email = $emergency_contact = '';
$selected_subjects = [];

// Helper function to check if username exists
function isUsernameAvailable($conn, $username) {
    if (empty($username)) {
        error_log("isUsernameAvailable: Empty username provided");
        return false; // Empty usernames are not available
    }
    
    error_log("isUsernameAvailable: Checking username: " . $username);
    
    // Check both users and applications tables for username uniqueness
    $stmt = $conn->prepare("SELECT COUNT(*) as count FROM users WHERE username = ?");
    $stmt->bind_param("s", $username);
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    
    if ($result['count'] > 0) {
        error_log("isUsernameAvailable: Username exists in users table");
        return false; // Username exists in users table
    }
    
    // Also check applications table - FIX: Execute the query and get fresh result
    $stmt2 = $conn->prepare("SELECT COUNT(*) as count FROM applications WHERE username = ?");
    $stmt2->bind_param("s", $username);
    $stmt2->execute();
    $result2 = $stmt2->get_result()->fetch_assoc();
    
    $is_available = $result2['count'] == 0;
    error_log("isUsernameAvailable: Username available: " . ($is_available ? 'YES' : 'NO'));
    return $is_available;
}

// Handle Google OAuth registration
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['google_credential']) && isset($_POST['grade_level'])) {
    $credential = $_POST['google_credential'];
    $grade_level = $_POST['grade_level'];
    

    
    // Determine role based on grade level
    $role = ($grade_level === 'Matric Rewrite') ? 'matric' : 'student';
    $redirect_folder = ($grade_level === 'Matric Rewrite') ? 'matric' : 'student';
    

    
    // Decode the JWT token
    $parts = explode('.', $credential);

    if (count($parts) == 3) {
        $payload = json_decode(base64_decode(str_replace(['-', '_'], ['+', '/'], $parts[1])), true);

        
        if (isset($payload['email'])) {
            $email = $payload['email'];
            $name = $payload['name'] ?? '';
            $picture = $payload['picture'] ?? '';
            

            
            // Check if user already exists
            $sql = "SELECT id FROM users WHERE email = ?";
            $stmt = mysqli_prepare($conn, $sql);
            mysqli_stmt_bind_param($stmt, "s", $email);
            mysqli_stmt_execute($stmt);
            $result = mysqli_stmt_get_result($stmt);
            
            if (mysqli_num_rows($result) > 0) {
                $response = [
                    'success' => false, 
                    'message' => 'Account already exists with this email address.',
                    'user_exists' => true,
                    'email' => $email,
                    'name' => $name
                ];
            } else {

                try {

                    // Start transaction for data consistency
                    mysqli_begin_transaction($conn);
                    
                    // Generate student number (same format for both roles)
                    $year = date('y');
                    $sequence = str_pad(rand(100, 999), 3, '0', STR_PAD_LEFT);
                    $random = str_pad(rand(10, 99), 2, '0', STR_PAD_LEFT);
                    $student_number = "WTA{$year}{$sequence}{$random}";
                    $debug_output[] = "✅ Generated student number: " . $student_number;
                    
                    // Generate unique username from email
                    $base_username = strtok($email, '@');
                    $username = $base_username;
                    $counter = 1;
                    $debug_output[] = "✅ Base username: " . $base_username;
                    
                    // Ensure username is unique
                    while (!isUsernameAvailable($conn, $username)) {
                        $username = $base_username . '_' . $counter;
                        $counter++;
                        if ($counter > 100) {
                            throw new Exception("Could not generate unique username after 100 attempts");
                        }
                    }

                    
                    // Create user account in users table with all Google OAuth data
                    $default_password = password_hash(bin2hex(random_bytes(16)), PASSWORD_DEFAULT);
                    $insert_user_sql = "INSERT INTO users (student_number, grade_level, username, name, email, profile_image, password, role, status, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'active', NOW())";

                    
                    $stmt = $conn->prepare($insert_user_sql);
                    if (!$stmt) {
                        throw new Exception("Failed to prepare SQL statement: " . $conn->error);
                    }
                    
                    $stmt->bind_param("ssssssss", $student_number, $grade_level, $username, $name, $email, $picture, $default_password, $role);

                    
                    if ($stmt->execute()) {
                        $user_id = $conn->insert_id;

                        
                        // Skip applications table insert during OAuth registration
                        // The applications table has many NOT NULL fields (phone, dob, gender, id_number, parent info, etc.)
                        // that we don't have during OAuth. The user will complete their full application later.
                        // All essential OAuth data is stored in the users table.

                        
                        // Commit transaction
                        mysqli_commit($conn);

                        
                        // Set session for the new user
                        $_SESSION['user_id'] = $user_id;
                        $_SESSION['role'] = $role;
                        $_SESSION['email'] = $email;
                        $_SESSION['name'] = $name;
                        $_SESSION['student_number'] = $student_number;
                        $_SESSION['grade_level'] = $grade_level;
                        $_SESSION['username'] = $username;
                        $_SESSION['login_time'] = time();
                        $_SESSION['needs_completion'] = true;

                        
                        error_log("Google OAuth Registration successful for: " . $email . " (ID: " . $user_id . ", Role: " . $role . ", Grade: " . $grade_level . ", Student#: " . $student_number . ")");
                        error_log("OAuth redirect: " . $redirect_folder . '/complete-application.php');
                        

                        
                        $response = ['success' => true, 'message' => 'Registration successful! Completing your profile...', 'redirect' => $redirect_folder . '/complete-application.php'];
                    } else {
                        mysqli_rollback($conn);
                        error_log("User insert failed: " . $stmt->error);
                        $response = ['success' => false, 'message' => 'Registration failed. Please try again.'];
                    }
                    
                } catch (Exception $e) {
                    mysqli_rollback($conn);
                    error_log("Google registration error: " . $e->getMessage());
                    $response = ['success' => false, 'message' => 'Registration failed. Please try again later.'];
                }
            }
        } else {
            $response = ['success' => false, 'message' => 'Failed to get user information from Google.'];
        }
    } else {
        $response = ['success' => false, 'message' => 'Invalid Google credential format.'];
    }

    // Output JSON response
    header('Content-Type: application/json');
    echo json_encode($response);
    exit;
}

// Process form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Check if this is an application form submission
    $is_application_submit = isset($_POST['submit_application']) || 
                            isset($_POST['username']) || 
                            isset($_POST['name']) || 
                            isset($_POST['email']);
    
if ($is_application_submit) {
    
    // Validate basic info
    $name = trim($_POST['name'] ?? '');
    if (empty($name)) {
        $errors['name'] = 'Full name is required';
    }
    
    $email = trim($_POST['email'] ?? '');
    if (empty($email)) {
        $errors['email'] = 'Email is required';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors['email'] = 'Invalid email format';
    }
    
    $username = trim($_POST['username'] ?? '');
    if (empty($username)) {
        $errors['username'] = 'Username is required';
    } elseif (strlen($username) < 5) {
        $errors['username'] = 'Username must be at least 5 characters';
    } elseif (strlen($username) > 50) {
        $errors['username'] = 'Username must be 50 characters or less';
    } elseif (!preg_match('/^[a-zA-Z0-9_]+$/', $username)) {
        $errors['username'] = 'Username can only contain letters, numbers, and underscores';
    } elseif (!isUsernameAvailable($conn, $username)) {
        $errors['username'] = 'Username is already taken';
    }
    
    $password = trim($_POST['password'] ?? '');
    if (empty($password)) {
        $errors['password'] = 'Password is required';
    } elseif (strlen($password) < 8) {
        $errors['password'] = 'Password must be at least 8 characters';
    }
    
    $confirm_password = trim($_POST['confirm_password'] ?? '');
    if ($password !== $confirm_password) {
        $errors['confirm_password'] = 'Passwords do not match';
    }
    
    $phone = trim($_POST['phone'] ?? '');
    if (empty($phone)) {
        $errors['phone'] = 'Phone number is required';
    }
    
    $dob = trim($_POST['dob'] ?? '');
    if (empty($dob)) {
        $errors['dob'] = 'Date of birth is required';
    }
    
    $gender = trim($_POST['gender'] ?? '');
    if (empty($gender)) {
        $errors['gender'] = 'Gender is required';
    }
    
    $id_number = trim($_POST['id_number'] ?? '');
    if (empty($id_number)) {
        $errors['id_number'] = 'ID number is required';
    }
    
    $parent_name = trim($_POST['parent_name'] ?? '');
    if (empty($parent_name)) {
        $errors['parent_name'] = 'Parent name is required';
    }
    
    $parent_phone = trim($_POST['parent_phone'] ?? '');
    if (empty($parent_phone)) {
        $errors['parent_phone'] = 'Parent phone is required';
    }
    
    $parent_email = trim($_POST['parent_email'] ?? '');
    if (empty($parent_email)) {
        $errors['parent_email'] = 'Parent email is required';
    }
    
    $emergency_contact = trim($_POST['emergency_contact'] ?? '');
    if (empty($emergency_contact)) {
        $errors['emergency_contact'] = 'Emergency contact is required';
    }
    
    $grade_level = trim($_POST['grade_level'] ?? '');
    if (empty($grade_level)) {
        $errors['grade_level'] = 'Grade level is required';
    }
    
    // Get other optional fields
    $address = trim($_POST['address'] ?? '');
    $education = trim($_POST['education'] ?? '');
    $message = trim($_POST['message'] ?? '');
    
    // Get selected subjects
    $selected_subjects = $_POST['subjects'] ?? [];
    if (empty($selected_subjects)) {
        $errors['subjects'] = 'Please select at least one subject';
    }
    
    // If no errors, proceed with insertion
    if (empty($errors)) {
        try {
            // Generate student number
            $year = date('y');
            $sequence = str_pad(rand(100, 999), 3, '0', STR_PAD_LEFT);
            $random = str_pad(rand(10, 99), 2, '0', STR_PAD_LEFT);
            $student_number = "25{$year}{$sequence}{$random}";
            
            // Final username check before insertion
            if (empty($username)) {
                throw new Exception("Username cannot be empty");
            }
            
            // Hash the password
            $hashed_password = password_hash($password, PASSWORD_DEFAULT);
            
            // Determine role based on grade level - PERMANENT FIX FOR ROLE ASSIGNMENT
            $role = 'student'; // Default to student role for ALL grades
            
            // CRITICAL: Only "Matric Rewrite" students get the 'matric' role
            // Grade 12 students should ALWAYS get 'student' role, never 'matric'
            if ($grade_level === 'Matric Rewrite') {
                $role = 'matric';
                error_log("APPLICATION: Assigning 'matric' role to Matric Rewrite student");
            } else {
                $role = 'student';
                error_log("APPLICATION: Assigning 'student' role to $grade_level student");
            }
            
            // Double-check: Ensure Grade 12 students never get 'matric' role
            if ($grade_level === 'Grade 12' && $role !== 'student') {
                $role = 'student'; // Force correct role for Grade 12
                error_log("APPLICATION: CRITICAL - Fixed incorrect role assignment for Grade 12 student");
            }
            
            if ($stmt->execute()) {
                $user_id = $conn->insert_id;
                
                // Also insert into applications table for record keeping
                $app_stmt = $conn->prepare("INSERT INTO applications (student_number, username, name, email, phone, address, dob, gender, id_number, parent_name, parent_phone, parent_email, emergency_contact, education, grade_level, message, status, account_activated) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'approved', 1)");
                $app_stmt->bind_param("ssssssssssssssss", 
                    $student_number, $username, $name, $email, $phone, $address, 
                    $dob, $gender, $id_number, $parent_name, $parent_phone, 
                    $parent_email, $emergency_contact, $education, $grade_level, $message
                );
                $app_stmt->execute();
                
                // Get the application ID for subjects insertion
                $application_id = $conn->insert_id;
                
                // Insert selected subjects using the application_id
                $subject_stmt = $conn->prepare("INSERT INTO application_subjects (application_id, subject_name) VALUES (?, ?)");
                foreach ($selected_subjects as $subject) {
                    $subject_stmt->bind_param("is", $application_id, $subject);
                    $subject_stmt->execute();
                }
                
                $success = true;
                
                // Set session variables for automatic login
                $_SESSION['user_id'] = $user_id;
                $_SESSION['role'] = $role;
                $_SESSION['email'] = $email;
                $_SESSION['name'] = $name;
                $_SESSION['user_status'] = 'active';
                
                // Send application confirmation emails BEFORE clearing form data
                $subjects_list = implode(', ', $selected_subjects);
                $subjects_count = count($selected_subjects);
                
                // Get ACTUAL registration fee from database (not hardcoded)
                $registration_fee = 300; // Safe fallback
                try {
                    // Convert grade level to database format
                    $grade_mapping = [
                        '8' => 'Grade 8', '9' => 'Grade 9', '10' => 'Grade 10', 
                        '11' => 'Grade 11', '12' => 'Grade 12', 'Matric Rewrite' => 'Matric Rewrite'
                    ];
                    $db_grade = $grade_mapping[$grade_level] ?? $grade_level;
                    
                    // Determine student type
                    $student_type = (stripos($db_grade, 'matric') !== false || stripos($db_grade, 'rewrite') !== false) ? 'matric' : 'student';
                    
                    $fee_query = "SELECT registration_fee FROM grade_fees WHERE grade_level = ? AND student_type = ? AND status = 'active' LIMIT 1";
                    $fee_stmt = mysqli_prepare($conn, $fee_query);
                    mysqli_stmt_bind_param($fee_stmt, "ss", $db_grade, $student_type);
                    mysqli_stmt_execute($fee_stmt);
                    $fee_result = mysqli_stmt_get_result($fee_stmt);
                    
                    if ($fee_row = mysqli_fetch_assoc($fee_result)) {
                        $registration_fee = $fee_row['registration_fee'];
                        error_log("✅ Using database registration fee: R" . number_format($registration_fee) . " for " . $db_grade);
                    } else {
                        error_log("⚠️ No database fee found for " . $db_grade . ", using fallback: R300");
                    }
                } catch (Exception $e) {
                    error_log("❌ Registration fee lookup error: " . $e->getMessage() . ", using fallback: R300");
                }
                
                // Create role-specific messaging
                $role_title = ($role === 'matric') ? 'Matric Rewrite Student' : 'Student';
                $base_url = 'https://wisdomtutoringacademy.co.za';
                $dashboard_url = ($role === 'matric') ? $base_url . '/matric/dashboard.php' : $base_url . '/student/dashboard.php';
                $payment_url = ($role === 'matric') ? $base_url . '/matric/payment-flow.php' : $base_url . '/student/payment-flow.php';
                // Send confirmation emails
                try {
                    if (file_exists('config/EmailService.php')) {
                        require_once 'config/EmailService.php';
                        
                        // Initialize EmailService for both student and parent emails
                        if (isset($conn) && !mysqli_connect_errno()) {
                            $emailService = new EmailService($conn);
                            
                            // Prepare additional data for comprehensive email template
                            $additional_data = [
                                'subjects_count' => $subjects_count,
                                'selected_subjects' => $selected_subjects,
                                'registration_fee' => $registration_fee,
                                'role' => $role,
                                'role_title' => $role_title,
                                'dashboard_url' => $dashboard_url,
                                'payment_url' => $payment_url
                            ];
                            
                            // 1. Send application confirmation email to STUDENT
                            $email_result = $emailService->sendApplicationEmail(
                                $name,
                                $email,
                                $grade_level,
                                $student_number,
                                $additional_data
                            );
                            
                            if ($email_result['success']) {
                                error_log("✅ Application confirmation email sent to student: " . $email);
                            } else {
                                error_log("❌ Application confirmation email failed for student: " . ($email_result['message'] ?? 'Unknown error'));
                            }
                            
                            // 2. Send parent notification email if parent email provided
                            if (!empty($parent_email) && filter_var($parent_email, FILTER_VALIDATE_EMAIL)) {
                                $parent_email_result = $emailService->sendParentApplicationNotification(
                                    $parent_name,
                                    $parent_email,
                                    $name,
                                    $grade_level,
                                    $student_number,
                                    $subjects_list,
                                    $registration_fee,
                                    $role_title
                                );
                                
                                if ($parent_email_result['success']) {
                                    error_log("✅ Parent notification email sent to: " . $parent_email);
                                } else {
                                    error_log("❌ Parent notification email failed: " . ($parent_email_result['message'] ?? 'Unknown error'));
                                }
                            }
                        }
                    }
                } catch (Exception $e) {
                    // Don't break application if email fails - just log the error
                    error_log("Application email system failed (application unaffected): " . $e->getMessage());
                }
                
                // Success message - Clean and professional for users
                $success_message = "✅ Application submitted successfully! Your student number is: " . $student_number . ". Welcome to Wisdom Tutoring Academy!";
                
                // Clear form data AFTER emails are sent
                $name = $email = $phone = $address = $education = $grade_level = $message = '';
                $username = $password = $confirm_password = $dob = $gender = $id_number = '';
                $parent_name = $parent_phone = $parent_email = $emergency_contact = '';
                $selected_subjects = [];
                
                // Set redirect URL for JavaScript redirect after success message
                $redirect_url = '';
                if ($role === 'student') {
                    $redirect_url = $base_url . '/student/payment-flow.php';
                } elseif ($role === 'matric') {
                    $redirect_url = $base_url . '/matric/payment-flow.php';
                }
                
                // Don't redirect immediately - let success message display first
                
            } else {
                throw new Exception("Database insertion failed: " . $stmt->error);
            }
            
        } catch (Exception $e) {
            // 🔒 SECURITY FIX: Sanitize database errors to prevent information disclosure
            $raw_error = $e->getMessage();
            error_log("Application submission error (hidden from user): " . $raw_error);
            
            // Convert database errors to user-friendly messages
            if (strpos($raw_error, 'Duplicate entry') !== false) {
                if (strpos($raw_error, 'email') !== false) {
                    $errors['database'] = 'This email address is already registered. Please use a different email or try logging in.';
                } elseif (strpos($raw_error, 'username') !== false) {
                    $errors['database'] = 'This username is already taken. Please choose a different username.';
                } elseif (strpos($raw_error, 'id_number') !== false) {
                    $errors['database'] = 'This ID number is already registered. Please contact support if you believe this is an error.';
                } else {
                    $errors['database'] = 'Some of the information you provided is already registered. Please check your details and try again.';
                }
            } elseif (strpos($raw_error, 'foreign key constraint') !== false) {
                $errors['database'] = 'There was an issue with your application. Please contact support for assistance.';
            } elseif (strpos($raw_error, 'Data too long') !== false) {
                $errors['database'] = 'Some of the information you provided is too long. Please shorten your entries and try again.';
            } elseif (strpos($raw_error, 'cannot be null') !== false || strpos($raw_error, 'required') !== false) {
                $errors['database'] = 'Please fill in all required fields and try again.';
            } else {
                // Generic error for any other database issues
                $errors['database'] = 'We encountered an issue processing your application. Please try again or contact support.';
            }
        }
    }
}
}

// Get available subjects
$subjects_query = "SELECT name, price FROM subjects WHERE status = 'active' ORDER BY name";
$subjects_result = mysqli_query($conn, $subjects_query);
$subjects = [];
if ($subjects_result) {
    while ($row = mysqli_fetch_assoc($subjects_result)) {
        $subjects[] = $row;
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    
    <!-- SEO and Favicon Integration -->
    <?php include 'includes/seo_head.php'; ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://unpkg.com/aos@2.3.1/dist/aos.css" rel="stylesheet">
    
    <!-- Google Sign-In API -->
    <script src="https://accounts.google.com/gsi/client" async defer></script>
    <style>
        :root {
            --primary-color: #6B9BD1;
            --secondary-color: #4A90B8;
            --accent-color: #87CEEB;
            --success-color: #7FB3D3;
            --light-color: #F5F5F5;
            --background-color: #F8F8FF;
            --dark-color: #2C2C2C;
            --white-color: #FFFFFF;
            --gray-color: #555555;
            --gray-light-color: #F0F8FF;
            --gray-dark-color: #404040;
            --gradient-bg: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            --transition: all 0.3s ease;
        }
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: #f5f5f5;
            min-height: 100vh;
            padding-top: 100px;
        }
        
        /* Navbar Styling */
        .navbar {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            box-shadow: 0 2px 20px rgba(0, 0, 0, 0.1);
            transition: all 0.3s ease;
        }
        
        .navbar-brand img {
            transition: transform 0.3s ease;
        }
        
        .navbar-brand:hover img {
            transform: scale(1.05);
        }
        
        .nav-link {
            color: #333 !important;
            font-weight: 500;
            position: relative;
            transition: all 0.3s ease;
            margin: 0 10px;
        }
        
        .nav-link::after {
            content: '';
            position: absolute;
            bottom: -5px;
            left: 50%;
            width: 0;
            height: 2px;
            background: var(--primary-color);
            transition: all 0.3s ease;
            transform: translateX(-50%);
        }
        
        .nav-link:hover::after {
            width: 100%;
        }
        
        .nav-link.login-btn {
            background: var(--primary-color);
            color: white !important;
            padding: 8px 20px;
            border-radius: 25px;
            transition: all 0.3s ease;
        }
        
        .nav-link.login-btn:hover {
            background: var(--secondary-color);
            transform: translateY(-2px);
        }
        
        /* Page Header */
        .page-header {
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            padding: 8rem 0 4rem;
            position: relative;
            overflow: hidden;
        }
        
        .floating-elements {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            pointer-events: none;
        }
        
        .float-element {
            animation: float 6s ease-in-out infinite;
        }
        
        .float-element:nth-child(2) {
            animation-delay: -2s;
        }
        
        .float-element:nth-child(3) {
            animation-delay: -4s;
        }
        
        .float-element:nth-child(4) {
            animation-delay: -1s;
        }
        
        @keyframes float {
            0%, 100% { transform: translateY(0px); }
            50% { transform: translateY(-20px); }
        }
        
        /* Application Container */
        .application-container {
            background: white;
            border-radius: 25px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.1);
            padding: 10px;
            margin: 50px 0;
            position: relative;
            overflow: hidden;
        }
        
        .application-container::before {
            content: '';
            position: absolute;
            top: -50%;
            left: -50%;
            width: 200%;
            height: 200%;
            background: radial-gradient(circle, rgba(67, 97, 238, 0.05) 0%, transparent 70%);
            animation: pulse 4s ease-in-out infinite;
        }
        
        @keyframes pulse {
            0%, 100% { transform: scale(1); }
            50% { transform: scale(1.05); }
        }
        
        /* Form Sections */
        .form-section {
            margin-bottom: 40px;
            padding: 5px;
            border-radius: 15px;
            background: rgba(255, 255, 255, 0.7);
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 255, 255, 0.3);
            transition: all 0.3s ease;
        }
        
        .form-section:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 40px rgba(0, 0, 0, 0.1);
        }
        
        .form-section h4 {
            color: var(--primary-color);
            border-bottom: 3px solid var(--primary-color);
            padding-bottom: 15px;
            margin-bottom: 25px;
            position: relative;
            font-weight: 600;
        }
        
        .form-section h4::after {
            content: '';
            position: absolute;
            bottom: -3px;
            left: 0;
            width: 60px;
            height: 3px;
            background: var(--accent-color);
            border-radius: 2px;
        }
        
        /* Form Controls */
        .form-control, .form-select {
            border: 2px solid #e9ecef;
            border-radius: 12px;
            padding: 15px 20px;
            font-size: 16px;
            transition: all 0.3s ease;
            background: rgba(255, 255, 255, 0.9);
        }
        
        .form-control:focus, .form-select:focus {
            border-color: var(--primary-color);
            box-shadow: 0 0 0 0.2rem rgba(67, 97, 238, 0.25);
            transform: translateY(-2px);
        }
        
        .form-label {
            font-weight: 600;
            color: #333;
            margin-bottom: 10px;
        }
        
        /* Subject Selection */
        .subject-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin-top: 20px;
        }
        
        .subject-card {
            background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
            border: 2px solid #e9ecef;
            border-radius: 15px;
            padding: 20px;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            cursor: pointer;
            position: relative;
            overflow: hidden;
        }
        
        .subject-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(67, 97, 238, 0.1), transparent);
            transition: left 0.5s;
        }
        
        .subject-card:hover::before {
            left: 100%;
        }
        
        .subject-card:hover {
            transform: translateY(-8px);
            box-shadow: 0 20px 40px rgba(67, 97, 238, 0.2);
            border-color: var(--primary-color);
        }
        
        .subject-card.selected {
            border-color: var(--primary-color);
            background: linear-gradient(135deg, #4361ee 0%, #3f37c9 100%);
            color: white;
            transform: translateY(-5px);
        }
        
        .subject-name {
            font-size: 18px;
            font-weight: 600;
            margin-bottom: 10px;
        }
        
        .subject-price {
            font-size: 24px;
            font-weight: 700;
            color: var(--accent-color);
        }
        
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
            border: none;
            padding: 15px 40px;
            font-size: 18px;
            font-weight: 600;
            border-radius: 25px;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }
        
        .btn-primary::before {
            content: '';
            position: absolute;
            top: 50%;
            left: 50%;
            width: 0;
            height: 0;
            background: rgba(255, 255, 255, 0.3);
            border-radius: 50%;
            transition: all 0.6s ease;
            transform: translate(-50%, -50%);
        }
        
        .btn-primary:hover::before {
            width: 300px;
            height: 300px;
        }
        
        .btn-primary:hover {
            transform: translateY(-3px);
            box-shadow: 0 10px 30px rgba(67, 97, 238, 0.4);
        }
        
        /* Alerts */
        .alert {
            border: none;
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 30px;
            animation: slideInDown 0.5s ease;
        }
        
        @keyframes slideInDown {
            from {
                transform: translateY(-30px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        .alert-success {
            background: linear-gradient(135deg, #d4edda 0%, #c3e6cb 100%);
            color: #155724;
            border-left: 5px solid #28a745;
        }
        
        .alert-danger {
            background: linear-gradient(135deg, #f8d7da 0%, #f5c6cb 100%);
            color: #721c24;
            border-left: 5px solid #dc3545;
        }
        
        /* Fee Display Card */
        .fee-display-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 2rem;
            border-radius: 15px;
            box-shadow: 0 10px 30px rgba(102, 126, 234, 0.3);
            margin-bottom: 1rem;
            position: relative;
            overflow: hidden;
        }
        
        .fee-display-card::before {
            content: '';
            position: absolute;
            top: -50%;
            right: -50%;
            width: 100%;
            height: 100%;
            background: radial-gradient(circle, rgba(255,255,255,0.1) 0%, transparent 70%);
            transform: rotate(45deg);
        }
        
        .fee-amount {
            font-size: 2rem;
            font-weight: bold;
            color: #ffd700;
            text-shadow: 0 2px 4px rgba(0,0,0,0.3);
        }
        
        .monthly-tuition {
            font-size: 0.9rem;
            color: rgba(255,255,255,0.8);
            margin-top: 0.5rem;
        }
        
        .fee-loading {
            color: #ffd700;
            font-size: 1.2rem;
        }
        
        /* Cost Calculator Card */
        .cost-calculator-card {
            background: linear-gradient(135deg, #56ab2f 0%, #a8e6cf 100%);
            color: white;
            padding: 1.5rem;
            border-radius: 12px;
            box-shadow: 0 8px 25px rgba(86, 171, 47, 0.3);
        }
        
        .cost-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 0.8rem;
            padding: 0.5rem 0;
            border-bottom: 1px solid rgba(255,255,255,0.2);
        }
        
        .cost-item.total {
            border-bottom: none;
            border-top: 2px solid rgba(255,255,255,0.4);
            margin-top: 1rem;
            padding-top: 1rem;
            font-size: 1.1rem;
        }
        
        .cost-value {
            font-weight: bold;
            color: #ffd700;
        }
        
        .cost-value.total {
            font-size: 1.3rem;
            color: #fff;
            text-shadow: 0 2px 4px rgba(0,0,0,0.3);
        }
        
        .subject-price {
            font-weight: 600;
            color: var(--primary-color);
        }
        
        /* Subject Grid Updates */
        .subject-grid.disabled {
            opacity: 0.5;
            pointer-events: none;
            transition: all 0.3s ease;
        }
        
        .subject-grid.enabled {
            opacity: 1;
            pointer-events: all;
            transition: all 0.3s ease;
        }
        
        /* Loading Animation */
        .loading {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(255, 255, 255, 0.9);
            z-index: 9999;
            align-items: center;
            justify-content: center;
        }
        
        .spinner {
            width: 50px;
            height: 50px;
            border: 5px solid #f3f3f3;
            border-top: 5px solid var(--primary-color);
            border-radius: 50%;
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        /* Footer Styling */
        .footer {
            background-color: var(--dark-color);
            color: white;
            padding: 6rem 0 2rem;
            position: relative;
        }

        .footer h5 {
            color: white;
            font-weight: 600;
            margin-bottom: 1.5rem;
            font-size: 1.2rem;
        }

        .footer ul li {
            margin-bottom: 0.75rem;
        }

        .footer ul li a {
            color: rgba(255, 255, 255, 0.7);
            transition: var(--transition);
        }

        .footer ul li a:hover {
            color: var(--accent-color);
            padding-left: 5px;
        }

        .footer .social-icons {
            margin-top: 1.5rem;
        }

        .footer .social-icons a {
            color: white;
            background: linear-gradient(45deg, var(--primary-color), var(--secondary-color));
            width: 40px;
            height: 40px;
            border-radius: 50%;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            margin-right: 0.8rem;
            font-size: 1rem;
            transition: var(--transition);
        }

        .footer .social-icons a:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 15px rgba(0, 0, 0, 0.2);
        }

        .footer hr {
            border-color: rgba(255, 255, 255, 0.1);
            margin: 2rem 0;
        }

        .footer .contact-info li {
            display: flex;
            align-items: flex-start;
            margin-bottom: 1rem;
        }

        .footer .contact-info li i {
            margin-right: 0.75rem;
            margin-top: 0.2rem;
            color: var(--accent-color);
        }
        
        /* Google Sign-In Styles */
        .google-signin-container {
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            border: 2px solid #dee2e6;
            border-radius: 15px;
            padding: 1.5rem;
            margin: 0 auto 2rem auto;
            max-width: 500px;
            text-align: center;
        }
        
        .google-signin-title {
            color: #495057;
            font-weight: 600;
            margin-bottom: 1rem;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.5rem;
        }
        
        .google-signin-subtitle {
            color: #6c757d;
            font-size: 0.9rem;
            margin-bottom: 1.5rem;
        }
        
        .divider {
            position: relative;
            text-align: center;
            margin: 2rem 0;
        }
        
        .divider::before {
            content: '';
            position: absolute;
            top: 50%;
            left: 0;
            right: 0;
            height: 1px;
            background: #dee2e6;
        }
        
        .divider span {
            background: white;
            padding: 0 1rem;
            color: #6c757d;
            font-weight: 500;
        }
        
    </style>
</head>
<body>
    <!-- Loading Screen -->
    <div class="loading" id="loadingScreen">
        <div class="spinner"></div>
    </div>

    <!-- Modern Navigation Bar -->
    <nav class="navbar navbar-expand-lg navbar-light fixed-top">
        <div class="container">
            <a class="navbar-brand" href="index.php" data-aos="fade-right">
                <img src="assets/images/logo.jpeg" alt="Wisdom Tutoring Academy Logo" height="60">
            </a>
            <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav" aria-controls="navbarNav" aria-expanded="false" aria-label="Toggle navigation" style="border: 2px solid #4361ee;">
                <i class="fas fa-bars" style="color: #4361ee; font-size: 1.5rem;"></i>
            </button>
            <div class="collapse navbar-collapse" id="navbarNav">
                <ul class="navbar-nav ms-auto">
                    <li class="nav-item" data-aos="fade-down" data-aos-delay="100">
                        <a class="nav-link" href="index.php">Home</a>
                    </li>
                    <li class="nav-item" data-aos="fade-down" data-aos-delay="200">
                        <a class="nav-link" href="about.php">About Us</a>
                    </li>
                    <li class="nav-item" data-aos="fade-down" data-aos-delay="300">
                        <a class="nav-link" href="courses.php">Courses</a>
                    </li>
                    <li class="nav-item" data-aos="fade-down" data-aos-delay="350">
                        <a class="nav-link" href="portfolio.php">Portfolio</a>
                    </li>
                    <li class="nav-item" data-aos="fade-down" data-aos-delay="400">
                        <a class="nav-link active" href="application.php">Apply Now</a>
                    </li>
                    <li class="nav-item" data-aos="fade-down" data-aos-delay="500">
                        <a class="nav-link" href="contact.php">Contact</a>
                    </li>
                    <li class="nav-item" data-aos="fade-down" data-aos-delay="600">
                        <a class="nav-link login-btn" href="login.php">Login</a>
                    </li>
                </ul>
            </div>
        </div>
    </nav>

    <!-- Clean Application Section -->
    <section class="py-5">
        <div class="container">
            <div class="row justify-content-center">
                <div class="col-md-8 col-lg-6 col-xl-5" data-aos="fade-up" data-aos-duration="1000">
                    <div class="card shadow-lg border-0 rounded-4 overflow-hidden" style="border-radius: 15px; transition: all 0.3s ease;" onmouseover="this.style.transform='translateY(-5px)';" onmouseout="this.style.transform='translateY(0)';">
                        <div class="card-header py-5 border-0 d-flex flex-column align-items-center justify-content-center" style="background: linear-gradient(135deg, #FDF7E1, #F7E7A7); color: #8B6914;">
                            <!-- Logo -->
                            <img src="assets/images/logo.jpeg" alt="Wisdom Tutoring Academy" class="rounded-circle mb-3" width="80" height="80" style="border: 3px solid #E6B800; box-shadow: 0 4px 12px rgba(230, 184, 0, 0.2); object-fit: cover;">
                            
                            <h4 class="fw-bold mb-2">Join Our Academy</h4>
                            <p class="mb-0 text-center">Start your journey to academic excellence</p>
                        </div>
                        <div class="card-body py-5 px-4 px-md-5">



                    <?php if ($success): ?>
                        <div class="alert alert-success" id="successMessage">
                            <i class="fas fa-check-circle me-2"></i>
                            <strong>🎉 Welcome to Wisdom Tutoring Academy!</strong><br>
                            <?php echo $success_message; ?>
                            <div class="mt-3">
                                <div class="progress mb-3">
                                    <div class="progress-bar progress-bar-striped progress-bar-animated bg-warning" 
                                         role="progressbar" style="width: 0%" id="redirectProgress"></div>
                                </div>
                                <p class="text-center mb-2">
                                    <i class="fas fa-clock me-1"></i>
                                    Redirecting to payment process in <span id="countdown">15</span> seconds...
                                </p>
                                <a href="<?php echo $redirect_url; ?>" class="btn btn-warning btn-lg" id="manualRedirect">
                                    <i class="fas fa-route me-2"></i>Continue to Payment Process
                                </a>
                            </div>
                            <p class="text-muted mt-2"><small>You need to complete payment to access your dashboard and course materials.</small></p>
                        </div>
                        
                        <script>
                        document.addEventListener('DOMContentLoaded', function() {
                            let countdown = 15;
                            const countdownElement = document.getElementById('countdown');
                            const progressBar = document.getElementById('redirectProgress');
                            const redirectUrl = '<?php echo $redirect_url; ?>';
                            
                            // Update progress bar and countdown
                            const timer = setInterval(function() {
                                countdown--;
                                countdownElement.textContent = countdown;
                                
                                // Update progress bar (100% - (countdown/15 * 100))
                                const progress = ((15 - countdown) / 15) * 100;
                                progressBar.style.width = progress + '%';
                                
                                if (countdown <= 0) {
                                    clearInterval(timer);
                                    // Redirect to payment flow
                                    window.location.href = redirectUrl;
                                }
                            }, 1000);
                            
                            // Smooth scroll to success message
                            document.getElementById('successMessage').scrollIntoView({ 
                                behavior: 'smooth', 
                                block: 'center' 
                            });
                        });
                        </script>
                    <?php endif; ?>

                    <?php if (!empty($errors['database'])): ?>
                        <div class="alert alert-danger">
                            <i class="fas fa-exclamation-circle me-2"></i>
                            <?php echo htmlspecialchars($errors['database']); ?>
                        </div>
                    <?php endif; ?>

                    <!-- Step 1: Grade Level Selection -->
                    <div id="gradeSelectionStep" class="mb-5" data-aos="fade-up" data-aos-delay="200">
                        <div class="text-center mb-4">
                            <h5 class="fw-bold text-primary mb-3">
                                <i class="fas fa-graduation-cap me-2"></i>
                                Step 1: Choose Your Grade Level
                            </h5>
                            <p class="text-muted">Select your current grade level to customize your registration experience</p>
                        </div>
                        
                        <div class="row justify-content-center">
                            <div class="col-md-6">
                                <select id="gradeSelector" class="form-select form-select-lg text-center" style="border-radius: 15px; border: 2px solid #e9ecef; padding: 1rem;">
                                    <option value="">Choose Your Grade Level</option>
                                    <option value="Grade 8">Grade 8</option>
                                    <option value="Grade 9">Grade 9</option>
                                    <option value="Grade 10">Grade 10</option>
                                    <option value="Grade 11">Grade 11</option>
                                    <option value="Grade 12">Grade 12</option>
                                    <option value="Matric Rewrite">Matric Rewrite</option>
                                </select>
                                
                                <!-- Grade Info Display -->
                                <div id="gradeInfo" class="mt-3 p-3 rounded-3 border-0" style="background: linear-gradient(135deg, #f8f9fa, #e9ecef); display: none;">
                                    <div class="d-flex align-items-center justify-content-center">
                                        <i class="fas fa-info-circle text-primary me-2"></i>
                                        <span id="gradeInfoText" class="text-muted"></span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Step 2: Google OAuth Registration Section (Hidden Initially) -->
                    <div id="googleOAuthStep" class="text-center mb-4" style="display: none;" data-aos="fade-up" data-aos-delay="200">
                        <div class="mb-4">
                            <h5 class="fw-bold text-primary mb-3">
                                <i class="fab fa-google me-2"></i>
                                Step 2: Create Your Account
                            </h5>
                            <p class="text-muted">Sign up with Google to create your account instantly</p>
                        </div>
                        
                        <!-- Google Sign-Up Button -->
                        <div class="d-flex justify-content-center mb-4">
                            <div id="g_id_onload"
                                 data-client_id="491045592970-idsrumvp3bn07s8hfti97qoncfkd8ush.apps.googleusercontent.com"
                                 data-callback="handleCredentialResponse"
                                 data-auto_prompt="false"
                                 data-cancel_on_tap_outside="false">
                            </div>
                            <div class="g_id_signin" 
                                 data-type="standard" 
                                 data-size="large" 
                                 data-theme="outline" 
                                 data-text="signup_with" 
                                 data-shape="rectangular" 
                                 data-logo_alignment="left"
                                 data-width="320"
                                 data-height="50">
                            </div>
                        </div>
                        
                        <!-- Custom styling to make button bigger -->
                        <style>
                        .g_id_signin {
                            transform: scale(1.2) !important;
                            margin: 10px !important;
                        }
                        .g_id_signin iframe {
                            min-width: 320px !important;
                            min-height: 50px !important;
                        }
                        </style>
                        
                        <!-- Security Notice -->
                        <div class="alert border-0 mb-4" role="alert" style="background: linear-gradient(135deg, #E8F4FD, #D6E9FA); color: #4A90B8; border-radius: 12px;">
                            <div class="d-flex align-items-center justify-content-center small">
                                <i class="fas fa-shield-alt me-2" style="color: #6B9BD1;"></i>
                                <span>Secure registration powered by Google Identity Services</span>
                            </div>
                        </div>
                        
                        <!-- Process Flow Cards -->
                        <div class="row text-center" data-aos="fade-up" data-aos-delay="500">
                            <div class="col-md-4 mb-3">
                                <div class="py-3">
                                    <div class="badge bg-success rounded-pill mb-2">✓ Step 1</div>
                                    <h6 class="fw-medium">Grade Selected</h6>
                                    <small class="text-muted" id="selectedGradeDisplay">Grade level chosen</small>
                                </div>
                            </div>
                            <div class="col-md-4 mb-3">
                                <div class="py-3">
                                    <div class="badge bg-primary rounded-pill mb-2">2</div>
                                    <h6 class="fw-medium">Create Account</h6>
                                    <small class="text-muted">Sign up with Google</small>
                                </div>
                            </div>
                            <div class="col-md-4 mb-3">
                                <div class="py-3">
                                    <div class="badge bg-secondary rounded-pill mb-2">3</div>
                                    <h6 class="fw-medium">Complete Profile</h6>
                                    <small class="text-muted">Guided form completion</small>
                                </div>
                            </div>
                        </div>
                    </div>

                            <!-- Clean ending section -->
                            <div class="text-center mt-5" data-aos="fade-up" data-aos-delay="600">
                                <div class="alert alert-info border-0" style="background: linear-gradient(135deg, #FFF9E6, #FFF4D1); color: #8B6914; border-radius: 12px;">
                                    <div class="d-flex align-items-center justify-content-center">
                                        <i class="fas fa-lightbulb me-2" style="color: #E6B800;"></i>
                                        <span>After signing up, you'll complete your profile in easy steps and choose your subjects</span>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Security Features Display -->
                            <div class="row text-center" data-aos="fade-up" data-aos-delay="300">
                                <div class="col-md-4 mb-3">
                                    <div class="py-3">
                                        <i class="fas fa-lock text-success fs-4 mb-2"></i>
                                        <h6 class="fw-medium">Secure</h6>
                                        <small class="text-muted">Protected by Google</small>
                                    </div>
                                </div>
                                <div class="col-md-4 mb-3">
                                    <div class="py-3">
                                        <i class="fas fa-bolt text-warning fs-4 mb-2"></i>
                                        <h6 class="fw-medium">Fast</h6>
                                        <small class="text-muted">One-click access</small>
                                    </div>
                                </div>
                                <div class="col-md-4 mb-3">
                                    <div class="py-3">
                                        <i class="fas fa-user-check text-primary fs-4 mb-2"></i>
                                        <h6 class="fw-medium">Verified</h6>
                                        <small class="text-muted">Registered accounts only</small>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="text-center mt-4 small text-muted" data-aos="fade-up" data-aos-delay="600">
                        <p>
                            Already have an account? 
                            <a href="login.php" class="text-decoration-none fw-medium" style="color: #6B9BD1;">
                                <i class="fas fa-sign-in-alt me-1"></i>Apply for Admission
                            </a>
                        </p>
                    </div>
                    <!-- Add spacing for better layout -->
                    <div class="py-4"></div>
                </div>
            </div>
        </div>
    </section>


    <!-- Footer -->
    <footer class="footer py-4">
        <div class="container">
            <div class="row align-items-center">
                <div class="col-md-6">
                    <div class="d-flex align-items-center">
                        <img src="assets/images/logo.jpeg" alt="Wisdom Tutoring Academy" class="rounded-circle me-3" width="40" height="40" style="object-fit: cover;">
                        <div>
                            <h6 class="mb-0 fw-bold">Wisdom Tutoring Academy</h6>
                            <small class="text-muted">Excellence in Education</small>
                        </div>
                    </div>
                </div>
                <div class="col-md-6 text-md-end mt-3 mt-md-0">
                    <div class="social-links">
                        <a href="#" class="text-muted me-3"><i class="fab fa-facebook-f"></i></a>
                        <a href="#" class="text-muted me-3"><i class="fab fa-twitter"></i></a>
                        <a href="#" class="text-muted me-3"><i class="fab fa-instagram"></i></a>
                        <a href="#" class="text-muted"><i class="fab fa-linkedin-in"></i></a>
                    </div>
                </div>
            </div>
            <hr class="my-4">
            <div class="row align-items-center">
                <div class="col-md-6">
                    <p class="text-muted small mb-0">&copy; 2024 Wisdom Tutoring Academy. All rights reserved.</p>
                </div>
                <div class="col-md-6 text-md-end mt-2 mt-md-0">
                    <a href="#" class="text-muted text-decoration-none small me-3">Privacy Policy</a>
                    <a href="#" class="text-muted text-decoration-none small me-3">Terms of Service</a>
                    <a href="#" class="text-muted text-decoration-none small">Contact</a>
                </div>
            </div>
        </div>
    </footer>

    <!-- Google Identity Services -->
    <script src="https://accounts.google.com/gsi/client" async defer></script>
    
    <!-- JavaScript Libraries -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/toastr.js/latest/toastr.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/aos/2.3.4/aos.js"></script>
    
    <!-- CMS JavaScript for inline editing -->
    <script src="assets/js/main.js"></script>
    <script src="assets/js/modern-scripts.js"></script>
    <script src="assets/js/cms-inline-editor.js"></script>

    <script>
        // Initialize AOS
        AOS.init({
            duration: 1000,
            once: true
        });

        // Hide loading screen
        window.addEventListener('load', function() {
            document.getElementById('loadingScreen').style.display = 'none';
        });

        // Grade selection handler
        document.addEventListener('DOMContentLoaded', function() {
            const gradeSelector = document.getElementById('gradeSelector');
            const gradeInfo = document.getElementById('gradeInfo');
            const gradeInfoText = document.getElementById('gradeInfoText');
            const googleOAuthStep = document.getElementById('googleOAuthStep');
            const selectedGradeDisplay = document.getElementById('selectedGradeDisplay');
            
            // Grade selection change handler
            gradeSelector.addEventListener('change', function() {
                const selectedGrade = this.value;
                
                if (selectedGrade) {
                    // Show grade info
                    const gradeMessages = {
                        'Grade 8': 'You will be enrolled as a Grade 8 student with access to student courses',
                        'Grade 9': 'You will be enrolled as a Grade 9 student with access to student courses', 
                        'Grade 10': 'You will be enrolled as a Grade 10 student with access to student courses',
                        'Grade 11': 'You will be enrolled as a Grade 11 student with access to student courses',
                        'Grade 12': 'You will be enrolled as a Grade 12 student with access to student courses',
                        'Matric Rewrite': 'You will be enrolled as a Matric Rewrite student with specialized matric courses'
                    };
                    
                    gradeInfoText.textContent = gradeMessages[selectedGrade] || 'Grade level selected';
                    gradeInfo.style.display = 'block';
                    
                    // Update selected grade display
                    selectedGradeDisplay.textContent = selectedGrade;
                    
                    // Show Google OAuth step with animation
                    setTimeout(() => {
                        googleOAuthStep.style.display = 'block';
                        googleOAuthStep.scrollIntoView({ behavior: 'smooth', block: 'start' });
                    }, 500);
                    
                    // Store selected grade for OAuth process
                    window.selectedGradeLevel = selectedGrade;
                } else {
                    // Hide elements if no grade selected
                    gradeInfo.style.display = 'none';
                    googleOAuthStep.style.display = 'none';
                    window.selectedGradeLevel = null;
                }
            });
        });

        // Google Sign-In callback function for registration
        function handleCredentialResponse(response) {
            
            // Get the selected grade level
            const selectedGrade = window.selectedGradeLevel;
            if (!selectedGrade) {
                toastr.error('Please select your grade level first.', 'Grade Level Required');
                return;
            }
            
            // Show loading state
            const loadingBtn = document.createElement('button');
            loadingBtn.className = 'btn btn-primary btn-lg w-100 d-flex align-items-center justify-content-center';
            loadingBtn.disabled = true;
            loadingBtn.innerHTML = `
                <div class="spinner-border spinner-border-sm me-2" role="status" aria-hidden="true"></div>
                Creating your account...
            `;
            
            const googleOAuthStep = document.getElementById('googleOAuthStep');
            const signInBtn = googleOAuthStep.querySelector('.g_id_signin');
            if (signInBtn) {
                signInBtn.style.display = 'none';
                googleOAuthStep.appendChild(loadingBtn);
            }
            

            
            // Send credentials to server
            fetch('application.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `google_credential=${encodeURIComponent(response.credential)}&grade_level=${encodeURIComponent(selectedGrade)}`
            })
            .then(response => response.json())
            .then(data => {
                
                if (data.success) {
                    toastr.success(data.message, 'Registration Successful!');
                    
                    // Show success message and redirect
                    setTimeout(() => {
                        window.location.href = data.redirect;
                    }, 2000);
                } else {
                    
                    // Special handling for existing users
                    if (data.user_exists) {
                        // Create beautiful existing user notification
                        const existingUserModal = document.createElement('div');
                        existingUserModal.innerHTML = `
                            <div class="modal fade show" style="display: block; background: rgba(0,0,0,0.4); z-index: 10000;" id="existingUserModal">
                                <div class="modal-dialog modal-dialog-centered modal-sm">
                                    <div class="modal-content" style="border-radius: 12px; border: none; box-shadow: none; max-width: 420px; background: #fef7e0;">
                                        <div class="border-0 py-3 position-relative" style="background: #fef7e0 !important; border-radius: 12px 12px 0 0;">
                                            <!-- Profile Picture in top right -->
                                            <div class="position-absolute" style="top: 10px; right: 15px;">
                                                <img src="${data.picture || ''}" alt="Profile" class="rounded-circle" style="width: 35px; height: 35px; border: 2px solid #000000;" onerror="this.style.display='none'">
                                            </div>
                                            <h6 class="d-flex align-items-center w-100 justify-content-center mb-0" style="color: #000000 !important; font-weight: 600;">
                                                <i class="fas fa-user-check me-2" style="color: #000000 !important;"></i>
                                                Account Already Exists
                                            </h6>
                                        </div>
                                        <div class="modal-body px-4 py-3 text-center" style="background: #fef7e0;">
                                            <h6 class="fw-bold mb-2" style="color: #000000; font-size: 0.95rem;">Welcome back, ${data.name || 'User'}!</h6>
                                            <p class="mb-3" style="color: #000000; font-size: 0.85rem; line-height: 1.4;">
                                                An account with <strong style="color: #000000;">${data.email}</strong> already exists.
                                            </p>
                                            <div class="alert border-0 py-2 px-3 mb-0" style="background: #fdf4d3; color: #000000; font-size: 0.8rem;">
                                                <i class="fas fa-info-circle me-1"></i>
                                                <strong>Good news!</strong> You can login directly.
                                            </div>
                                        </div>
                                        <div class="modal-footer border-0 pt-0 pb-3 justify-content-center" style="background: #fef7e0;">
                                            <button type="button" class="btn btn-outline-dark btn-sm me-2" style="font-size: 0.8rem; padding: 0.4rem 0.8rem; background: transparent; border-color: #000000; color: #000000;" onclick="closeExistingUserModal()">
                                                <i class="fas fa-arrow-left me-1"></i>
                                                Go Back
                                            </button>
                                            <a href="login.php" class="btn btn-sm" style="background: #000000; border-color: #000000; color: #fef7e0; font-size: 0.8rem; padding: 0.4rem 1rem;">
                                                <i class="fas fa-sign-in-alt me-1"></i>
                                                Login Instead
                                            </a>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        `;
                        
                        document.body.appendChild(existingUserModal);
                        
                        // Add global close function
                        window.closeExistingUserModal = function() {
                            document.body.removeChild(existingUserModal);
                            // Reset the form
                            if (signInBtn) {
                                signInBtn.style.display = 'block';
                                loadingBtn.remove();
                            }
                        };
                        
                        // Close on backdrop click
                        existingUserModal.addEventListener('click', function(e) {
                            if (e.target === existingUserModal) {
                                window.closeExistingUserModal();
                            }
                        });
                        
                    } else {
                        // Regular error notification
                        toastr.error(data.message, 'Registration Failed');
                        
                        // Reset the form
                        if (signInBtn) {
                            signInBtn.style.display = 'block';
                            loadingBtn.remove();
                        }
                    }
                }
            })
            .catch(error => {
                console.error('Registration error:', error);
                toastr.error('Registration failed. Please try again.', 'Error');
                
                // Reset the form
                if (signInBtn) {
                    signInBtn.style.display = 'block';
                    loadingBtn.remove();
                }
            });
        }

        // Grade Selection Handler - Show Google OAuth step when grade is selected
        document.addEventListener('DOMContentLoaded', function() {
            const gradeSelector = document.getElementById('gradeSelector');
            const googleOAuthStep = document.getElementById('googleOAuthStep');
            const gradeInfo = document.getElementById('gradeInfo');
            const gradeInfoText = document.getElementById('gradeInfoText');
            
            // Handle grade selection change
            if (gradeSelector) {
                gradeSelector.addEventListener('change', function() {
                    const selectedGrade = this.value;
                    window.selectedGradeLevel = selectedGrade; // Store globally for OAuth handler
                    
                    if (selectedGrade) {
                        // Show grade info
                        if (gradeInfo && gradeInfoText) {
                            gradeInfoText.textContent = `Selected: ${selectedGrade}`;
                            gradeInfo.style.display = 'block';
                        }
                        
                        // Show Google OAuth step
                        if (googleOAuthStep) {
                            googleOAuthStep.style.display = 'block';
                            
                            // Smooth scroll to OAuth step
                            setTimeout(() => {
                                googleOAuthStep.scrollIntoView({ 
                                    behavior: 'smooth', 
                                    block: 'center' 
                                });
                            }, 100);
                        }
                    } else {
                        // Hide steps if no grade selected
                        if (gradeInfo) gradeInfo.style.display = 'none';
                        if (googleOAuthStep) googleOAuthStep.style.display = 'none';
                        window.selectedGradeLevel = null;
                    }
                });
            }
            
            // Legacy form elements (for the manual form if it exists)
            const gradeSelect = document.getElementById('grade_level');
            const btnPrice = document.getElementById('btnPrice');
            const submitBtn = document.querySelector('.submit-btn');
            const form = document.querySelector('form');
            const registrationFeeDisplay = document.getElementById('registrationFeeDisplay');
            const monthlyTuitionDisplay = document.getElementById('monthlyTuitionDisplay');
            const subjectGrid = document.getElementById('subjectGrid');
            
            let currentRegistrationFee = 0;
            let currentGradeLevel = '';
            let currentMonthlyTuition = 0;

            // Update course duration options based on grade level
            function updateCourseDurationOptions(gradeLevel) {
                const courseDurationSelect = document.getElementById('courseDuration');
                
                // Clear existing options and show placeholder
                courseDurationSelect.innerHTML = '<option value="" disabled selected>Select subjects first to see payment plan</option>';
                courseDurationSelect.disabled = true;
                courseDurationSelect.style.backgroundColor = '#f8f9fa';
                courseDurationSelect.style.cursor = 'not-allowed';
                
                // Add helper text to explain the automatic payment plan system
                const helpText = document.getElementById('paymentPlanHelpText') || createPaymentPlanHelpText();
                helpText.innerHTML = `
                    <div class="alert alert-light border">
                        <i class="fas fa-info-circle text-primary me-2"></i>
                        <strong>Payment Plan Information:</strong><br>
                        Payment plans are automatically determined based on the number of subjects you select:<br>
                        • <strong>1 subject:</strong> Full payment only<br>
                        • <strong>2 subjects:</strong> 2 months payment plan<br>
                        • <strong>3-4 subjects:</strong> 3 months payment plan<br>
                        • <strong>5+ subjects:</strong> 4 months payment plan<br>
                        <small class="text-muted">All students pay the same first month amount: R1,500 (Registration + Monthly Tuition)</small>
                    </div>
                `;
                helpText.style.display = 'block';
                
                // Update cost calculation after changing duration options
                updateCostCalculation();
            }
            
            // Create payment plan help text element if it doesn't exist
            function createPaymentPlanHelpText() {
                const helpText = document.createElement('div');
                helpText.id = 'paymentPlanHelpText';
                helpText.style.display = 'none';
                
                const courseDurationSelect = document.getElementById('courseDuration');
                if (courseDurationSelect && courseDurationSelect.parentNode) {
                    courseDurationSelect.parentNode.insertBefore(helpText, courseDurationSelect.nextSibling);
                }
                
                return helpText;
            }

            // Update registration fee based on grade level
            function updateRegistrationFee() {
                const selectedGrade = gradeSelect.value;
                
                if (!selectedGrade) {
                    registrationFeeDisplay.textContent = 'Select grade level';
                    monthlyTuitionDisplay.textContent = '';
                    btnPrice.textContent = 'R 0.00';
                    currentRegistrationFee = 0;
                    currentGradeLevel = '';
                    
                    // Reset course duration to default options
                    updateCourseDurationOptions('');
                    
                    // Disable subject grid
                    subjectGrid.innerHTML = `
                        <div class="text-center p-4">
                            <i class="fas fa-arrow-up text-muted mb-2" style="font-size: 2rem;"></i>
                            <p class="text-muted">Please select your grade level first to see available subjects</p>
                        </div>
                    `;
                    subjectGrid.style.opacity = '0.5';
                    subjectGrid.style.pointerEvents = 'none';
                    return;
                }
                
                // Show loading state
                registrationFeeDisplay.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Loading...';
                monthlyTuitionDisplay.textContent = '';
                
                // Fetch registration fee for selected grade
                fetch(`ajax/get_registration_fee.php?grade=${encodeURIComponent(selectedGrade)}`)
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            currentRegistrationFee = data.registration_fee;
                            currentMonthlyTuition = data.tuition_fee_monthly;
                            currentGradeLevel = data.grade_level;
                            
                            registrationFeeDisplay.textContent = data.registration_fee_formatted;
                            monthlyTuitionDisplay.textContent = `Monthly Tuition: ${data.tuition_fee_formatted}`;
                            
                            // Enable submit button
                            submitBtn.classList.add('has-subjects');
                            
                            // Show cost calculator
                            document.getElementById('costCalculator').style.display = 'block';
                            
                            // Update course duration options based on grade level
                            updateCourseDurationOptions(selectedGrade);
                            
                            // Load subjects for this grade
                            loadSubjectsForGrade(selectedGrade);
                            
                            // Update cost calculation (including button price)
                            updateCostCalculation();
                        } else {
                            registrationFeeDisplay.textContent = 'Fee not available';
                            monthlyTuitionDisplay.textContent = data.message || 'Please contact admin';
                            btnPrice.textContent = 'R 0.00';
                            currentRegistrationFee = 0;
                            
                            // Show error in subject grid
                            subjectGrid.innerHTML = `
                                <div class="text-center p-4">
                                    <i class="fas fa-exclamation-triangle text-warning mb-2" style="font-size: 2rem;"></i>
                                    <p class="text-muted">No subjects available for this grade level</p>
                                </div>
                            `;
                        }
                    })
                    .catch(error => {
                        console.error('Error fetching registration fee:', error);
                        registrationFeeDisplay.textContent = 'Error loading fee';
                        monthlyTuitionDisplay.textContent = 'Please try again';
                        btnPrice.textContent = 'R 0.00';
                        currentRegistrationFee = 0;
                    });
            }
            
            // Update cost calculation
            function updateCostCalculation() {
                const selectedSubjects = document.querySelectorAll('.subject-checkbox:checked');
                const subjectCount = selectedSubjects.length;
                
                // Auto-determine payment duration based on number of subjects selected
                let paymentDuration = 1; // Default to 1 month (full payment)
                let paymentPlan = 'Full Payment';
                
                if (subjectCount === 1) {
                    paymentDuration = 1; // Full payment only
                    paymentPlan = 'Full Payment';
                } else if (subjectCount === 2) {
                    paymentDuration = 2; // 2 months payment plan
                    paymentPlan = '2 Months Payment Plan';
                } else if (subjectCount === 3 || subjectCount === 4) {
                    paymentDuration = 3; // 3 months payment plan  
                    paymentPlan = '3 Months Payment Plan';
                } else if (subjectCount >= 5) {
                    paymentDuration = 4; // 4 months payment plan
                    paymentPlan = '4 Months Payment Plan';
                }
                
                // Update the duration dropdown to reflect auto-selected payment plan
                const courseDurationSelect = document.getElementById('courseDuration');
                if (courseDurationSelect && subjectCount > 0) {
                    courseDurationSelect.innerHTML = `<option value="${paymentDuration}" selected>${paymentPlan}</option>`;
                    courseDurationSelect.disabled = true; // Disable manual selection
                    courseDurationSelect.style.backgroundColor = '#f8f9fa';
                    courseDurationSelect.style.cursor = 'not-allowed';
                }
                
                let totalSubjectCost = 0;
                selectedSubjects.forEach(checkbox => {
                    const price = parseFloat(checkbox.dataset.price || 0);
                    totalSubjectCost += price;
                });
                
                // Update calculator display
                document.getElementById('calcRegistrationFee').textContent = 
                    `R ${currentRegistrationFee.toLocaleString('en-ZA', { minimumFractionDigits: 2, maximumFractionDigits: 2 })}`;
                
                document.getElementById('selectedSubjectsCount').textContent = selectedSubjects.length;
                
                // Calculate total semester cost for all selected subjects
                // Each subject price is per semester (not per month)
                let totalSemesterSubjectsCost = 0;
                selectedSubjects.forEach(checkbox => {
                    const subjectPrice = parseFloat(checkbox.dataset.price || 0);
                    totalSemesterSubjectsCost += subjectPrice; // Each subject price is per semester
                });
                
                // Calculate total course cost (Registration + Total semester subjects cost)
                const totalCourseCost = currentRegistrationFee + totalSemesterSubjectsCost;
                
                document.getElementById('totalCourseCost').textContent = 
                    `R ${totalCourseCost.toLocaleString('en-ZA', { minimumFractionDigits: 2, maximumFractionDigits: 2 })}`;
                
                // Calculate upfront payment (Registration fee + First month's base tuition)
                // All students pay the same first month: Registration R250 + Base Monthly Tuition R1,250 = R1,500
                const upfrontPayment = currentRegistrationFee + currentMonthlyTuition;
                
                // Calculate remaining balance - this is the total cost minus the first month payment
                const remainingBalance = totalCourseCost - upfrontPayment;
                
                // Calculate monthly installment for remaining months
                const remainingMonths = paymentDuration > 1 ? paymentDuration - 1 : 0;
                const monthlyInstallment = remainingMonths > 0 ? remainingBalance / remainingMonths : 0;
                
                // Update submit button price (always R1,500 for first month)
                if (btnPrice) {
                    btnPrice.textContent = `R ${upfrontPayment.toLocaleString('en-ZA', { minimumFractionDigits: 2, maximumFractionDigits: 2 })}`;
                }
                
                // Show payment plan information
                const paymentPlanInfo = document.getElementById('paymentPlanInfo') || createPaymentPlanInfo();
                if (subjectCount > 0) {
                    if (paymentDuration === 1) {
                        paymentPlanInfo.innerHTML = `
                            <div class="alert alert-success">
                                <i class="fas fa-check-circle me-2"></i>
                                <strong>Full Payment:</strong> Total course cost R${totalCourseCost.toLocaleString('en-ZA', { minimumFractionDigits: 2, maximumFractionDigits: 2 })} paid upfront.
                            </div>
                        `;
                    } else {
                        paymentPlanInfo.innerHTML = `
                            <div class="alert alert-info">
                                <i class="fas fa-calendar-alt me-2"></i>
                                <strong>${paymentPlan}:</strong><br>
                                • <strong>First Month:</strong> R${upfrontPayment.toLocaleString('en-ZA', { minimumFractionDigits: 2, maximumFractionDigits: 2 })} (Registration + Monthly Tuition)<br>
                                • <strong>Remaining Balance:</strong> R${remainingBalance.toLocaleString('en-ZA', { minimumFractionDigits: 2, maximumFractionDigits: 2 })} over ${paymentDuration - 1} months<br>
                                • <strong>Monthly Installment:</strong> R${monthlyInstallment.toLocaleString('en-ZA', { minimumFractionDigits: 2, maximumFractionDigits: 2 })} per month
                            </div>
                        `;
                    }
                    paymentPlanInfo.style.display = 'block';
                } else {
                    paymentPlanInfo.style.display = 'none';
                }
            }
            
            // Create payment plan info element if it doesn't exist
            function createPaymentPlanInfo() {
                const paymentPlanInfo = document.createElement('div');
                paymentPlanInfo.id = 'paymentPlanInfo';
                paymentPlanInfo.style.display = 'none';
                
                const costCalculator = document.getElementById('costCalculator');
                if (costCalculator) {
                    costCalculator.appendChild(paymentPlanInfo);
                }
                
                return paymentPlanInfo;
            }
            
            // Update subject validation feedback
            function updateSubjectValidation() {
                const selectedSubjects = document.querySelectorAll('.subject-checkbox:checked');
                const subjectAlert = document.getElementById('subjectSelectionAlert');
                const submitBtn = document.querySelector('.submit-btn');
                
                if (selectedSubjects.length === 0) {
                    // Show info alert encouraging selection
                    subjectAlert.classList.remove('alert-danger');
                    subjectAlert.classList.add('alert-info');
                    subjectAlert.innerHTML = '<i class="fas fa-info-circle me-2"></i>Please select at least one subject to continue with your application.';
                    subjectAlert.style.display = 'block';
                    
                    // Disable submit button
                    submitBtn.classList.add('btn-outline-primary');
                    submitBtn.classList.remove('btn-primary');
                    submitBtn.style.opacity = '0.6';
                } else {
                    // Hide alert and enable submit
                    subjectAlert.style.display = 'none';
                    submitBtn.classList.remove('btn-outline-primary');
                    submitBtn.classList.add('btn-primary');
                    submitBtn.style.opacity = '1';
                }
            }
            
            // Load subjects for selected grade
            function loadSubjectsForGrade(gradeLevel) {
                fetch(`ajax/get_grade_subjects.php?grade=${encodeURIComponent(gradeLevel)}`)
                    .then(response => response.json())
                    .then(data => {
                        if (data.success && data.subjects.length > 0) {
                            let subjectsHtml = '';
                            data.subjects.forEach(subject => {
                                const subjectPrice = subject.price || 0;
                                subjectsHtml += `
                                    <div class="subject-card" data-subject="${subject.name}" data-price="${subjectPrice}">
                                        <div class="form-check">
                                            <input class="form-check-input subject-checkbox" type="checkbox" name="subjects[]" 
                                                   value="${subject.name}" id="subject_${subject.id}" 
                                                   data-price="${subjectPrice}">
                                            <label class="form-check-label w-100" for="subject_${subject.id}">
                                                <div class="subject-name">${subject.name}</div>
                                                <div class="subject-price">R ${Number(subjectPrice).toLocaleString('en-ZA', { minimumFractionDigits: 2, maximumFractionDigits: 2 })}</div>
                                            </label>
                                        </div>
                                    </div>
                                `;
                            });
                            
                            // Add event listeners to subject checkboxes for cost calculation and validation
                            setTimeout(() => {
                                document.querySelectorAll('.subject-checkbox').forEach(checkbox => {
                                    checkbox.addEventListener('change', function() {
                                        updateCostCalculation();
                                        updateSubjectValidation();
                                    });
                                });
                                
                                // Initial validation check
                                updateSubjectValidation();
                                
                                // Initial cost calculation
                                updateCostCalculation();
                            }, 100);
                            subjectGrid.innerHTML = subjectsHtml;
                            subjectGrid.style.opacity = '1';
                            subjectGrid.style.pointerEvents = 'all';
                        } else {
                            subjectGrid.innerHTML = `
                                <div class="text-center p-4">
                                    <i class="fas fa-info-circle text-info mb-2" style="font-size: 2rem;"></i>
                                    <p class="text-muted">No subjects available for this grade level yet</p>
                                </div>
                            `;
                        }
                    })
                    .catch(error => {
                        console.error('Error loading subjects:', error);
                        subjectGrid.innerHTML = `
                            <div class="text-center p-4">
                                <i class="fas fa-exclamation-triangle text-warning mb-2" style="font-size: 2rem;"></i>
                                <p class="text-muted">Error loading subjects. Please try again.</p>
                            </div>
                        `;
                    });
            }

            // Add event listener to grade level select
            if (gradeSelect) {
                gradeSelect.addEventListener('change', updateRegistrationFee);
                
                // Initialize on page load if grade is already selected
                if (gradeSelect.value) {
                    updateRegistrationFee();
                }
            }
            
            // Add event listener to course duration
            const courseDurationSelect = document.getElementById('courseDuration');
            if (courseDurationSelect) {
                courseDurationSelect.addEventListener('change', updateCostCalculation);
            }

            // Form submission with loading state
            form.addEventListener('submit', function(e) {
                const btnText = submitBtn.querySelector('.btn-text');
                const btnLoading = submitBtn.querySelector('.btn-loading');
                
                btnText.classList.add('d-none');
                btnLoading.classList.remove('d-none');
                submitBtn.disabled = true;
                
                // Re-enable button after 30 seconds (timeout)
                setTimeout(() => {
                    btnText.classList.remove('d-none');
                    btnLoading.classList.add('d-none');
                    submitBtn.disabled = false;
                }, 30000);
            });
            
            // Add form submission validation
            form.addEventListener('submit', function(e) {
                const selectedSubjects = document.querySelectorAll('.subject-checkbox:checked');
                const subjectAlert = document.getElementById('subjectSelectionAlert');
                
                if (selectedSubjects.length === 0) {
                    e.preventDefault(); // Prevent form submission
                    
                    // Show error alert
                    subjectAlert.classList.remove('alert-info');
                    subjectAlert.classList.add('alert-danger');
                    subjectAlert.innerHTML = '<i class="fas fa-exclamation-triangle me-2"></i>Please select at least one subject before submitting your application.';
                    subjectAlert.style.display = 'block';
                    
                    // Scroll to the subject section
                    document.querySelector('.form-section:has(#subjectGrid)').scrollIntoView({
                        behavior: 'smooth',
                        block: 'center'
                    });
                    
                    // Show toast error
                    toastr.error('Please select at least one subject to continue with your application.', 'Subject Selection Required');
                    
                    return false;
                } else {
                    // Hide alert if subjects are selected
                    subjectAlert.style.display = 'none';
                }
            });
            
            // Initialize fee display
            updateRegistrationFee();
        });

        // Toastr configuration
        toastr.options = {
            "closeButton": true,
            "debug": false,
            "newestOnTop": true,
            "progressBar": true,
            "positionClass": "toast-top-right",
            "preventDuplicates": false,
            "onclick": null,
            "showDuration": "300",
            "hideDuration": "1000",
            "timeOut": "5000",
            "extendedTimeOut": "1000",
            "showEasing": "swing",
            "hideEasing": "linear",
            "showMethod": "fadeIn",
            "hideMethod": "fadeOut"
        };

        // Show success/error messages and auto-redirect
        <?php if ($success): ?>
            toastr.success('Welcome to Wisdom Tutoring Academy! You are now logged in.', 'Success!', {
                timeOut: 5000,
                closeButton: true,
                progressBar: true
            });
            
            // Show payment notification
            setTimeout(() => {
                toastr.info('Please complete payment to access your dashboard...', 'Payment Required', {
                    timeOut: 3000
                });
            }, 1000);
            
            // Auto-redirect to appropriate payment flow chart after 5 seconds
            setTimeout(() => {
                <?php if (isset($_SESSION['role']) && $_SESSION['role'] === 'matric'): ?>
                    window.location.href = 'matric/payment-flow.php';
                <?php else: ?>
                    window.location.href = 'student/payment-flow.php';
                <?php endif; ?>
            }, 5000);
        <?php endif; ?>

        <?php if (!empty($errors)): ?>
            toastr.error('Please fix the errors in the form and try again.');
        <?php endif; ?>
    </script>

    <!-- Footer -->
    <footer class="footer py-4">
        <div class="container">
            <div class="row">
                <div class="col-lg-4 mb-4 mb-lg-0">
                    <img src="assets/images/logo.jpeg" alt="Wisdom Tutoring Academy Logo" height="60" class="mb-3">
                    <?php echo renderEditableText('footer', 'company_tagline', 'We learn. We thrive. We excel.', 'p', ''); ?>
                    <div class="social-icons">
                    <a href="https://www.facebook.com/share/14FqYEg8ps4/?mibextid=wwXIfr" class="me-2"><i class="fab fa-facebook-f"></i></a>
                      
                      <a href="https://www.instagram.com/wisdom_tutoringacademy?igsh=MWd5MWZsMXBqM242cw%3D%3D&utm_source=qr" class="me-2"><i class="fab fa-instagram"></i></a>
                    </div>
                </div>
                <div class="col-lg-2 col-md-6 mb-4 mb-lg-0">
                    <?php echo renderEditableText('footer', 'quick_links_title', 'Quick Links', 'h5', ''); ?>
                    <ul class="list-unstyled">
                    <li><a href="index.php">Home</a></li>
                        <li><a href="about.php">About Us</a></li>
                        <li><a href="courses.php">Courses</a></li>
                        <li><a href="portfolio.php">Portfolio</a></li>
                        <li><a href="contact.php">Contact</a></li>
                    </ul>
                </div>
                <div class="col-lg-3 col-md-6 mb-4 mb-lg-0">
                    <?php echo renderEditableText('footer', 'programs_title', 'Programs', 'h5', ''); ?>
                    <ul class="list-unstyled">
                        <li><?php echo renderEditableText('footer', 'program1', 'Matric Rewrite', 'a', ''); ?></li>
                        <li><?php echo renderEditableText('footer', 'program2', 'Grade 8-12 Support', 'a', ''); ?></li>
                        <li><?php echo renderEditableText('footer', 'program3', 'Online Assessments', 'a', ''); ?></li>
                        <li><?php echo renderEditableText('footer', 'program4', 'Virtual Classrooms', 'a', ''); ?></li>
                    </ul>
                </div>
                <div class="col-lg-3 col-md-6">
                    <!--
                    <?php echo renderEditableText('footer', 'contact_title', 'Contact Us', 'h5', ''); ?>
                    <ul class="list-unstyled contact-info">
                        <li><i class="fas fa-map-marker-alt me-2"></i> <?php echo renderEditableText('footer', 'contact_address', '123 Education Street, Learning City', 'span', ''); ?></li>
                        <li><i class="fas fa-phone me-2"></i> <?php echo renderEditableText('footer', 'contact_phone', '(012) 345 6789', 'span', ''); ?></li>
                        <li><i class="fas fa-envelope me-2"></i> <?php echo renderEditableText('footer', 'contact_email', 'info@wisdomtutoring.ac.za', 'span', ''); ?></li>
                    </ul>
                    -->
                    
                    <ul class="list-unstyled contact-info">
                     <li><i class="fas fa-map-marker-alt me-2"></i> 
                        Address: Wisdom Tutoring Academy, Plantation Street 
                        Cofimvaba, Eastern Cape 5380 
                        South Africa 

                       
                        </li>
                        <li><i class="fas fa-phone me-2"></i>  Contact: +27 63 2617881 / +27 66 4195231</li>
                        <li><i class="fas fa-envelope me-2"></i>
                        Email: infor@wisdomtutoringacademy.co.za</li>
                    </ul>
                </div>
            </div>
            <hr>
            <div class="row">
                <div class="col-md-6 text-center text-md-start">
                    <p class="mb-0">&copy; <?php echo date('Y'); ?> Wisdom Tutoring Academy. All rights reserved.</p>
                </div>
                <div class="col-md-6 text-center text-md-end">
                </div>
            </div>
            <div class="row mt-3">
                <div class="col-12 text-center">
                    <p class="mb-0" style="color: var(--gray-color); font-size: 0.9rem;">Developed by <strong>Webectak</strong></p>
                    <p class="mb-0 mt-2" style="color: var(--gray-color); font-size: 0.8rem;">
                        <i class="fas fa-copyright me-1"></i>
                        Unauthorized copying or reproduction is strictly prohibited. For licensing permissions, contact <strong>Wisdom Tutoring Academy</strong>.
                    </p>
                </div>
            </div>
        </div>
    </footer>

</body>
</html>
