/**
 * Calendar Functionality - Shared functions for all roles
 */

// Update calendar grid based on view type and data
function updateCalendarGrid(data, viewType) {
    const calendarContainer = document.getElementById('calendar-main');
    if (!calendarContainer) return;
    
    // Clear existing content
    calendarContainer.innerHTML = '';
    
    // Select rendering function based on view
    switch (viewType) {
        case 'month':
            renderMonthView(calendarContainer, data);
            break;
        case 'week':
            renderWeekView(calendarContainer, data);
            break;
        case 'day':
            renderDayView(calendarContainer, data);
            break;
        case 'list':
            renderListView(calendarContainer, data);
            break;
        default:
            renderMonthView(calendarContainer, data);
    }
}

// Render Month View
function renderMonthView(container, data) {
    const { monthName, firstDayOfMonth, daysInMonth, events, today, currentMonth, currentYear, month, year } = data;
    
    // Create month view table
    const table = document.createElement('table');
    table.className = 'calendar-table';
    
    // Create header
    const thead = document.createElement('thead');
    const headerRow = document.createElement('tr');
    const days = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'];
    days.forEach(day => {
        const th = document.createElement('th');
        th.textContent = day;
        headerRow.appendChild(th);
    });
    thead.appendChild(headerRow);
    table.appendChild(thead);
    
    // Create body
    const tbody = document.createElement('tbody');
    let dayCount = 1;
    const totalCells = parseInt(firstDayOfMonth) + parseInt(daysInMonth);
    const totalRows = Math.ceil(totalCells / 7);
    
    for (let i = 0; i < totalRows; i++) {
        const row = document.createElement('tr');
        for (let j = 0; j < 7; j++) {
            const cellNumber = i * 7 + j;
            const cell = document.createElement('td');
            
            if (cellNumber < firstDayOfMonth || dayCount > daysInMonth) {
                cell.className = 'empty';
            } else {
                const isToday = (today == dayCount && currentMonth == month && currentYear == year);
                cell.className = 'day-cell' + (isToday ? ' today' : '');
                
                // Day number
                const dayNumber = document.createElement('div');
                dayNumber.className = 'day-number';
                dayNumber.textContent = dayCount;
                cell.appendChild(dayNumber);
                
                // Day events container
                const dayEvents = document.createElement('div');
                dayEvents.className = 'day-events';
                
                // Add events for this day
                events.forEach(event => {
                    const eventDate = new Date(event.event_date).getDate();
                    if (eventDate == dayCount) {
                        const eventDiv = document.createElement('div');
                        eventDiv.className = 'calendar-event ' + event.type;
                        
                        const eventTitle = document.createElement('span');
                        eventTitle.className = 'event-title';
                        eventTitle.textContent = event.title;
                        eventDiv.appendChild(eventTitle);
                        
                        // Add click event to show modal
                        eventDiv.addEventListener('click', () => {
                            const startTime = formatTime(event.start_time);
                            const endTime = formatTime(event.end_time);
                            const timeRange = `${startTime} - ${endTime}`;
                            showEventModal(event.title, timeRange, event.course_title, event.description);
                        });
                        
                        dayEvents.appendChild(eventDiv);
                    }
                });
                
                cell.appendChild(dayEvents);
                dayCount++;
            }
            
            row.appendChild(cell);
        }
        tbody.appendChild(row);
    }
    
    table.appendChild(tbody);
    container.appendChild(table);
}

// Render Week View
function renderWeekView(container, data) {
    const { events, month, year } = data;
    
    // Create week container
    const weekContainer = document.createElement('div');
    weekContainer.className = 'week-view';
    
    // Get the current date
    const currentDate = new Date(year, month - 1, new Date().getDate());
    const startOfWeek = new Date(currentDate);
    startOfWeek.setDate(currentDate.getDate() - currentDate.getDay());
    
    // Create header with days of the week
    const header = document.createElement('div');
    header.className = 'week-header';
    
    for (let i = 0; i < 7; i++) {
        const day = new Date(startOfWeek);
        day.setDate(startOfWeek.getDate() + i);
        
        const dayColumn = document.createElement('div');
        dayColumn.className = 'day-column';
        
        const dayHeader = document.createElement('div');
        dayHeader.className = 'day-header' + (day.toDateString() === new Date().toDateString() ? ' today' : '');
        dayHeader.innerHTML = `
            <div class="day-name">${day.toLocaleDateString('en-US', { weekday: 'short' })}</div>
            <div class="day-number">${day.getDate()}</div>
        `;
        dayColumn.appendChild(dayHeader);
        
        // Add events for this day
        const dayEvents = document.createElement('div');
        dayEvents.className = 'day-events';
        
        events.forEach(event => {
            const eventDate = new Date(event.event_date);
            if (eventDate.toDateString() === day.toDateString()) {
                const eventDiv = document.createElement('div');
                eventDiv.className = 'calendar-event ' + event.type;
                
                const eventTime = document.createElement('div');
                eventTime.className = 'event-time';
                eventTime.textContent = formatTime(event.start_time);
                eventDiv.appendChild(eventTime);
                
                const eventTitle = document.createElement('div');
                eventTitle.className = 'event-title';
                eventTitle.textContent = event.title;
                eventDiv.appendChild(eventTitle);
                
                // Add click event to show modal
                eventDiv.addEventListener('click', () => {
                    const startTime = formatTime(event.start_time);
                    const endTime = formatTime(event.end_time);
                    const timeRange = `${startTime} - ${endTime}`;
                    showEventModal(event.title, timeRange, event.course_title, event.description);
                });
                
                dayEvents.appendChild(eventDiv);
            }
        });
        
        dayColumn.appendChild(dayEvents);
        header.appendChild(dayColumn);
    }
    
    weekContainer.appendChild(header);
    container.appendChild(weekContainer);
}

// Render Day View
function renderDayView(container, data) {
    const { events, month, year } = data;
    
    // Create day view container
    const dayContainer = document.createElement('div');
    dayContainer.className = 'day-view';
    
    // Get the current date
    const currentDate = new Date(year, month - 1, new Date().getDate());
    
    // Create header with the selected day
    const header = document.createElement('div');
    header.className = 'day-header';
    header.innerHTML = `
        <h3>${currentDate.toLocaleDateString('en-US', { weekday: 'long', month: 'long', day: 'numeric' })}</h3>
    `;
    dayContainer.appendChild(header);
    
    // Create timeline
    const timeline = document.createElement('div');
    timeline.className = 'day-timeline';
    
    // Create hours (8AM to 9PM)
    for (let hour = 8; hour <= 21; hour++) {
        const hourDiv = document.createElement('div');
        hourDiv.className = 'hour-block';
        
        const hourLabel = document.createElement('div');
        hourLabel.className = 'hour-label';
        hourLabel.textContent = `${hour % 12 === 0 ? 12 : hour % 12}${hour < 12 ? 'AM' : 'PM'}`;
        hourDiv.appendChild(hourLabel);
        
        const hourContent = document.createElement('div');
        hourContent.className = 'hour-content';
        hourDiv.appendChild(hourContent);
        
        timeline.appendChild(hourDiv);
    }
    
    dayContainer.appendChild(timeline);
    
    // Add events for the selected day
    const filteredEvents = events.filter(event => {
        const eventDate = new Date(event.event_date);
        return eventDate.toDateString() === currentDate.toDateString();
    });
    
    if (filteredEvents.length > 0) {
        const eventsList = document.createElement('div');
        eventsList.className = 'day-events-list';
        
        filteredEvents.forEach(event => {
            const eventDiv = document.createElement('div');
            eventDiv.className = 'calendar-event ' + event.type;
            
            const eventTime = document.createElement('div');
            eventTime.className = 'event-time';
            eventTime.textContent = `${formatTime(event.start_time)} - ${formatTime(event.end_time)}`;
            eventDiv.appendChild(eventTime);
            
            const eventTitle = document.createElement('div');
            eventTitle.className = 'event-title';
            eventTitle.textContent = event.title;
            eventDiv.appendChild(eventTitle);
            
            const eventCourse = document.createElement('div');
            eventCourse.className = 'event-course';
            eventCourse.textContent = event.course_title;
            eventDiv.appendChild(eventCourse);
            
            // Add click event to show modal
            eventDiv.addEventListener('click', () => {
                const startTime = formatTime(event.start_time);
                const endTime = formatTime(event.end_time);
                const timeRange = `${startTime} - ${endTime}`;
                showEventModal(event.title, timeRange, event.course_title, event.description);
            });
            
            eventsList.appendChild(eventDiv);
        });
        
        dayContainer.appendChild(eventsList);
    } else {
        const noEvents = document.createElement('div');
        noEvents.className = 'no-events-message';
        noEvents.textContent = 'No events scheduled for today';
        dayContainer.appendChild(noEvents);
    }
    
    container.appendChild(dayContainer);
}

// Render List View
function renderListView(container, data) {
    const { events, month, year } = data;
    
    // Create list view container
    const listContainer = document.createElement('div');
    listContainer.className = 'list-view';
    
    // Group events by date
    const eventsByDate = {};
    events.forEach(event => {
        const date = event.event_date;
        if (!eventsByDate[date]) {
            eventsByDate[date] = [];
        }
        eventsByDate[date].push(event);
    });
    
    // Sort dates
    const sortedDates = Object.keys(eventsByDate).sort();
    
    if (sortedDates.length > 0) {
        sortedDates.forEach(date => {
            const dateEvents = eventsByDate[date];
            const dateObj = new Date(date);
            
            const dateHeader = document.createElement('div');
            dateHeader.className = 'date-header';
            dateHeader.innerHTML = `
                <h3>${dateObj.toLocaleDateString('en-US', { weekday: 'long', month: 'long', day: 'numeric' })}</h3>
            `;
            listContainer.appendChild(dateHeader);
            
            const eventsList = document.createElement('div');
            eventsList.className = 'events-list';
            
            dateEvents.forEach(event => {
                const eventItem = document.createElement('div');
                eventItem.className = 'event-item ' + event.type;
                
                const eventTime = document.createElement('div');
                eventTime.className = 'event-time';
                eventTime.textContent = `${formatTime(event.start_time)} - ${formatTime(event.end_time)}`;
                eventItem.appendChild(eventTime);
                
                const eventContent = document.createElement('div');
                eventContent.className = 'event-content';
                
                const eventTitle = document.createElement('div');
                eventTitle.className = 'event-title';
                eventTitle.textContent = event.title;
                eventContent.appendChild(eventTitle);
                
                const eventCourse = document.createElement('div');
                eventCourse.className = 'event-course';
                eventCourse.textContent = event.course_title;
                eventContent.appendChild(eventCourse);
                
                eventItem.appendChild(eventContent);
                
                // Add click event to show modal
                eventItem.addEventListener('click', () => {
                    const startTime = formatTime(event.start_time);
                    const endTime = formatTime(event.end_time);
                    const timeRange = `${startTime} - ${endTime}`;
                    showEventModal(event.title, timeRange, event.course_title, event.description);
                });
                
                eventsList.appendChild(eventItem);
            });
            
            listContainer.appendChild(eventsList);
        });
    } else {
        const noEvents = document.createElement('div');
        noEvents.className = 'no-events-message';
        noEvents.textContent = 'No events scheduled for this month';
        listContainer.appendChild(noEvents);
    }
    
    container.appendChild(listContainer);
}

// Update events display in sidebar
function updateEventsDisplay(events) {
    const eventsContainer = document.querySelector('.upcoming-events-content');
    if (!eventsContainer) return;
    
    if (events.length > 0) {
        let eventsHTML = '';
        events.slice(0, 5).forEach(event => {
            let badgeClass = 'recurring';
            let badgeText = 'Class';
            
            if (event.type === 'assignment') {
                badgeClass = 'important';
                badgeText = 'Assignment';
            } else if (event.type === 'exam') {
                badgeClass = 'reminder';
                badgeText = 'Exam';
            } else if (event.type === 'meeting') {
                badgeClass = 'recurring';
                badgeText = 'Meeting';
            } else if (event.type === 'office_hours') {
                badgeClass = 'reminder';
                badgeText = 'Office Hours';
            } else if (event.type === 'grading') {
                badgeClass = 'important';
                badgeText = 'Grading';
            } else if (event.type === 'training') {
                badgeClass = 'reminder';
                badgeText = 'Training';
            }
            
            const eventDate = new Date(event.event_date);
            const formattedDate = eventDate.toLocaleDateString('en-US', { month: 'short', day: 'numeric', year: 'numeric' });
            
            eventsHTML += `
                <div class="event-item" onclick="showEventModal('${escapeHTML(event.title)}', '${formatTime(event.start_time)} - ${formatTime(event.end_time)}', '${escapeHTML(event.course_title)}', '${escapeHTML(event.description)}')">
                    <div class="event-title">
                        <span class="event-badge ${badgeClass}">${badgeText}</span>
                        ${escapeHTML(event.title)}
                    </div>
                    <div class="event-course">
                        <i class="fas fa-bookmark"></i>
                        ${escapeHTML(event.course_title)}
                    </div>
                    <div class="event-date">
                        <i class="fas fa-calendar"></i>
                        ${formattedDate}
                    </div>
                </div>
            `;
        });
        
        eventsContainer.innerHTML = eventsHTML;
    } else {
        eventsContainer.innerHTML = `
            <div class="no-events">
                <p>No upcoming events scheduled for the upcoming days</p>
                <button class="btn-primary">Add Event</button>
            </div>
        `;
    }
}

// Helper function to format time from 24h to 12h format
function formatTime(timeString) {
    if (!timeString) return '';
    const [hours, minutes] = timeString.split(':');
    const hour = parseInt(hours);
    const ampm = hour >= 12 ? 'PM' : 'AM';
    const hour12 = hour % 12 || 12;
    return `${hour12}:${minutes} ${ampm}`;
}

// Helper function to escape HTML special characters
function escapeHTML(str) {
    if (!str) return '';
    return str
        .replace(/&/g, "&amp;")
        .replace(/</g, "&lt;")
        .replace(/>/g, "&gt;")
        .replace(/"/g, "&quot;")
        .replace(/'/g, "&#039;");
}
