/**
 * Panadite Academy - Teacher Dashboard JavaScript
 * Handles AJAX requests, charts, and interactive elements
 */

$(document).ready(function() {
    'use strict';
    
    // Initialize tooltips
    $('[data-bs-toggle="tooltip"]').tooltip();
    
    // Initialize popovers
    $('[data-bs-toggle="popover"]').popover();
    
    // Toggle sidebar function
    $('#sidebarToggle').on('click', function(e) {
        e.preventDefault();
        $('.sidebar').toggleClass('collapsed');
        $('.main-content').toggleClass('expanded');
        
        // Store sidebar state in localStorage
        localStorage.setItem('sidebar-collapsed', $('.sidebar').hasClass('collapsed'));
        
        // Force repaint to fix any rendering issues
        setTimeout(function(){
            $(window).trigger('resize');
        }, 300);
    });
    
    // Check localStorage for sidebar state
    if(localStorage.getItem('sidebar-collapsed') === 'true') {
        $('.sidebar').addClass('collapsed');
        $('.main-content').addClass('expanded');
    }
    
    // Initialize earnings chart
    if($('#earningsChart').length > 0) {
        initEarningsChart();
    }
    
    // AJAX Dashboard Data Loading
    loadDashboardData();
    
    // Course status toggle (Published/Draft)
    $('.course-status-toggle').on('change', function() {
        const courseId = $(this).data('course-id');
        const isPublished = $(this).is(':checked') ? 1 : 0;
        
        updateCourseStatus(courseId, isPublished);
    });
    
    // Video status toggle (Published/Draft)
    $('.video-status-toggle').on('change', function() {
        const videoId = $(this).data('video-id');
        const isPublished = $(this).is(':checked') ? 1 : 0;
        
        updateVideoStatus(videoId, isPublished);
    });
    
    // Init date range picker for reports
    if($('#reportDateRange').length > 0) {
        $('#reportDateRange').daterangepicker({
            ranges: {
                'Today': [moment(), moment()],
                'Yesterday': [moment().subtract(1, 'days'), moment().subtract(1, 'days')],
                'Last 7 Days': [moment().subtract(6, 'days'), moment()],
                'Last 30 Days': [moment().subtract(29, 'days'), moment()],
                'This Month': [moment().startOf('month'), moment().endOf('month')],
                'Last Month': [moment().subtract(1, 'month').startOf('month'), moment().subtract(1, 'month').endOf('month')]
            },
            startDate: moment().subtract(29, 'days'),
            endDate: moment()
        }, function(start, end) {
            $('#reportDateRangeText').html(start.format('MMMM D, YYYY') + ' - ' + end.format('MMMM D, YYYY'));
            loadReportData(start.format('YYYY-MM-DD'), end.format('YYYY-MM-DD'));
        });
    }
    
    // Notification dropdown animation
    $('.dropdown-notifications').on('show.bs.dropdown', function() {
        $(this).find('.dropdown-menu').first().stop(true, true).slideDown(300);
    });
    
    $('.dropdown-notifications').on('hide.bs.dropdown', function() {
        $(this).find('.dropdown-menu').first().stop(true, true).slideUp(200);
    });
    
    // Mark notifications as read
    $('.mark-as-read').on('click', function(e) {
        e.preventDefault();
        const notificationId = $(this).data('notification-id');
        
        if(notificationId === 'all') {
            markAllNotificationsAsRead();
        } else {
            markNotificationAsRead(notificationId);
        }
    });
    
    // Course filter
    $('#courseFilter').on('change', function() {
        const courseId = $(this).val();
        loadCourseVideos(courseId);
    });
});

/**
 * Load main dashboard data via AJAX
 */
function loadDashboardData() {
    $.ajax({
        url: 'ajax/get_dashboard_data.php',
        type: 'GET',
        dataType: 'json',
        beforeSend: function() {
            // Show loading indicators
            $('.dashboard-stats .stat-card').addClass('loading');
            $('#dashboardLoading').fadeIn(200);
        },
        success: function(response) {
            if(response.success) {
                // Update stats
                $('#totalCourses').text(response.data.courseCount || 0);
                $('#totalVideos').text(response.data.videoCount || 0);
                $('#totalEarnings').text('R' + (response.data.totalEarnings || 0).toFixed(2));
                $('#totalPurchases').text(response.data.purchaseCount || 0);
                $('#monthlyEarnings').text('R' + (response.data.monthlyEarnings || 0).toFixed(2));
                
                // Update charts
                updateEarningsChart(response.data.earningsData);
                
                // Update course list
                if(response.data.recentCourses && response.data.recentCourses.length > 0) {
                    renderCourseCards(response.data.recentCourses);
                } else {
                    $('#courseList').html('<div class="alert alert-info">You haven\'t created any courses yet. <a href="add_course.php" class="alert-link">Create your first course</a> to get started.</div>');
                }
                
                // Update video list
                if(response.data.recentVideos && response.data.recentVideos.length > 0) {
                    renderVideoTable(response.data.recentVideos);
                } else {
                    $('#videoList').html('<div class="alert alert-info">You haven\'t uploaded any videos yet. <a href="add_video.php" class="alert-link">Upload your first video</a> to start earning.</div>');
                }
                
                // Update student enrollments
                if(response.data.recentEnrollments && response.data.recentEnrollments.length > 0) {
                    renderEnrollments(response.data.recentEnrollments);
                } else {
                    $('#enrollmentList').hide();
                }
                
                // Update notifications count
                if(response.data.unreadNotifications > 0) {
                    $('#notificationCount').text(response.data.unreadNotifications).show();
                } else {
                    $('#notificationCount').hide();
                }
            } else {
                showErrorMessage('Failed to load dashboard data.');
            }
        },
        error: function() {
            showErrorMessage('Network error. Please try again.');
        },
        complete: function() {
            // Hide loading indicators
            $('.dashboard-stats .stat-card').removeClass('loading');
            $('#dashboardLoading').fadeOut(200);
        }
    });
}

/**
 * Initialize earnings chart
 */
function initEarningsChart() {
    const ctx = document.getElementById('earningsChart').getContext('2d');
    
    // Default empty chart
    window.earningsChart = new Chart(ctx, {
        type: 'line',
        data: {
            labels: [],
            datasets: [{
                label: 'Earnings',
                data: [],
                backgroundColor: 'rgba(39, 107, 228, 0.1)',
                borderColor: '#276BE4',
                borderWidth: 2,
                pointBackgroundColor: '#276BE4',
                pointRadius: 4,
                pointHoverRadius: 6,
                tension: 0.3
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    display: false
                },
                tooltip: {
                    backgroundColor: '#fff',
                    titleColor: '#495057',
                    bodyColor: '#495057',
                    borderColor: '#e9ecef',
                    borderWidth: 1,
                    padding: 12,
                    displayColors: false,
                    callbacks: {
                        label: function(context) {
                            return 'R' + context.parsed.y.toFixed(2);
                        }
                    }
                }
            },
            scales: {
                x: {
                    grid: {
                        display: false
                    },
                    ticks: {
                        color: '#6c757d'
                    }
                },
                y: {
                    beginAtZero: true,
                    grid: {
                        color: 'rgba(0,0,0,0.05)'
                    },
                    ticks: {
                        color: '#6c757d',
                        callback: function(value) {
                            return 'R' + value;
                        }
                    }
                }
            }
        }
    });
}

/**
 * Update earnings chart with new data
 */
function updateEarningsChart(earningsData) {
    if (!window.earningsChart || !earningsData) return;
    
    const labels = earningsData.map(item => item.date);
    const data = earningsData.map(item => item.amount);
    
    window.earningsChart.data.labels = labels;
    window.earningsChart.data.datasets[0].data = data;
    window.earningsChart.update();
}

/**
 * Render course cards
 */
function renderCourseCards(courses) {
    let html = '';
    
    courses.forEach(function(course) {
        const thumbnail = course.thumbnail || '/assets/images/default-thumbnail.jpg';
        const statusClass = course.is_published ? 'success' : 'warning';
        const statusText = course.is_published ? 'Published' : 'Draft';
        
        html += `
        <div class="col-md-6 col-lg-3 mb-4">
            <div class="content-card card h-100">
                <img src="${thumbnail}" class="card-img-top" alt="${course.title}">
                <div class="card-body">
                    <span class="badge bg-${statusClass} badge-status mb-2">${statusText}</span>
                    <h5 class="card-title">${course.title}</h5>
                    <p class="card-text small">${course.description.substring(0, 80)}...</p>
                </div>
                <div class="card-footer d-flex justify-content-between align-items-center">
                    <span><i class="fas fa-users me-1"></i> ${course.student_count || 0} students</span>
                    <a href="edit_course.php?id=${course.course_id}" class="btn btn-sm btn-primary">Manage</a>
                </div>
            </div>
        </div>`;
    });
    
    $('#courseList').html(html);
}

/**
 * Render video table
 */
function renderVideoTable(videos) {
    let html = `
    <div class="table-responsive">
        <table class="table table-hover data-table">
            <thead>
                <tr>
                    <th>Title</th>
                    <th>Course</th>
                    <th>Price</th>
                    <th>Status</th>
                    <th>Purchases</th>
                    <th>Date Added</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>`;
    
    videos.forEach(function(video) {
        const thumbnail = video.thumbnail || '/assets/images/default-thumbnail.jpg';
        const statusClass = video.is_published ? 'success' : 'warning';
        const statusText = video.is_published ? 'Published' : 'Draft';
        const formattedDate = new Date(video.created_at).toLocaleDateString('en-US', { month: 'short', day: 'numeric', year: 'numeric' });
        
        html += `
        <tr>
            <td>
                <div class="d-flex align-items-center">
                    <img src="${thumbnail}" alt="${video.title}" width="40" height="30" 
                         class="me-2" style="object-fit: cover; border-radius: 4px;">
                    <span>${video.title}</span>
                </div>
            </td>
            <td>${video.course_title}</td>
            <td>R${parseFloat(video.price).toFixed(2)}</td>
            <td>
                <span class="badge bg-${statusClass}">${statusText}</span>
            </td>
            <td>${video.purchases || 0}</td>
            <td>${formattedDate}</td>
            <td>
                <a href="edit_video.php?id=${video.video_id}" class="btn btn-sm btn-outline-primary">
                    <i class="fas fa-edit"></i>
                </a>
            </td>
        </tr>`;
    });
    
    html += `
            </tbody>
        </table>
    </div>`;
    
    $('#videoList').html(html);
}

/**
 * Render student enrollments
 */
function renderEnrollments(enrollments) {
    let html = '';
    
    enrollments.forEach(function(enrollment) {
        const enrollmentDate = new Date(enrollment.enrolled_date).toLocaleDateString('en-US', { month: 'short', day: 'numeric', year: 'numeric' });
        
        html += `
        <li class="list-group-item px-0">
            <div class="activity-item">
                <div class="activity-icon activity-primary">
                    <i class="fas fa-user-graduate"></i>
                </div>
                <div class="activity-content">
                    <div class="activity-title">${enrollment.first_name} ${enrollment.last_name} enrolled in <b>${enrollment.course_title}</b></div>
                    <div class="activity-time">${enrollmentDate}</div>
                </div>
            </div>
        </li>`;
    });
    
    $('#enrollmentItems').html(html);
    $('#enrollmentList').show();
}

/**
 * Update course publish status
 */
function updateCourseStatus(courseId, isPublished) {
    $.ajax({
        url: 'ajax/update_course_status.php',
        type: 'POST',
        data: {
            course_id: courseId,
            is_published: isPublished
        },
        dataType: 'json',
        success: function(response) {
            if(response.success) {
                showSuccessMessage(response.message || 'Course status updated successfully.');
            } else {
                showErrorMessage(response.message || 'Failed to update course status.');
                // Revert the toggle
                $('.course-status-toggle[data-course-id="' + courseId + '"]').prop('checked', !isPublished);
            }
        },
        error: function() {
            showErrorMessage('Network error. Please try again.');
            // Revert the toggle
            $('.course-status-toggle[data-course-id="' + courseId + '"]').prop('checked', !isPublished);
        }
    });
}

/**
 * Update video publish status
 */
function updateVideoStatus(videoId, isPublished) {
    $.ajax({
        url: 'ajax/update_video_status.php',
        type: 'POST',
        data: {
            video_id: videoId,
            is_published: isPublished
        },
        dataType: 'json',
        success: function(response) {
            if(response.success) {
                showSuccessMessage(response.message || 'Video status updated successfully.');
            } else {
                showErrorMessage(response.message || 'Failed to update video status.');
                // Revert the toggle
                $('.video-status-toggle[data-video-id="' + videoId + '"]').prop('checked', !isPublished);
            }
        },
        error: function() {
            showErrorMessage('Network error. Please try again.');
            // Revert the toggle
            $('.video-status-toggle[data-video-id="' + videoId + '"]').prop('checked', !isPublished);
        }
    });
}

/**
 * Load report data by date range
 */
function loadReportData(startDate, endDate) {
    $.ajax({
        url: 'ajax/get_report_data.php',
        type: 'GET',
        data: {
            start_date: startDate,
            end_date: endDate
        },
        dataType: 'json',
        beforeSend: function() {
            $('#reportLoading').fadeIn(200);
        },
        success: function(response) {
            if(response.success) {
                // Update report stats
                $('#reportEarnings').text('R' + (response.data.earnings || 0).toFixed(2));
                $('#reportPurchases').text(response.data.purchases || 0);
                $('#reportEnrollments').text(response.data.enrollments || 0);
                $('#reportNewStudents').text(response.data.newStudents || 0);
                
                // Update report chart
                if(window.reportChart && response.data.dailyData) {
                    updateReportChart(response.data.dailyData);
                }
            } else {
                showErrorMessage('Failed to load report data.');
            }
        },
        error: function() {
            showErrorMessage('Network error. Please try again.');
        },
        complete: function() {
            $('#reportLoading').fadeOut(200);
        }
    });
}

/**
 * Mark a notification as read
 */
function markNotificationAsRead(notificationId) {
    $.ajax({
        url: 'ajax/mark_notification_read.php',
        type: 'POST',
        data: {
            notification_id: notificationId
        },
        dataType: 'json',
        success: function(response) {
            if(response.success) {
                // Update UI
                $('#notification-' + notificationId).removeClass('unread');
                
                // Update notification count
                const currentCount = parseInt($('#notificationCount').text()) || 0;
                if(currentCount > 1) {
                    $('#notificationCount').text(currentCount - 1);
                } else {
                    $('#notificationCount').hide();
                }
            }
        }
    });
}

/**
 * Mark all notifications as read
 */
function markAllNotificationsAsRead() {
    $.ajax({
        url: 'ajax/mark_notification_read.php',
        type: 'POST',
        data: {
            mark_all: true
        },
        dataType: 'json',
        success: function(response) {
            if(response.success) {
                // Update UI
                $('.notification-item').removeClass('unread');
                
                // Update notification count
                $('#notificationCount').hide();
            }
        }
    });
}

/**
 * Load videos filtered by course
 */
function loadCourseVideos(courseId) {
    $.ajax({
        url: 'ajax/get_course_videos.php',
        type: 'GET',
        data: {
            course_id: courseId
        },
        dataType: 'json',
        beforeSend: function() {
            $('#videoListLoading').fadeIn(200);
        },
        success: function(response) {
            if(response.success) {
                if(response.data.videos && response.data.videos.length > 0) {
                    renderVideoTable(response.data.videos);
                } else {
                    $('#videoList').html('<div class="alert alert-info">No videos found for this course. <a href="add_video.php" class="alert-link">Upload a video</a> to get started.</div>');
                }
            } else {
                showErrorMessage('Failed to load videos.');
            }
        },
        error: function() {
            showErrorMessage('Network error. Please try again.');
        },
        complete: function() {
            $('#videoListLoading').fadeOut(200);
        }
    });
}

/**
 * Show success message toast
 */
function showSuccessMessage(message) {
    Toastify({
        text: message,
        duration: 3000,
        close: true,
        gravity: "top",
        position: "right",
        backgroundColor: "#28a745",
        stopOnFocus: true,
    }).showToast();
}

/**
 * Show error message toast
 */
function showErrorMessage(message) {
    Toastify({
        text: message,
        duration: 3000,
        close: true,
        gravity: "top",
        position: "right",
        backgroundColor: "#dc3545",
        stopOnFocus: true,
    }).showToast();
}
