<?php
/**
 * Panadite Academy - Guest Shopping Cart Page
 * Shows cart for guests and prompts them to register/login to complete purchase
 */

// Include SEO headers with favicon for cart page
$seo_title = 'Shopping Cart - Panadite Academy | Secure Course Checkout';
$seo_description = 'Review your selected courses and complete secure checkout at Panadite Academy. Start your professional learning journey with expert-led online courses.';
require_once __DIR__ . '/includes/seo_header.php';

session_start();
require_once 'includes/db.php';
require_once 'auth/functions.php';
require_once 'includes/id_encryption.php';

// Check if user is authenticated
$currentUser = getCurrentUser();
$isAuthenticated = $currentUser !== null;

// If authenticated, redirect to student cart
if ($isAuthenticated) {
    header('Location: /student/cart.php?from_guest=1');
    exit;
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Shopping Cart - Panadite Academy</title>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    
    <!-- Font Awesome Icons -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    
    <!-- Custom CSS -->
    <style>
        :root {
            --primary-blue: #2563eb;
            --primary-orange: #f59e0b;
            --dark-text: #1f2937;
            --light-text: #6b7280;
            --border-color: #e5e7eb;
            --success-color: #10b981;
            --warning-color: #f59e0b;
            --danger-color: #ef4444;
        }

        * {
            box-sizing: border-box;
        }

        body {
            background-color: #f5f5f5; /* Whitesmoke background */
            min-height: 100vh;
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
            margin: 0;
        }

        .navbar-brand {
            font-weight: 700;
            color: var(--primary-blue) !important;
        }

        .modern-cart-container {
            min-height: calc(100vh - 76px);
            padding: 0 2rem 2rem;
            max-width: 1200px;
            margin: 0 auto;
        }

        .cart-main-wrapper {
            background: white;
            border-radius: 8px;
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
            width: 100%;
            overflow: hidden;
        }

        .page-header {
            padding: 2rem 2rem 1rem;
            max-width: 1200px;
            margin: 0 auto;
        }

        .page-header h1 {
            font-size: 2rem;
            font-weight: 700;
            color: var(--text-dark);
            margin: 0;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .page-header p {
            font-size: 1rem;
            color: #6b7280;
            margin: 0.5rem 0 0;
        }

        /* Font Awesome Icons Styling */
        .fa, .fas, .far, .fal, .fab {
            line-height: 1;
            vertical-align: -.125em;
        }

        .page-header .fas {
            font-size: 1.5rem;
            margin-right: 0.5rem;
        }

        .empty-icon {
            font-size: 4rem;
            color: #9ca3af;
            margin-bottom: 1rem;
        }

        .browse-btn .fas {
            margin-right: 0.5rem;
        }

        .cart-content {
            display: grid;
            grid-template-columns: 1fr 300px;
            gap: 3rem;
            padding: 0 2rem 2rem;
            max-width: 1200px;
            margin: 0 auto;
        }

        .cart-items-section {
            background: white;
        }

        .checkout-section {
            background: #f8f9fa;
            padding: 1.5rem;
            border: 1px solid #dee2e6;
            height: fit-content;
        }

        .section-header {
            padding: 1.5rem 2rem;
            border-bottom: 1px solid var(--border-color);
        }

        .section-title {
            font-size: 1.3rem;
            font-weight: 600;
            color: var(--dark-text);
            margin: 0;
        }

        .section-content {
            padding: 2rem;
        }

        .section-title {
            font-size: 1.5rem;
            font-weight: 700;
            color: var(--dark-text);
            margin-bottom: 1rem;
        }

        .cart-count {
            font-size: 1.1rem;
            color: var(--light-text);
            font-weight: normal;
        }

        .cart-items-list {
            display: flex;
            flex-direction: column;
            gap: 1.5rem;
        }

        .cart-item {
            display: flex;
            align-items: flex-start;
            gap: 1rem;
            padding: 1.5rem 0;
            border-bottom: 1px solid #e5e7eb;
        }

        .cart-item:last-child {
            border-bottom: none;
        }

        .item-thumbnail {
            width: 80px;
            height: 60px;
            object-fit: cover;
            flex-shrink: 0;
        }

        .item-info {
            flex: 1;
        }

        .item-title {
            font-size: 1rem;
            font-weight: 600;
            color: var(--dark-text);
            margin-bottom: 0.4rem;
            line-height: 1.4;
        }

        .item-instructor {
            font-size: 0.85rem;
            color: var(--light-text);
            margin-bottom: 0.4rem;
        }

        .item-description {
            font-size: 0.8rem;
            color: #6b7280;
            line-height: 1.4;
            margin: 6px 0;
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
            overflow: hidden;
        }

        .item-level {
            display: inline-block;
            font-size: 0.7rem;
            font-weight: 500;
            padding: 0.25rem 0.6rem;
            border-radius: 4px;
            background: var(--primary-orange);
            color: white;
        }

        .item-actions {
            display: flex;
            align-items: center;
            gap: 1rem;
            margin-left: auto;
        }

        .item-price {
            font-size: 1.1rem;
            font-weight: 700;
            color: var(--primary-blue);
        }

        .remove-btn {
            color: var(--primary-blue);
            background: none;
            border: none;
            font-size: 0.9rem;
            cursor: pointer;
            text-decoration: underline;
            padding: 0;
        }

        .remove-btn:hover {
            color: var(--danger-color);
        }

        .checkout-summary {
            margin-bottom: 1.5rem;
        }

        .summary-title {
            font-size: 1.1rem;
            font-weight: 700;
            color: var(--dark-text);
            margin-bottom: 1rem;
            text-align: right;
        }

        .summary-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 0.75rem;
            font-size: 0.95rem;
        }

        .summary-total {
            font-size: 1.4rem;
            font-weight: 700;
            color: var(--dark-text);
            border-top: 1px solid #e5e7eb;
            padding-top: 1rem;
            margin-top: 1rem;
        }

        .checkout-btn {
            width: 100%;
            background: var(--primary-blue);
            color: white;
            border: none;
            padding: 0.875rem;
            font-size: 1rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.2s ease;
        }

        .checkout-btn:hover {
            background: #1d4ed8;
        }

        .checkout-note {
            font-size: 0.875rem;
            color: var(--light-text);
            text-align: center;
            margin: 0.75rem 0 0;
        }

        .auth-required-section {
            margin: 1.5rem 0;
            padding: 1.5rem;
            background: #f8f9fa;
            border-radius: 8px;
            border: 1px solid #e5e7eb;
        }

        .auth-note {
            font-size: 0.9rem;
            color: var(--text-dark);
            text-align: center;
            margin: 0 0 1.5rem;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.5rem;
        }

        .auth-note .fas {
            color: var(--primary-blue);
        }

        .auth-buttons-group {
            display: flex;
            flex-direction: column;
            gap: 0.75rem;
        }

        .auth-btn {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.5rem;
            padding: 0.75rem 1rem;
            border-radius: 6px;
            text-decoration: none;
            font-weight: 600;
            font-size: 0.875rem;
            transition: all 0.2s ease;
            border: 2px solid transparent;
        }

        .login-btn {
            background: var(--primary-blue);
            color: white;
        }

        .login-btn:hover {
            background: #1d4ed8;
            color: white;
            text-decoration: none;
        }

        .register-btn {
            background: var(--primary-orange);
            color: white;
        }

        .register-btn:hover {
            background: #d97706;
            color: white;
            text-decoration: none;
        }

        .auth-prompt {
            text-align: center;
            padding: 2rem;
            background: #f8f9fa;
            margin-top: 1.5rem;
            border: 1px solid #dee2e6;
        }

        .auth-icon {
            font-size: 3rem;
            color: var(--primary-blue);
            margin-bottom: 1.5rem;
        }

        .auth-title {
            font-size: 1.4rem;
            font-weight: 700;
            color: var(--dark-text);
            margin-bottom: 1rem;
        }

        .auth-description {
            font-size: 0.95rem;
            color: var(--light-text);
            margin-bottom: 1.5rem;
            line-height: 1.6;
        }

        .auth-buttons {
            display: flex;
            gap: 1rem;
            justify-content: center;
            flex-wrap: wrap;
        }

        .btn-auth {
            padding: 0.7rem 1.5rem;
            border-radius: 12px;
            font-weight: 600;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            transition: all 0.3s ease;
            font-size: 0.9rem;
        }

        .btn-primary {
            background: var(--primary-blue);
            color: white;
            border: 2px solid var(--primary-blue);
        }

        .btn-primary:hover {
            background: #1d4ed8;
            color: white;
        }

        .btn-secondary {
            background: var(--primary-orange);
            color: white;
            border: 2px solid var(--primary-orange);
        }

        .btn-secondary:hover {
            background: #d97706;
            color: white;
        }

        .empty-state {
            text-align: center;
            padding: 4rem 1rem;
            color: var(--light-text);
        }

        .empty-icon {
            font-size: 4rem;
            color: var(--light-text);
            margin-bottom: 1rem;
            opacity: 0.5;
        }

        .empty-title {
            font-size: 1.5rem;
            font-weight: 600;
            margin-bottom: 1rem;
            color: var(--dark-text);
        }

        .empty-description {
            margin-bottom: 2rem;
            line-height: 1.6;
        }

        .browse-btn {
            background: var(--primary-orange);
            color: white;
            padding: 0.8rem 2rem;
            text-decoration: none;
            font-weight: 600;
            transition: all 0.2s ease;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }

        .browse-btn:hover {
            background: #d97706;
            color: white;
        }



        .empty-cart {
            text-align: center;
            padding: 4rem 2rem;
        }

        .empty-icon {
            font-size: 5rem;
            color: var(--light-text);
            margin-bottom: 1.5rem;
        }

        .empty-title {
            font-size: 1.5rem;
            font-weight: 600;
            color: var(--dark-text);
            margin-bottom: 1rem;
        }

        .empty-description {
            color: var(--light-text);
            margin-bottom: 2rem;
        }

        .browse-btn {
            background: var(--primary-orange);
            color: white;
            padding: 0.875rem 2rem;
            border-radius: 8px;
            text-decoration: none;
            font-weight: 600;
            transition: all 0.2s ease;
        }

        .browse-btn:hover {
            background: #d97706;
            transform: translateY(-1px);
            color: white;
        }

        @media (max-width: 968px) {
            .cart-content {
                grid-template-columns: 1fr;
                gap: 1.5rem;
            }

            .checkout-section {
                position: static;
                order: -1;
            }

            .modern-cart-container {
                padding: 1rem;
            }
        }

        @media (max-width: 768px) {
            .cart-header h1 {
                font-size: 1.8rem;
            }

            .cart-items-section,
            .tips-section {
                padding: 1.5rem;
            }

            .cart-item {
                flex-direction: column;
                align-items: flex-start;
            }

            .item-actions {
                width: 100%;
                justify-content: space-between;
                margin-top: 1rem;
            }

            .auth-buttons {
                flex-direction: column;
                align-items: center;
            }

            .btn-auth {
                width: 100%;
                max-width: 280px;
                justify-content: center;
            }
        }
    </style>
</head>

<body>
    <!-- Include Navbar -->
    <?php include_once 'includes/navbar.php'; ?>

    <!-- Page Header -->
    <div class="page-header">
        <h1>
            <i class="fas fa-shopping-bag"></i>
            Your Shopping Cart
        </h1>
        <p>Review your selected courses and complete your purchase</p>
    </div>

    <!-- Modern Cart Container -->
    <div class="modern-cart-container">
        <div class="cart-main-wrapper">
            <!-- Cart Content -->
            <div class="cart-content">
                <!-- Cart Items Section -->
                <div class="cart-items-section">
                    <h1 class="section-title">
                        Shopping Cart <span class="cart-count">(<span id="cart-item-count">0</span> Course<span id="course-plural">s</span>)</span>
                    </h1>
                    
                    <!-- Cart Items Container -->
                    <div id="cart-items-container" class="cart-items-list">
                        <!-- Cart items will be loaded here by JavaScript -->
                    </div>

                    <!-- Empty Cart State -->
                    <div id="empty-cart-state" class="empty-state" style="display: none;">
                        <i class="fas fa-shopping-cart empty-icon"></i>
                        <h3 class="empty-title">Your cart is empty</h3>
                        <p class="empty-description">
                            Discover amazing courses and start your learning journey today!
                        </p>
                        <a href="/courses.php" class="browse-btn">
                            <i class="fas fa-search"></i>
                            Browse Courses
                        </a>
                    </div>


                </div>

                <!-- Checkout Section -->
                <div class="checkout-section">
                    <div class="checkout-summary">
                        <h3 class="summary-title">Total</h3>
                        <div class="summary-row summary-total">
                            <span>Total:</span>
                            <span id="cart-total">R0.00</span>
                        </div>
                    </div>
                    
                    <!-- Authentication Required -->
                    <div class="auth-required-section">
                        <p class="auth-note">
                            <i class="fas fa-lock"></i>
                            Please log in or create an account to complete your purchase
                        </p>
                        
                        <div class="auth-buttons-group">
                            <a href="#" class="auth-btn login-btn" onclick="this.href = prepareCartTransfer('/auth/login.php?redirect_to=student/checkout.php')">
                                <i class="fas fa-sign-in-alt"></i>
                                Log In
                            </a>
                            <a href="#" class="auth-btn register-btn" onclick="this.href = prepareCartTransfer('/auth/register.php?redirect_to=student/checkout.php')">
                                <i class="fas fa-user-plus"></i>
                                Create Account
                            </a>
                        </div>
                    </div>
                    
                    <p class="checkout-note">
                        You won't be charged yet
                    </p>
                </div>
            </div>
        </div>
    </div>

    <!-- Footer -->
    <?php include_once 'includes/footer.php'; ?>

    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    
    <!-- Guest Cart System -->
    <script src="/js/guest-cart-system.js"></script>

    <!-- Cart Page JavaScript -->
    <script>
        class GuestCartPage {
            constructor() {
                this.init();
            }

            init() {
                this.loadCartItems();
                this.bindEvents();
                this.setupCheckoutButton();
            }

            loadCartItems() {
                console.log('Loading cart items...');
                const cart = this.getGuestCart();
                console.log('Cart items loaded:', cart);
                
                const container = document.getElementById('cart-items-container');
                const countElement = document.getElementById('cart-item-count');
                const coursePlural = document.getElementById('course-plural');
                const emptyState = document.getElementById('empty-cart-state');
                const checkoutSection = document.querySelector('.checkout-section');

                console.log('Container element:', container);
                console.log('Cart length:', cart.length);

                if (cart.length === 0) {
                    console.log('Cart is empty, showing empty state');
                    if (emptyState) emptyState.style.display = 'block';
                    if (checkoutSection) checkoutSection.style.display = 'none';
                    if (countElement) countElement.textContent = '0';
                    return;
                }

                // Show cart items
                container.innerHTML = cart.map((item, index) => `
                    <div class="cart-item" data-item-index="${index}">
                        <img src="${item.thumbnail}" alt="${item.title}" class="item-thumbnail">
                        <div class="item-info">
                            <h4 class="item-title">${item.title}</h4>
                            <p class="item-instructor">By ${item.instructor}</p>
                            ${item.description ? `<div class="item-description">${item.description}</div>` : ''}
                            <div style="margin-top: 0.5rem;">
                                <span class="item-level">${item.level}</span>
                            </div>
                        </div>
                        <div class="item-actions">
                            <div class="item-price">R${parseFloat(item.price || 0).toFixed(2)}</div>
                            <button class="remove-btn" onclick="guestCartPage.removeItem(${index})">
                                Remove
                            </button>
                        </div>
                    </div>
                `).join('');

                // Update counts
                if (countElement) {
                    countElement.textContent = cart.length;
                } else {
                    console.log('countElement not found!');
                }
                
                if (coursePlural) {
                    coursePlural.textContent = cart.length === 1 ? '' : 's';
                } else {
                    console.log('coursePlural not found!');
                }
                
                // Call updateTotal and handle any errors
                try {
                    this.updateTotal(cart);
                } catch (error) {
                    console.log('Error in updateTotal:', error);
                    const debugElement = document.getElementById('debug-cart-data');
                    if (debugElement) {
                        debugElement.innerHTML = `ERROR in updateTotal: ${error.message}`;
                    }
                }

                // Show sections
                if (emptyState) emptyState.style.display = 'none';
                if (checkoutSection) checkoutSection.style.display = 'block';
            }

            updateTotal(cart) {
                let total = 0;
                
                cart.forEach(item => {
                    // Handle different price formats
                    let price = 0;
                    if (item.price) {
                        if (typeof item.price === 'string') {
                            // Remove currency symbols and parse
                            const cleanPrice = item.price.replace(/[R,\s]/g, '');
                            price = parseFloat(cleanPrice) || 0;
                        } else {
                            price = parseFloat(item.price) || 0;
                        }
                    }
                    total += price;
                });
                
                // Update total display
                const totalElement = document.getElementById('cart-total');
                if (totalElement) {
                    totalElement.textContent = `R${total.toFixed(2)}`;
                }
            }

            removeItem(index) {
                let cart = this.getGuestCart();
                const removedItem = cart[index];
                
                cart.splice(index, 1);
                localStorage.setItem('panadite_guest_cart', JSON.stringify(cart));
                
                this.loadCartItems();
                
                // Update cart counter in navbar
                if (window.panaditeCart) {
                    window.panaditeCart.updateCartCounter();
                }
                
                // Show success message
                this.showRemoveMessage(removedItem.title);

                // Show notification
                this.showNotification(`"${removedItem.title}" removed from cart`, 'info');
            }

            showRemoveMessage(itemTitle) {
                // You can implement a toast notification here if needed
                console.log(`"${itemTitle}" removed from cart`);
            }

            setupCheckoutButton() {
                // No checkout button needed since we have auth buttons
            }

            getGuestCart() {
                try {
                    const cart = localStorage.getItem('panadite_guest_cart');
                    console.log('Raw cart from localStorage:', cart);
                    const parsedCart = cart ? JSON.parse(cart) : [];
                    console.log('Parsed cart:', parsedCart);
                    return parsedCart;
                } catch (error) {
                    console.error('Error reading cart:', error);
                    return [];
                }
            }

            bindEvents() {
                // Handle remove button clicks
                document.addEventListener('click', (e) => {
                    if (e.target.classList.contains('remove-btn')) {
                        const index = parseInt(e.target.getAttribute('data-index'));
                        if (!isNaN(index)) {
                            this.removeItem(index);
                        }
                    }
                });
            }

            showNotification(message, type = 'success') {
                // Simple notification system
                const notification = document.createElement('div');
                notification.className = `alert alert-${type === 'success' ? 'success' : 'info'} alert-dismissible fade show`;
                notification.style.cssText = 'position: fixed; top: 20px; right: 20px; z-index: 1050; max-width: 300px;';
                notification.innerHTML = `
                    ${message}
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                `;
                document.body.appendChild(notification);
                
                // Auto remove after 3 seconds
                setTimeout(() => {
                    if (notification.parentNode) {
                        notification.parentNode.removeChild(notification);
                    }
                }, 3000);
            }
        }

        // Guest Cart Transfer Function
        function prepareCartTransfer(authUrl) {
            const cart = JSON.parse(localStorage.getItem('panadite_guest_cart') || '[]');
            
            if (cart.length > 0) {
                // Store cart data in session storage for transfer
                sessionStorage.setItem('guest_cart_transfer', JSON.stringify(cart));
                console.log('Cart prepared for transfer:', cart);
            }
            
            return authUrl;
        }

        // Initialize when DOM is ready
        document.addEventListener('DOMContentLoaded', () => {
            window.guestCartPage = new GuestCartPage();
        });
    </script>
</body>
</html>
