<?php
/**
 * Panadite Academy - Forgot Password System
 * Complete password reset functionality with email verification
 */

// Start session and include necessary files
session_start();
require_once __DIR__ . '/../includes/site_config.php';
require_once __DIR__ . '/../includes/db_connection.php';
require_once __DIR__ . '/../config/email_verification_service.php';

// Initialize variables
$error = '';
$success = '';
$email_sent = false;

// Process form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email = trim($_POST['email']);
    
    if (empty($email)) {
        $error = 'Please enter your email address.';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = 'Please enter a valid email address.';
    } else {
        // Check if user exists in database
        $stmt = $conn->prepare("SELECT user_id, first_name, last_name, email FROM users WHERE email = ?");
        $stmt->bind_param("s", $email);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            $user = $result->fetch_assoc();
            
            // Generate secure reset token
            $reset_token = bin2hex(random_bytes(32));
            $expires_at = date('Y-m-d H:i:s', strtotime('+1 hour'));
            
            // Store reset token in database
            $token_stmt = $conn->prepare("INSERT INTO password_reset_tokens (user_id, token, expires_at) VALUES (?, ?, ?) ON DUPLICATE KEY UPDATE token = ?, expires_at = ?");
            $token_stmt->bind_param("issss", $user['user_id'], $reset_token, $expires_at, $reset_token, $expires_at);
            
            if ($token_stmt->execute()) {
                // Send password reset email
                $reset_link = $site_url . '/auth/reset-password.php?token=' . $reset_token;
                $user_name = $user['first_name'] . ' ' . $user['last_name'];
                
                $subject = "🔑 Password Reset Request - Panadite Academy";
                $message = "
                <div style='max-width:600px; margin:0 auto; font-family:Arial,sans-serif;'>
                    <div style='background:linear-gradient(135deg, #276BE4, #1e5bcc); color:white; padding:30px; text-align:center; border-radius:10px 10px 0 0;'>
                        <h1 style='margin:0; font-size:28px;'>🔑 Password Reset</h1>
                        <p style='margin:10px 0 0 0; opacity:0.9;'>Panadite Academy</p>
                    </div>
                    <div style='padding:30px; background:white; border:1px solid #e1e5e9; border-top:none; border-radius:0 0 10px 10px;'>
                        <h2 style='color:#333; margin-top:0;'>Hi " . htmlspecialchars($user_name) . "! 👋</h2>
                        <p style='color:#666; line-height:1.6;'>We received a request to reset the password for your Panadite Academy account.</p>
                        
                        <div style='background:#f8f9fa; padding:20px; border-radius:8px; margin:25px 0; border-left:4px solid #276BE4;'>
                            <h3 style='color:#333; margin-top:0;'>Reset Your Password</h3>
                            <p style='color:#666; margin-bottom:20px;'>Click the button below to reset your password. This link will expire in 1 hour for security.</p>
                            <div style='text-align:center;'>
                                <a href='" . $reset_link . "' 
                                   style='background:linear-gradient(135deg, #276BE4, #1e5bcc); color:white; text-decoration:none; 
                                          padding:15px 30px; border-radius:8px; font-weight:bold; display:inline-block; 
                                          font-size:16px; transition:all 0.3s ease;'>
                                    🔑 Reset My Password
                                </a>
                            </div>
                        </div>
                        
                        <div style='background:#fff3cd; padding:15px; border-radius:8px; border:1px solid #ffeaa7; margin:20px 0;'>
                            <h4 style='color:#856404; margin-top:0;'>⚠️ Important Security Information</h4>
                            <ul style='color:#856404; margin-bottom:0;'>
                                <li>This link expires in 1 hour</li>
                                <li>If you didn't request this, please ignore this email</li>
                                <li>Never share this link with anyone</li>
                            </ul>
                        </div>
                        
                        <p style='color:#999; font-size:14px; margin-top:30px;'>
                            If the button doesn't work, copy and paste this link into your browser:<br>
                            <a href='" . $reset_link . "' style='color:#276BE4; word-break:break-all;'>" . $reset_link . "</a>
                        </p>
                        
                        <hr style='border:none; border-top:1px solid #eee; margin:30px 0;'>
                        <p style='color:#999; font-size:12px; text-align:center; margin:0;'>
                            © " . date('Y') . " Panadite Academy. All rights reserved.<br>
                            296 Pretorius Street, Pretoria Central, 0001, South Africa
                        </p>
                    </div>
                </div>";
                
                // Send email
                if (function_exists('sendEmailNow')) {
                    $email_result = sendEmailNow($email, $subject, $message);
                    if ($email_result) {
                        $success = "Password reset instructions have been sent to your email address. Please check your inbox and follow the instructions.";
                        $email_sent = true;
                    } else {
                        $error = "Failed to send password reset email. Please try again later.";
                    }
                } else {
                    $error = "Email service is currently unavailable. Please try again later.";
                }
            } else {
                $error = "Failed to generate reset token. Please try again later.";
            }
        } else {
            // For security, don't reveal if email exists or not
            $success = "If an account with that email address exists, we have sent you password reset instructions.";
            $email_sent = true;
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Forgot Password - Panadite Academy</title>
    <meta name="description" content="Reset your Panadite Academy password. Enter your email address and we'll send you a secure link to reset your password.">
    
    <!-- Bootstrap 5.2.3 -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <!-- Animate.css -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css">
    
    <style>
        :root {
            --primary-color: #276BE4;
            --secondary-color: #FFA915;
            --success-color: #28a745;
            --danger-color: #dc3545;
            --light-bg: #f8f9fa;
            --white: #ffffff;
            --text-dark: #333333;
            --text-muted: #6c757d;
            --border-color: #e1e5e9;
        }

        body {
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            min-height: 100vh;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }

        .forgot-password-container {
            max-width: 500px;
            margin: 50px auto;
            background: var(--white);
            border-radius: 20px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
            overflow: hidden;
            position: relative;
        }

        .forgot-password-container::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: linear-gradient(90deg, var(--primary-color), var(--secondary-color));
        }

        .logo-container {
            text-align: center;
            padding: 40px 40px 20px 40px;
        }

        .logo-container img {
            max-width: 120px;
            height: auto;
            margin-bottom: 15px;
            border-radius: 50%;
            border: 3px solid var(--primary-color);
            padding: 5px;
        }

        .tagline {
            color: var(--secondary-color);
            font-style: italic;
            font-size: 16px;
            font-weight: 500;
            margin-bottom: 0;
        }

        .forgot-password-form {
            padding: 20px 40px 40px 40px;
        }

        .section-title {
            text-align: center;
            margin-bottom: 30px;
        }

        .section-title h2 {
            color: var(--text-dark);
            font-weight: 700;
            font-size: 28px;
            margin-bottom: 10px;
        }

        .section-title p {
            color: var(--text-muted);
            font-size: 16px;
            margin-bottom: 0;
        }

        .form-control {
            border: 2px solid var(--border-color);
            border-radius: 12px;
            padding: 14px 16px;
            font-size: 16px;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            background-color: #fafbfc;
        }

        .form-control:focus {
            border-color: var(--primary-color);
            background-color: rgba(39, 107, 228, 0.05);
            box-shadow: 0 0 0 0.2rem rgba(39, 107, 228, 0.25);
            transform: translateY(-2px);
        }

        .input-group-text {
            background: linear-gradient(135deg, var(--primary-color), #1e5bcc);
            border: none;
            color: white;
            border-radius: 12px 0 0 12px;
            padding: 14px 16px;
        }

        .input-group .form-control {
            border-left: none;
            border-radius: 0 12px 12px 0;
        }

        .btn-reset {
            background: linear-gradient(135deg, var(--primary-color), #1e5bcc);
            border: none;
            border-radius: 12px;
            color: white;
            font-weight: 600;
            padding: 16px 24px;
            font-size: 17px;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            width: 100%;
            height: 54px;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
        }

        .btn-reset:hover {
            background: linear-gradient(135deg, #1e5bcc, #1752b8);
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(39, 107, 228, 0.3);
        }

        .alert {
            border-radius: 12px;
            border: none;
            padding: 16px 20px;
            margin-bottom: 25px;
            font-weight: 500;
        }

        .alert-success {
            background: linear-gradient(135deg, #d4edda, #c3e6cb);
            color: #155724;
            border-left: 4px solid var(--success-color);
        }

        .alert-danger {
            background: linear-gradient(135deg, #f8d7da, #f5c6cb);
            color: #721c24;
            border-left: 4px solid var(--danger-color);
        }

        .forgot-password-links {
            text-align: center;
            margin-top: 25px;
            padding-top: 20px;
            border-top: 1px solid var(--border-color);
        }

        .forgot-password-links a {
            color: var(--primary-color);
            text-decoration: none;
            font-weight: 500;
            font-size: 15px;
            transition: all 0.3s ease;
            margin: 0 10px;
        }

        .forgot-password-links a:hover {
            color: #1e5bcc;
            text-decoration: underline;
        }

        .success-icon {
            font-size: 64px;
            color: var(--success-color);
            margin-bottom: 20px;
            animation: bounce 1s ease-out;
        }

        @keyframes bounce {
            0%, 20%, 53%, 80%, 100% {
                animation-timing-function: cubic-bezier(0.215, 0.61, 0.355, 1);
                transform: translate3d(0, 0, 0);
            }
            40%, 43% {
                animation-timing-function: cubic-bezier(0.755, 0.05, 0.855, 0.06);
                transform: translate3d(0, -30px, 0);
            }
            70% {
                animation-timing-function: cubic-bezier(0.755, 0.05, 0.855, 0.06);
                transform: translate3d(0, -15px, 0);
            }
            90% {
                transform: translate3d(0, -4px, 0);
            }
        }

        @media (max-width: 768px) {
            .forgot-password-container {
                margin: 20px;
                border-radius: 15px;
            }
            
            .forgot-password-form {
                padding: 20px 30px;
            }
            
            .logo-container {
                padding: 30px 30px 15px 30px;
            }
        }
    </style>
</head>
<body>
    <!-- Include navbar -->
    <?php include '../includes/navbar.php'; ?>
    
    <div class="container">
        <div class="forgot-password-container animate__animated animate__fadeIn">
            <div class="logo-container">
                <img src="/assets/images/logo.jpeg" alt="Panadite Academy Logo" class="animate__animated animate__zoomIn">
                <p class="tagline animate__animated animate__fadeIn animate__delay-1s">Get ahead of the curve</p>
            </div>
            
            <div class="forgot-password-form">
                <?php if ($email_sent): ?>
                    <!-- Success State -->
                    <div class="text-center animate__animated animate__fadeIn">
                        <div class="success-icon">
                            <i class="fas fa-envelope-circle-check"></i>
                        </div>
                        <div class="section-title">
                            <h2>Email Sent! 📧</h2>
                            <p>Check your inbox for password reset instructions</p>
                        </div>
                        
                        <?php if ($success): ?>
                        <div class="alert alert-success animate__animated animate__slideInDown">
                            <i class="fas fa-check-circle me-2"></i>
                            <?php echo $success; ?>
                        </div>
                        <?php endif; ?>
                        
                        <div class="forgot-password-links animate__animated animate__fadeIn animate__delay-1s">
                            <a href="login.php">
                                <i class="fas fa-arrow-left me-1"></i>
                                Back to Login
                            </a>
                            <a href="/index.php">
                                <i class="fas fa-home me-1"></i>
                                Home
                            </a>
                        </div>
                    </div>
                <?php else: ?>
                    <!-- Request Form -->
                    <div class="section-title animate__animated animate__fadeIn">
                        <h2>Forgot Password? 🔐</h2>
                        <p>Enter your email address and we'll send you a reset link</p>
                    </div>
                    
                    <?php if ($error): ?>
                    <div class="alert alert-danger animate__animated animate__shake">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        <?php echo $error; ?>
                    </div>
                    <?php endif; ?>
                    
                    <form method="POST" action="" class="animate__animated animate__fadeIn animate__delay-1s">
                        <div class="mb-4">
                            <label for="email" class="form-label">Email Address</label>
                            <div class="input-group">
                                <span class="input-group-text">
                                    <i class="fas fa-envelope"></i>
                                </span>
                                <input type="email" 
                                       class="form-control" 
                                       id="email" 
                                       name="email" 
                                       required 
                                       placeholder="Enter your registered email address"
                                       value="<?php echo isset($_POST['email']) ? htmlspecialchars($_POST['email']) : ''; ?>">
                            </div>
                            <div class="form-text">
                                <i class="fas fa-info-circle me-1"></i>
                                We'll send password reset instructions to this email address
                            </div>
                        </div>
                        
                        <button type="submit" class="btn btn-reset">
                            <i class="fas fa-paper-plane"></i>
                            Send Reset Instructions
                        </button>
                    </form>
                    
                    <div class="forgot-password-links animate__animated animate__fadeIn animate__delay-1s">
                        <a href="login.php">
                            <i class="fas fa-arrow-left me-1"></i>
                            Back to Login
                        </a>
                        <a href="register.php">
                            <i class="fas fa-user-plus me-1"></i>
                            Create Account
                        </a>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <!-- Include footer -->
    <?php include '../includes/footer.php'; ?>
    
    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const emailInput = document.getElementById('email');
            
            if (emailInput) {
                emailInput.addEventListener('focus', function() {
                    this.closest('.input-group').style.transform = 'translateY(-3px)';
                });
                
                emailInput.addEventListener('blur', function() {
                    this.closest('.input-group').style.transform = 'translateY(0)';
                });
                
                // Auto-focus email input if not on success page
                if (!<?php echo $email_sent ? 'true' : 'false'; ?>) {
                    setTimeout(() => emailInput.focus(), 500);
                }
            }
        });
    </script>
</body>
</html>
