<?php
/**
 * Authentication Functions
 * Panadite Academy
 */

// Use a custom error handler to log critical errors
set_error_handler(function($errno, $errstr, $errfile, $errline) {
    if (error_reporting() === 0) {
        return false;
    }
    error_log("Error $errno: $errstr in $errfile on line $errline");
});

// Function to check if tables exist
if (!function_exists('tableExists')) {
    function tableExists($conn, $tableName) {
        if (!$conn) return false;
        $result = $conn->query("SHOW TABLES LIKE '{$tableName}'");
        return ($result && $result->num_rows > 0);
    }
}

// Use centralized session handling instead of managing it here
require_once __DIR__ . '/../includes/session_start.php';

// Include configuration after session start
require_once __DIR__ . '/../includes/site_config.php';
require_once __DIR__ . '/../includes/config.php';

/**
 * Require user to be logged in and have specific role
 */
function requireRole($requiredRole) {
    // Check if user is logged in
    if (!isLoggedIn()) {
        header('Location: /auth/login.php?redirect=' . urlencode($_SERVER['REQUEST_URI']));
        exit();
    }
    
    // Check role - fix session variable detection
    $userRole = $_SESSION['user_role'] ?? $_SESSION['role'] ?? '';
    
    // Debug logging for role detection
    error_log('🔐 ROLE CHECK - Required: ' . $requiredRole . ', User: ' . $userRole);
    error_log('🔐 SESSION DATA: ' . print_r($_SESSION, true));
    
    if ($userRole !== $requiredRole) {
        // 🚨 SECURITY FIX: Redirect to correct LIVE SERVER paths
        switch ($userRole) {
            case 'admin':
                header('Location: /admin/dashboard.php');
                break;
            case 'teacher':
                header('Location: /teacher/dashboard.php');
                break;
            case 'student':
                header('Location: /student/dashboard.php');
                break;
            default:
                // If no valid role, redirect to login
                header('Location: /auth/login.php');
        }
        exit();
    }
}

/**
 * Register a new user
 */
function registerUser($username, $email, $password, $firstName, $lastName, $role) {
    global $conn;
    
    // Validate input
    if (empty($username) || empty($email) || empty($password) || empty($firstName) || empty($lastName)) {
        return array('success' => false, 'message' => 'All fields are required');
    }
    
    // Check if database connection is available
    if ($conn === null || isset($GLOBALS['db_connection_failed']) && $GLOBALS['db_connection_failed']) {
        return array('success' => false, 'message' => 'Database connection is not available. Please check PHP MySQL extension installation.');
    }
    
    // Validate email
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        return array('success' => false, 'message' => 'Invalid email format');
    }
    
    // Check if username already exists
    try {
        $stmt = $conn->prepare("SELECT user_id FROM users WHERE username = ?");
        $stmt->bind_param("s", $username);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result->num_rows > 0) {
            return array('success' => false, 'message' => 'Username already exists');
        }
    } catch (Exception $e) {
        error_log('Registration error: ' . $e->getMessage());
        return array('success' => false, 'message' => 'Registration failed due to technical issues. Please try again later.');
    }
    
    // Check if email already exists
    try {
        $stmt = $conn->prepare("SELECT user_id FROM users WHERE email = ?");
        $stmt->bind_param("s", $email);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result->num_rows > 0) {
            return array('success' => false, 'message' => 'Email already exists');
        }
    } catch (Exception $e) {
        error_log('Registration email check error: ' . $e->getMessage());
        return array('success' => false, 'message' => 'Registration failed due to technical issues. Please try again later.');
    }
    
    // Hash password
    $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
    
    // Insert user
    try {
        $stmt = $conn->prepare("INSERT INTO users (username, email, password, first_name, last_name, user_role) VALUES (?, ?, ?, ?, ?, ?)");
        $stmt->bind_param("ssssss", $username, $email, $hashedPassword, $firstName, $lastName, $role);
        
        if ($stmt->execute()) {
            $userId = $stmt->insert_id;
            
            try {
                // Create role-specific profile
                if ($role === 'student') {
                    // Check if student_profiles table exists
                    if (tableExists($conn, 'student_profiles')) {
                        $profileStmt = $conn->prepare("INSERT INTO student_profiles (student_id) VALUES (?)");
                        
                        // Check if prepare was successful
                        if ($profileStmt !== false) {
                            $profileStmt->bind_param("i", $userId);
                            $profileStmt->execute();
                        } else {
                            error_log("Failed to prepare student profile statement: " . $conn->error);
                        }
                    } else {
                        error_log("Table student_profiles does not exist in the database");
                    }
                } else if ($role === 'teacher') {
                    // Get facilitator-specific fields from POST
                    $courseTitle = isset($_POST['course_title']) ? $_POST['course_title'] : '';
                    $courseDescription = isset($_POST['course_description']) ? $_POST['course_description'] : '';
                    $qualifications = isset($_POST['qualifications']) ? $_POST['qualifications'] : '';
                    
                    // Set initial status to 'pending' for admin approval
                    $status = 'pending';
                    
                    // CRITICAL FIX: Create record in 'teachers' table for social links support
                    if (tableExists($conn, 'teachers')) {
                        // Insert teacher record with basic info and empty social links
                        $teacherStmt = $conn->prepare("INSERT INTO teachers (user_id, first_name, last_name, bio, expertise, created_at, updated_at) VALUES (?, ?, ?, ?, ?, NOW(), NOW())");
                        
                        if ($teacherStmt !== false) {
                            $defaultBio = $courseDescription ?: 'Welcome to my teaching profile! I am passionate about education and helping students achieve their learning goals.';
                            $defaultExpertise = $courseTitle ?: 'Education Specialist';
                            
                            $teacherStmt->bind_param("issss", $userId, $firstName, $lastName, $defaultBio, $defaultExpertise);
                            $teacherStmt->execute();
                        } else {
                            error_log("Failed to prepare teacher statement: " . $conn->error);
                        }
                    } else {
                        error_log("Table teachers does not exist in the database");
                    }
                    
                    // Also check if teacher_profiles table exists (legacy support)
                    if (tableExists($conn, 'teacher_profiles')) {
                        // Insert teacher profile with course information
                        $profileStmt = $conn->prepare("INSERT INTO teacher_profiles (teacher_id, bio, qualifications, expertise, status) VALUES (?, ?, ?, ?, ?)");
                        
                        // Check if prepare was successful
                        if ($profileStmt !== false) {
                            $profileStmt->bind_param("issss", $userId, $courseDescription, $qualifications, $courseTitle, $status);
                            $profileStmt->execute();
                        } else {
                            error_log("Failed to prepare teacher profile statement: " . $conn->error);
                        }
                    } else {
                        error_log("Table teacher_profiles does not exist in the database");
                    }
                } else if ($role === 'admin') {
                    // Check if admin_profiles table exists
                    if (tableExists($conn, 'admin_profiles')) {
                        $profileStmt = $conn->prepare("INSERT INTO admin_profiles (admin_id, permissions) VALUES (?, 'limited')");
                        
                        // Check if prepare was successful
                        if ($profileStmt !== false) {
                            $profileStmt->bind_param("i", $userId);
                            $profileStmt->execute();
                        } else {
                            error_log("Failed to prepare admin profile statement: " . $conn->error);
                        }
                    } else {
                        error_log("Table admin_profiles does not exist in the database");
                    }
                }
            } catch (Exception $e) {
                // Even if profile creation fails, the user account was created
                error_log('Profile creation error: ' . $e->getMessage());
            }
            
            return array('success' => true, 'user_id' => $userId, 'message' => 'Registration successful');
        } else {
            return array('success' => false, 'message' => 'Registration failed: ' . $stmt->error);
        }
    } catch (Exception $e) {
        error_log('Registration error: ' . $e->getMessage());
        return array('success' => false, 'message' => 'Registration failed due to technical issues. Please try again later.');
    }
}

/**
 * Login a user
 */
function loginUser($username, $password) {
    global $conn; // Use the global database connection
    
    // Validate input
    if (empty($username) || empty($password)) {
        return array('success' => false, 'message' => 'Username and password are required');
    }
    
    // EMERGENCY LOGINS for when MySQL extensions aren't available
    // This allows access even when the database connection doesn't work
    $emergency_users = [
        // Admin emergency login
        'admin' => [
            'password' => 'admin123',
            'user_id' => 1,
            'first_name' => 'Admin',
            'last_name' => 'User',
            'user_role' => 'admin'
        ],
        // Teacher emergency login
        'teacher' => [
            'password' => 'teacher123',
            'user_id' => 2,
            'first_name' => 'Teacher',
            'last_name' => 'User',
            'user_role' => 'teacher'
        ],
        // Student emergency login
        'student' => [
            'password' => 'student123',
            'user_id' => 3,
            'first_name' => 'Student',
            'last_name' => 'User',
            'user_role' => 'student'
        ]
    ];
    
    // Check if username exists in emergency users
    if (isset($emergency_users[$username]) && $emergency_users[$username]['password'] === $password) {
        $user = $emergency_users[$username];
        
        // Set session variables
        $_SESSION['user_id'] = $user['user_id'];
        $_SESSION['username'] = $username;
        $_SESSION['first_name'] = $user['first_name'];
        $_SESSION['last_name'] = $user['last_name'];
        $_SESSION['user_role'] = $user['user_role'];
        $_SESSION['logged_in'] = true;
        $_SESSION['emergency_login'] = true;
        
        return array('success' => true, 'message' => 'Emergency '.$user['user_role'].' login successful', 'user' => $user);
    }
    
    // Check if we have a valid database connection
    if (!$conn) {
        // No database connection available, offer emergency login
        return array(
            'success' => false, 
            'message' => 'Database connection failed. Use emergency login with one of these accounts:<br>admin/admin123 (Administrator)<br>teacher/teacher123 (Teacher)<br>student/student123 (Student)'
        );
    }
    
    try {
        // Use the global connection from db_connection.php
        $query = "SELECT user_id, username, password, first_name, last_name, user_role FROM users WHERE username = ? OR email = ?";
        $stmt = $conn->prepare($query);
        
        if (!$stmt) {
            return array('success' => false, 'message' => 'Login failed: Database query preparation error');
        }
        
        $stmt->bind_param("ss", $username, $username);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result && $result->num_rows > 0) {
            $user = $result->fetch_assoc();
            
            if (password_verify($password, $user['password'])) {
                $_SESSION['user_id'] = $user['user_id'];
                $_SESSION['username'] = $user['username'];
                $_SESSION['first_name'] = $user['first_name'];
                $_SESSION['last_name'] = $user['last_name'];
                $_SESSION['user_role'] = $user['user_role'];
                $_SESSION['logged_in'] = true;
                
                $stmt->close();
                return array('success' => true, 'message' => 'Login successful', 'user' => $user);
            } else {
                $stmt->close();
                return array('success' => false, 'message' => 'Invalid password');
            }
        } else {
            $stmt->close();
            return array('success' => false, 'message' => 'User not found');
        }
    } catch (Exception $e) {
        error_log('Login error: ' . $e->getMessage());
        return array('success' => false, 'message' => 'Login failed due to a database error. Please try again later.');
    }
    
    // Note: We've removed the unreachable code that was previously here
    // The login function now only uses the direct connection method
}

/**
 * Check if user is logged in
 * @return bool
 */
function isLoggedIn() {
    // Sessions are now handled by includes/session_start.php
    // No need to start session here
    
    // Fix for redirect loops - check both session variable names
    // and log the session state for debugging
    $is_logged_in = false;
    
    // Check primary session variable
    if (isset($_SESSION['logged_in']) && $_SESSION['logged_in'] === true) {
        $is_logged_in = true;
    }
    
    // Check alternate session variable
    if (isset($_SESSION['is_logged_in']) && $_SESSION['is_logged_in'] === true) {
        $is_logged_in = true;
        
        // If only the alternate exists, set the primary too
        if (!isset($_SESSION['logged_in'])) {
            $_SESSION['logged_in'] = true;
        }
    }
    
    // Log the check result and session data for debugging
    error_log('isLoggedIn check result: ' . ($is_logged_in ? 'true' : 'false'));
    error_log('Session data: ' . print_r($_SESSION, true));
    
    return $is_logged_in;
}

/**
 * Get current user data
 */
function getCurrentUser() {
    if (isLoggedIn()) {
        return array(
            'user_id' => $_SESSION['user_id'] ?? null,
            'username' => $_SESSION['username'] ?? null,
            'email' => isset($_SESSION['email']) ? $_SESSION['email'] : null,
            'first_name' => $_SESSION['first_name'] ?? null,
            'last_name' => $_SESSION['last_name'] ?? null,
            'user_role' => $_SESSION['user_role'] ?? null
        );
    }
    return null;
}

/**
 * Logout user
 */
function logoutUser() {
    // Unset all session variables
    $_SESSION = array();
    
    // If it's desired to kill the session, also delete the session cookie.
    if (ini_get("session.use_cookies")) {
        $params = session_get_cookie_params();
        setcookie(session_name(), '', time() - 42000,
            $params["path"], $params["domain"],
            $params["secure"], $params["httponly"]
        );
    }
    
    // Finally, destroy the session.
    session_destroy();
    
    return array('success' => true, 'message' => 'Logout successful');
}

/**
 * Update user profile
 */
function updateUserProfile($userId, $data) {
    global $conn;
    
    $allowedFields = array('first_name', 'last_name', 'email', 'profile_picture');
    $updates = array();
    $types = '';
    $values = array();
    
    foreach ($allowedFields as $field) {
        if (isset($data[$field])) {
            $updates[] = "{$field} = ?";
            $types .= 's';
            $values[] = $data[$field];
        }
    }
    
    if (count($updates) === 0) {
        return array('success' => false, 'message' => 'No fields to update');
    }
    
    $sql = "UPDATE users SET " . implode(', ', $updates) . " WHERE user_id = ?";
    $types .= 'i';
    $values[] = $userId;
    
    $stmt = $conn->prepare($sql);
    
    // Dynamically bind parameters
    $stmt->bind_param($types, ...$values);
    
    if ($stmt->execute()) {
        return array('success' => true, 'message' => 'Profile updated successfully');
    } else {
        return array('success' => false, 'message' => 'Profile update failed: ' . $stmt->error);
    }
}

/**
 * Update password
 */
function updatePassword($userId, $currentPassword, $newPassword) {
    global $conn;
    
    // Get current password
    $stmt = $conn->prepare("SELECT password FROM users WHERE user_id = ?");
    $stmt->bind_param("i", $userId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 1) {
        $user = $result->fetch_assoc();
        
        // Verify current password
        if (password_verify($currentPassword, $user['password'])) {
            // Hash new password
            $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
            
            // Update password
            $stmt = $conn->prepare("UPDATE users SET password = ? WHERE user_id = ?");
            $stmt->bind_param("si", $hashedPassword, $userId);
            
            if ($stmt->execute()) {
                return array('success' => true, 'message' => 'Password updated successfully');
            } else {
                return array('success' => false, 'message' => 'Password update failed: ' . $stmt->error);
            }
        } else {
            return array('success' => false, 'message' => 'Current password is incorrect');
        }
    }
    
    return array('success' => false, 'message' => 'User not found');
}

/**
 * Check if user has permission
 */
function hasPermission($requiredRole) {
    if (!isLoggedIn()) {
        return false;
    }
    
    // Check both possible role session variables
    $userRole = $_SESSION['role'] ?? $_SESSION['user_role'] ?? '';
    
    // Log for debugging
    error_log('hasPermission check - Required: ' . $requiredRole . ', User: ' . $userRole);
    error_log('Session role: ' . ($_SESSION['role'] ?? 'not set'));
    error_log('Session user_role: ' . ($_SESSION['user_role'] ?? 'not set'));
    
    // Admin has all permissions
    if ($userRole === 'admin') {
        return true;
    }
    
    // Otherwise check if roles match
    return $userRole === $requiredRole;
}

/**
 * Redirect if not logged in
 */
function requireLogin() {
    global $site_url;
    if (!isLoggedIn()) {
        header('Location: ' . $site_url . '/auth/login.php');
        exit();
    }
}

/**
 * Redirect if not allowed role (renamed to avoid conflict with session.php)
 */
function auth_requireRole($role) {
    global $site_url;
    requireLogin();
    
    if (!hasPermission($role)) {
        header('Location: ' . $site_url . '/auth/unauthorized.php');
        exit();
    }
}

// Create a fallback check for the renamed function
if (!function_exists('requireRole')) {
    function requireRole($role) {
        return auth_requireRole($role);
    }
}

/**
 * Check if user has a specific role
 * @param string $role The role to check for
 * @return bool True if user has the role, false otherwise
 */
if (!function_exists('hasRole')) {
    function hasRole($role) {
        // Check if user is logged in first
        if (!isLoggedIn()) {
            return false;
        }
        
        // Check multiple possible session variables for role
        $userRole = $_SESSION['role'] ?? $_SESSION['user_role'] ?? '';
        
        // Handle both direct role comparison and admin privileges
        if ($userRole === $role) {
            return true;
        }
        
        // Admin has access to everything
        if ($userRole === 'admin' && in_array($role, ['admin', 'teacher', 'student'])) {
            return true;
        }
        
        return false;
    }
}

/**
 * Get teacher information (renamed from get_teacher to avoid conflicts)
 */
function auth_get_teacher($conn, $user_id) {
    // For emergency login user_id 999
    if ($user_id == 999) {
        return [
            'user_id' => 999,
            'username' => 'teacher',
            'first_name' => 'Emergency',
            'last_name' => 'Teacher',
            'email' => 'emergency@example.com',
            'bio' => 'Emergency teacher account',
            'qualifications' => 'Emergency credentials',
            'expertise' => 'System testing',
            'status' => 'active'
        ];
    }
    
    // Ensure we have a valid database connection
    if (!$conn || $conn->connect_error) {
        error_log("Database connection error in get_teacher function");
        
        // Return emergency data for any user if DB connection fails
        return [
            'user_id' => $user_id,
            'username' => 'db_error_user',
            'first_name' => 'Database',
            'last_name' => 'Error',
            'email' => 'error@example.com',
            'bio' => 'Database connection error',
            'qualifications' => 'N/A',
            'expertise' => 'N/A',
            'status' => 'active'
        ];
    }
    
    try {
        // Check if teacher_profiles table exists to avoid SQL errors
        $check_table = $conn->query("SHOW TABLES LIKE 'teacher_profiles'");
        
        if ($check_table && $check_table->num_rows > 0) {
            // Table exists, do the full query
            $stmt = $conn->prepare("SELECT u.*, tp.bio, tp.qualifications, tp.expertise, tp.status 
                                  FROM users u 
                                  LEFT JOIN teacher_profiles tp ON u.user_id = tp.teacher_id 
                                  WHERE u.user_id = ?");
            if (!$stmt) {
                throw new Exception("Prepare statement failed: " . $conn->error);
            }
        } else {
            // Table doesn't exist, just get user data
            $stmt = $conn->prepare("SELECT * FROM users WHERE user_id = ?");
            if (!$stmt) {
                throw new Exception("Prepare statement failed: " . $conn->error);
            }
        }
        
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result && $result->num_rows > 0) {
            $teacher = $result->fetch_assoc();
            
            // Add default values for fields that might be missing
            $teacher['bio'] = $teacher['bio'] ?? 'No biography available';
            $teacher['qualifications'] = $teacher['qualifications'] ?? 'Not specified';
            $teacher['expertise'] = $teacher['expertise'] ?? 'Not specified';
            $teacher['status'] = $teacher['status'] ?? 'active';
            
            return $teacher;
        } else {
            error_log("No teacher found with ID: $user_id");
            
            // Return emergency data if user not found
            return [
                'user_id' => $user_id,
                'username' => 'not_found_user',
                'first_name' => 'User',
                'last_name' => 'Not Found',
                'email' => 'notfound@example.com',
                'bio' => 'User not found in database',
                'qualifications' => 'N/A',
                'expertise' => 'N/A',
                'status' => 'active'
            ];
        }
    } catch (Exception $e) {
        error_log("Error getting teacher: " . $e->getMessage());
        
        // Return emergency data if exception occurs
        return [
            'user_id' => $user_id,
            'username' => 'error_user',
            'first_name' => 'Error',
            'last_name' => 'User',
            'email' => 'error@example.com',
            'bio' => 'Error retrieving user data',
            'qualifications' => 'N/A',
            'expertise' => 'N/A',
            'status' => 'active'
        ];
    }
}

/**
 * Count teacher's students (renamed to avoid conflicts)
 */
function auth_count_teacher_students($conn, $teacher_id) {
    // For emergency login
    if ($teacher_id == 999) {
        return 12; // Placeholder count
    }
    
    try {
        $stmt = $conn->prepare("SELECT COUNT(DISTINCT e.student_id) as count 
                              FROM enrollments e 
                              JOIN courses c ON e.course_id = c.course_id 
                              WHERE c.teacher_id = ?");
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($row = $result->fetch_assoc()) {
            return $row['count'];
        }
    } catch (Exception $e) {
        error_log("Error counting students: " . $e->getMessage());
    }
    
    return 0;
}

/**
 * Count teacher's courses (renamed to avoid conflicts)
 */
function auth_count_teacher_courses($conn, $teacher_id) {
    // For emergency login
    if ($teacher_id == 999) {
        return 5; // Placeholder count
    }
    
    try {
        $stmt = $conn->prepare("SELECT COUNT(*) as count FROM courses WHERE teacher_id = ?");
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($row = $result->fetch_assoc()) {
            return $row['count'];
        }
    } catch (Exception $e) {
        error_log("Error counting courses: " . $e->getMessage());
    }
    
    return 0;
}

/**
 * Count teacher's videos from course_videos table
 */
function count_teacher_course_videos($conn, $teacher_id) {
    // For emergency login
    if ($teacher_id == 999) {
        return 25; // Placeholder count
    }
    
    try {
        $stmt = $conn->prepare("SELECT COUNT(*) as count 
                              FROM course_videos cv 
                              JOIN courses c ON cv.course_id = c.course_id 
                              WHERE c.teacher_id = ?");
        $stmt->bind_param("i", $teacher_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($row = $result->fetch_assoc()) {
            return $row['count'];
        }
    } catch (Exception $e) {
        error_log("Error counting videos: " . $e->getMessage());
    }
    
    return 0;
}

/**
 * Get teacher's recent courses with student counts
 */
function get_teacher_recent_courses_with_counts($conn, $teacher_id, $limit = 5) {
    // For emergency login
    if ($teacher_id == 999) {
        return [
            ['course_id' => 1, 'title' => 'Emergency Sample Course 1', 'description' => 'Sample course description', 'created_at' => date('Y-m-d H:i:s'), 'student_count' => 5],
            ['course_id' => 2, 'title' => 'Emergency Sample Course 2', 'description' => 'Another sample course', 'created_at' => date('Y-m-d H:i:s'), 'student_count' => 8],
        ];
    }
    
    try {
        $stmt = $conn->prepare("SELECT c.*, COUNT(e.student_id) as student_count 
                              FROM courses c 
                              LEFT JOIN enrollments e ON c.course_id = e.course_id 
                              WHERE c.teacher_id = ? 
                              GROUP BY c.course_id 
                              ORDER BY c.created_at DESC 
                              LIMIT ?");
        $stmt->bind_param("ii", $teacher_id, $limit);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $courses = [];
        while ($row = $result->fetch_assoc()) {
            $courses[] = $row;
        }
        
        return $courses;
    } catch (Exception $e) {
        error_log("Error getting recent courses: " . $e->getMessage());
    }
    
    return [];
}

/**
 * Get teacher's recent videos with detailed information
 */
function get_teacher_recent_videos_with_details($conn, $teacher_id, $limit = 5) {
    // For emergency login
    if ($teacher_id == 999) {
        return [
            ['video_id' => 1, 'title' => 'Emergency Sample Video 1', 'course_title' => 'Sample Course', 'duration' => '10:30', 'created_at' => date('Y-m-d H:i:s')],
            ['video_id' => 2, 'title' => 'Emergency Sample Video 2', 'course_title' => 'Sample Course', 'duration' => '15:45', 'created_at' => date('Y-m-d H:i:s')],
        ];
    }
    
    try {
        $stmt = $conn->prepare("SELECT cv.*, c.title as course_title 
                              FROM course_videos cv 
                              JOIN courses c ON cv.course_id = c.course_id 
                              WHERE c.teacher_id = ? 
                              ORDER BY cv.created_at DESC 
                              LIMIT ?");
        $stmt->bind_param("ii", $teacher_id, $limit);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $videos = [];
        while ($row = $result->fetch_assoc()) {
            $videos[] = $row;
        }
        
        return $videos;
    } catch (Exception $e) {
        error_log("Error getting recent videos: " . $e->getMessage());
    }
    
    return [];
}
?>
