<?php
/**
 * Panadite Academy - Enhanced Login System
 * Integrated with SessionManager and AuthManager for robust session handling
 */

// Include SEO headers with favicon for login page
$seo_title = 'Login - Panadite Academy | Access Your Learning Dashboard';
$seo_description = 'Login to your Panadite Academy account to access your courses, track progress, and continue your professional learning journey.';
require_once __DIR__ . '/../includes/seo_header.php';

// Turn off output buffering to prevent header issues
ob_end_clean();
if (ob_get_level() == 0) ob_start();

// Start error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// ==========================================
// ENHANCED SESSION AND AUTH MANAGEMENT
// ==========================================
// Integrated session management directly to avoid external dependencies

// Start secure session configuration
if (session_status() !== PHP_SESSION_ACTIVE) {
    // Enhanced session configuration for security and Linux server compatibility
    ini_set('session.cookie_httponly', 1);
    ini_set('session.use_only_cookies', 1);
    // Adjust secure flag based on HTTPS availability
    ini_set('session.cookie_secure', isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 1 : 0);
    ini_set('session.cookie_samesite', 'Lax');
    ini_set('session.use_strict_mode', 1);
    ini_set('session.cookie_lifetime', 7200); // 2 hours
    ini_set('session.gc_maxlifetime', 7200); // 2 hours
    ini_set('session.cookie_domain', ''); // Let PHP determine the domain
    ini_set('session.cookie_path', '/'); // Ensure cookie works for entire site

    // Set proper session save path for Linux server
    $temp_dir = sys_get_temp_dir();
    if (!is_writable($temp_dir)) {
        // Try alternative paths for Linux server
        $alternative_paths = ['/tmp', '/var/tmp', dirname(__FILE__) . '/tmp'];
        foreach ($alternative_paths as $path) {
            if (is_dir($path) && is_writable($path)) {
                $temp_dir = $path;
                break;
            }
        }
    }
    ini_set('session.save_path', $temp_dir);
    

    
    // Start session
    session_start();
}

// Initialize session security
if (!isset($_SESSION['created_at'])) {
    $_SESSION['created_at'] = time();
    $_SESSION['last_regeneration'] = time();
    session_regenerate_id(true);
} else {
    // Regenerate session ID periodically for security
    if (time() - ($_SESSION['last_regeneration'] ?? 0) > 300) { // 5 minutes
        $_SESSION['last_regeneration'] = time();
        session_regenerate_id(true);
    }
    
    // Check session lifetime
    if (time() - $_SESSION['created_at'] > 7200) { // 2 hours
        session_destroy();
        session_start();
    }
}

require_once __DIR__ . '/functions.php';
require_once __DIR__ . '/../includes/site_config.php';

// Include fast email service for login alert notifications
require_once __DIR__ . '/../config/fast_email_service.php';

// Bridge database connections - Email service needs PDO, but login uses MySQLi
if (!isset($pdo)) {
    try {
        // Create PDO connection using the same credentials as MySQLi
        $pdo = new PDO("mysql:host=" . DB_SERVER . ";dbname=" . DB_NAME, DB_USERNAME, DB_PASSWORD);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        error_log("PDO connection created successfully for email service");
    } catch (Exception $e) {
        error_log("Failed to create PDO connection for email service: " . $e->getMessage());
        $pdo = null;
    }
}

// Try to include database connection with error handling
try {
    require_once __DIR__ . '/../includes/db_connection.php';
    if (isset($conn) && $conn) {
        
    }
} catch (Exception $e) {
    // Continue with emergency credentials fallback
}

// Handle session clearing if requested
if (isset($_GET['clear_session']) && $_GET['clear_session'] === 'true') {
    error_log('DEBUG - Clearing session as requested');
    // Clear session completely
    $_SESSION = array();
    if (ini_get("session.use_cookies")) {
        $params = session_get_cookie_params();
        setcookie(session_name(), '', time() - 42000,
            $params["path"], $params["domain"],
            $params["secure"], $params["httponly"]
        );
    }
    session_destroy();
    header('Location: login.php');
    exit();
}

// Check for redirect parameter (including cart/checkout redirect)
$redirect = '';
$redirect_to_cart = false;

if (isset($_GET['redirect_to'])) {
    if ($_GET['redirect_to'] === 'cart') {
        $redirect_to_cart = true;
        $redirect = '/student/cart.php?cart_redirect=1';
    } else if ($_GET['redirect_to'] === 'student/checkout.php') {
        $redirect_to_cart = true;
        $redirect = '/student/checkout.php?from_guest_cart=1';
    } else {
        $redirect = '/' . $_GET['redirect_to'];
    }
} else if (isset($_GET['redirect'])) {
    $redirect = $_GET['redirect'];
}

if ($redirect) {
    // Store redirect in session to persist through session clearing
    $_SESSION['stored_redirect'] = $redirect;
    $_SESSION['redirect_to_cart'] = $redirect_to_cart;
} else if (isset($_SESSION['stored_redirect'])) {
    // Retrieve redirect from session if it was stored before
    $redirect = $_SESSION['stored_redirect'];
    $redirect_to_cart = $_SESSION['redirect_to_cart'] ?? false;
}

/**
 * Send login alert notification to user
 */
function sendLoginAlertEmail($user_data) {
    global $conn;
    
    try {
        $user_name = $user_data['first_name'] . ' ' . $user_data['last_name'];
        $user_email = $user_data['email'];
        $user_role = ucfirst($user_data['user_role']);
        $login_time = date('M j, Y \a\t g:i A');
        $user_ip = $_SERVER['REMOTE_ADDR'] ?? 'Unknown';
        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
        $browser_info = substr($user_agent, 0, 100); // Truncate for display
        
        // Create email content
        $subject = "🔐 Login Alert: New Sign-In to Your Panadite Academy Account";
        $message = "
        <div style='max-width:600px; margin:0 auto; font-family:Arial,sans-serif;'>
            <div style='background:#667eea; color:white; padding:20px; text-align:center;'>
                <h1>🔐 Login Alert</h1>
            </div>
            <div style='padding:20px; background:white;'>
                <h2>Hi " . htmlspecialchars($user_name) . "! 👋</h2>
                <p>We're letting you know that someone just signed into your Panadite Academy account.</p>
                <div style='background:#f8f9fa; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h3>Login Details:</h3>
                    <p><strong>Account:</strong> " . htmlspecialchars($user_email) . " (" . htmlspecialchars($user_role) . ")</p>
                    <p><strong>Time:</strong> " . $login_time . "</p>
                    <p><strong>IP Address:</strong> " . htmlspecialchars($user_ip) . "</p>
                    <p><strong>Browser:</strong> " . htmlspecialchars($browser_info) . "</p>
                </div>
                <div style='background:#e8f5e8; padding:15px; border-radius:5px; margin:20px 0;'>
                    <h4>✅ Was This You?</h4>
                    <p>If this was you, no action is needed. You can safely ignore this email.</p>
                    <p>If this wasn't you, please contact our support team immediately and change your password.</p>
                </div>
                <p style='text-align:center;'>
                    <a href='https://panaditeacademy.com/student/profile.php' 
                       style='background:#667eea; color:white; text-decoration:none; padding:15px 25px; border-radius:8px; font-weight:bold; display:inline-block;'>
                        View Account Settings
                    </a>
                </p>
            </div>
        </div>";
        
        // Try direct email sending
        if (function_exists('sendEmailNow')) {
            $result = sendEmailNow($user_email, $subject, $message);
            if ($result) {
                return true;
            }
        }
        
        // Fallback to native PHP mail
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: noreply@panaditeacademy.com" . "\r\n";
        
        return mail($user_email, $subject, $message, $headers);
        
    } catch (Exception $e) {
        error_log("Error sending login alert email: " . $e->getMessage());
        return false;
    }
}

// Define isLoggedIn function inline if it doesn't exist
if (!function_exists('isLoggedIn')) {
    function isLoggedIn() {
        return isset($_SESSION['logged_in']) && $_SESSION['logged_in'] === true && 
               isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
    }
}

// Check if already logged in
if (isset($_SESSION['logged_in']) && $_SESSION['logged_in'] === true && isset($_SESSION['user_id'])) {
    $username = $_SESSION['username'] ?? 'unknown';
    $role = $_SESSION['user_role'] ?? 'unknown';
    error_log('DEBUG - User already logged in: ' . $username . ' (Role: ' . $role . ')');
    
    // Handle redirect logic
    if (!empty($redirect)) {
        header('Location: ' . $redirect);
    } else {
        // Redirect to role-based dashboard
        if ($role === 'admin') {
            $dashboard_url = $site_url . '/admin/dashboard.php';
        } else if ($role === 'teacher') {
            $dashboard_url = $site_url . '/teacher/dashboard.php';
        } else {
            $dashboard_url = $site_url . '/student/dashboard.php';
        }
        header('Location: ' . $dashboard_url);
    }
    exit();
}

$error = '';
$success = '';

// Check for logout success message
if (isset($_GET['logout']) && $_GET['logout'] == '1') {
    $success = 'You have been logged out successfully';
}



// Handle login form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email = trim($_POST['email'] ?? '');
    $password = trim($_POST['password'] ?? '');
    

    
    if (empty($email) || empty($password)) {
        $error = 'Email and password are required';

    } else {

        
        // All users (including admin) now use database authentication
        {
            // Database authentication for all other users
            try {
                // Check database connection with more specific error handling
                if (!isset($conn) || !$conn) {
                    error_log('DEBUG - Database connection not available');
                    throw new Exception('Database connection not available');
                }
                
                // Test the connection
                if ($conn->connect_error) {
                    error_log('DEBUG - Database connection error: ' . $conn->connect_error);
                    throw new Exception('Database connection failed: ' . $conn->connect_error);
                }
                
                // Prepare statement with error checking - Login with email only
                $sql = "SELECT user_id, username, email, password, first_name, last_name, user_role FROM users WHERE email = ?";
                $stmt = $conn->prepare($sql);
                
                if (!$stmt) {
                    error_log('DEBUG - Prepare failed: ' . $conn->error);
                    throw new Exception('Database query preparation failed: ' . $conn->error);
                }
                
                $stmt->bind_param("s", $email);
                
                if (!$stmt->execute()) {
                    error_log('DEBUG - Execute failed: ' . $stmt->error);
                    throw new Exception('Database query execution failed: ' . $stmt->error);
                }
                
                $result = $stmt->get_result();
                
                if ($result->num_rows === 0) {
                    $error = 'Invalid email or password';
                } else {
                    $user = $result->fetch_assoc();
                    
                    if (password_verify($password, $user['password'])) {
                        // Create user session (same logic as emergency credentials)
                        // Enhanced session data creation
                        $sessionData = [
                            'user_id' => $user['user_id'],
                            'username' => $user['username'],
                            'email' => $user['email'],
                            'first_name' => $user['first_name'],
                            'last_name' => $user['last_name'],
                            'user_role' => $user['user_role'],
                            'logged_in' => true,
                            'is_logged_in' => true,
                            'login_time' => time(),
                            'created_at' => time(),
                            'last_regeneration' => time()
                        ];
                        
                        // Set all session data at once
                        foreach ($sessionData as $key => $value) {
                            $_SESSION[$key] = $value;
                        }

                        // Send login alert notification email - USING EXACT SAME APPROACH AS OTHER NOTIFICATIONS
                        $email_result = sendLoginAlertEmail($user);
                        error_log("Login alert email sent: " . ($email_result ? "SUCCESS" : "FAILED"));
                        
                        // Manual cookie setting as backup (same as emergency credentials)
                        if (!headers_sent()) {
                            session_write_close();
                            session_start();
                            $session_id = session_id();
                            
                            // Force cookie settings with multiple fallbacks for cross-directory access
                            $cookie_options = [
                                'expires' => time() + 7200,
                                'path' => '/',  // CRITICAL: Root path for entire site
                                'domain' => $_SERVER['HTTP_HOST'] ?? $_SERVER['SERVER_NAME'] ?? '',
                                'secure' => isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on',
                                'httponly' => true,
                                'samesite' => 'Lax'
                            ];
                            
                            // Set cookie multiple ways to ensure it sticks
                            setcookie('PHPSESSID', $session_id, $cookie_options);
                            setcookie(session_name(), $session_id, $cookie_options);
                            
                            // Also set via header as additional fallback
                            $cookie_header = session_name() . '=' . $session_id . '; Path=/; HttpOnly; SameSite=Lax';
                            if ($cookie_options['secure']) {
                                $cookie_header .= '; Secure';
                            }
                            header('Set-Cookie: ' . $cookie_header, false);
                            

                        }
                        
                        // Handle redirect logic - check for redirect parameter first
                        if (!empty($redirect)) {
                            error_log('DEBUG - Redirecting to specified URL: ' . $redirect);
                            
                            // Special handling for cart/checkout redirects - need to verify cart has items
                            if (strpos($redirect, 'checkout.php') !== false || strpos($redirect, 'cart.php') !== false) {
                                // Redirect to cart page with a flag to check contents and decide final destination
                                $cart_check_url = $site_url . '/student/cart.php?check_cart_redirect=1&original_redirect=' . urlencode($redirect);
                                header('Location: ' . $cart_check_url);
                                exit();
                            }
                            
                            // For non-cart redirects, proceed normally
                            header('Location: ' . $redirect);
                            exit();
                        }
                        
                        // Role-based redirect
                        $role = $user['user_role'];
                        $dashboard_url = '';
                        
                        if ($role === 'admin') {
                            $dashboard_url = $site_url . '/admin/dashboard.php';
                        } else if ($role === 'teacher') {
                            $dashboard_url = $site_url . '/teacher/dashboard.php';
                        } else {
                            // For students, always check cart before final redirect
                            $dashboard_url = $site_url . '/student/cart.php?check_cart_redirect=1&original_redirect=' . urlencode($site_url . '/student/dashboard.php');
                        }
                        
                        error_log('DEBUG - Redirecting to: ' . $dashboard_url);
                        header('Location: ' . $dashboard_url);
                        exit();
                    } else {
                        $error = 'Invalid username/email or password';
                    }
                }
                $stmt->close();
                
            } catch (Exception $e) {
                // If database fails, log the error but still allow emergency credentials
                error_log('DEBUG - Database authentication failed: ' . $e->getMessage());
                $error = 'Database authentication unavailable. Please use emergency credentials or try again later.';
            }
        }
    }
}

?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Login - Panadite Academy</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css">
    <!-- Updated Font Awesome with integrity check to ensure proper loading -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" integrity="sha512-iecdLmaskl7CVkqkXNQ/ZH/XLlvWZOJyj7Yy7tcenmpD1ypASozpmT/E0iPtmFIB46ZmdtAc9eNBvH0H/ZpiBw==" crossorigin="anonymous" referrerpolicy="no-referrer" />
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css">
    <style>
        body {
            background-color: #f8f9fa;
            min-height: 100vh;
        }
        
        .login-container {
            max-width: 480px;
            margin: 40px auto;
            padding: 40px;
            background-color: #fff;
            border-radius: 20px;
            box-shadow: 0 8px 40px rgba(39, 107, 228, 0.1);
            border: 1px solid rgba(39, 107, 228, 0.05);
        }
        
        .logo-container {
            text-align: center;
            margin-bottom: 35px;
        }
        
        .logo-container img {
            max-height: 100px;
            margin-bottom: 20px;
            filter: drop-shadow(0 2px 4px rgba(39, 107, 228, 0.1));
        }
        
        .form-label {
            font-weight: 600;
            color: #2d3436;
            margin-bottom: 10px;
            font-size: 0.95rem;
        }
        
        .form-control {
            border: 2px solid #e9ecef;
            border-radius: 12px;
            padding: 14px 16px;
            transition: all 0.3s ease;
            font-size: 16px;
            background-color: #fafbfc;
            height: 52px;
        }
        
        .form-control:focus {
            border-color: #276BE4;
            box-shadow: 0 0 0 0.25rem rgba(39, 107, 228, 0.15);
            background-color: #ffffff;
            transform: translateY(-1px);
        }
        
        .input-group-text {
            background-color: #fafbfc;
            border: 2px solid #e9ecef;
            border-radius: 12px 0 0 12px;
            color: #276BE4;
            font-size: 18px;
            padding: 14px 16px;
            width: 52px;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .input-group .form-control {
            border-left: none;
            border-radius: 0 12px 12px 0;
        }
        
        .input-group:focus-within .input-group-text {
            border-color: #276BE4;
            background-color: rgba(39, 107, 228, 0.05);
            color: #1e5bcc;
        }
        
        .btn-login {
            background: linear-gradient(135deg, #276BE4, #1e5bcc);
            border: none;
            border-radius: 12px;
            color: white;
            font-weight: 600;
            padding: 16px 24px;
            font-size: 17px;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            width: 100%;
            height: 52px;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
            margin-top: 8px;
        }
        
        .btn-login:hover {
            background: linear-gradient(135deg, #1e5bcc, #1752b8);
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(39, 107, 228, 0.3);
        }
        
        .login-links {
            text-align: center;
            margin-top: 25px;
            padding-top: 20px;
            border-top: 1px solid #f1f3f4;
        }
        
        .login-links a {
            color: #276BE4;
            text-decoration: none;
            font-weight: 500;
            font-size: 15px;
            transition: all 0.3s ease;
        }
        
        .login-links a:hover {
            color: #1e5bcc;
            text-decoration: underline;
        }
        
        .tagline {
            text-align: center;
            color: #FFA915;
            font-style: italic;
            margin-bottom: 25px;
            letter-spacing: 0.5px;
            font-size: 1.1em;
            font-weight: 500;
        }
        
        .password-toggle {
            position: absolute;
            right: 10px;
            top: 50%;
            transform: translateY(-50%);
            cursor: pointer;
            color: #6c757d;
            z-index: 10;
        }
        
        .password-container {
            position: relative;
        }
            transition: all 0.3s ease;
        }
        
        .input-group:focus-within + .form-label::after,
        .form-control:focus ~ .form-label::after {
            width: 30px;
        }
        
        .input-group {
            transition: all 0.3s ease;
        }
        
        .input-group:focus-within {
            transform: translateY(-3px);
        }
        
        @keyframes pulseGlow {
            0% { box-shadow: 0 0 0 0 rgba(39, 107, 228, 0.4); }
            70% { box-shadow: 0 0 0 10px rgba(39, 107, 228, 0); }
            100% { box-shadow: 0 0 0 0 rgba(39, 107, 228, 0); }
        }
        
        .pulse-animation {
            animation: pulseGlow 2s infinite;
        }
    </style>
</head>
<body>
    <!-- Include the modern navbar -->
    <?php include '../includes/navbar.php'; ?>
    
    <div class="container">
        <div class="login-container">
            <div class="logo-container animate__animated animate__fadeIn">
                <img src="/assets/images/logo.jpeg" alt="Panadite Academy Logo" class="img-fluid">
                <p class="tagline animate__animated animate__fadeIn animate__delay-1s">Get ahead of the curve</p>
            </div>
            
            <?php if ($error): ?>
            <div class="alert alert-danger" role="alert">
                <?php echo $error; ?>
            </div>
            <?php endif; ?>
            
            <?php if ($success): ?>
            <div class="alert alert-success" role="alert">
                <?php echo $success; ?>
            </div>
            <?php endif; ?>
            
            <?php if (!empty($debug_info)): ?>
            <div class="alert alert-info" role="alert">
                <strong>Debug Information:</strong>
                <ul>
                    <?php foreach($debug_info as $info): ?>
                    <li><?php echo htmlspecialchars($info); ?></li>
                    <?php endforeach; ?>
                </ul>
                <p><strong>SESSION Data:</strong> <?php echo htmlspecialchars(print_r($_SESSION, true)); ?></p>
            </div>
            <?php endif; ?>
            

            
            <form class="login-form animate__animated animate__fadeIn animate__delay-1s" method="POST" action="<?php echo htmlspecialchars($_SERVER['PHP_SELF']) . (!empty($redirect) ? '?redirect=' . urlencode($redirect) : ''); ?>" id="loginForm">
                <div class="mb-3">
                    <label for="email" class="form-label">Email</label>
                    <div class="input-group">
                        <span class="input-group-text"><i class="fas fa-envelope"></i></span>
                        <input type="email" class="form-control" id="email" name="email" required placeholder="Enter your email address">
                    </div>
                </div>
                
                <div class="mb-3">
                    <label for="password" class="form-label">Password</label>
                    <div class="input-group password-container">
                        <span class="input-group-text"><i class="fas fa-lock"></i></span>
                        <input type="password" class="form-control" id="password" name="password" required>
                        <span class="password-toggle" onclick="togglePassword('password')">
                            <i class="far fa-eye"></i>
                        </span>
                    </div>
                </div>
                

                
                <button type="submit" class="btn btn-login w-100 animate__animated animate__fadeIn animate__delay-2s" id="loginBtn">
                    Login <i class="fas fa-arrow-right-to-bracket ms-2"></i>
                </button>
            </form>
            
            
            <div class="login-links animate__animated animate__fadeIn animate__delay-1s">
                <a href="../forgot-password.php" class="d-block mb-2">Forgot your password?</a>
                <a href="register.php">Don't have an account? Register</a><br>
                <a href="/index.php">Back to Home</a>
            </div>
        </div>
    </div>





    <footer class="mt-5 position-relative">
        <!-- Top gradient bar -->
        <div class="position-absolute top-0 start-0 w-100" style="height: 4px; background: linear-gradient(90deg, #276BE4, #FFA915);"></div>
        
        <!-- Main footer content -->
        <div class="bg-white py-5">
            <div class="container">
                <div class="row gy-4">
                    <!-- Logo and description -->
                    <div class="col-lg-4 col-md-6 mb-4">
                        <div class="mb-4">
                            <img src="/assets/images/logo.jpeg" alt="Panadite Academy" style="height: 50px;">
                        </div>
                        <p class="mb-2" style="color: #333; font-weight: 500;">Empowering learners through high-quality educational content.</p>
                        <p style="color: #333;">Get ahead of the curve with our premium videos and courses.</p>
                    </div>
                    
                    <!-- Quick links -->
                    <div class="col-lg-2 col-md-6 mb-4">
                        <h5 class="fw-bold mb-3 pb-2 position-relative" style="color: #222;">
                            Quick Links
                            <span class="position-absolute bottom-0 start-0" style="height: 2px; width: 40px; background: #FFA915;"></span>
                        </h5>
                        <ul class="list-unstyled">
                            <li class="mb-2"><a href="/index.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Home</a></li>
                            <li class="mb-2"><a href="/courses.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Courses</a></li>
                            <li class="mb-2"><a href="/videos.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Videos</a></li>
                            <li class="mb-2"><a href="/about.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>About Us</a></li>
                            <li class="mb-2"><a href="/contact.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Contact</a></li>
                        </ul>
                    </div>
                    
                    <!-- Facilitator links -->
                    <div class="col-lg-2 col-md-6 mb-4">
                        <h5 class="fw-bold mb-3 pb-2 position-relative" style="color: #222;">
                            Facilitator
                            <span class="position-absolute bottom-0 start-0" style="height: 2px; width: 40px; background: #FFA915;"></span>
                        </h5>
                        <ul class="list-unstyled">
                            <li class="mb-2"><a href="/auth/register.php?role=teacher" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Become a Facilitator</a></li>
                            <li class="mb-2"><a href="/teacher/dashboard.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Teacher Dashboard</a></li>
                            <li class="mb-2"><a href="/teacher/sales.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Your Sales</a></li>
                            <li class="mb-2"><a href="/teacher/help.php" class="text-decoration-none" style="color: #333;"><i class="fas fa-angle-right me-2" style="color: #276BE4;"></i>Teacher Help</a></li>
                        </ul>
                    </div>
                    
                    <!-- Contact info -->
                    <div class="col-lg-4 col-md-6">
                        <h5 class="fw-bold mb-3 pb-2 position-relative" style="color: #222;">
                            Contact
                            <span class="position-absolute bottom-0 start-0" style="height: 2px; width: 40px; background: #FFA915;"></span>
                        </h5>
                        <ul class="list-unstyled mb-4">
                            <li class="mb-3 d-flex align-items-center">
                                <span class="d-flex align-items-center justify-content-center me-3" style="width: 36px; height: 36px; background: rgba(39,107,228,0.1); 
    </script>    
border-radius: 50%;">
                                    <i class="fas fa-map-marker-alt" style="color: #276BE4;"></i>
                                </span>
                                <span style="color: #333;">296 Pretorius Street, Corner Lilian Ngoyi, Pretoria Central, 0001, South Africa</span>
                            </li>
                            <li class="mb-3 d-flex align-items-center">
                                <span class="d-flex align-items-center justify-content-center me-3" style="width: 36px; height: 36px; background: rgba(39,107,228,0.1); border-radius: 50%;">
                                    <i class="fas fa-envelope" style="color: #276BE4;"></i>
                                </span>
                                <a href="mailto:admin@panaditeacademy.com" class="text-decoration-none" style="color: #333;">admin@panaditeacademy.com</a>
                            </li>
                            <li class="mb-3 d-flex align-items-center">
                                <span class="d-flex align-items-center justify-content-center me-3" style="width: 36px; height: 36px; background: rgba(39,107,228,0.1); border-radius: 50%;">
                                    <i class="fas fa-phone" style="color: #276BE4;"></i>
                                </span>
                                <a href="tel:+27747223141" class="text-decoration-none" style="color: #333;">+27 74 722 3141</a>
                            </li>
                        </ul>
                        
                        <!-- Social icons -->
                        <div class="d-flex">
                            <a href="#" class="d-flex align-items-center justify-content-center me-2" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-facebook-f"></i>
                            </a>
                            <a href="#" class="d-flex align-items-center justify-content-center me-2" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-twitter"></i>
                            </a>
                            <a href="#" class="d-flex align-items-center justify-content-center me-2" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-instagram"></i>
                            </a>
                            <a href="#" class="d-flex align-items-center justify-content-center" style="width: 40px; height: 40px; background: rgba(39,107,228,0.1); border-radius: 50%; color: #276BE4; transition: all 0.3s ease;">
                                <i class="fab fa-linkedin-in"></i>
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Copyright section -->
        <div style="background: #f8f9fa; border-top: 1px solid #eee;" class="py-3">
            <div class="container">
                <div class="row align-items-center">
                    <div class="col-md-6 mb-3 mb-md-0">
                        <p class="mb-0" style="color: #333; font-weight: 500;">&copy; <?php echo date('Y'); ?> Panadite Academy. All rights reserved.</p>
                    </div>
                    <div class="col-md-6 text-md-end">
                        <a href="/privacy-policy.php" class="text-decoration-none me-3" style="color: #333;">Privacy Policy</a>
                        <a href="/terms.php" class="text-decoration-none me-3" style="color: #333;">Terms of Use</a>
                        <a href="/sitemap.php" class="text-decoration-none" style="color: #333;">Sitemap</a>
                    </div>
                </div>
            </div>
        </div>
    </footer>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function togglePassword(inputId) {
            const input = document.getElementById(inputId);
            const icon = input.nextElementSibling.querySelector('i');
            
            if (input.type === 'password') {
                input.type = 'text';
                icon.classList.remove('fa-eye');
                icon.classList.add('fa-eye-slash');
            } else {
                input.type = 'password';
                icon.classList.remove('fa-eye-slash');
                icon.classList.add('fa-eye');
            }
        }
        

        
        // Add form submission animation
        document.getElementById('loginForm').addEventListener('submit', function(e) {
            document.getElementById('loginBtn').innerHTML = 'Logging in... <i class="fas fa-spinner fa-spin ms-2"></i>';
            document.getElementById('loginBtn').disabled = true;
        });
        
        $(document).ready(function() {
            // Animate floating icons and shapes
            animateFloatingElements();
            
            // Add staggered animations to form elements
            $('.form-control').each(function(index) {
                $(this).addClass('animate__animated animate__fadeInUp')
                       .css('animation-delay', (index * 100 + 300) + 'ms');
            });
            
            // Add focus animations
            $('.form-control').on('focus', function() {
                $(this).parent().addClass('pulse-animation');
            }).on('blur', function() {
                $(this).parent().removeClass('pulse-animation');
            });
            
            // Add floating effect to login container
            $('.login-container').on('mousemove', function(e) {
                const container = $(this);
                const containerWidth = container.width();
                const containerHeight = container.height();
                const centerX = container.offset().left + containerWidth/2;
                const centerY = container.offset().top + containerHeight/2;
                const mouseX = e.pageX - centerX;
                const mouseY = e.pageY - centerY;
                const rotateY = (mouseX / containerWidth) * 5;
                const rotateX = ((mouseY / containerHeight) * -5);
                
                container.css('transform', 
                    `translateY(-5px) perspective(1000px) rotateX(${rotateX}deg) rotateY(${rotateY}deg)`
                );
            }).on('mouseleave', function() {
                $(this).css('transform', 'translateY(-5px)');
            });
            
            // Add shine effect on button hover
            $('.btn-login').on('mouseenter', function() {
                $(this).addClass('animate__animated animate__pulse');
                $(this).find('i').css({'opacity': '1', 'transform': 'translateX(3px)'});
            }).on('mouseleave', function() {
                $(this).removeClass('animate__animated animate__pulse');
                $(this).find('i').css({'transform': 'translateX(0)'});
            });
            
            // Ensure the icon is visible with a small animation to draw attention
            $('.btn-login i').css({'opacity': '1', 'transition': 'transform 0.3s ease'});
            
            // Add a quick flicker animation to ensure the icon is noticed
            setTimeout(function() {
                $('.btn-login i').css({'transform': 'scale(1.2)'});
                setTimeout(function() {
                    $('.btn-login i').css({'transform': 'scale(1)'});
                }, 300);
            }, 1000);
            
            // Validate login form before submission
            $('.login-form').on('submit', function() {
                const email = $('#email').val();
                const password = $('#password').val();
                
                if (!email || !password) {
                    return false;
                }
                
                $('.btn-login').html('<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Signing in...');
                return true;
            });
        });
        // Function to animate floating elements
        function animateFloatingElements() {
            $('.floating-icon, .shape').each(function(index) {
                var element = $(this);
                // Set random initial positions
                var startX = Math.random() * 10 - 5; // -5 to +5
                var startY = Math.random() * 10 - 5; // -5 to +5
                
                // Animate with random duration between 5-10 seconds
                var duration = 5000 + Math.random() * 5000;
                
                // Function to create floating animation
                function floatElement() {
                    // Generate new random positions
                    var moveX = Math.random() * 20 - 10; // -10 to +10
                    var moveY = Math.random() * 20 - 10; // -10 to +10
                    var rotate = Math.random() * 10 - 5; // -5 to +5 degrees
                    
                    element.animate({
                        transform: 'translate(' + moveX + 'px, ' + moveY + 'px) rotate(' + rotate + 'deg)'
                    }, duration, 'linear', function() {
                        floatElement(); // Infinite loop
                    });
                }
                
                // Start with a delay based on index
                setTimeout(function() {
                    element.css('transform', 'translate(' + startX + 'px, ' + startY + 'px)');
                    floatElement();
                }, index * 300);
            });
        }
    </script>
</body>
</html>

       