<?php
/**
 * Panadite Academy - Password Reset Page
 * Complete the password reset process with secure token verification
 */

// Start session and include necessary files
session_start();
require_once __DIR__ . '/../includes/site_config.php';
require_once __DIR__ . '/../includes/db_connection.php';

// Initialize variables
$error = '';
$success = '';
$valid_token = false;
$token = '';
$user_data = null;

// Get token from URL
if (isset($_GET['token']) && !empty($_GET['token'])) {
    $token = trim($_GET['token']);
    
    // Verify token exists and is not expired
    $stmt = $conn->prepare("
        SELECT prt.*, u.user_id, u.first_name, u.last_name, u.email 
        FROM password_reset_tokens prt 
        JOIN users u ON prt.user_id = u.user_id 
        WHERE prt.token = ? AND prt.expires_at > NOW() AND prt.used = 0
    ");
    $stmt->bind_param("s", $token);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        $user_data = $result->fetch_assoc();
        $valid_token = true;
    } else {
        $error = 'Invalid or expired reset token. Please request a new password reset.';
    }
} else {
    $error = 'No reset token provided. Please check your email for the correct link.';
}

// Process password reset form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $valid_token) {
    $new_password = trim($_POST['new_password']);
    $confirm_password = trim($_POST['confirm_password']);
    
    if (empty($new_password)) {
        $error = 'Please enter a new password.';
    } elseif (strlen($new_password) < 8) {
        $error = 'Password must be at least 8 characters long.';
    } elseif ($new_password !== $confirm_password) {
        $error = 'Passwords do not match.';
    } else {
        // Hash the new password
        $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
        
        // Update user's password
        $update_stmt = $conn->prepare("UPDATE users SET password = ? WHERE user_id = ?");
        $update_stmt->bind_param("si", $hashed_password, $user_data['user_id']);
        
        if ($update_stmt->execute()) {
            // Mark token as used
            $token_stmt = $conn->prepare("UPDATE password_reset_tokens SET used = 1 WHERE token = ?");
            $token_stmt->bind_param("s", $token);
            $token_stmt->execute();
            
            $success = "Your password has been successfully updated! You can now login with your new password.";
        } else {
            $error = "Failed to update password. Please try again.";
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reset Password - Panadite Academy</title>
    <meta name="description" content="Reset your Panadite Academy password securely with our password reset system.">
    
    <!-- Bootstrap 5.2.3 -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <!-- Animate.css -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css">
    
    <style>
        :root {
            --primary-color: #276BE4;
            --secondary-color: #FFA915;
            --success-color: #28a745;
            --danger-color: #dc3545;
            --warning-color: #ffc107;
            --light-bg: #f8f9fa;
            --white: #ffffff;
            --text-dark: #333333;
            --text-muted: #6c757d;
            --border-color: #e1e5e9;
        }

        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }

        .reset-password-container {
            max-width: 520px;
            margin: 50px auto;
            background: var(--white);
            border-radius: 20px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.15);
            overflow: hidden;
            position: relative;
        }

        .reset-password-container::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: linear-gradient(90deg, var(--primary-color), var(--secondary-color));
        }

        .logo-container {
            text-align: center;
            padding: 40px 40px 20px 40px;
            background: linear-gradient(135deg, #f8f9fa, #e9ecef);
        }

        .logo-container img {
            max-width: 100px;
            height: auto;
            margin-bottom: 15px;
            border-radius: 50%;
            border: 3px solid var(--primary-color);
            padding: 5px;
        }

        .tagline {
            color: var(--secondary-color);
            font-style: italic;
            font-size: 14px;
            font-weight: 500;
            margin-bottom: 0;
        }

        .reset-password-form {
            padding: 30px 40px 40px 40px;
        }

        .section-title {
            text-align: center;
            margin-bottom: 30px;
        }

        .section-title h2 {
            color: var(--text-dark);
            font-weight: 700;
            font-size: 26px;
            margin-bottom: 8px;
        }

        .section-title p {
            color: var(--text-muted);
            font-size: 15px;
            margin-bottom: 0;
        }

        .user-info {
            background: linear-gradient(135deg, #e3f2fd, #f3e5f5);
            border: 1px solid #e1bee7;
            border-radius: 12px;
            padding: 15px 20px;
            margin-bottom: 25px;
            text-align: center;
        }

        .user-info h5 {
            color: var(--primary-color);
            margin-bottom: 5px;
            font-weight: 600;
        }

        .user-info p {
            color: var(--text-muted);
            margin-bottom: 0;
            font-size: 14px;
        }

        .form-control {
            border: 2px solid var(--border-color);
            border-radius: 12px;
            padding: 14px 16px;
            font-size: 16px;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            background-color: #fafbfc;
        }

        .form-control:focus {
            border-color: var(--primary-color);
            background-color: rgba(39, 107, 228, 0.05);
            box-shadow: 0 0 0 0.2rem rgba(39, 107, 228, 0.25);
            transform: translateY(-2px);
        }

        .input-group-text {
            background: linear-gradient(135deg, var(--primary-color), #1e5bcc);
            border: none;
            color: white;
            border-radius: 12px 0 0 12px;
            padding: 14px 16px;
        }

        .input-group .form-control {
            border-left: none;
            border-radius: 0 12px 12px 0;
        }

        .password-strength {
            margin-top: 8px;
        }

        .password-strength-meter {
            height: 4px;
            background: #e9ecef;
            border-radius: 2px;
            overflow: hidden;
            transition: all 0.3s ease;
        }

        .strength-weak .password-strength-meter {
            background: linear-gradient(90deg, var(--danger-color) 33%, #e9ecef 33%);
        }

        .strength-medium .password-strength-meter {
            background: linear-gradient(90deg, var(--warning-color) 66%, #e9ecef 66%);
        }

        .strength-strong .password-strength-meter {
            background: var(--success-color);
        }

        .btn-reset {
            background: linear-gradient(135deg, var(--primary-color), #1e5bcc);
            border: none;
            border-radius: 12px;
            color: white;
            font-weight: 600;
            padding: 16px 24px;
            font-size: 17px;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            width: 100%;
            height: 54px;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
        }

        .btn-reset:hover {
            background: linear-gradient(135deg, #1e5bcc, #1752b8);
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(39, 107, 228, 0.3);
        }

        .btn-success-final {
            background: linear-gradient(135deg, var(--success-color), #20c997);
        }

        .btn-success-final:hover {
            background: linear-gradient(135deg, #20c997, #198754);
        }

        .alert {
            border-radius: 12px;
            border: none;
            padding: 16px 20px;
            margin-bottom: 25px;
            font-weight: 500;
        }

        .alert-success {
            background: linear-gradient(135deg, #d4edda, #c3e6cb);
            color: #155724;
            border-left: 4px solid var(--success-color);
        }

        .alert-danger {
            background: linear-gradient(135deg, #f8d7da, #f5c6cb);
            color: #721c24;
            border-left: 4px solid var(--danger-color);
        }

        .password-toggle {
            position: absolute;
            right: 15px;
            top: 50%;
            transform: translateY(-50%);
            cursor: pointer;
            color: var(--text-muted);
            z-index: 10;
            transition: color 0.3s ease;
        }

        .password-toggle:hover {
            color: var(--primary-color);
        }

        .password-container {
            position: relative;
        }

        .reset-links {
            text-align: center;
            margin-top: 25px;
            padding-top: 20px;
            border-top: 1px solid var(--border-color);
        }

        .reset-links a {
            color: var(--primary-color);
            text-decoration: none;
            font-weight: 500;
            font-size: 15px;
            transition: all 0.3s ease;
            margin: 0 10px;
        }

        .reset-links a:hover {
            color: #1e5bcc;
            text-decoration: underline;
        }

        .success-checkmark {
            font-size: 64px;
            color: var(--success-color);
            margin-bottom: 20px;
            animation: checkmark 0.8s ease-out;
        }

        @keyframes checkmark {
            0% {
                transform: scale(0);
                opacity: 0;
            }
            50% {
                transform: scale(1.2);
                opacity: 0.8;
            }
            100% {
                transform: scale(1);
                opacity: 1;
            }
        }

        .error-icon {
            font-size: 48px;
            color: var(--danger-color);
            margin-bottom: 20px;
        }

        @media (max-width: 768px) {
            .reset-password-container {
                margin: 20px;
                border-radius: 15px;
            }
            
            .reset-password-form {
                padding: 20px 30px;
            }
            
            .logo-container {
                padding: 30px 30px 15px 30px;
            }
        }
    </style>
</head>
<body>
    <!-- Include navbar -->
    <?php include '../includes/navbar.php'; ?>
    
    <div class="container">
        <div class="reset-password-container animate__animated animate__fadeIn">
            <div class="logo-container">
                <img src="/assets/images/logo.jpeg" alt="Panadite Academy Logo" class="animate__animated animate__zoomIn">
                <p class="tagline animate__animated animate__fadeIn animate__delay-1s">Get ahead of the curve</p>
            </div>
            
            <div class="reset-password-form">
                <?php if (!$valid_token): ?>
                    <!-- Invalid Token State -->
                    <div class="text-center animate__animated animate__fadeIn">
                        <div class="error-icon">
                            <i class="fas fa-exclamation-triangle"></i>
                        </div>
                        <div class="section-title">
                            <h2>Invalid Reset Link ❌</h2>
                            <p>This password reset link is invalid or has expired</p>
                        </div>
                        
                        <div class="alert alert-danger animate__animated animate__slideInDown">
                            <i class="fas fa-exclamation-triangle me-2"></i>
                            <?php echo $error; ?>
                        </div>
                        
                        <div class="reset-links animate__animated animate__fadeIn animate__delay-1s">
                            <a href="forgot-password.php" class="btn btn-reset">
                                <i class="fas fa-redo"></i>
                                Request New Reset Link
                            </a>
                        </div>
                    </div>
                    
                <?php elseif ($success): ?>
                    <!-- Success State -->
                    <div class="text-center animate__animated animate__fadeIn">
                        <div class="success-checkmark">
                            <i class="fas fa-check-circle"></i>
                        </div>
                        <div class="section-title">
                            <h2>Password Updated! 🎉</h2>
                            <p>Your password has been successfully changed</p>
                        </div>
                        
                        <div class="alert alert-success animate__animated animate__slideInDown">
                            <i class="fas fa-check-circle me-2"></i>
                            <?php echo $success; ?>
                        </div>
                        
                        <div class="reset-links animate__animated animate__fadeIn animate__delay-1s">
                            <a href="login.php" class="btn btn-success-final">
                                <i class="fas fa-sign-in-alt"></i>
                                Login Now
                            </a>
                        </div>
                    </div>
                    
                <?php else: ?>
                    <!-- Reset Form -->
                    <div class="section-title animate__animated animate__fadeIn">
                        <h2>Reset Password 🔐</h2>
                        <p>Enter your new password below</p>
                    </div>
                    
                    <?php if ($user_data): ?>
                    <div class="user-info animate__animated animate__slideInDown">
                        <h5><i class="fas fa-user me-2"></i><?php echo htmlspecialchars($user_data['first_name'] . ' ' . $user_data['last_name']); ?></h5>
                        <p><?php echo htmlspecialchars($user_data['email']); ?></p>
                    </div>
                    <?php endif; ?>
                    
                    <?php if ($error): ?>
                    <div class="alert alert-danger animate__animated animate__shake">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        <?php echo $error; ?>
                    </div>
                    <?php endif; ?>
                    
                    <form method="POST" action="" class="animate__animated animate__fadeIn animate__delay-1s">
                        <input type="hidden" name="token" value="<?php echo htmlspecialchars($token); ?>">
                        
                        <div class="mb-4">
                            <label for="new_password" class="form-label">New Password</label>
                            <div class="input-group password-container">
                                <span class="input-group-text">
                                    <i class="fas fa-lock"></i>
                                </span>
                                <input type="password" 
                                       class="form-control" 
                                       id="new_password" 
                                       name="new_password" 
                                       required 
                                       placeholder="Enter your new password"
                                       minlength="8">
                                <span class="password-toggle" onclick="togglePassword('new_password')">
                                    <i class="far fa-eye"></i>
                                </span>
                            </div>
                            <div class="password-strength" id="password-strength">
                                <div class="password-strength-meter"></div>
                                <small class="text-muted">Password should be at least 8 characters long</small>
                            </div>
                        </div>
                        
                        <div class="mb-4">
                            <label for="confirm_password" class="form-label">Confirm New Password</label>
                            <div class="input-group password-container">
                                <span class="input-group-text">
                                    <i class="fas fa-check-double"></i>
                                </span>
                                <input type="password" 
                                       class="form-control" 
                                       id="confirm_password" 
                                       name="confirm_password" 
                                       required 
                                       placeholder="Confirm your new password">
                                <span class="password-toggle" onclick="togglePassword('confirm_password')">
                                    <i class="far fa-eye"></i>
                                </span>
                            </div>
                            <div id="password-match" class="form-text"></div>
                        </div>
                        
                        <button type="submit" class="btn btn-reset animate__animated animate__pulse animate__infinite animate__slow">
                            <i class="fas fa-key"></i>
                            Update Password
                        </button>
                    </form>
                    
                    <div class="reset-links animate__animated animate__fadeIn animate__delay-1s">
                        <a href="login.php">
                            <i class="fas fa-arrow-left me-1"></i>
                            Back to Login
                        </a>
                        <a href="/index.php">
                            <i class="fas fa-home me-1"></i>
                            Home
                        </a>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <!-- Include footer -->
    <?php include '../includes/footer.php'; ?>
    
    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
        function togglePassword(inputId) {
            const input = document.getElementById(inputId);
            const icon = input.parentElement.querySelector('.password-toggle i');
            
            if (input.type === 'password') {
                input.type = 'text';
                icon.classList.remove('fa-eye');
                icon.classList.add('fa-eye-slash');
            } else {
                input.type = 'password';
                icon.classList.remove('fa-eye-slash');
                icon.classList.add('fa-eye');
            }
        }

        document.addEventListener('DOMContentLoaded', function() {
            const newPasswordInput = document.getElementById('new_password');
            const confirmPasswordInput = document.getElementById('confirm_password');
            const passwordStrength = document.getElementById('password-strength');
            const passwordMatch = document.getElementById('password-match');
            
            if (newPasswordInput && passwordStrength) {
                newPasswordInput.addEventListener('input', function() {
                    const password = this.value;
                    let strength = 0;
                    
                    // Password strength calculation
                    if (password.length >= 8) strength += 1;
                    if (password.match(/[a-z]+/)) strength += 1;
                    if (password.match(/[A-Z]+/)) strength += 1;
                    if (password.match(/[0-9]+/)) strength += 1;
                    if (password.match(/[^a-zA-Z0-9]+/)) strength += 1;
                    
                    // Remove existing strength classes
                    passwordStrength.classList.remove('strength-weak', 'strength-medium', 'strength-strong');
                    
                    if (strength < 3) {
                        passwordStrength.classList.add('strength-weak');
                    } else if (strength < 5) {
                        passwordStrength.classList.add('strength-medium');
                    } else {
                        passwordStrength.classList.add('strength-strong');
                    }
                });
                
                // Auto-focus new password input
                setTimeout(() => newPasswordInput.focus(), 500);
            }
            
            if (confirmPasswordInput && passwordMatch) {
                confirmPasswordInput.addEventListener('input', function() {
                    const newPassword = newPasswordInput.value;
                    const confirmPassword = this.value;
                    
                    if (confirmPassword.length > 0) {
                        if (newPassword === confirmPassword) {
                            passwordMatch.innerHTML = '<i class="fas fa-check text-success me-1"></i><span class="text-success">Passwords match!</span>';
                        } else {
                            passwordMatch.innerHTML = '<i class="fas fa-times text-danger me-1"></i><span class="text-danger">Passwords do not match</span>';
                        }
                    } else {
                        passwordMatch.innerHTML = '';
                    }
                });
            }
        });
    </script>
</body>
</html>
