<?php
// CMS Core Functions for Panadite Academy

// Database connection check
function ensureCMSDatabase() {
    global $conn;
    if (!$conn || $conn->connect_error) {
        return false;
    }
    return true;
}

// Admin Authentication Functions
function getCMSAdmin($admin_id) {
    global $conn;
    if (!ensureCMSDatabase()) return false;
    
    $stmt = $conn->prepare("SELECT * FROM cms_admins WHERE id = ? AND is_active = 1");
    $stmt->bind_param("i", $admin_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 1) {
        return $result->fetch_assoc();
    }
    return false;
}

function authenticateCMSAdmin($username, $password) {
    global $conn;
    if (!ensureCMSDatabase()) return false;
    
    $stmt = $conn->prepare("SELECT * FROM cms_admins WHERE username = ? AND is_active = 1");
    $stmt->bind_param("s", $username);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 1) {
        $admin = $result->fetch_assoc();
        if (password_verify($password, $admin['password_hash'])) {
            // Update last login
            $update_stmt = $conn->prepare("UPDATE cms_admins SET last_login = CURRENT_TIMESTAMP WHERE id = ?");
            $update_stmt->bind_param("i", $admin['id']);
            $update_stmt->execute();
            
            return $admin;
        }
    }
    return false;
}

// Content Management Functions
function getCMSContent($page_name = null, $section_key = null) {
    global $conn;
    if (!ensureCMSDatabase()) return [];
    
    $sql = "SELECT * FROM cms_content WHERE is_active = 1";
    $params = [];
    $types = "";
    
    if ($page_name) {
        $sql .= " AND page_name = ?";
        $params[] = $page_name;
        $types .= "s";
    }
    
    if ($section_key) {
        $sql .= " AND section_key = ?";
        $params[] = $section_key;
        $types .= "s";
    }
    
    $sql .= " ORDER BY page_name, display_order, section_key";
    
    $stmt = $conn->prepare($sql);
    if (!empty($params)) {
        $stmt->bind_param($types, ...$params);
    }
    $stmt->execute();
    $result = $stmt->get_result();
    
    $content = [];
    while ($row = $result->fetch_assoc()) {
        if ($section_key) {
            return $row; // Return single item if specific section requested
        }
        $content[] = $row;
    }
    
    return $content;
}

function updateCMSContent($page_name, $section_key, $content_value, $content_type = 'text', $admin_id = null) {
    global $conn;
    if (!ensureCMSDatabase()) return false;
    
    // Get old value for logging
    $old_content = getCMSContent($page_name, $section_key);
    $old_value = $old_content ? $old_content['content_value'] : null;
    
    // Update or insert content
    $stmt = $conn->prepare("
        INSERT INTO cms_content (page_name, section_key, content_type, content_value, updated_at) 
        VALUES (?, ?, ?, ?, CURRENT_TIMESTAMP)
        ON DUPLICATE KEY UPDATE 
        content_value = VALUES(content_value), 
        content_type = VALUES(content_type),
        updated_at = CURRENT_TIMESTAMP
    ");
    
    $stmt->bind_param("ssss", $page_name, $section_key, $content_type, $content_value);
    $success = $stmt->execute();
    
    if ($success && $admin_id) {
        logCMSActivity($admin_id, 'content_update', 'cms_content', null, $old_value, $content_value);
    }
    
    return $success;
}

// Image Management Functions  
function uploadCMSImage($image_key, $file, $alt_text = '', $caption = '', $admin_id = null) {
    global $conn;
    if (!ensureCMSDatabase()) return false;
    
    // Validate file
    $allowed_types = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
    if (!in_array($file['type'], $allowed_types)) {
        return ['error' => 'Invalid file type. Only JPEG, PNG, GIF, and WebP are allowed.'];
    }
    
    $max_size = 5 * 1024 * 1024; // 5MB
    if ($file['size'] > $max_size) {
        return ['error' => 'File too large. Maximum size is 5MB.'];
    }
    
    // Create upload directory if it doesn't exist
    $upload_dir = '../../uploads/cms/';
    if (!file_exists($upload_dir)) {
        mkdir($upload_dir, 0755, true);
    }
    
    // Generate unique filename
    $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
    $stored_filename = $image_key . '_' . time() . '.' . $extension;
    $file_path = $upload_dir . $stored_filename;
    
    if (move_uploaded_file($file['tmp_name'], $file_path)) {
        // Get image dimensions
        list($width, $height) = getimagesize($file_path);
        
        // Save to database
        $stmt = $conn->prepare("
            INSERT INTO cms_images (image_key, original_filename, stored_filename, file_path, alt_text, caption, file_size, mime_type, width, height)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE
            original_filename = VALUES(original_filename),
            stored_filename = VALUES(stored_filename),
            file_path = VALUES(file_path),
            alt_text = VALUES(alt_text),
            caption = VALUES(caption),
            file_size = VALUES(file_size),
            mime_type = VALUES(mime_type),
            width = VALUES(width),
            height = VALUES(height),
            upload_date = CURRENT_TIMESTAMP
        ");
        
        $stmt->bind_param("ssssssisii", 
            $image_key, 
            $file['name'], 
            $stored_filename, 
            'uploads/cms/' . $stored_filename,
            $alt_text, 
            $caption, 
            $file['size'], 
            $file['type'], 
            $width, 
            $height
        );
        
        $success = $stmt->execute();
        
        if ($success && $admin_id) {
            logCMSActivity($admin_id, 'image_upload', 'cms_images', null, null, $image_key);
        }
        
        return $success ? ['success' => true, 'file_path' => 'uploads/cms/' . $stored_filename] : ['error' => 'Database error'];
    }
    
    return ['error' => 'Failed to upload file'];
}

function getCMSImage($image_key) {
    global $conn;
    if (!ensureCMSDatabase()) return false;
    
    $stmt = $conn->prepare("SELECT * FROM cms_images WHERE image_key = ? AND is_active = 1");
    $stmt->bind_param("s", $image_key);
    $stmt->execute();
    $result = $stmt->get_result();
    
    return $result->num_rows === 1 ? $result->fetch_assoc() : false;
}

function getAllCMSImages() {
    global $conn;
    if (!ensureCMSDatabase()) return [];
    
    $result = $conn->query("SELECT * FROM cms_images WHERE is_active = 1 ORDER BY upload_date DESC");
    
    $images = [];
    while ($row = $result->fetch_assoc()) {
        $images[] = $row;
    }
    
    return $images;
}

// Activity Logging
function logCMSActivity($admin_id, $action_type, $target_table = null, $target_id = null, $old_value = null, $new_value = null) {
    global $conn;
    if (!ensureCMSDatabase()) return false;
    
    $ip_address = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
    
    $stmt = $conn->prepare("
        INSERT INTO cms_activity_log (admin_id, action_type, target_table, target_id, old_value, new_value, ip_address, user_agent)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?)
    ");
    
    $stmt->bind_param("ississss", $admin_id, $action_type, $target_table, $target_id, $old_value, $new_value, $ip_address, $user_agent);
    return $stmt->execute();
}

function getRecentCMSActivity($limit = 10) {
    global $conn;
    if (!ensureCMSDatabase()) return [];
    
    $stmt = $conn->prepare("
        SELECT al.*, a.username, a.full_name 
        FROM cms_activity_log al 
        LEFT JOIN cms_admins a ON al.admin_id = a.id 
        ORDER BY al.timestamp DESC 
        LIMIT ?
    ");
    $stmt->bind_param("i", $limit);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $activities = [];
    while ($row = $result->fetch_assoc()) {
        $activities[] = $row;
    }
    
    return $activities;
}

// Statistics Functions
function getCMSContentStats() {
    global $conn;
    if (!ensureCMSDatabase()) return ['total_content' => 0, 'total_images' => 0, 'active_pages' => 0];
    
    $stats = [];
    
    // Total content sections
    $result = $conn->query("SELECT COUNT(*) as count FROM cms_content WHERE is_active = 1");
    $stats['total_content'] = $result->fetch_assoc()['count'];
    
    // Total images
    $result = $conn->query("SELECT COUNT(*) as count FROM cms_images WHERE is_active = 1");
    $stats['total_images'] = $result->fetch_assoc()['count'];
    
    // Active pages
    $result = $conn->query("SELECT COUNT(DISTINCT page_name) as count FROM cms_content WHERE is_active = 1");
    $stats['active_pages'] = $result->fetch_assoc()['count'];
    
    return $stats;
}

// Helper Functions
function formatActivityType($action_type) {
    $types = [
        'content_update' => 'Content Updated',
        'image_upload' => 'Image Uploaded',
        'image_delete' => 'Image Deleted',
        'login' => 'Admin Login',
        'logout' => 'Admin Logout'
    ];
    
    return $types[$action_type] ?? ucfirst(str_replace('_', ' ', $action_type));
}

function getActivityIcon($action_type) {
    $icons = [
        'content_update' => 'edit',
        'image_upload' => 'upload',
        'image_delete' => 'trash',
        'login' => 'sign-in-alt',
        'logout' => 'sign-out-alt'
    ];
    
    return $icons[$action_type] ?? 'circle';
}

function formatTimeAgo($timestamp) {
    $time = time() - strtotime($timestamp);
    
    if ($time < 60) return 'just now';
    if ($time < 3600) return floor($time / 60) . ' minutes ago';
    if ($time < 86400) return floor($time / 3600) . ' hours ago';
    if ($time < 2592000) return floor($time / 86400) . ' days ago';
    
    return date('M j, Y', strtotime($timestamp));
}

// Content Helper for Frontend
function cms($page_name = null, $section_key = null, $default_content = '') {
    $content = getCMSContent($page_name, $section_key);
    $content_html = '';
    
    if ($page_name && $section_key) {
        // getCMSContent returns a single row when section_key is specified
        if (is_array($content) && isset($content['content_value'])) {
            $content_html = $content['content_value'];
        } else {
            $content_html = $default_content;
        }
    } elseif ($page_name && is_array($content)) {
        // Return all content for the page
        foreach ($content as $section) {
            if (isset($section['content_value'])) {
                $content_html .= $section['content_value'] . "\n";
            }
        }
        $content_html = trim($content_html);
    } else {
        $content_html = $default_content;
    }
    
    // Add inline editing attributes if admin is logged in
    if (isAdminLoggedIn() && $page_name && $section_key) {
        return '<span data-cms-content data-cms-page="' . htmlspecialchars($page_name) . '" data-cms-section="' . htmlspecialchars($section_key) . '">' . $content_html . '</span>';
    }
    
    return $content_html;
}

function cms_image($image_key, $default = '') {
    $image = getCMSImage($image_key);
    return $image ? $image['file_path'] : $default;
}

function isAdminLoggedIn() {
    // Check CMS admin login (separate CMS system)
    if (isset($_SESSION['cms_admin_id']) && !empty($_SESSION['cms_admin_id'])) {
        return true;
    }
    
    // Check main application admin login (unified system)
    if (isset($_SESSION['user_role']) && $_SESSION['user_role'] === 'admin' && isset($_SESSION['logged_in']) && $_SESSION['logged_in'] == 1) {
        return true;
    }
    
    return false;
}
?>
