<?php
// CMS Image Helper Functions for Inline Editing
require_once 'cms_functions.php';

/**
 * Generate editable image HTML with CMS support
 * @param string $image_key - Unique identifier for the image
 * @param string $default_src - Default image path if no CMS image exists
 * @param string $alt_text - Alternative text for accessibility
 * @param array $attributes - Additional HTML attributes
 * @return string - HTML img tag with CMS editing capabilities
 */
function cms_editable_image($image_key, $default_src = '/assets/images/placeholder-icon.png', $alt_text = '', $attributes = []) {
    // Get image from CMS database
    $cms_image = getCMSImage($image_key);
    $src = $cms_image ? $cms_image['file_path'] : $default_src;
    $alt = $cms_image && $cms_image['alt_text'] ? $cms_image['alt_text'] : $alt_text;
    
    // Build attributes
    $attr_string = '';
    foreach ($attributes as $key => $value) {
        $attr_string .= ' ' . htmlspecialchars($key) . '="' . htmlspecialchars($value) . '"';
    }
    
    // Add CMS editing attributes if admin is logged in
    $cms_attrs = '';
    if (isAdminLoggedIn()) {
        $cms_attrs = ' data-cms-image-key="' . htmlspecialchars($image_key) . '" class="cms-editable-image"';
    }
    
    return '<img src="' . htmlspecialchars($src) . '" alt="' . htmlspecialchars($alt) . '"' . $cms_attrs . $attr_string . '>';
}

/**
 * Include CMS image editing assets (JS/CSS)
 * Call this in the head section of pages that need image editing
 */
function include_cms_image_assets() {
    if (!isAdminLoggedIn()) return '';
    
    return '
    <!-- CMS Image Editing Assets -->
    <script src="/cms/functions/inline-image-editor.js" defer></script>
    <script>
        // Set admin mode flag for JavaScript
        localStorage.setItem("cms_admin_logged_in", "true");
        document.body.setAttribute("data-admin-mode", "true");
    </script>
    ';
}

/**
 * Generate team member card with editable image
 * @param string $member_key - Unique key for the team member
 * @param array $member_data - Member information (title, description)
 * @return string - HTML for team member card
 */
function cms_team_member_card($member_key, $member_data = []) {
    $title = $member_data['title'] ?? 'Team Member';
    $description = $member_data['description'] ?? 'Professional description';
    
    // Use CMS function for editable text content
    $editable_title = cms('about', $member_key . '_title', $title);
    $editable_description = cms('about', $member_key . '_description', $description);
    
    // Generate editable image
    $image_html = cms_editable_image(
        $member_key . '_icon',
        '/assets/images/team-placeholder.png',
        $title . ' Icon',
        ['class' => 'team-icon-img']
    );
    
    return '
    <div class="col-md-4 mb-4">
        <div class="team-member-card text-center">
            <div class="team-icon">
                ' . $image_html . '
            </div>
            <h5 class="mt-3">' . $editable_title . '</h5>
            <p class="text-muted">' . $editable_description . '</p>
        </div>
    </div>';
}

/**
 * Setup CMS database if not exists (call once)
 */
function setup_cms_database() {
    global $conn;
    
    // Check if cms_images table exists
    $result = $conn->query("SHOW TABLES LIKE 'cms_images'");
    if ($result->num_rows == 0) {
        // Create cms_images table
        $sql = "CREATE TABLE IF NOT EXISTS cms_images (
            id INT AUTO_INCREMENT PRIMARY KEY,
            image_key VARCHAR(100) NOT NULL UNIQUE,
            original_filename VARCHAR(255),
            stored_filename VARCHAR(255),
            file_path VARCHAR(500),
            alt_text TEXT,
            caption TEXT,
            file_size INT,
            mime_type VARCHAR(50),
            width INT,
            height INT,
            upload_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            is_active BOOLEAN DEFAULT TRUE
        )";
        
        if ($conn->query($sql)) {
            error_log("CMS Images table created successfully");
        } else {
            error_log("Error creating cms_images table: " . $conn->error);
        }
    }
    
    // Create uploads directory
    $upload_dir = __DIR__ . '/../../uploads/cms_images/';
    if (!file_exists($upload_dir)) {
        mkdir($upload_dir, 0755, true);
        error_log("Created CMS images upload directory");
    }
}

// Auto-setup on include
setup_cms_database();
?>
