<?php
/**
 * Enhanced Video Player Component for URL Videos
 * Integrates with existing video_progress tracking system
 * Supports all platforms: YouTube, TikTok, Instagram, Facebook, Vimeo, etc.
 */

// Ensure we have video data
if (!isset($video) || empty($video)) {
    echo '<div class="alert alert-danger">Video data not available</div>';
    return;
}

// Get student progress if logged in
$student_progress = 0;
$is_completed = false;
$last_position = 0;

if (isset($_SESSION['user_id']) && $_SESSION['user_role'] === 'student') {
    $student_id = $_SESSION['user_id'];
    $progress_stmt = $conn->prepare("
        SELECT completion_percentage, completed, last_position 
        FROM video_progress 
        WHERE video_id = ? AND student_id = ?
    ");
    if ($progress_stmt) {
        $progress_stmt->bind_param("ii", $video['video_id'], $student_id);
        $progress_stmt->execute();
        $progress_result = $progress_stmt->get_result();
        if ($progress_data = $progress_result->fetch_assoc()) {
            $student_progress = $progress_data['completion_percentage'] ?? 0;
            $is_completed = $progress_data['completed'] ?? false;
            $last_position = $progress_data['last_position'] ?? 0;
        }
    }
}

$video_duration = $video['duration'] ?? '0:00';
$platform = $video['platform'] ?? 'unknown';
$embed_code = $video['embed_code'] ?? '';
?>

<div class="url-video-player-container">
    
    <!-- Video Player Area -->
    <div class="card mb-4">
        <div class="card-header d-flex justify-content-between align-items-center">
            <h5 class="mb-0">
                <i class="fas fa-play-circle text-primary me-2"></i>
                <?= htmlspecialchars($video['title']) ?>
            </h5>
            
            <!-- Platform Badge -->
            <?php
            $platform_badges = [
                'youtube' => ['fab fa-youtube', 'badge-danger'],
                'vimeo' => ['fab fa-vimeo-v', 'badge-info'],
                'tiktok' => ['fab fa-tiktok', 'badge-dark'],
                'instagram' => ['fab fa-instagram', 'badge-gradient-instagram'],
                'facebook' => ['fab fa-facebook', 'badge-primary'],
                'twitter' => ['fab fa-twitter', 'badge-info']
            ];
            
            $platform_lower = strtolower($platform);
            $badge_info = $platform_badges[$platform_lower] ?? ['fas fa-globe', 'badge-secondary'];
            ?>
            
            <div class="platform-info">
                <span class="badge <?= $badge_info[1] ?> platform-badge">
                    <i class="<?= $badge_info[0] ?> me-1"></i>
                    <?= ucfirst($platform) ?>
                </span>
            </div>
        </div>
        
        <div class="card-body p-0">
            <!-- Video Embed Area -->
            <div class="video-embed-container" id="videoPreviewArea">
                <?php if (!empty($embed_code)): ?>
                    <div class="embed-responsive embed-responsive-16by9">
                        <?= $embed_code ?>
                    </div>
                <?php else: ?>
                    <div class="no-embed-fallback text-center py-5">
                        <i class="fas fa-video fa-3x text-muted mb-3"></i>
                        <h5 class="text-muted">Video Preview Not Available</h5>
                        <p class="text-muted">Click the link below to watch this video</p>
                        <a href="<?= htmlspecialchars($video['video_url']) ?>" target="_blank" class="btn btn-primary">
                            <i class="fas fa-external-link-alt me-2"></i>Watch Video
                        </a>
                    </div>
                <?php endif; ?>
            </div>
            
            <!-- Progress Control Panel for URL Videos -->
            <div class="url-video-controls bg-light border-top p-3">
                <div class="row align-items-center">
                    
                    <!-- Progress Display -->
                    <div class="col-md-6">
                        <div class="progress-info">
                            <div class="d-flex justify-content-between mb-2">
                                <span class="text-muted">
                                    <i class="fas fa-clock me-1"></i>
                                    Duration: <strong><?= htmlspecialchars($video_duration) ?></strong>
                                </span>
                                <span class="progress-text">
                                    Progress: <strong><?= $student_progress ?>%</strong>
                                </span>
                            </div>
                            
                            <div class="progress mb-2" style="height: 8px;">
                                <div class="progress-bar video-progress <?= $is_completed ? 'bg-success' : 'bg-primary' ?>" 
                                     role="progressbar" 
                                     style="width: <?= $student_progress ?>%"
                                     aria-valuenow="<?= $student_progress ?>" 
                                     aria-valuemin="0" 
                                     aria-valuemax="100">
                                </div>
                            </div>
                            
                            <div class="completion-status text-muted small">
                                <?php if ($is_completed): ?>
                                    <i class="fas fa-check-circle text-success me-1"></i>
                                    Video completed
                                <?php elseif ($student_progress > 0): ?>
                                    <i class="fas fa-play-circle text-primary me-1"></i>
                                    In progress
                                <?php else: ?>
                                    <i class="fas fa-circle text-muted me-1"></i>
                                    Not started
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Manual Control Buttons -->
                    <div class="col-md-6">
                        <div class="control-buttons text-md-end">
                            
                            <?php if (isset($_SESSION['user_id']) && $_SESSION['user_role'] === 'student'): ?>
                                
                                <!-- Start Watching Button -->
                                <?php if ($student_progress == 0): ?>
                                    <button type="button" class="btn btn-success btn-sm me-2" id="startWatchingBtn">
                                        <i class="fas fa-play me-1"></i>Start Watching
                                    </button>
                                <?php endif; ?>
                                
                                <!-- Continue Watching Button -->
                                <?php if ($student_progress > 0 && !$is_completed): ?>
                                    <button type="button" class="btn btn-primary btn-sm me-2" id="continueWatchingBtn">
                                        <i class="fas fa-play me-1"></i>Continue Watching
                                    </button>
                                <?php endif; ?>
                                
                                <!-- Mark as Completed Button -->
                                <?php if (!$is_completed): ?>
                                    <button type="button" class="btn btn-outline-success btn-sm me-2" id="markCompletedBtn">
                                        <i class="fas fa-check me-1"></i>Mark Complete
                                    </button>
                                <?php endif; ?>
                                
                                <!-- Reset Progress Button -->
                                <?php if ($student_progress > 0): ?>
                                    <button type="button" class="btn btn-outline-secondary btn-sm" id="resetProgressBtn">
                                        <i class="fas fa-undo me-1"></i>Reset
                                    </button>
                                <?php endif; ?>
                                
                            <?php else: ?>
                                <div class="alert alert-info mb-0 py-2">
                                    <small><i class="fas fa-info-circle me-1"></i>Login as student to track progress</small>
                                </div>
                            <?php endif; ?>
                            
                        </div>
                    </div>
                    
                </div>
                
                <!-- Tracking Status Display -->
                <div id="trackingStatus" class="tracking-status mt-2" style="display: none;">
                    <div class="alert alert-info mb-0 py-2">
                        <i class="fas fa-satellite-dish me-2"></i>
                        <span id="trackingStatusText">Initializing video tracking...</span>
                        <div class="spinner-border spinner-border-sm ms-2" role="status" id="trackingSpinner">
                            <span class="visually-hidden">Loading...</span>
                        </div>
                    </div>
                </div>
                
            </div>
        </div>
    </div>
    
    <!-- Video Information -->
    <div class="card mb-4">
        <div class="card-body">
            <div class="row">
                <div class="col-md-8">
                    <h6 class="text-muted mb-2">Description</h6>
                    <p><?= htmlspecialchars($video['description'] ?? 'No description available.') ?></p>
                </div>
                <div class="col-md-4">
                    <div class="video-meta">
                        <div class="row g-2">
                            <div class="col-6">
                                <div class="meta-item text-center p-2 bg-light rounded">
                                    <i class="fas fa-eye text-primary"></i>
                                    <div class="small mt-1">
                                        <strong><?= number_format($video['views'] ?? 0) ?></strong><br>
                                        <span class="text-muted">Views</span>
                                    </div>
                                </div>
                            </div>
                            <div class="col-6">
                                <div class="meta-item text-center p-2 bg-light rounded">
                                    <i class="fas fa-calendar text-info"></i>
                                    <div class="small mt-1">
                                        <strong><?= date('M j', strtotime($video['created_at'] ?? 'now')) ?></strong><br>
                                        <span class="text-muted">Added</span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Progress Tracking Tip -->
    <div class="card border-info">
        <div class="card-body py-3">
            <div class="row align-items-center">
                <div class="col-md-8">
                    <h6 class="mb-1 text-info">
                        <i class="fas fa-lightbulb me-2"></i>How Progress Tracking Works for URL Videos
                    </h6>
                    <p class="mb-0 small text-muted">
                        Click "Start Watching" to begin tracking, then interact with the video. 
                        The system monitors your engagement and updates progress automatically. 
                        Mark as complete when finished watching.
                    </p>
                </div>
                <div class="col-md-4 text-md-end">
                    <div class="platform-specific-info">
                        <small class="text-muted">
                            <i class="<?= $badge_info[0] ?> me-1"></i>
                            <?= ucfirst($platform) ?> Video
                        </small>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
</div>

<!-- Include the Enhanced Video Tracker -->
<script src="../js/embedded-video-tracker.js"></script>
<script>
// Debug: Check if tracker loaded
console.log('🔍 Checking if EmbeddedVideoTracker loaded...');
console.log('EmbeddedVideoTracker available:', typeof EmbeddedVideoTracker !== 'undefined');
console.log('initEmbeddedVideoTracking function available:', typeof window.initEmbeddedVideoTracking !== 'undefined');
</script>

<script>
document.addEventListener('DOMContentLoaded', function() {
    
    <?php if (isset($_SESSION['user_id']) && $_SESSION['user_role'] === 'student'): ?>
    
    const videoId = <?= json_encode($video['video_id']) ?>;
    const courseId = <?= json_encode($video['course_id']) ?>;
    const videoDuration = <?= json_encode($video_duration) ?>;
    
    let videoTracker = null;
    
    // Initialize tracking buttons
    const startBtn = document.getElementById('startWatchingBtn');
    const continueBtn = document.getElementById('continueWatchingBtn');
    const completeBtn = document.getElementById('markCompletedBtn');
    const resetBtn = document.getElementById('resetProgressBtn');
    const trackingStatus = document.getElementById('trackingStatus');
    const statusText = document.getElementById('trackingStatusText');
    const spinner = document.getElementById('trackingSpinner');
    
    // Start watching button
    if (startBtn) {
        startBtn.addEventListener('click', function() {
            startVideoTracking();
            updateTrackingStatus('Tracking started - video progress will be monitored');
        });
    }
    
    // Continue watching button  
    if (continueBtn) {
        continueBtn.addEventListener('click', function() {
            startVideoTracking();
            updateTrackingStatus('Tracking resumed - continuing from where you left off');
        });
    }
    
    // Mark completed button
    if (completeBtn) {
        completeBtn.addEventListener('click', function() {
            if (videoTracker) {
                videoTracker.markCompleted();
            } else {
                // Manual completion without tracker
                updateVideoProgress(100, videoDuration, true);
            }
            updateTrackingStatus('Video marked as completed!');
        });
    }
    
    // Reset progress button
    if (resetBtn) {
        resetBtn.addEventListener('click', function() {
            if (confirm('Are you sure you want to reset your progress for this video?')) {
                updateVideoProgress(0, 0, false);
                location.reload(); // Refresh to show updated progress
            }
        });
    }
    
    // Start video tracking
    function startVideoTracking() {
        if (!videoTracker) {
            trackingStatus.style.display = 'block';
            updateTrackingStatus('Initializing video tracking system...');
            
            console.log('🚀 Starting video tracking...');
            console.log('Video ID:', videoId);
            console.log('Course ID:', courseId);
            console.log('Duration:', videoDuration);
            console.log('Tracker function available:', typeof window.initEmbeddedVideoTracking);
            
            // Check if main tracker is available
            if (typeof window.initEmbeddedVideoTracking === 'function') {
                try {
                    console.log('✅ Initializing main tracker...');
                    videoTracker = window.initEmbeddedVideoTracking(videoId, courseId, videoDuration);
                    
                    setTimeout(() => {
                        updateTrackingStatus('Advanced tracking active - monitoring video engagement', false);
                        hideTrackingStatus(3000);
                    }, 1000);
                    
                } catch (error) {
                    console.error('❌ Main tracker failed:', error);
                    initializeFallbackTracker();
                }
            } else {
                console.warn('⚠️ Main tracker not available, using fallback');
                initializeFallbackTracker();
            }
        }
    }
    
    // Fallback tracking system that always works
    function initializeFallbackTracker() {
        console.log('🔄 Initializing fallback tracking system...');
        updateTrackingStatus('Fallback tracking active - using manual progress system', false);
        
        // Create a simple tracker object
        videoTracker = {
            startTime: Date.now(),
            isActive: true,
            progressInterval: null,
            
            start: function() {
                console.log('📊 Fallback tracker started');
                this.isActive = true;
                this.startSimulatedProgress();
            },
            
            startSimulatedProgress: function() {
                // Start a timer to simulate progress based on user activity
                this.progressInterval = setInterval(() => {
                    if (this.isActive && isUserEngaged()) {
                        const elapsed = (Date.now() - this.startTime) / 1000; // seconds
                        const durationSeconds = parseDurationToSeconds(videoDuration);
                        const progress = Math.min(Math.round((elapsed / durationSeconds) * 100), 99);
                        
                        if (progress > 0 && progress % 5 === 0) { // Update every 5%
                            console.log(`📈 Simulated progress: ${progress}%`);
                            updateVideoProgress(progress, elapsed, false);
                        }
                    }
                }, 2000); // Check every 2 seconds
            },
            
            markCompleted: function() {
                console.log('✅ Manual completion triggered');
                this.isActive = false;
                if (this.progressInterval) {
                    clearInterval(this.progressInterval);
                }
                updateVideoProgress(100, parseDurationToSeconds(videoDuration), true);
            }
        };
        
        // Start the fallback tracker
        videoTracker.start();
        
        hideTrackingStatus(3000);
    }
    
    // Check if user is engaged (simplified version)
    function isUserEngaged() {
        // Check if page is visible and video area might be in view
        return !document.hidden && document.hasFocus();
    }
    
    // Parse duration string to seconds
    function parseDurationToSeconds(duration) {
        const parts = duration.split(':');
        if (parts.length === 2) {
            return parseInt(parts[0]) * 60 + parseInt(parts[1]);
        } else if (parts.length === 3) {
            return parseInt(parts[0]) * 3600 + parseInt(parts[1]) * 60 + parseInt(parts[2]);
        }
        return 60; // Default 1 minute if parsing fails
    }
    
    // Update tracking status display
    function updateTrackingStatus(message, showSpinner = true) {
        statusText.textContent = message;
        spinner.style.display = showSpinner ? 'inline-block' : 'none';
    }
    
    // Hide tracking status after delay
    function hideTrackingStatus(delay = 2000) {
        setTimeout(() => {
            trackingStatus.style.display = 'none';
        }, delay);
    }
    
    // Manual progress update function
    function updateVideoProgress(progress, currentTime, completed) {
        const formData = new FormData();
        formData.append('video_id', videoId);
        formData.append('course_id', courseId);
        formData.append('progress_percentage', progress);
        formData.append('current_time', currentTime);
        formData.append('completed', completed ? 1 : 0);
        
        fetch('ajax/update_video_progress.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                console.log('✅ Manual progress update successful:', data);
                updateProgressUI(data);
            } else {
                console.error('❌ Progress update failed:', data);
            }
        })
        .catch(error => {
            console.error('❌ Progress update error:', error);
        });
    }
    
    // Update progress UI
    function updateProgressUI(data) {
        const progressBar = document.querySelector('.video-progress');
        const progressText = document.querySelector('.progress-text strong');
        const completionStatus = document.querySelector('.completion-status');
        
        if (progressBar && progressText) {
            progressBar.style.width = `${data.course_progress}%`;
            progressBar.setAttribute('aria-valuenow', data.course_progress);
            progressText.textContent = `${data.course_progress}%`;
            
            if (data.course_progress >= 100) {
                progressBar.classList.remove('bg-primary');
                progressBar.classList.add('bg-success');
                
                if (completionStatus) {
                    completionStatus.innerHTML = '<i class="fas fa-check-circle text-success me-1"></i>Video completed';
                }
            }
        }
    }
    
    // Listen for progress updates from tracker
    window.addEventListener('videoProgressUpdate', function(event) {
        console.log('📊 Progress update received:', event.detail);
        updateProgressUI(event.detail);
    });

    // Initialize automatic progress tracking for supported video platforms
    <?php if (isset($video['video_url']) && !empty($video['video_url'])): ?>
    try {
        // Initialize VideoAPIIntegration for automatic progress tracking
        if (typeof VideoAPIIntegration !== 'undefined') {
            const videoAPI = new VideoAPIIntegration();
            const videoUrl = <?= json_encode($video['video_url']) ?>;
            const videoId = <?= json_encode($video['video_id']) ?>;
            const courseId = <?= json_encode($video['course_id']) ?>;
            
            console.log('🎬 Attempting to initialize automatic progress tracking...');
            console.log('Video URL:', videoUrl);
            console.log('Video ID:', videoId);
            console.log('Course ID:', courseId);
            
            // Initialize progress tracking with custom options
            const trackingSuccess = videoAPI.initializeProgressTracking(videoUrl, videoId, courseId, {
                updateInterval: 3000,  // Update every 3 seconds
                progressThreshold: 3,  // Update when progress changes by 3%
                autoStart: true
            });
            
            if (trackingSuccess) {
                console.log('✅ Automatic video progress tracking initialized successfully');
                updateTrackingStatus('🎯 Automatic progress tracking enabled', false);
                hideTrackingStatus(3000);
            } else {
                console.log('ℹ️ Platform does not support automatic progress tracking - manual controls available');
            }
        } else {
            console.log('⚠️ VideoAPIIntegration not loaded - manual tracking only');
        }
    } catch (error) {
        console.error('❌ Error initializing automatic progress tracking:', error);
        console.log('📌 Falling back to manual progress tracking');
    }
    <?php endif; ?>
    
    <?php else: ?>
    console.log('👤 No student session - progress tracking disabled');
    <?php endif; ?>
    
});
</script>

<style>
.url-video-player-container .platform-badge {
    font-size: 0.75rem;
    padding: 0.25rem 0.5rem;
}

.badge-gradient-instagram {
    background: linear-gradient(45deg, #f09433 0%,#e6683c 25%,#dc2743 50%,#cc2366 75%,#bc1888 100%);
}

.url-video-controls {
    border-top: 1px solid #dee2e6;
}

.progress {
    border-radius: 10px;
    overflow: hidden;
}

.progress-bar {
    transition: width 0.3s ease;
}

.meta-item {
    transition: all 0.3s ease;
}

.meta-item:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(0,0,0,0.1);
}

.tracking-status {
    animation: fadeIn 0.3s ease;
}

@keyframes fadeIn {
    from { opacity: 0; transform: translateY(-10px); }
    to { opacity: 1; transform: translateY(0); }
}

.control-buttons .btn {
    margin: 2px;
}

.embed-responsive {
    position: relative;
    width: 100%;
    padding-bottom: 56.25%; /* 16:9 aspect ratio */
    overflow: hidden;
}

.embed-responsive iframe,
.embed-responsive video,
.embed-responsive embed,
.embed-responsive object {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    border: 0;
}

.no-embed-fallback {
    min-height: 300px;
    display: flex;
    flex-direction: column;
    justify-content: center;
    align-items: center;
}
</style>
