<?php
/**
 * Certificate Generation Hooks
 * Automatic certificate generation for course completion and SITA accreditation
 */

require_once __DIR__ . '/../admin/api/generate_certificate.php';

class CertificateHooks {
    private $conn;
    
    public function __construct($connection) {
        $this->conn = $connection;
    }
    
    /**
     * Generate completion certificate when student completes a course
     */
    public function generateCompletionCertificate($student_id, $course_id) {
        try {
            // Check if completion certificate already exists
            $existing_stmt = $this->conn->prepare("SELECT certificate_id FROM student_certificates WHERE student_id = ? AND course_id = ? AND certificate_type = 'completion'");
            $existing_stmt->bind_param("ii", $student_id, $course_id);
            $existing_stmt->execute();
            
            if ($existing_stmt->get_result()->num_rows > 0) {
                return ['success' => false, 'message' => 'Completion certificate already exists'];
            }
            
            // Check if auto-generation is enabled
            $auto_generate = $this->getSetting('auto_generate_completion', '1');
            if ($auto_generate !== '1') {
                return ['success' => false, 'message' => 'Auto-generation disabled'];
            }
            
            // Generate certificate
            $generator = new CertificateGenerator($this->conn);
            $result = $generator->generateCertificate($student_id, 'completion', $course_id);
            
            // Send notification email to student
            $this->sendCertificateNotification($student_id, $result, 'completion');
            
            return ['success' => true, 'certificate' => $result];
            
        } catch (Exception $e) {
            error_log("Certificate Generation Error: " . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Generate accreditation certificate when SITA assessment is passed
     */
    public function generateAccreditationCertificate($student_id, $course_id, $assessment_id) {
        try {
            // Check if accreditation certificate already exists
            $existing_stmt = $this->conn->prepare("SELECT certificate_id FROM student_certificates WHERE student_id = ? AND course_id = ? AND certificate_type = 'accreditation'");
            $existing_stmt->bind_param("ii", $student_id, $course_id);
            $existing_stmt->execute();
            
            if ($existing_stmt->get_result()->num_rows > 0) {
                return ['success' => false, 'message' => 'Accreditation certificate already exists'];
            }
            
            // Check if auto-generation is enabled
            $auto_generate = $this->getSetting('auto_generate_accreditation', '1');
            if ($auto_generate !== '1') {
                return ['success' => false, 'message' => 'Auto-generation disabled'];
            }
            
            // Generate certificate
            $generator = new CertificateGenerator($this->conn);
            $result = $generator->generateCertificate($student_id, 'accreditation', $course_id);
            
            // Update accreditation request status
            $update_stmt = $this->conn->prepare("UPDATE accreditation_requests SET status = 'certificate_issued', certificate_number = ?, certificate_issued_date = NOW() WHERE student_id = ? AND course_id = ?");
            $update_stmt->bind_param("sii", $result['certificate_number'], $student_id, $course_id);
            $update_stmt->execute();
            
            // Send notification email to student
            $this->sendCertificateNotification($student_id, $result, 'accreditation');
            
            return ['success' => true, 'certificate' => $result];
            
        } catch (Exception $e) {
            error_log("Accreditation Certificate Generation Error: " . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Generate certificate for any type
     */
    public function generateCustomCertificate($student_id, $certificate_type, $course_id = null, $template_id = null) {
        try {
            $generator = new CertificateGenerator($this->conn);
            $result = $generator->generateCertificate($student_id, $certificate_type, $course_id, $template_id);
            
            // Send notification email to student
            $this->sendCertificateNotification($student_id, $result, $certificate_type);
            
            return ['success' => true, 'certificate' => $result];
            
        } catch (Exception $e) {
            error_log("Custom Certificate Generation Error: " . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Send certificate notification email to student
     */
    private function sendCertificateNotification($student_id, $certificate_data, $certificate_type) {
        // Get student info
        $student_stmt = $this->conn->prepare("SELECT first_name, last_name, email FROM users WHERE user_id = ?");
        $student_stmt->bind_param("i", $student_id);
        $student_stmt->execute();
        $student = $student_stmt->get_result()->fetch_assoc();
        
        if (!$student) return;
        
        // Prepare email content based on certificate type
        $type_messages = [
            'completion' => [
                'subject' => '🎓 Your Course Completion Certificate is Ready!',
                'title' => 'Congratulations! Your Course Completion Certificate is Ready',
                'message' => 'You have successfully completed your course and earned your certificate!'
            ],
            'accreditation' => [
                'subject' => '🏆 Your SITA Accreditation Certificate is Ready!',
                'title' => 'Congratulations! Your Professional Accreditation Certificate is Ready',
                'message' => 'You have successfully completed the SITA accreditation process and earned your professional certificate!'
            ],
            'competence' => [
                'subject' => '⭐ Your Competence Certificate is Ready!',
                'title' => 'Your Competence Certificate is Ready',
                'message' => 'You have demonstrated competence in your field and earned this certificate!'
            ]
        ];
        
        $email_config = $type_messages[$certificate_type] ?? $type_messages['completion'];
        
        $email_body = "
        <div style='font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; background: #ffffff;'>
            <div style='background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 2rem; text-align: center;'>
                <h1 style='color: white; margin: 0; font-size: 1.8rem;'>{$email_config['title']}</h1>
            </div>
            
            <div style='padding: 2rem;'>
                <p style='font-size: 1.1rem; color: #333;'>Dear {$student['first_name']},</p>
                
                <p style='line-height: 1.6; color: #555;'>{$email_config['message']}</p>
                
                <div style='background: #f8f9fa; border-left: 4px solid #28a745; padding: 1.5rem; margin: 2rem 0; border-radius: 0 8px 8px 0;'>
                    <h3 style='color: #28a745; margin-top: 0;'>📜 Certificate Details</h3>
                    <p><strong>Certificate Number:</strong> {$certificate_data['certificate_number']}</p>
                    <p><strong>Issue Date:</strong> " . date('F j, Y') . "</p>
                    <p><strong>Type:</strong> " . ucfirst($certificate_type) . " Certificate</p>
                </div>
                
                <div style='text-align: center; margin: 2rem 0;'>
                    <a href='" . $_SERVER['HTTP_HOST'] . "/student/certificates.php' 
                       style='background: #007bff; color: white; padding: 12px 30px; text-decoration: none; border-radius: 25px; font-weight: bold; display: inline-block;'>
                        🎓 View My Certificate
                    </a>
                </div>
                
                <p style='color: #666; font-size: 0.9rem;'>You can view, download, and share your certificate anytime from your student dashboard.</p>
                
                <hr style='margin: 2rem 0; border: none; border-top: 1px solid #eee;'>
                
                <div style='text-align: center; color: #888; font-size: 0.85rem;'>
                    <p><strong>Panadite Academy</strong><br>
                    Professional Development & Certification<br>
                    This certificate is digitally generated and verified.</p>
                </div>
            </div>
        </div>
        ";
        
        // Queue email using existing email service
        require_once __DIR__ . '/fast_email_service.php';
        queueEmail(
            $student['email'],
            $email_config['subject'],
            $email_body,
            'certificate_ready',
            'high'
        );
    }
    
    /**
     * Get certificate setting
     */
    private function getSetting($key, $default = '') {
        $stmt = $this->conn->prepare("SELECT setting_value FROM certificate_settings WHERE setting_key = ?");
        $stmt->bind_param("s", $key);
        $stmt->execute();
        $result = $stmt->get_result()->fetch_assoc();
        
        return $result ? $result['setting_value'] : $default;
    }
}

// Global function to trigger course completion certificate
function triggerCompletionCertificate($student_id, $course_id, $conn) {
    $hooks = new CertificateHooks($conn);
    return $hooks->generateCompletionCertificate($student_id, $course_id);
}

// Global function to trigger accreditation certificate
function triggerAccreditationCertificate($student_id, $course_id, $assessment_id, $conn) {
    $hooks = new CertificateHooks($conn);
    return $hooks->generateAccreditationCertificate($student_id, $course_id, $assessment_id);
}

// Global function to trigger custom certificate
function triggerCustomCertificate($student_id, $certificate_type, $course_id, $template_id, $conn) {
    $hooks = new CertificateHooks($conn);
    return $hooks->generateCustomCertificate($student_id, $certificate_type, $course_id, $template_id);
}
?>
