<?php
/**
 * Email Inbox Service for Panadite Academy
 * IMAP-based email fetching and management
 */

require_once 'db_connect.php';
require_once 'fast_email_service.php';

/**
 * Get IMAP settings from email configuration
 */
function getImapSettings() {
    $emailSettings = getEmailSettings();
    
    // IMAP settings based on SMTP settings
    $settings = [
        'imap_host' => $emailSettings['smtp_host'],
        'imap_port' => 993, // Standard IMAP SSL port
        'imap_encryption' => 'ssl',
        'imap_username' => $emailSettings['smtp_username'],
        'imap_password' => $emailSettings['smtp_password'],
        'mailbox' => 'INBOX'
    ];
    
    return $settings;
}

/**
 * Connect to IMAP server
 */
function connectIMAP() {
    $settings = getImapSettings();
    
    // Build IMAP connection string
    $hostname = "{{$settings['imap_host']}:{$settings['imap_port']}/imap/ssl/novalidate-cert}";
    $mailbox = $hostname . $settings['mailbox'];
    
    // Connect to IMAP
    $connection = imap_open($mailbox, $settings['imap_username'], $settings['imap_password']);
    
    if (!$connection) {
        error_log("IMAP Connection failed: " . imap_last_error());
        return false;
    }
    
    return $connection;
}

/**
 * Get email list from inbox with search and filter support
 */
function getEmailList($limit = 50, $offset = 0, $filters = []) {
    $connection = connectIMAP();
    
    if (!$connection) {
        return ['success' => false, 'error' => 'Failed to connect to email server'];
    }
    
    try {
        $emails = [];
        $totalEmails = 0;
        $matchingEmailIds = [];
        
        // Extract filters
        $search = isset($filters['search']) ? trim($filters['search']) : '';
        $status = isset($filters['status']) ? trim($filters['status']) : '';
        $dateFrom = isset($filters['date_from']) ? trim($filters['date_from']) : '';
        $dateTo = isset($filters['date_to']) ? trim($filters['date_to']) : '';
        
        $hasFilters = !empty($search) || !empty($status) || !empty($dateFrom) || !empty($dateTo);
        
        if ($hasFilters) {
            // Build IMAP search criteria
            $searchCriteria = [];
            
            // Add search terms if provided
            if (!empty($search)) {
                $searchTerms = [
                    'SUBJECT "' . $search . '"',
                    'FROM "' . $search . '"',
                    'BODY "' . $search . '"'
                ];
                $searchCriteria = array_merge($searchCriteria, $searchTerms);
            }
            
            // Add status filter
            if ($status === 'unread') {
                $searchCriteria[] = 'UNSEEN';
            } elseif ($status === 'read') {
                $searchCriteria[] = 'SEEN';
            }
            
            // Add date filters
            if (!empty($dateFrom)) {
                $searchCriteria[] = 'SINCE "' . date('j-M-Y', strtotime($dateFrom)) . '"';
            }
            if (!empty($dateTo)) {
                $searchCriteria[] = 'BEFORE "' . date('j-M-Y', strtotime($dateTo . ' +1 day')) . '"';
            }
            
            // Combine search results
            $allMatches = [];
            
            if (!empty($search)) {
                // For search terms, combine results from multiple criteria
                $searchTerms = [
                    'SUBJECT "' . $search . '"',
                    'FROM "' . $search . '"',
                    'BODY "' . $search . '"'
                ];
                
                foreach ($searchTerms as $term) {
                    $baseCriteria = [$term];
                    
                    // Add other filters to each search term
                    if ($status === 'unread') {
                        $baseCriteria[] = 'UNSEEN';
                    } elseif ($status === 'read') {
                        $baseCriteria[] = 'SEEN';
                    }
                    
                    if (!empty($dateFrom)) {
                        $baseCriteria[] = 'SINCE "' . date('j-M-Y', strtotime($dateFrom)) . '"';
                    }
                    if (!empty($dateTo)) {
                        $baseCriteria[] = 'BEFORE "' . date('j-M-Y', strtotime($dateTo . ' +1 day')) . '"';
                    }
                    
                    $criteriaString = implode(' ', $baseCriteria);
                    $matches = imap_search($connection, $criteriaString, SE_UID);
                    if ($matches) {
                        $allMatches = array_merge($allMatches, $matches);
                    }
                }
            } else {
                // No search term, just apply other filters
                $criteriaString = implode(' ', $searchCriteria);
                if (!empty($criteriaString)) {
                    $allMatches = imap_search($connection, $criteriaString, SE_UID);
                    if (!$allMatches) $allMatches = [];
                } else {
                    // No filters at all, get all messages
                    $allMatches = imap_search($connection, 'ALL', SE_UID);
                    if (!$allMatches) $allMatches = [];
                }
            }
            
            // Remove duplicates and sort by UID (newest first)
            $matchingEmailIds = array_unique($allMatches);
            rsort($matchingEmailIds);
            
            $totalEmails = count($matchingEmailIds);
            
            if ($totalEmails == 0) {
                imap_close($connection);
                return ['success' => true, 'emails' => [], 'total' => 0];
            }
            
            // Apply pagination to filtered results
            $pagedIds = array_slice($matchingEmailIds, $offset, $limit);
            
            foreach ($pagedIds as $uid) {
                $msgNum = imap_msgno($connection, $uid);
                if ($msgNum) {
                    $header = imap_headerinfo($connection, $msgNum);
                    
                    $email = [
                        'id' => $msgNum,
                        'uid' => $uid,
                        'subject' => isset($header->subject) ? decode_mime_string($header->subject) : 'No Subject',
                        'from' => isset($header->from[0]) ? $header->from[0]->mailbox . '@' . $header->from[0]->host : 'Unknown',
                        'from_name' => isset($header->from[0]->personal) ? decode_mime_string($header->from[0]->personal) : '',
                        'date' => isset($header->date) ? date('Y-m-d H:i:s', strtotime($header->date)) : '',
                        'size' => isset($header->Size) ? formatBytes($header->Size) : '0 B',
                        'seen' => isset($header->Unseen) ? false : true,
                        'flagged' => isset($header->Flagged) ? true : false
                    ];
                    
                    $emails[] = $email;
                }
            }
        } else {
            // Normal email listing (no search)
            $totalEmails = imap_num_msg($connection);
            
            if ($totalEmails == 0) {
                imap_close($connection);
                return ['success' => true, 'emails' => [], 'total' => 0];
            }
            
            // Calculate range (newest first)
            $start = max(1, $totalEmails - $offset - $limit + 1);
            $end = $totalEmails - $offset;
            
            for ($i = $end; $i >= $start; $i--) {
                $header = imap_headerinfo($connection, $i);
                
                $email = [
                    'id' => $i,
                    'subject' => isset($header->subject) ? decode_mime_string($header->subject) : 'No Subject',
                    'from' => isset($header->from[0]) ? $header->from[0]->mailbox . '@' . $header->from[0]->host : 'Unknown',
                    'from_name' => isset($header->from[0]->personal) ? decode_mime_string($header->from[0]->personal) : '',
                    'date' => isset($header->date) ? date('Y-m-d H:i:s', strtotime($header->date)) : '',
                    'size' => isset($header->Size) ? formatBytes($header->Size) : '0 B',
                    'seen' => isset($header->Unseen) ? false : true,
                    'flagged' => isset($header->Flagged) ? true : false
                ];
                
                $emails[] = $email;
            }
        }
        
        imap_close($connection);
        
        return ['success' => true, 'emails' => $emails, 'total' => $totalEmails];
        
    } catch (Exception $e) {
        imap_close($connection);
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

/**
 * Get single email details
 */
function getEmailDetails($emailId) {
    $connection = connectIMAP();
    
    if (!$connection) {
        return ['success' => false, 'error' => 'Failed to connect to email server'];
    }
    
    try {
        $header = imap_headerinfo($connection, $emailId);
        $structure = imap_fetchstructure($connection, $emailId);
        
        // Get email body
        $body = '';
        $htmlBody = '';
        
        if (isset($structure->parts)) {
            // Multipart email
            foreach ($structure->parts as $partIndex => $part) {
                $partNumber = $partIndex + 1;
                $partBody = imap_fetchbody($connection, $emailId, $partNumber);
                
                // Decode based on encoding
                if ($part->encoding == 4) {
                    $partBody = quoted_printable_decode($partBody);
                } elseif ($part->encoding == 3) {
                    $partBody = base64_decode($partBody);
                }
                
                if ($part->subtype == 'PLAIN') {
                    $body = $partBody;
                } elseif ($part->subtype == 'HTML') {
                    $htmlBody = $partBody;
                }
            }
        } else {
            // Single part email
            $body = imap_fetchbody($connection, $emailId, 1);
            
            if ($structure->encoding == 4) {
                $body = quoted_printable_decode($body);
            } elseif ($structure->encoding == 3) {
                $body = base64_decode($body);
            }
        }
        
        $email = [
            'id' => $emailId,
            'subject' => isset($header->subject) ? decode_mime_string($header->subject) : 'No Subject',
            'from' => isset($header->from[0]) ? $header->from[0]->mailbox . '@' . $header->from[0]->host : 'Unknown',
            'from_name' => isset($header->from[0]->personal) ? decode_mime_string($header->from[0]->personal) : '',
            'to' => isset($header->to[0]) ? $header->to[0]->mailbox . '@' . $header->to[0]->host : 'Unknown',
            'date' => isset($header->date) ? date('Y-m-d H:i:s', strtotime($header->date)) : '',
            'size' => isset($header->Size) ? formatBytes($header->Size) : '0 B',
            'body' => $body,
            'html_body' => $htmlBody,
            'seen' => isset($header->Unseen) ? false : true,
            'flagged' => isset($header->Flagged) ? true : false
        ];
        
        // Mark as read
        imap_setflag_full($connection, $emailId, "\\Seen");
        
        imap_close($connection);
        
        return ['success' => true, 'email' => $email];
        
    } catch (Exception $e) {
        imap_close($connection);
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

/**
 * Delete email
 */
function deleteEmail($emailId) {
    $connection = connectIMAP();
    
    if (!$connection) {
        return ['success' => false, 'error' => 'Failed to connect to email server'];
    }
    
    try {
        imap_delete($connection, $emailId);
        imap_expunge($connection);
        imap_close($connection);
        
        return ['success' => true];
        
    } catch (Exception $e) {
        imap_close($connection);
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

/**
 * Mark email as read/unread
 */
function markEmailRead($emailId, $read = true) {
    $connection = connectIMAP();
    
    if (!$connection) {
        return ['success' => false, 'error' => 'Failed to connect to email server'];
    }
    
    try {
        if ($read) {
            imap_setflag_full($connection, $emailId, "\\Seen");
        } else {
            imap_clearflag_full($connection, $emailId, "\\Seen");
        }
        
        imap_close($connection);
        
        return ['success' => true];
        
    } catch (Exception $e) {
        imap_close($connection);
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

/**
 * Decode MIME encoded strings
 */
function decode_mime_string($string) {
    $decoded = imap_mime_header_decode($string);
    $result = '';
    
    foreach ($decoded as $element) {
        $result .= $element->text;
    }
    
    return $result;
}

/**
 * Format bytes to human readable
 */
function formatBytes($bytes) {
    if ($bytes >= 1073741824) {
        return number_format($bytes / 1073741824, 2) . ' GB';
    } elseif ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    } else {
        return $bytes . ' B';
    }
}

/**
 * Send email using existing SMTP service
 */
function sendEmailFromComposer($to, $subject, $body, $cc = '', $bcc = '') {
    return sendEmailNow($to, $subject, $body);
}

/**
 * Get inbox statistics
 */
function getInboxStats() {
    $connection = connectIMAP();
    
    if (!$connection) {
        return ['success' => false, 'error' => 'Failed to connect to email server'];
    }
    
    try {
        $status = imap_status($connection, "{{$connection->mailbox}}", SA_ALL);
        
        $stats = [
            'total' => $status->messages,
            'unread' => $status->unseen,
            'recent' => $status->recent
        ];
        
        imap_close($connection);
        
        return ['success' => true, 'stats' => $stats];
        
    } catch (Exception $e) {
        imap_close($connection);
        return ['success' => false, 'error' => $e->getMessage()];
    }
}
?>
