<?php
/**
 * Global utility functions for Panadite Academy
 */

// Detect environment - checks if we're on a local or production server
$is_local = isset($_SERVER['HTTP_HOST']) && ($_SERVER['HTTP_HOST'] == 'localhost' || 
           strpos($_SERVER['HTTP_HOST'], '127.0.0.1') !== false ||
           strpos($_SERVER['HTTP_HOST'], '192.168.') !== false);

// Set appropriate session path based on environment
if (session_status() === PHP_SESSION_NONE) {
    // Configure session parameters before starting session
    if (!$is_local) {
        // On production server, use default path
        ini_set('session.save_path', '/tmp');
    } else {
        // On local Windows server
        ini_set('session.save_path', 'C:\Windows\Temp');
    }
    
    // Start the session with proper settings
    session_start([  
        'cookie_httponly' => true,
        'cookie_secure' => !$is_local, // Only HTTPS on production
        'cookie_samesite' => 'Lax'
    ]);
}

/**
 * Format price for display
 *
 * @param float $price
 * @return string Formatted price
 */
function formatPrice($price) {
    if ($price <= 0) {
        return 'Free';
    }
    return 'R' . number_format($price, 2);
}

/**
 * Format date for display
 *
 * @param string $dateTime Date/time string
 * @param string $format PHP date format
 * @return string Formatted date
 */
function formatDate($dateTime, $format = 'j M Y') {
    $date = new DateTime($dateTime);
    return $date->format($format);
}

/**
 * Get time elapsed string (e.g., "2 days ago")
 *
 * @param string $datetime Date/time string
 * @return string Time elapsed
 */
function timeElapsed($datetime) {
    $now = new DateTime();
    $ago = new DateTime($datetime);
    $diff = $now->diff($ago);

    $diff->w = floor($diff->d / 7);
    $diff->d -= $diff->w * 7;

    $string = array(
        'y' => 'year',
        'm' => 'month',
        'w' => 'week',
        'd' => 'day',
        'h' => 'hour',
        'i' => 'minute',
        's' => 'second',
    );
    
    foreach ($string as $k => &$v) {
        if ($diff->$k) {
            $v = $diff->$k . ' ' . $v . ($diff->$k > 1 ? 's' : '');
        } else {
            unset($string[$k]);
        }
    }

    $string = array_slice($string, 0, 1);
    return $string ? implode(', ', $string) . ' ago' : 'just now';
}

/**
 * Format video duration (converts seconds to MM:SS format)
 *
 * @param string $duration Duration string (could be in seconds or already formatted)
 * @return string Formatted duration
 */
function formatDuration($duration) {
    // Check if duration is already formatted (contains a colon)
    if (strpos($duration, ':') !== false) {
        return $duration;
    }
    
    // Convert seconds to MM:SS format
    $minutes = floor($duration / 60);
    $seconds = $duration % 60;
    return sprintf('%02d:%02d', $minutes, $seconds);
}

/**
 * Truncate text to specified length
 *
 * @param string $text Text to truncate
 * @param int $length Maximum length
 * @param string $append String to append if truncated
 * @return string Truncated text
 */
function truncateText($text, $length = 100, $append = '...') {
    if (strlen($text) <= $length) {
        return $text;
    }
    
    $text = substr($text, 0, $length);
    $text = substr($text, 0, strrpos($text, ' '));
    return $text . $append;
}

/**
 * Check if user is logged in (config version)
 * RENAMED to avoid conflicts with auth/functions.php
 * @return bool True if logged in, false otherwise
 */
function config_isLoggedIn() {
    return isset($_SESSION['user_id']);
}

/**
 * Get current user information
 */
if (!function_exists('getCurrentUser')) {
    function getCurrentUser() {
        global $pdo;
        
        if (!isset($_SESSION['user_id'])) {
            return null;
        }
        
        try {
            $stmt = $pdo->prepare("SELECT id, username, email, role, full_name, created_at FROM users WHERE id = ?");
            $stmt->execute([$_SESSION['user_id']]);
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("Error getting current user: " . $e->getMessage());
            return null;
        }
    }
}

/**
 * Check if current user is an admin
 */
function isAdmin() {
    $user = getCurrentUser();
    return $user && isset($user['role']) && $user['role'] === 'admin';
}

/**
 * Check if current user is a teacher/facilitator
 */
function isTeacher() {
    $user = getCurrentUser();
    return $user && isset($user['role']) && in_array($user['role'], ['admin', 'teacher', 'facilitator']);
}

/**
 * Check if current user is a student
 */
function isStudent() {
    $user = getCurrentUser();
    return $user && isset($user['role']) && $user['role'] === 'student';
}

/**
 * Check if current user has specific role
 *
 * @param string|array $roles Role(s) to check
 * @return bool True if user has role, false otherwise
 */
if (!function_exists('hasRole')) {
    function hasRole($roles) {
        if (!config_isLoggedIn()) {
            return false;
        }
        
        if (is_string($roles)) {
            $roles = [$roles];
        }
        
        $user = getCurrentUser();
        return $user && isset($user['role']) && in_array($user['role'], $roles);
    }
}

/**
 * Generate pagination HTML
 *
 * @param int $currentPage Current page number
 * @param int $totalPages Total number of pages
 * @param string $baseUrl Base URL for pagination links
 * @return string HTML for pagination controls
 */
function generatePagination($currentPage, $totalPages, $baseUrl = '?page=') {
    if ($totalPages <= 1) {
        return '';
    }
    
    $html = '<nav aria-label="Page navigation">';
    $html .= '<ul class="pagination justify-content-center">';
    
    // Previous button
    $prevDisabled = ($currentPage <= 1) ? 'disabled' : '';
    $html .= '<li class="page-item ' . $prevDisabled . '">';
    $html .= '<a class="page-link" href="' . $baseUrl . ($currentPage - 1) . '" aria-label="Previous">';
    $html .= '<span aria-hidden="true">&laquo;</span>';
    $html .= '</a></li>';
    
    // Page numbers
    $startPage = max(1, $currentPage - 2);
    $endPage = min($totalPages, $currentPage + 2);
    
    if ($startPage > 1) {
        $html .= '<li class="page-item"><a class="page-link" href="' . $baseUrl . '1">1</a></li>';
        if ($startPage > 2) {
            $html .= '<li class="page-item disabled"><span class="page-link">...</span></li>';
        }
    }
    
    for ($i = $startPage; $i <= $endPage; $i++) {
        $activeClass = ($i == $currentPage) ? 'active' : '';
        $html .= '<li class="page-item ' . $activeClass . '">';
        $html .= '<a class="page-link" href="' . $baseUrl . $i . '">' . $i . '</a></li>';
    }
    
    if ($endPage < $totalPages) {
        if ($endPage < $totalPages - 1) {
            $html .= '<li class="page-item disabled"><span class="page-link">...</span></li>';
        }
        $html .= '<li class="page-item"><a class="page-link" href="' . $baseUrl . $totalPages . '">' . $totalPages . '</a></li>';
    }
    
    // Next button
    $nextDisabled = ($currentPage >= $totalPages) ? 'disabled' : '';
    $html .= '<li class="page-item ' . $nextDisabled . '">';
    $html .= '<a class="page-link" href="' . $baseUrl . ($currentPage + 1) . '" aria-label="Next">';
    $html .= '<span aria-hidden="true">&raquo;</span>';
    $html .= '</a></li>';
    
    $html .= '</ul></nav>';
    return $html;
}

/**
 * Filter input data (for security)
 *
 * @param mixed $data Data to filter
 * @return mixed Filtered data
 */
function filterInput($data) {
    $data = trim($data);
    $data = stripslashes($data);
    $data = htmlspecialchars($data);
    return $data;
}

/**
 * Set flash message to be displayed on the next page
 *
 * @param string $type Message type (success, danger, warning, info)
 * @param string $message Message content
 */
function setFlashMessage($type, $message) {
    $_SESSION['flash_message'] = [
        'type' => $type,
        'message' => $message
    ];
}

/**
 * Display and clear flash message
 *
 * @return string HTML for flash message or empty string if no message
 */
function displayFlashMessage() {
    if (isset($_SESSION['flash_message'])) {
        $type = $_SESSION['flash_message']['type'];
        $message = $_SESSION['flash_message']['message'];
        
        unset($_SESSION['flash_message']);
        
        return '
        <div class="alert alert-' . $type . ' alert-dismissible fade show" role="alert">
            ' . $message . '
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>';
    }
    
    return '';
}


