<?php
// Enable error reporting for debugging
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Include SEO headers with favicon for course details page
$seo_title = 'Course Details - Panadite Academy | Professional Online Learning';
$seo_description = 'Explore course details, curriculum, instructor information, and enrollment options at Panadite Academy. Start your professional development journey today.';
require_once __DIR__ . '/includes/seo_header.php';

// Debug checkpoint 1
echo "<!-- Debug: Starting course.php -->\n";

// Fix session handling for live server
if (session_status() == PHP_SESSION_NONE) {
    // Force correct session path for Linux server
    ini_set('session.save_path', '/tmp');
    
    // Alternative fallback paths for different server configurations
    if (!is_writable('/tmp')) {
        $temp_paths = ['/var/tmp', sys_get_temp_dir()];
        foreach ($temp_paths as $path) {
            if (is_dir($path) && is_writable($path)) {
                ini_set('session.save_path', $path);
                break;
            }
        }
    }
    
    session_start();
}

try {
    // Include database connection
    require_once 'config/db_connect.php';
    echo "<!-- Debug: Database connection included -->\n";
    
    require_once 'config/functions.php';
    require_once 'includes/id_encryption.php'; // Include ID encryption utility
    echo "<!-- Debug: Functions included -->\n";
} catch (Exception $e) {
    die("Error loading files: " . $e->getMessage());
}

// Get current user if logged in
$current_user = getCurrentUser();
$is_logged_in = $current_user !== null;

// Initialize variables - Handle encrypted IDs
if (isset($_GET['id'])) {
    $original_id = $_GET['id'];
    
    // URL decode the ID first in case it was URL encoded
    $id_param = urldecode($_GET['id']);
    
    // Check if IdEncryption class exists
    if (class_exists('IdEncryption')) {
        // Try to decrypt the ID first (new encrypted format)
        try {
            $decrypted_id = IdEncryption::decrypt($id_param);
            
            if ($decrypted_id !== false && is_numeric($decrypted_id)) {
                $course_id = (int)$decrypted_id;
            } else {
                // Fallback to old integer format for backward compatibility
                if (is_numeric($original_id)) {
                    $course_id = (int)$original_id;
                } else {
                    $course_id = 0; // This will cause "Course Not Found"
                }
            }
        } catch (Exception $e) {
            // Fallback to old integer format for backward compatibility
            if (is_numeric($original_id)) {
                $course_id = (int)$original_id;
            } else {
                $course_id = 0;
            }
        }
    } else {
        // Fallback to old integer format for backward compatibility
        if (is_numeric($original_id)) {
            $course_id = (int)$original_id;
        } else {
            $course_id = 0;
        }
    }
} else {
    $course_id = 0;
}
$course = null;
$related_courses = [];
$error = false;
$is_enrolled = false;

if (!$course_id) {
    header('Location: courses.php');
    exit;
}

// Check if user is enrolled in this course
if ($is_logged_in) {
    $user_id = $_SESSION['user_id'];
    
    // First check if the enrollments table exists
    $tableCheckQuery = "SHOW TABLES LIKE 'enrollments'";
    $tableCheckResult = $conn->query($tableCheckQuery);
    
    if ($tableCheckResult->num_rows > 0) {
        try {
            // First try with user_id (the column name we created)
            $enrollQuery = "SELECT * FROM enrollments WHERE user_id = ? AND course_id = ?";
            $enrollStmt = $conn->prepare($enrollQuery);
            $enrollStmt->bind_param("ii", $user_id, $course_id);
            $enrollStmt->execute();
            $enrollResult = $enrollStmt->get_result();
            
            if ($enrollResult->num_rows > 0) {
                $is_enrolled = true;
                $has_purchased = true;
            }
        } catch (mysqli_sql_exception $e) {
            // If that fails, try with student_id (which appears to be the existing column name)
            try {
                $enrollQuery = "SELECT * FROM enrollments WHERE student_id = ? AND course_id = ?";
                $enrollStmt = $conn->prepare($enrollQuery);
                $enrollStmt->bind_param("ii", $user_id, $course_id);
                $enrollStmt->execute();
                $enrollResult = $enrollStmt->get_result();
                
                if ($enrollResult->num_rows > 0) {
                    $is_enrolled = true;
                    $has_purchased = true;
                }
            } catch (mysqli_sql_exception $e) {
                // Both failed, so table structure is unknown
                error_log("Enrollment check error: " . $e->getMessage());
            }
        }
    }
}

// Initialize variables
// Use the already set $is_logged_in value from line 7 instead of overwriting it
// $is_logged_in was already set to isset($_SESSION['user_id']) at the top of the file
$user_id = $is_logged_in ? $_SESSION['user_id'] : 0;
$error = false;
$course = null;
$videos = [];
$instructor = null;
$total_duration = 0;
$video_count = 0;
$enrolled_students = 0;

// Debug current session
error_log("Session check in course.php: User ID: " . ($is_logged_in ? $_SESSION['user_id'] : 'Not logged in'));
error_log("Session role: " . ($_SESSION['user_role'] ?? 'No role'));

// Check if course ID is provided and valid (already handled above with encryption)
if (!$course_id || $course_id <= 0) {
    $error = true;
} else {
    // course_id is already set from encryption handling above
    
    try {
        // Get course information with instructor details
        $stmt = $conn->prepare("
            SELECT c.*, 
                   u.first_name, u.last_name,
                   cat.category_name
            FROM courses c 
            LEFT JOIN users u ON c.teacher_id = u.user_id 
            LEFT JOIN categories cat ON c.category_id = cat.category_id
            WHERE c.course_id = ?
        ");
        
        if (!$stmt) {
            throw new Exception("Failed to prepare course query");
        }
        
        $stmt->bind_param("i", $course_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows === 0) {
            $error = true;
        } else {
            $course = $result->fetch_assoc();
            
            // Format instructor name
            $course['instructor_name'] = trim($course['first_name'] . ' ' . $course['last_name']);
            if (empty($course['instructor_name'])) {
                $course['instructor_name'] = 'Panadite Academy Instructor';
            }
            
            // Format price
            $course['price'] = floatval($course['price'] ?? 0);
            $course['price_formatted'] = $course['price'] > 0 ? 'R' . number_format($course['price'], 2) : 'Free';
            
            // Check if user is enrolled
            if ($is_logged_in) {
                $enroll_stmt = $conn->prepare("SELECT enrollment_id FROM enrollments WHERE student_id = ? AND course_id = ?");
                $enroll_stmt->bind_param("ii", $current_user['user_id'], $course_id);
                $enroll_stmt->execute();
                $is_enrolled = $enroll_stmt->get_result()->num_rows > 0;
                $enroll_stmt->close();
            }
            
            // Get course videos/lessons count
            $video_stmt = $conn->prepare("SELECT COUNT(*) as video_count FROM videos WHERE course_id = ?");
            $video_stmt->bind_param("i", $course_id);
            $video_stmt->execute();
            $video_result = $video_stmt->get_result();
            $course['video_count'] = $video_result->fetch_assoc()['video_count'];
            $video_stmt->close();
            
            // Get enrollment count
            $student_stmt = $conn->prepare("SELECT COUNT(*) as student_count FROM enrollments WHERE course_id = ?");
            $student_stmt->bind_param("i", $course_id);
            $student_stmt->execute();
            $student_result = $student_stmt->get_result();
            $course['student_count'] = $student_result->fetch_assoc()['student_count'];
            $student_stmt->close();
            
            // Get related courses (same category, different course)
            echo "<!-- Debug: Course category_id: " . ($course['category_id'] ?? 'NULL') . " -->\n";
            
            if (!empty($course['category_id'])) {
                $related_stmt = $conn->prepare("
                    SELECT c.course_id, c.title, c.description, c.price, c.thumbnail,
                           u.first_name, u.last_name,
                           COUNT(e.enrollment_id) as enrollment_count
                    FROM courses c 
                    LEFT JOIN users u ON c.teacher_id = u.user_id 
                    LEFT JOIN enrollments e ON c.course_id = e.course_id
                    WHERE c.category_id = ? AND c.course_id != ? AND c.is_published = 1
                    GROUP BY c.course_id
                    ORDER BY enrollment_count DESC
                    LIMIT 4
                ");
                
                if ($related_stmt) {
                    $related_stmt->bind_param("ii", $course['category_id'], $course_id);
                    $related_stmt->execute();
                    $related_result = $related_stmt->get_result();
                    
                    while ($related = $related_result->fetch_assoc()) {
                        $related['instructor_name'] = trim($related['first_name'] . ' ' . $related['last_name']);
                        $related['price_formatted'] = floatval($related['price']) > 0 ? 'R' . number_format($related['price'], 2) : 'Free';
                        $related_courses[] = $related;
                    }
                    $related_stmt->close();
                    echo "<!-- Debug: Found " . count($related_courses) . " related courses by category -->\n";
                } else {
                    error_log("Failed to prepare related courses query: " . $conn->error);
                }
            }
            
            // If no related courses found by category, get any other courses
            if (empty($related_courses)) {
                echo "<!-- Debug: No category-based courses, getting any other courses -->\n";
                $fallback_stmt = $conn->prepare("
                    SELECT c.course_id, c.title, c.description, c.price, c.thumbnail,
                           u.first_name, u.last_name,
                           COUNT(e.enrollment_id) as enrollment_count
                    FROM courses c 
                    LEFT JOIN users u ON c.teacher_id = u.user_id 
                    LEFT JOIN enrollments e ON c.course_id = e.course_id
                    WHERE c.course_id != ? AND c.is_published = 1
                    GROUP BY c.course_id
                    ORDER BY enrollment_count DESC
                    LIMIT 3
                ");
                
                if ($fallback_stmt) {
                    $fallback_stmt->bind_param("i", $course_id);
                    $fallback_stmt->execute();
                    $fallback_result = $fallback_stmt->get_result();
                    
                    while ($related = $fallback_result->fetch_assoc()) {
                        $related['instructor_name'] = trim($related['first_name'] . ' ' . $related['last_name']);
                        $related['price_formatted'] = floatval($related['price']) > 0 ? 'R' . number_format($related['price'], 2) : 'Free';
                        $related_courses[] = $related;
                    }
                    $fallback_stmt->close();
                    echo "<!-- Debug: Found " . count($related_courses) . " fallback courses -->\n";
                } else {
                    error_log("Failed to prepare fallback courses query: " . $conn->error);
                }
            }
        }
        $stmt->close();
        
    } catch (Exception $e) {
        error_log("Course page error: " . $e->getMessage());
        $error = true;
    }
    
    // Get course videos
    $video_stmt = $conn->prepare("SELECT video_id, title, description, thumbnail, duration, views 
                                FROM videos 
                                WHERE course_id = ? 
                                ORDER BY created_at DESC");
    // Removed is_published and order_index as they might not exist
    $video_stmt->bind_param("i", $course_id);
    $video_stmt->execute();
    $video_result = $video_stmt->get_result();
    
    while ($video = $video_result->fetch_assoc()) {
        $videos[] = $video;
        
        // Calculate total duration (if duration is a valid number)
        if (isset($video['duration']) && is_numeric($video['duration'])) {
            $total_duration += intval($video['duration']);
        }
    }
    
    $video_count = count($videos);
    
    // Get enrolled student count with error handling
    $student_count = 0;
    try {
        $enrollment_stmt = $conn->prepare("SELECT COUNT(*) AS student_count FROM enrollments WHERE course_id = ?");
        if ($enrollment_stmt) {
            $enrollment_stmt->bind_param("i", $course_id);
            $enrollment_stmt->execute();
            $enrollment_result = $enrollment_stmt->get_result();
            $student_count = ($enrollment_result->fetch_assoc())['student_count'] ?? 0;
            $enrollment_stmt->close();
        }
    } catch (Exception $e) {
        error_log("Error getting enrollment count: " . $e->getMessage());
    }
    
    $enrolled_students = $student_count;
}
// Format total duration
$hours = floor($total_duration / 3600);
$minutes = floor(($total_duration % 3600) / 60);
$formatted_duration = '';

if ($hours > 0) {
    $formatted_duration = $hours . 'h ' . $minutes . 'm';
} else {
    $formatted_duration = $minutes . ' minutes';
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $error ? 'Course Not Found' : htmlspecialchars($course['title']); ?> - Panadite Academy</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.1.1/css/all.min.css" rel="stylesheet">
    <link rel="stylesheet" href="assets/css/style.css">
    <style>
        :root {
            --primary-color: #4e73df;
            --secondary-color: #224abe;
            --light-color: #f8f9fc;
            --dark-color: #5a5c69;
            --success-color: #1cc88a;
            --info-color: #36b9cc;
            --warning-color: #f6c23e;
            --danger-color: #e74a3b;
            --border-radius: 10px;
            --box-shadow: 0 10px 20px rgba(58, 59, 69, 0.1);
            --transition: all 0.3s ease;
        }
        
        body {
            font-family: 'Nunito', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
            background-color: #f0f2f5;
            color: #444;
            line-height: 1.6;
        }
        
        .course-banner {
            background: linear-gradient(120deg, #276BE4 0%, #1a4fc7 70%, #0b3ca1 100%);
            position: relative;
            overflow: hidden;
            padding: 100px 0 80px;
            margin-bottom: 40px;
            box-shadow: 0 15px 30px rgba(0,0,0,0.1);
            color: white;
        }
        
        .navbar {
            background-color: white;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        }
        
        .navbar-brand img {
            height: 40px;
        }
        
        .page-container {
            padding: 2rem 0;
            position: relative;
            z-index: 5;
            margin-top: -50px;
        }
        
        /* Banner Styles */
        .course-banner {
            position: relative;
            height: var(--banner-height);
            width: 100%;
            overflow: hidden;
            color: white;
        }
        
        .banner-background {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            width: 100%;
            height: 100%;
            z-index: 0;
        }
        
        .banner-background::before {
            content: '';
            position: absolute;
            top: -10%;
            left: -10%;
            width: 120%;
            height: 120%;
            background-image: url("data:image/svg+xml,%3Csvg width='80' height='80' viewBox='0 0 100 100' xmlns='http://www.w3.org/2000/svg'%3E%3Cpath d='M11 18c3.866 0 7-3.134 7-7s-3.134-7-7-7-7 3.134-7 7 3.134 7 7 7zm48 25c3.866 0 7-3.134 7-7s-3.134-7-7-7-7 3.134-7 7 3.134 7 7 7zm-43-7c1.657 0 3-1.343 3-3s-1.343-3-3-3-3 1.343-3 3 1.343 3 3 3zm63 31c1.657 0 3-1.343 3-3s-1.343-3-3-3-3 1.343-3 3 1.343 3 3 3zM34 90c1.657 0 3-1.343 3-3s-1.343-3-3-3-3 1.343-3 3 1.343 3 3 3z' fill='%23ffffff' fill-opacity='0.1' fill-rule='evenodd'/%3E%3C/svg%3E");
            opacity: 0.7;
            animation: moveBackground 50s linear infinite;
        }
        
        .banner-background::after {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: radial-gradient(circle at top right, rgba(255,255,255,0.15) 0%, rgba(39,107,228,0) 60%);
        }
        
        @keyframes gradientAnimation {
            0% { background-position: 0% 50%; }
            100% { background-position: 100% 50%; }
        }
        
        @keyframes moveBackground {
            0% { transform: translate(0, 0); }
            50% { transform: translate(-5%, -5%); }
            100% { transform: translate(0, 0); }
        }
        
        @keyframes float {
            0% { transform: translateY(0px); }
            50% { transform: translateY(-20px); }
            100% { transform: translateY(0px); }
        }

        /* Related Courses Clean Design */
        .related-courses .card {
            transition: all 0.2s ease;
            border-radius: 8px !important;
            overflow: hidden;
            border: 1px solid #e9ecef !important;
        }
        
        .related-courses .card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.1) !important;
        }
        
        .related-courses .card-img-top {
            border-radius: 8px 8px 0 0;
            transition: transform 0.2s ease;
        }
        
        .related-courses .card:hover .card-img-top {
            transform: scale(1.02);
        }
        
        .related-courses .course-overlay {
            border-radius: 8px 8px 0 0;
            background: rgba(0,0,0,0.6);
        }
        
        .related-courses .card-title {
            font-size: 0.9rem !important;
            line-height: 1.2 !important;
            height: 2.4rem;
            overflow: hidden;
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
            margin-bottom: 0.5rem !important;
        }
        
        .related-courses .card-text {
            font-size: 0.75rem !important;
            line-height: 1.3 !important;
            height: 2.6rem;
            overflow: hidden;
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
            margin-bottom: 0.75rem !important;
        }
        
        .related-courses .badge {
            font-weight: 600;
            border-radius: 12px;
            padding: 3px 8px;
            font-size: 0.65rem !important;
        }
        
        .related-courses .instructor-info {
            font-size: 0.7rem !important;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
            max-width: 120px;
        }
        
        .related-courses .price-display {
            font-size: 0.9rem !important;
            font-weight: 700 !important;
            text-align: center;
        }
        
        .related-courses .card-body {
            padding: 0.75rem !important;
        }

        /* Typing Animation Styles */
        .typing-text {
            font-weight: 700;
            color: #ff8c00 !important;
            display: inline-flex;
            align-items: baseline;
            white-space: nowrap;
            flex-shrink: 0;
        }

        .typed-text {
            display: inline;
        }

        .cursor {
            display: inline;
            background-color: #ff8c00;
            margin-left: 2px;
            width: 3px;
            height: 1em;
            animation: blink 1s infinite;
        }

        @keyframes blink {
            0%, 50% { opacity: 1; }
            51%, 100% { opacity: 0; }
        }

        .cursor.typing {
            animation: none;
            opacity: 1;
        }
        
        .animate-fade-in {
            opacity: 0;
            animation: fadeIn 0.8s ease forwards;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        .banner-title {
            font-weight: 800;
            font-size: 3.2rem;
            margin-bottom: 25px;
            text-shadow: 0 2px 15px rgba(0, 0, 0, 0.15);
            letter-spacing: -0.5px;
            background: linear-gradient(120deg, #ffffff 0%, #f0f4ff 100%);
            -webkit-background-clip: text;
            background-clip: text;
            color: #ffffff;
            transform: translateZ(0);
            position: relative;
        }
        
        .banner-meta {
            position: relative;
        }
        
        .banner-meta .d-flex > div {
            background-color: rgba(255, 255, 255, 0.1);
            backdrop-filter: blur(5px);
            border-radius: 50px;
            padding: 10px 18px;
            margin-right: 10px;
            margin-bottom: 10px;
            transition: all 0.3s ease;
            border: 1px solid rgba(255, 255, 255, 0.2);
        }
        
        .banner-meta .d-flex > div:hover {
            background-color: rgba(255, 255, 255, 0.2);
            transform: translateY(-3px);
            box-shadow: 0 7px 15px rgba(0, 0, 0, 0.1);
        }
        
        .course-content-section {
            background-color: #fff;
            border-radius: var(--border-radius);
            padding: 40px;
            margin-bottom: 30px;
            box-shadow: var(--box-shadow);
            position: relative;
            overflow: hidden;
        }
        
        .banner-content {
            position: relative;
            z-index: 3;
            padding-top: 60px;
        }
        
        .banner-title {
            font-size: 3rem;
            font-weight: 800;
            margin-top: 20px;
            margin-bottom: 30px;
            max-width: 800px;
            text-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .banner-meta {
            margin-top: 20px;
        }
        
        .banner-overlay {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: radial-gradient(circle, rgba(0,0,0,0) 0%, rgba(0,0,0,0.3) 100%);
            z-index: 2;
        }
            margin-top: 20px;
        }
        
        .banner-shape {
            position: absolute;
            bottom: -2px;
            left: 0;
            width: 100%;
            height: 50px;
            background-color: #f8f9fc;
            clip-path: polygon(0 100%, 100% 100%, 100% 0);
            z-index: 3;
        }
        
        /* Enhanced animations */
        @keyframes fadeIn {
            0% {
                opacity: 1;
                transform: scale(0.98);
            }
            100% {
                opacity: 1;
                transform: scale(1);
            }
        }
        
        @keyframes slideUp {
            0% {
                opacity: 1;
                transform: translateY(10px);
            }
            100% {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        @keyframes slideInLeft {
            0% {
                opacity: 1;
                transform: translateX(-10px);
            }
            100% {
                opacity: 1;
                transform: translateX(0);
            }
        }
        
        @keyframes slideInRight {
            0% {
                opacity: 1;
                transform: translateX(10px);
            }
            100% {
                opacity: 1;
                transform: translateX(0);
            }
        }
        
        @keyframes float {
            0%, 100% {
                transform: translateY(0px);
            }
            50% {
                transform: translateY(-10px);
            }
        }
        
        @keyframes pulse {
            0% {
                transform: scale(1);
            }
            50% {
                transform: scale(1.05);
            }
            100% {
                transform: scale(1);
            }
        }
        
        @keyframes shimmer {
            0% {
                background-position: -200% 0;
            }
            100% {
                background-position: 200% 0;
            }
        }
        
        /* Animation classes - simplified for better visibility */
        .animate-fade-in {
            animation: fadeIn 1s ease forwards;
            animation-fill-mode: both;
        }
        
        .animate-slide-up {
            animation: slideUp 1s ease forwards;
            animation-fill-mode: both;
        }
        
        .animate-slide-left {
            animation: slideInLeft 1s ease forwards;
            animation-fill-mode: both;
        }
        
        .animate-slide-right {
            animation: slideInRight 1s ease forwards;
            animation-fill-mode: both;
        }
        
        .animate-float {
            animation: float 3s ease-in-out infinite;
        }
        
        .animate-pulse {
            animation: pulse 2s ease-in-out infinite;
        }
        
        /* Animation delays */
        .animate-delay-1 { animation-delay: 0.3s; }
        .animate-delay-2 { animation-delay: 0.6s; }
        .animate-delay-3 { animation-delay: 0.9s; }
        .animate-delay-4 { animation-delay: 1.2s; }
        .animate-delay-5 { animation-delay: 1.5s; }
        
        /* Button animations */
        .btn-animated {
            position: relative;
            overflow: hidden;
            transition: all 0.3s ease;
        }
        
        .btn-animated:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(78, 115, 223, 0.3);
        }
        
        .btn-animated::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
            transition: left 0.5s;
        }
        
        .btn-animated:hover::before {
            left: 100%;
        }
        
        /* Meta icon animations */
        .meta-icon {
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .meta-icon:hover {
            background: #4e73df !important;
            transform: scale(1.1);
        }
        
        .meta-icon:hover i {
            color: white !important;
        }
        
        /* Course image hover effect */
        .course-image-container {
            overflow: hidden;
            border-radius: 16px;
        }
        
        .course-image {
            transition: transform 0.4s ease;
        }
        
        .course-image:hover {
            transform: scale(1.05);
        }
        
        /* Certificate badge floating animation */
        .certificate-badge {
            animation: float 2.5s ease-in-out infinite;
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .certificate-badge:hover {
            animation-play-state: paused;
            transform: scale(1.08) translateY(-5px);
            box-shadow: 0 15px 35px rgba(0,0,0,0.2);
        }
        
        /* Course content card styles */
        .course-card {
            border-radius: var(--border-radius);
            box-shadow: var(--card-shadow);
            transition: all var(--transition-speed) ease;
            border: none;
            background-color: white;
            height: 100%;
        }
        
        .course-price {
            font-size: 2.5rem;
            font-weight: 700;
            color: var(--primary-color);
        }
        
        .course-meta {
            display: flex;
            flex-wrap: wrap;
            gap: 1rem;
            margin-bottom: 1.5rem;
            font-size: 0.9rem;
            color: #666;
        }
        
        .course-meta-item {
            display: flex;
            align-items: center;
        }
        
        .course-meta-item i {
            margin-right: 0.5rem;
            color: var(--primary-color);
        }
        
        /* Course content styles */
        .course-outline {
            margin-top: 2rem;
        }
        
        .course-section {
            margin-bottom: 1.5rem;
        }
        
        .course-section-header {
            background-color: var(--primary-light);
            padding: 1rem 1.5rem;
            border-radius: var(--border-radius);
            margin-bottom: 1rem;
            display: flex;
            align-items: center;
            justify-content: space-between;
            cursor: pointer;
        }
        
        .course-section-title {
            margin-bottom: 0;
            font-weight: 600;
            font-size: 1.1rem;
            color: var(--primary-color);
        }
        
        .course-section-content {
            padding: 0 1rem;
        }
        
        .course-video-item {
            display: flex;
            align-items: center;
            padding: 20px;
            border-radius: 10px;
            background-color: #fff;
            box-shadow: 0 3px 15px rgba(0,0,0,0.05);
            margin-bottom: 15px;
            transition: all 0.3s ease;
            border-left: 4px solid transparent;
        }
        
        .course-video-item:hover {
            transform: translateX(5px);
            border-left: 4px solid var(--primary-color);
            box-shadow: 0 5px 20px rgba(0,0,0,0.08);
        }
        
        .course-video-icon {
            width: 50px;
            height: 50px;
            min-width: 50px;
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
            color: white;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 20px;
            box-shadow: 0 5px 15px rgba(78, 115, 223, 0.3);
            transition: all 0.3s ease;
        }
        
        a:hover .course-video-icon {
            transform: scale(1.1);
        }
        
        .course-video-title {
            flex-grow: 1;
            font-weight: 500;
            font-size: 0.95rem;
        }
        
        .course-video-duration {
            flex-shrink: 0;
            color: #666;
            font-size: 0.9rem;
        }
        
        .instructor-section {
            margin-top: 3rem;
        }
        
        .instructor-card {
            display: flex;
            align-items: flex-start;
            padding: 2.5rem;
            border-radius: var(--border-radius);
            background-color: white;
            box-shadow: var(--box-shadow);
            position: relative;
            border-left: 5px solid var(--primary-color);
            transition: all 0.3s ease;
        }
        
        .instructor-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 30px rgba(0,0,0,0.1);
        }
        
        .instructor-avatar {
            width: 120px;
            height: 120px;
            border-radius: 50%;
            overflow: hidden;
            margin-right: 2rem;
            flex-shrink: 0;
            border: 5px solid rgba(78, 115, 223, 0.1);
        }
        
        .instructor-avatar img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }
        
        .instructor-info h3 {
            margin-bottom: 0.5rem;
            font-size: 1.5rem;
        }
        
        .instructor-bio {
            margin-top: 1rem;
            line-height: 1.6;
        }
        
        /* Button styles */
        .btn-enroll {
            padding: 0.75rem 2rem;
            font-size: 1rem;
            font-weight: 600;
            border-radius: 50px;
            transition: all var(--transition-speed) ease;
        }
        
        .btn-enroll:hover {
            transform: translateY(-3px);
            box-shadow: 0 10px 20px rgba(78, 115, 223, 0.3);
        }
        
        .btn-primary {
            background-color: var(--primary-color);
            border-color: var(--primary-color);
        }
        
        .btn-primary:hover {
            background-color: #2e59d9;
            border-color: #2e59d9;
        }
        
        .btn-success {
            background-color: var(--success-color);
            border-color: var(--success-color);
        }
        
        .btn-success:hover {
            background-color: #15a978;
            border-color: #15a978;
        }
        
        .course-stats-card {
            border-radius: var(--border-radius);
            overflow: hidden;
            border: none;
        }
        
        .course-feature-list li {
            padding: 0.9rem 0;
            display: flex;
            align-items: center;
            border-bottom: 1px dashed rgba(0,0,0,0.08);
            transition: all 0.2s ease;
        }
        
        .course-feature-list li:last-child {
            border-bottom: none;
        }
        
        .course-feature-list li:hover {
            transform: translateX(5px);
        }
        
        .course-feature-list li i {
            color: var(--primary-color);
            margin-right: 0.85rem;
            background-color: rgba(78, 115, 223, 0.1);
            width: 32px;
            height: 32px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 0.9rem;
        }
        
        /* What you'll learn section */
        .learning-objectives {
            margin-top: 3rem;
            background-color: #fff;
            padding: 2.5rem;
            border-radius: var(--border-radius);
            box-shadow: var(--box-shadow);
            position: relative;
            overflow: hidden;
        }
        
        .learning-objectives::before {
            content: '';
            position: absolute;
            top: 0;
            right: 0;
            width: 150px;
            height: 150px;
            background: radial-gradient(circle, rgba(78, 115, 223, 0.1) 0%, rgba(78, 115, 223, 0) 70%);
            border-radius: 0 0 0 100%;
            z-index: 1;
        }
        
        .learning-objectives-list {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 1.5rem;
            margin-top: 2rem;
            position: relative;
            z-index: 2;
        }
        
        .learning-objective-item {
            display: flex;
            align-items: flex-start;
            transition: all 0.3s ease;
            padding: 0.5rem 0.75rem;
            border-radius: 8px;
        }
        
        .learning-objective-item:hover {
            background-color: rgba(78, 115, 223, 0.05);
            transform: translateX(5px);
        }
        
        .learning-objective-item i {
            color: var(--primary-color);
            margin-right: 1rem;
            margin-top: 0.25rem;
            flex-shrink: 0;
            font-size: 1rem;
            background-color: rgba(78, 115, 223, 0.1);
            width: 28px;
            height: 28px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        /* Error page styles */
        .error-container {
            text-align: center;
            padding: 5rem 0;
            max-width: 600px;
            margin: 0 auto;
        }
        
        .error-title {
            font-size: 2.5rem;
            color: var(--secondary-color);
            margin-bottom: 1.5rem;
        }
        
        .error-message {
            font-size: 1.1rem;
            color: #666;
            margin-bottom: 2rem;
        }
        
        /* Related Courses Section */
        .related-courses {
            background: #f8f9fa;
        }
        
        .course-card {
            transition: all 0.3s ease;
            border-radius: 12px;
            overflow: hidden;
        }
        
        .course-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 35px rgba(0,0,0,0.1) !important;
        }
        
        .course-thumbnail {
            transition: transform 0.3s ease;
        }
        
        .course-card:hover .course-thumbnail {
            transform: scale(1.05);
        }
        
        .course-overlay {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0,0,0,0.7);
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0;
            transition: opacity 0.3s ease;
        }
        
        .course-card:hover .course-overlay {
            opacity: 1;
        }
        
        .course-card .card-title a:hover {
            color: var(--primary-color) !important;
        }
        
        .course-price {
            font-weight: 600;
        }
        
        .related-courses h3 {
            color: #2c3e50;
            font-weight: 700;
            position: relative;
        }
        
        .related-courses h3::after {
            content: '';
            position: absolute;
            bottom: -10px;
            left: 50%;
            transform: translateX(-50%);
            width: 60px;
            height: 3px;
            background: linear-gradient(90deg, var(--primary-color), var(--secondary-color));
            border-radius: 2px;
        }

        /* Clean Modern Tab Styling */
        #courseTabs {
            background: whitesmoke !important;
            border: 1px solid rgba(0, 0, 0, 0.08) !important;
            border-radius: 12px !important;
            padding: 6px !important;
            box-shadow: 0 2px 12px rgba(0, 0, 0, 0.06) !important;
        }

        #courseTabs .nav-link {
            border-radius: 8px !important;
            font-weight: 500 !important;
            padding: 12px 20px !important;
            transition: all 0.3s ease !important;
            color: #64748b !important;
            background: transparent !important;
            border: none !important;
            margin: 0 2px !important;
        }

        #courseTabs .nav-link:hover {
            background: rgba(255, 248, 220, 0.7) !important;
            color: #d97706 !important;
            transform: translateY(-1px) !important;
            box-shadow: 0 2px 8px rgba(217, 119, 6, 0.15) !important;
        }

        #courseTabs .nav-link.active {
            background: linear-gradient(135deg, #fffbeb 0%, #fef3c7 100%) !important;
            color: #d97706 !important;
            font-weight: 600 !important;
            box-shadow: 0 4px 12px rgba(217, 119, 6, 0.2) !important;
            border: 1px solid rgba(217, 119, 6, 0.2) !important;
        }

        /* Tab Content Container Styling */
        .tab-content-panel {
            background: #ffffff !important;
            border-radius: 12px !important;
            padding: 32px !important;
            box-shadow: 0 2px 16px rgba(0, 0, 0, 0.06) !important;
            border: 1px solid rgba(0, 0, 0, 0.08) !important;
            margin-top: 20px !important;
        }

        /* Enhanced Course Cards Inside Tabs */
        .tab-content-panel .card {
            border: 1px solid rgba(0, 0, 0, 0.06) !important;
            border-radius: 12px !important;
            box-shadow: 0 2px 12px rgba(0, 0, 0, 0.04) !important;
        }

        .tab-content-panel .card-body {
            padding: 24px !important;
        }

        .tab-content-panel h3 {
            color: #1f2937 !important;
            font-weight: 700 !important;
            margin-bottom: 20px !important;
        }

        .tab-content-panel h4 {
            color: #374151 !important;
            font-weight: 600 !important;
            margin-bottom: 16px !important;
        }

        /* Accordion Styling for Curriculum */
        .accordion-item {
            border: 1px solid rgba(0, 0, 0, 0.06) !important;
            border-radius: 8px !important;
            margin-bottom: 12px !important;
            overflow: hidden !important;
        }

        .accordion-button {
            background: #fafafa !important;
            border: none !important;
            font-weight: 500 !important;
            color: #374151 !important;
            padding: 16px 20px !important;
        }

        .accordion-button:hover {
            background: rgba(255, 248, 220, 0.5) !important;
        }

        .accordion-button:not(.collapsed) {
            background: linear-gradient(135deg, #fffbeb 0%, #fef3c7 100%) !important;
            color: #d97706 !important;
            box-shadow: none !important;
        }

        .accordion-body {
            background: #ffffff !important;
            border-top: 1px solid rgba(0, 0, 0, 0.06) !important;
            padding: 20px !important;
        }

        /* Social Links Styling in Instructor Tab */
        .social-links .bg-light {
            background: rgba(255, 248, 220, 0.3) !important;
            border: 1px solid rgba(217, 119, 6, 0.1) !important;
        }

        /* Responsive Design */
        @media (max-width: 768px) {
            #courseTabs .nav-link {
                padding: 10px 16px !important;
                font-size: 0.9rem !important;
            }
            
            .tab-content-panel {
                padding: 20px !important;
            }
        }
    </style>
</head>
<body>
    <!-- Include navbar -->
    <?php include 'includes/navbar.php'; ?>

    <?php if ($error): ?>
    <!-- Error Message -->
    <div class="container py-5">
        <div class="text-center">
            <h1 class="display-4 text-muted">Course Not Found</h1>
            <p class="lead">The course you're looking for doesn't exist.</p>
            <a href="courses.php" class="btn btn-primary">Browse Courses</a>
        </div>
    </div>
    <?php else: ?>
    
    <!-- Clean Course Header -->
    <section class="py-5" style="background: #f8f9fc; min-height: 50vh;">
        <div class="container">
            <div class="row align-items-center">
                <div class="col-lg-8">
                    <!-- Course Badge -->
                    <div class="mb-4 animate-fade-in">
                        <span class="badge rounded-pill px-3 py-2" 
                              style="background: #e3f2fd; color: #1976d2; font-size: 0.85rem; font-weight: 600;">
                            <i class="fas fa-graduation-cap me-1"></i>
                            <?php echo htmlspecialchars($course['category_name'] ?? 'Professional Course'); ?>
                        </span>
                    </div>
                    
                    <!-- Course Title with Typing Animation -->
                    <h1 class="display-5 fw-bold text-dark mb-4 animate-slide-left animate-delay-1" style="line-height: 1.2; display: flex; align-items: baseline; flex-wrap: wrap;">
                        <span>Master&nbsp;</span>
                        <span id="typing-text" class="typing-text" style="color: #ff8c00;">
                            <span class="typed-text"></span>
                            <span class="cursor">|</span>
                        </span>
                    </h1>
                    
                    <!-- Course Description -->
                    <p class="lead text-muted mb-4 animate-slide-left animate-delay-2" style="font-size: 1.1rem; line-height: 1.6; max-width: 600px;">
                        Advance your career with comprehensive training in 
                        <?php echo htmlspecialchars(strtolower($course['title'])); ?>. 
                        Join thousands of learners building in-demand skills.
                    </p>
                    

                    
                    <!-- Action Buttons -->
                    <div class="d-flex gap-3 animate-slide-left animate-delay-4">
                        <a href="auth/register.php" class="btn btn-primary btn-lg px-4 py-2 btn-animated" style="border-radius: 8px; font-weight: 600; text-decoration: none;">
                            <i class="fas fa-play me-2"></i>Start Learning
                        </a>
                    </div>
                </div>
                
                <!-- Course Visual -->
                <div class="col-lg-4">
                    <div class="position-relative animate-slide-right animate-delay-2">
                        <div class="course-image-container">
                            <?php if (!empty($course['thumbnail'])): ?>
                                <img src="<?php echo htmlspecialchars($course['thumbnail']); ?>" 
                                     class="img-fluid rounded-3 shadow-sm course-image" 
                                     alt="<?php echo htmlspecialchars($course['title']); ?>"
                                     style="max-width: 400px; border-radius: 16px;">
                            <?php else: ?>
                                <div class="bg-gradient-primary text-white rounded-3 shadow-sm d-flex align-items-center justify-content-center course-image"
                                     style="height: 250px; max-width: 400px; border-radius: 16px;">
                                    <div class="text-center">
                                        <i class="fas fa-play-circle mb-2 animate-pulse" style="font-size: 3rem;"></i>
                                        <div style="font-size: 1.1rem; font-weight: 600;">Course Preview</div>
                                    </div>
                                </div>
                            <?php endif; ?>
                        </div>
                        
                        <!-- Floating Certificate Badge -->
                        <div class="position-absolute bottom-0 start-0 bg-white rounded-3 shadow p-3 m-3 certificate-badge" 
                             style="border-radius: 12px; min-width: 150px;">
                            <div class="d-flex align-items-center">
                                <div class="bg-success rounded-circle p-2 me-2 animate-pulse" style="width: 32px; height: 32px;">
                                    <i class="fas fa-certificate text-white" style="font-size: 0.8rem;"></i>
                                </div>
                                <div>
                                    <div class="fw-bold text-dark" style="font-size: 0.9rem;">Certificate</div>
                                    <div class="text-muted" style="font-size: 0.75rem;">Completion</div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Course Content -->
    <div class="container my-5">
        <div class="row">
            <!-- Main Content -->
            <div class="col-lg-8">
                <!-- Course Navigation Tabs -->
                <div class="mb-4">
                    <ul class="nav nav-pills nav-fill bg-light rounded p-1" id="courseTabs" style="background: #f8f9fa !important;">
                        <li class="nav-item">
                            <a class="nav-link active" href="#overview" onclick="showTab('overview')" data-tab="overview">Course Overview</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="#curriculum" onclick="showTab('curriculum')" data-tab="curriculum">Curriculum</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="#instructor" onclick="showTab('instructor')" data-tab="instructor">Instructor</a>
                        </li>
                    </ul>
                </div>

                <!-- Tab Content Panels -->
                
                <!-- Course Overview Tab -->
                <div id="overview-content" class="tab-content-panel">
                    <div class="card border-0 shadow-sm mb-4">
                        <div class="card-body p-4">
                            <h3 class="mb-4">About This Course</h3>
                            <p class="lead text-muted mb-4"><?php echo nl2br(htmlspecialchars($course['description'] ?? '')); ?></p>
                            
                            <h4 class="mb-3">What You'll Learn</h4>
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="d-flex align-items-start mb-3">
                                        <i class="fas fa-check text-success me-3 mt-1"></i>
                                        <span>Master fundamental concepts and advanced techniques</span>
                                    </div>
                                    <div class="d-flex align-items-start mb-3">
                                        <i class="fas fa-check text-success me-3 mt-1"></i>
                                        <span>Apply knowledge in real-world projects</span>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="d-flex align-items-start mb-3">
                                        <i class="fas fa-check text-success me-3 mt-1"></i>
                                        <span>Build a professional portfolio</span>
                                    </div>
                                    <div class="d-flex align-items-start mb-3">
                                        <i class="fas fa-check text-success me-3 mt-1"></i>
                                        <span>Get industry-ready certification</span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Curriculum Tab -->
                <div id="curriculum-content" class="tab-content-panel" style="display: none;">
                    <div class="card border-0 shadow-sm mb-4">
                        <div class="card-body p-4">
                            <h3 class="mb-4"><i class="fas fa-list-alt me-2 text-primary"></i>Course Curriculum</h3>
                            
                            <?php
                            // Get real course curriculum from database - same as student page
                            $lessons = [];
                            
                            if (isset($conn)) {
                                $video_query = "
                                    SELECT v.*, 
                                           COALESCE(vp.completion_percentage, 0) as watch_progress,
                                           CASE WHEN vp.completed = 1 THEN 1 ELSE 0 END as completed,
                                           CASE WHEN vp.last_position > 0 THEN vp.last_position ELSE 0 END as last_position
                                    FROM videos v
                                    LEFT JOIN video_progress vp ON v.video_id = vp.video_id AND vp.student_id = ?
                                    WHERE v.course_id = ? AND v.is_published = 1
                                    ORDER BY v.video_order ASC, v.created_at ASC
                                ";
                                
                                if ($stmt = $conn->prepare($video_query)) {
                                    $user_id_for_progress = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : 0;
                                    $stmt->bind_param("ii", $user_id_for_progress, $course_id);
                                    $stmt->execute();
                                    $result = $stmt->get_result();
                                    
                                    while ($row = $result->fetch_assoc()) {
                                        // Handle video URLs and thumbnails
                                        if (!empty($row['external_url'])) {
                                            $external_url = $row['external_url'];
                                            
                                            // Convert YouTube URLs to embed format
                                            if (strpos($external_url, 'youtube.com/watch') !== false) {
                                                parse_str(parse_url($external_url, PHP_URL_QUERY), $query);
                                                if (isset($query['v'])) {
                                                    $row['video_url'] = 'https://www.youtube.com/embed/' . $query['v'];
                                                    // Set YouTube thumbnail
                                                    $row['thumbnail'] = 'https://img.youtube.com/vi/' . $query['v'] . '/default.jpg';
                                                }
                                            } elseif (strpos($external_url, 'youtu.be') !== false) {
                                                $video_id = basename(parse_url($external_url, PHP_URL_PATH));
                                                $row['video_url'] = 'https://www.youtube.com/embed/' . $video_id;
                                                // Set YouTube thumbnail
                                                $row['thumbnail'] = 'https://img.youtube.com/vi/' . $video_id . '/default.jpg';
                                            } else {
                                                $row['video_url'] = $external_url;
                                                $row['thumbnail'] = '/assets/images/default-video-thumbnail.jpg';
                                            }
                                            $row['is_external'] = true;
                                        } elseif (!empty($row['video_url'])) {
                                            // File upload - fix video URL path
                                            if (strpos($row['video_url'], 'http') !== 0) {
                                                $row['video_url'] = '/uploads/videos/' . basename($row['video_url']);
                                            }
                                            $row['is_external'] = false;
                                            $row['thumbnail'] = !empty($row['thumbnail']) ? $row['thumbnail'] : '/assets/images/default-video-thumbnail.jpg';
                                        } else {
                                            $row['thumbnail'] = '/assets/images/default-video-thumbnail.jpg';
                                        }
                                        
                                        // Get related content for each video
                                        $row['has_quiz'] = false;
                                        $row['has_assignment'] = false;
                                        $row['has_materials'] = false;
                                        $row['quiz_count'] = 0;
                                        $row['material_count'] = 0;
                                        $row['assignment_count'] = 0;
                                        
                                        // Check quizzes
                                        $quiz_check = $conn->prepare("SELECT COUNT(*) as count FROM quizzes WHERE video_id = ? AND is_published = 1");
                                        if ($quiz_check) {
                                            $quiz_check->bind_param("i", $row['video_id']);
                                            $quiz_check->execute();
                                            $quiz_result = $quiz_check->get_result();
                                            if ($quiz_data = $quiz_result->fetch_assoc()) {
                                                $row['has_quiz'] = $quiz_data['count'] > 0;
                                                $row['quiz_count'] = $quiz_data['count'];
                                            }
                                            $quiz_check->close();
                                        }
                                        
                                        // Check assignments
                                        $assignment_check = $conn->prepare("SELECT COUNT(*) as count FROM assignments WHERE video_id = ?");
                                        if ($assignment_check) {
                                            $assignment_check->bind_param("i", $row['video_id']);
                                            $assignment_check->execute();
                                            $assignment_result = $assignment_check->get_result();
                                            if ($assignment_data = $assignment_result->fetch_assoc()) {
                                                $row['has_assignment'] = $assignment_data['count'] > 0;
                                                $row['assignment_count'] = $assignment_data['count'];
                                            }
                                            $assignment_check->close();
                                        }
                                        
                                        // Check materials
                                        $material_check = $conn->prepare("SELECT COUNT(*) as count FROM course_materials WHERE video_id = ?");
                                        if ($material_check) {
                                            $material_check->bind_param("i", $row['video_id']);
                                            $material_check->execute();
                                            $material_result = $material_check->get_result();
                                            if ($material_data = $material_result->fetch_assoc()) {
                                                $row['has_materials'] = $material_data['count'] > 0;
                                                $row['material_count'] = $material_data['count'];
                                            }
                                            $material_check->close();
                                        }
                                        
                                        $row['lesson_id'] = $row['video_id'];
                                        $lessons[] = $row;
                                    }
                                    $stmt->close();
                                }
                            }
                            
                            // Get course-level materials and quizzes
                            $course_materials = [];
                            $course_quizzes = [];
                            
                            if (isset($conn)) {
                                // Get course materials not tied to specific videos
                                $materials_query = "SELECT * FROM course_materials WHERE course_id = ? AND (video_id IS NULL OR video_id = 0) ORDER BY created_at ASC";
                                if ($stmt = $conn->prepare($materials_query)) {
                                    $stmt->bind_param("i", $course_id);
                                    $stmt->execute();
                                    $result = $stmt->get_result();
                                    while ($row = $result->fetch_assoc()) {
                                        $course_materials[] = $row;
                                    }
                                    $stmt->close();
                                }
                                
                                // Get course quizzes not tied to specific videos
                                $quizzes_query = "SELECT * FROM quizzes WHERE course_id = ? AND (video_id IS NULL OR video_id = 0) AND is_published = 1 ORDER BY created_at ASC";
                                if ($stmt = $conn->prepare($quizzes_query)) {
                                    $stmt->bind_param("i", $course_id);
                                    $stmt->execute();
                                    $result = $stmt->get_result();
                                    while ($row = $result->fetch_assoc()) {
                                        $course_quizzes[] = $row;
                                    }
                                    $stmt->close();
                                }
                            }
                            ?>
                            
                            <!-- Course Content Accordion -->
                            <div class="accordion" id="courseContentAccordion">
                                
                                <?php if (!empty($course_quizzes)): ?>
                                <!-- Course Quizzes Section -->
                                <div class="accordion-item">
                                    <h2 class="accordion-header" id="headingQuizzes">
                                        <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapseQuizzes" aria-expanded="false" aria-controls="collapseQuizzes">
                                            <div class="d-flex align-items-center w-100">
                                                <i class="fas fa-question-circle text-warning me-3" style="font-size: 1.2rem;"></i>
                                                <div class="flex-grow-1">
                                                    <div class="fw-semibold">Quizzes</div>
                                                    <small class="text-muted"><?php echo count($course_quizzes); ?> quiz(es) available</small>
                                                </div>
                                                <span class="badge bg-warning text-dark"><?php echo count($course_quizzes); ?></span>
                                            </div>
                                        </button>
                                    </h2>
                                    <div id="collapseQuizzes" class="accordion-collapse collapse" aria-labelledby="headingQuizzes" data-bs-parent="#courseContentAccordion">
                                        <div class="accordion-body">
                                            <?php foreach ($course_quizzes as $quiz): ?>
                                            <div class="d-flex align-items-center mb-3 p-3 bg-light rounded">
                                                <i class="fas fa-quiz text-warning me-3"></i>
                                                <div class="flex-grow-1">
                                                    <h6 class="mb-1"><?php echo htmlspecialchars($quiz['title']); ?></h6>
                                                    <small class="text-muted">
                                                        <?php if (!empty($quiz['time_limit'])): ?>
                                                            <i class="fas fa-clock me-1"></i> <?php echo $quiz['time_limit']; ?> minutes
                                                        <?php endif; ?>
                                                        <?php if (!empty($quiz['passing_score'])): ?>
                                                            | Passing: <?php echo $quiz['passing_score']; ?>%
                                                        <?php endif; ?>
                                                    </small>
                                                </div>
                                                <?php if (!$is_enrolled): ?>
                                                    <span class="badge bg-light text-muted"><i class="fas fa-lock"></i> Locked</span>
                                                <?php else: ?>
                                                    <button class="btn btn-sm btn-outline-warning">Take Quiz</button>
                                                <?php endif; ?>
                                            </div>
                                            <?php endforeach; ?>
                                        </div>
                                    </div>
                                </div>
                                <?php endif; ?>
                                
                                <?php if (!empty($course_materials)): ?>
                                <!-- Course Materials Section -->
                                <div class="accordion-item">
                                    <h2 class="accordion-header" id="headingMaterials">
                                        <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapseMaterials" aria-expanded="false" aria-controls="collapseMaterials">
                                            <div class="d-flex align-items-center w-100">
                                                <i class="fas fa-file-download text-success me-3" style="font-size: 1.2rem;"></i>
                                                <div class="flex-grow-1">
                                                    <div class="fw-semibold">Materials</div>
                                                    <small class="text-muted"><?php echo count($course_materials); ?> file(s) available</small>
                                                </div>
                                                <span class="badge bg-success"><?php echo count($course_materials); ?></span>
                                            </div>
                                        </button>
                                    </h2>
                                    <div id="collapseMaterials" class="accordion-collapse collapse" aria-labelledby="headingMaterials" data-bs-parent="#courseContentAccordion">
                                        <div class="accordion-body">
                                            <?php foreach ($course_materials as $material): ?>
                                            <div class="d-flex align-items-center mb-3 p-3 bg-light rounded">
                                                <i class="fas fa-file-alt text-success me-3"></i>
                                                <div class="flex-grow-1">
                                                    <h6 class="mb-1"><?php echo htmlspecialchars($material['title']); ?></h6>
                                                    <small class="text-muted">
                                                        <?php if (!empty($material['file_type'])): ?>
                                                            <?php echo strtoupper($material['file_type']); ?> file
                                                        <?php endif; ?>
                                                    </small>
                                                </div>
                                                <?php if (!$is_enrolled): ?>
                                                    <span class="badge bg-light text-muted"><i class="fas fa-lock"></i> Locked</span>
                                                <?php else: ?>
                                                    <a href="<?php echo htmlspecialchars($material['file_path']); ?>" class="btn btn-sm btn-outline-success" download>
                                                        <i class="fas fa-download"></i> Download
                                                    </a>
                                                <?php endif; ?>
                                            </div>
                                            <?php endforeach; ?>
                                        </div>
                                    </div>
                                </div>
                                <?php endif; ?>
                                
                                <?php if (!empty($lessons)): ?>
                                <!-- Individual Video Lessons -->
                                <?php foreach ($lessons as $index => $lesson): ?>
                                <div class="accordion-item">
                                    <h2 class="accordion-header" id="headingLesson<?= $lesson['lesson_id'] ?>">
                                        <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapseLesson<?= $lesson['lesson_id'] ?>" aria-expanded="false" aria-controls="collapseLesson<?= $lesson['lesson_id'] ?>">
                                            <div class="d-flex align-items-center w-100">
                                                <div class="lesson-thumbnail-small me-3 position-relative">
                                                    <?php if (!empty($lesson['thumbnail'])): ?>
                                                        <img src="<?= htmlspecialchars($lesson['thumbnail']) ?>" alt="Lesson <?= $index + 1 ?>" class="rounded" style="width: 50px; height: 35px; object-fit: cover;">
                                                    <?php else: ?>
                                                        <div class="thumbnail-placeholder-small d-flex align-items-center justify-content-center" style="width: 50px; height: 35px; background: #f8f9fa; border-radius: 4px;">
                                                            <i class="fas fa-play text-muted"></i>
                                                        </div>
                                                    <?php endif; ?>
                                                    <?php if ($lesson['completed']): ?>
                                                        <div class="completion-check position-absolute" style="top: -5px; right: -5px;">
                                                            <i class="fas fa-check-circle text-success" style="font-size: 0.8rem;"></i>
                                                        </div>
                                                    <?php endif; ?>
                                                </div>
                                                <div class="flex-grow-1" style="min-width: 0;">
                                                    <div class="fw-semibold" style="white-space: nowrap; overflow: hidden; text-overflow: ellipsis;">
                                                        <?= htmlspecialchars($lesson['title']) ?>
                                                    </div>
                                                    <div class="d-flex align-items-center justify-content-between">
                                                        <small class="text-muted">
                                                            <i class="fas fa-clock"></i> <?= !empty($lesson['duration']) ? htmlspecialchars($lesson['duration']) : '5:00' ?>
                                                        </small>
                                                        <div class="d-flex align-items-center">
                                                            <?php if ($lesson['has_quiz']): ?>
                                                                <span class="badge bg-warning me-1" style="font-size: 0.6rem;" title="Quiz Available">
                                                                    <i class="fas fa-question-circle"></i> <?= $lesson['quiz_count'] ?>
                                                                </span>
                                                            <?php endif; ?>
                                                            <?php if ($lesson['has_assignment']): ?>
                                                                <span class="badge bg-info me-1" style="font-size: 0.6rem;" title="Assignment Available">
                                                                    <i class="fas fa-tasks"></i> <?= $lesson['assignment_count'] ?>
                                                                </span>
                                                            <?php endif; ?>
                                                            <?php if ($lesson['has_materials']): ?>
                                                                <span class="badge bg-success" style="font-size: 0.6rem;" title="Materials Available">
                                                                    <i class="fas fa-file"></i> <?= $lesson['material_count'] ?>
                                                                </span>
                                                            <?php endif; ?>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </button>
                                    </h2>
                                    <div id="collapseLesson<?= $lesson['lesson_id'] ?>" class="accordion-collapse collapse" aria-labelledby="headingLesson<?= $lesson['lesson_id'] ?>" data-bs-parent="#courseContentAccordion">
                                        <div class="accordion-body">
                                            <div class="lesson-content">
                                                <?php if (!empty($lesson['description'])): ?>
                                                    <p class="text-muted mb-3"><?= htmlspecialchars($lesson['description']) ?></p>
                                                <?php endif; ?>
                                                
                                                <div class="lesson-actions mb-3">
                                                    <?php if (!$is_enrolled): ?>
                                                        <div class="alert alert-warning">
                                                            <i class="fas fa-lock me-2"></i>
                                                            Enroll to access this lesson and all course content
                                                        </div>
                                                    <?php else: ?>
                                                        <button class="btn btn-primary btn-sm me-2">
                                                            <i class="fas fa-play"></i> Start Lesson
                                                        </button>
                                                        <?php if ($lesson['watch_progress'] > 0): ?>
                                                            <span class="badge bg-info">
                                                                <i class="fas fa-eye"></i> <?= round($lesson['watch_progress']) ?>% watched
                                                            </span>
                                                        <?php endif; ?>
                                                    <?php endif; ?>
                                                </div>
                                                
                                                <?php if ($lesson['has_quiz'] || $lesson['has_assignment'] || $lesson['has_materials']): ?>
                                                <div class="lesson-resources">
                                                    <h6 class="mb-2">Resources & Activities</h6>
                                                    
                                                    <?php if ($lesson['has_quiz']): ?>
                                                        <div class="resource-item mb-2">
                                                            <i class="fas fa-question-circle text-warning me-2"></i>
                                                            <span>Interactive Quiz (<?= $lesson['quiz_count'] ?> available)</span>
                                                            <?php if (!$is_enrolled): ?>
                                                                <span class="badge bg-light text-muted ms-2"><i class="fas fa-lock"></i></span>
                                                            <?php endif; ?>
                                                        </div>
                                                    <?php endif; ?>
                                                    
                                                    <?php if ($lesson['has_assignment']): ?>
                                                        <div class="resource-item mb-2">
                                                            <i class="fas fa-tasks text-info me-2"></i>
                                                            <span>Assignment (<?= $lesson['assignment_count'] ?> available)</span>
                                                            <?php if (!$is_enrolled): ?>
                                                                <span class="badge bg-light text-muted ms-2"><i class="fas fa-lock"></i></span>
                                                            <?php endif; ?>
                                                        </div>
                                                    <?php endif; ?>
                                                    
                                                    <?php if ($lesson['has_materials']): ?>
                                                        <div class="resource-item mb-2">
                                                            <i class="fas fa-file text-success me-2"></i>
                                                            <span>Downloadable Materials (<?= $lesson['material_count'] ?> files)</span>
                                                            <?php if (!$is_enrolled): ?>
                                                                <span class="badge bg-light text-muted ms-2"><i class="fas fa-lock"></i></span>
                                                            <?php endif; ?>
                                                        </div>
                                                    <?php endif; ?>
                                                </div>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                                
                                <?php else: ?>
                                <!-- No Lessons Available -->
                                <div class="accordion-item">
                                    <div class="empty-state p-4">
                                        <div class="text-center text-muted">
                                            <i class="fas fa-video" style="font-size: 3rem; margin-bottom: 1rem;"></i>
                                            <h5>No lessons available yet</h5>
                                            <p class="mb-0">Course content is being prepared. Check back soon!</p>
                                        </div>
                                    </div>
                                </div>
                                <?php endif; ?>
                                
                            </div> <!-- End accordion -->
                        </div>
                    </div>
                </div>

                <!-- Instructor Tab -->
                <div id="instructor-content" class="tab-content-panel" style="display: none;">
                    <?php
                    // Get real instructor information from database
                    $instructor_data = null;
                    $instructor_stats = [
                        'student_count' => '500+',
                        'rating' => '4.9',
                        'course_count' => '5+'
                    ];
                    
                    // Debug: Check what instructor_id we have
                    echo "<!-- Debug: Course instructor_id: " . ($course['instructor_id'] ?? 'NULL') . " -->\n";
                    echo "<!-- Debug: Course instructor_name: " . ($course['instructor_name'] ?? 'NULL') . " -->\n";
                    
                    if (isset($conn)) {
                        // First, try to find instructor by instructor_id from course
                        if (!empty($course['instructor_id'])) {
                            // Get instructor data from users table + social links from teachers table
                            $instructor_query = "
                                SELECT u.user_id, u.first_name, u.last_name, u.email, u.bio, u.profile_picture,
                                       u.cell_number as phone, 
                                       t.website, t.social_linkedin, t.social_twitter, t.social_facebook, t.social_youtube
                                FROM users u 
                                LEFT JOIN teachers t ON u.user_id = t.user_id
                                WHERE u.user_id = ?
                            ";
                            
                            if ($stmt = $conn->prepare($instructor_query)) {
                                $stmt->bind_param("i", $course['instructor_id']);
                                $stmt->execute();
                                $result = $stmt->get_result();
                                
                                echo "<!-- Debug: Query executed, rows found: " . $result->num_rows . " -->\n";
                                
                                if ($result->num_rows > 0) {
                                    $instructor_data = $result->fetch_assoc();
                                    echo "<!-- Debug: Found instructor: " . ($instructor_data['first_name'] ?? 'NULL') . " " . ($instructor_data['last_name'] ?? 'NULL') . " -->\n";
                                } else {
                                    echo "<!-- Debug: No instructor found with ID " . $course['instructor_id'] . " -->\n";
                                }
                                $stmt->close();
                            }
                        } 
                        
                        // If no instructor found by ID, try searching by name
                        if (!$instructor_data && !empty($course['instructor_name'])) {
                            echo "<!-- Debug: Trying to find instructor by name: " . $course['instructor_name'] . " -->\n";
                            $name_parts = explode(' ', trim($course['instructor_name']), 2);
                            $first_name = $name_parts[0];
                            $last_name = isset($name_parts[1]) ? $name_parts[1] : '';
                            
                            $instructor_name_query = "
                                SELECT u.user_id, u.first_name, u.last_name, u.email, u.bio, u.profile_picture,
                                       u.cell_number as phone, 
                                       t.website, t.social_linkedin, t.social_twitter, t.social_facebook, t.social_youtube
                                FROM users u 
                                LEFT JOIN teachers t ON u.user_id = t.user_id
                                WHERE u.first_name LIKE ? " . (!empty($last_name) ? "AND u.last_name LIKE ?" : "") . "
                                ORDER BY u.user_role = 'teacher' DESC
                                LIMIT 1
                            ";
                            
                            if ($stmt = $conn->prepare($instructor_name_query)) {
                                if (!empty($last_name)) {
                                    $stmt->bind_param("ss", $first_name, $last_name);
                                } else {
                                    $stmt->bind_param("s", $first_name);
                                }
                                $stmt->execute();
                                $result = $stmt->get_result();
                                
                                echo "<!-- Debug: Name search query executed, rows found: " . $result->num_rows . " -->\n";
                                
                                if ($result->num_rows > 0) {
                                    $instructor_data = $result->fetch_assoc();
                                    echo "<!-- Debug: Found instructor by name: " . ($instructor_data['first_name'] ?? 'NULL') . " " . ($instructor_data['last_name'] ?? 'NULL') . " -->\n";
                                }
                                $stmt->close();
                            }
                        }
                        
                        // If we found an instructor, get real stats
                        if ($instructor_data) {
                            // Get real instructor stats
                            // Get student count for this instructor
                            $student_count_query = "
                                SELECT COUNT(DISTINCT e.student_id) as total_students
                                FROM enrollments e
                                INNER JOIN courses c ON e.course_id = c.course_id
                                WHERE c.instructor_id = ?
                            ";
                            if ($stats_stmt = $conn->prepare($student_count_query)) {
                                $stats_stmt->bind_param("i", $instructor_data['user_id']);
                                $stats_stmt->execute();
                                $stats_result = $stats_stmt->get_result();
                                if ($stats_data = $stats_result->fetch_assoc()) {
                                    $instructor_stats['student_count'] = $stats_data['total_students'] > 0 ? $stats_data['total_students'] . '+' : '0';
                                }
                                $stats_stmt->close();
                            }
                            
                            // Get course count for this instructor
                            $course_count_query = "SELECT COUNT(*) as total_courses FROM courses WHERE instructor_id = ?";
                            if ($course_stmt = $conn->prepare($course_count_query)) {
                                $course_stmt->bind_param("i", $instructor_data['user_id']);
                                $course_stmt->execute();
                                $course_result = $course_stmt->get_result();
                                if ($course_data = $course_result->fetch_assoc()) {
                                    $instructor_stats['course_count'] = $course_data['total_courses'];
                                }
                                $course_stmt->close();
                            }
                            
                            // Get average rating (if reviews table exists)
                            $rating_query = "
                                SELECT AVG(r.rating) as avg_rating, COUNT(r.review_id) as review_count
                                FROM reviews r
                                INNER JOIN courses c ON r.course_id = c.course_id
                                WHERE c.instructor_id = ?
                            ";
                            if ($rating_stmt = $conn->prepare($rating_query)) {
                                $rating_stmt->bind_param("i", $instructor_data['user_id']);
                                $rating_stmt->execute();
                                $rating_result = $rating_stmt->get_result();
                                if ($rating_data = $rating_result->fetch_assoc()) {
                                    $instructor_stats['rating'] = $rating_data['avg_rating'] ? number_format($rating_data['avg_rating'], 1) : '4.9';
                                }
                                $rating_stmt->close();
                            }
                        }
                    }
                    
                    // Set fallback data if no instructor found
                    if (!$instructor_data) {
                        $instructor_data = [
                            'first_name' => $course['instructor_name'] ?? 'Professional',
                            'last_name' => 'Instructor',
                            'bio' => 'Welcome to my teaching profile! I am passionate about education and helping students achieve their learning goals through hands-on instruction and real-world applications.',
                            'profile_picture' => null,
                            'website' => null,
                            'social_linkedin' => null,
                            'social_twitter' => null,
                            'social_facebook' => null,
                            'social_youtube' => null
                        ];
                    }
                    
                    $full_name = trim(($instructor_data['first_name'] ?? '') . ' ' . ($instructor_data['last_name'] ?? ''));
                    if (empty($full_name)) {
                        $full_name = $course['instructor_name'] ?? 'Professional Instructor';
                    }
                    ?>
                    
                    <div class="card border-0 shadow-sm mb-4">
                        <div class="card-body p-4">
                            <h3 class="mb-4"><i class="fas fa-user-tie me-2 text-primary"></i>Meet Your Instructor</h3>
                            
                            <div class="row align-items-center mb-4">
                                <div class="col-md-3 text-center">
                                    <?php if (!empty($instructor_data['profile_picture'])): ?>
                                        <img src="<?= htmlspecialchars($instructor_data['profile_picture']) ?>" 
                                             alt="<?= htmlspecialchars($full_name) ?>"
                                             class="rounded-circle mx-auto mb-3"
                                             style="width: 100px; height: 100px; object-fit: cover;">
                                    <?php else: ?>
                                        <div class="bg-primary rounded-circle d-flex align-items-center justify-content-center mx-auto mb-3" style="width: 100px; height: 100px;">
                                            <span class="text-white fw-bold" style="font-size: 2rem;">
                                                <?php 
                                                $initials = '';
                                                if (!empty($instructor_data['first_name'])) {
                                                    $initials .= strtoupper(substr($instructor_data['first_name'], 0, 1));
                                                }
                                                if (!empty($instructor_data['last_name'])) {
                                                    $initials .= strtoupper(substr($instructor_data['last_name'], 0, 1));
                                                }
                                                echo $initials ?: 'PI';
                                                ?>
                                            </span>
                                        </div>
                                    <?php endif; ?>
                                </div>
                                <div class="col-md-9">
                                    <h4 class="fw-bold text-dark mb-2"><?= htmlspecialchars($full_name) ?></h4>
                                    <p class="text-muted mb-3">Expert Educator & Industry Professional</p>
                                </div>
                            </div>
                            
                            <h5 class="mb-3">About the Instructor</h5>
                            <div class="text-muted mb-4" style="line-height: 1.6;">
                                <?php 
                                $bio = $instructor_data['bio'] ?? '';
                                if (empty($bio)) {
                                    $bio = $full_name . ' brings years of professional experience and expertise in this field. They are dedicated to helping students master practical skills and achieve their learning goals through hands-on instruction and real-world applications.';
                                }
                                echo nl2br(htmlspecialchars($bio));
                                ?>
                            </div>
                            
                            <?php 
                            // Check if instructor has any social links
                            $has_social_links = !empty($instructor_data['website']) || 
                                              !empty($instructor_data['social_linkedin']) || 
                                              !empty($instructor_data['social_twitter']) || 
                                              !empty($instructor_data['social_facebook']) || 
                                              !empty($instructor_data['social_youtube']);
                            ?>
                            
                            <?php if ($has_social_links): ?>
                            <h6 class="mb-3">Social Links</h6>
                            <div class="social-links mb-4">
                                <div class="row g-3">
                                    <?php if (!empty($instructor_data['website'])): ?>
                                    <div class="col-md-6">
                                        <div class="d-flex align-items-center p-3 bg-light rounded">
                                            <div class="social-icon me-3">
                                                <i class="fas fa-globe text-primary" style="font-size: 1.5rem; width: 24px;"></i>
                                            </div>
                                            <div class="flex-grow-1">
                                                <h6 class="mb-1">Website</h6>
                                                <a href="<?= htmlspecialchars($instructor_data['website']) ?>" 
                                                   target="_blank" 
                                                   class="text-muted text-decoration-none small"
                                                   style="word-break: break-all;">
                                                    <?= htmlspecialchars($instructor_data['website']) ?>
                                                    <i class="fas fa-external-link-alt ms-1" style="font-size: 0.7rem;"></i>
                                                </a>
                                            </div>
                                        </div>
                                    </div>
                                    <?php endif; ?>
                                    
                                    <?php if (!empty($instructor_data['social_linkedin'])): ?>
                                    <div class="col-md-6">
                                        <div class="d-flex align-items-center p-3 bg-light rounded">
                                            <div class="social-icon me-3">
                                                <i class="fab fa-linkedin text-primary" style="font-size: 1.5rem; width: 24px;"></i>
                                            </div>
                                            <div class="flex-grow-1">
                                                <h6 class="mb-1">LinkedIn</h6>
                                                <a href="<?= htmlspecialchars($instructor_data['social_linkedin']) ?>" 
                                                   target="_blank" 
                                                   class="text-muted text-decoration-none small"
                                                   style="word-break: break-all;">
                                                    <?= htmlspecialchars($instructor_data['social_linkedin']) ?>
                                                    <i class="fas fa-external-link-alt ms-1" style="font-size: 0.7rem;"></i>
                                                </a>
                                            </div>
                                        </div>
                                    </div>
                                    <?php endif; ?>
                                    
                                    <?php if (!empty($instructor_data['social_twitter'])): ?>
                                    <div class="col-md-6">
                                        <div class="d-flex align-items-center p-3 bg-light rounded">
                                            <div class="social-icon me-3">
                                                <i class="fab fa-twitter text-info" style="font-size: 1.5rem; width: 24px;"></i>
                                            </div>
                                            <div class="flex-grow-1">
                                                <h6 class="mb-1">Twitter</h6>
                                                <a href="<?= htmlspecialchars($instructor_data['social_twitter']) ?>" 
                                                   target="_blank" 
                                                   class="text-muted text-decoration-none small"
                                                   style="word-break: break-all;">
                                                    <?= htmlspecialchars($instructor_data['social_twitter']) ?>
                                                    <i class="fas fa-external-link-alt ms-1" style="font-size: 0.7rem;"></i>
                                                </a>
                                            </div>
                                        </div>
                                    </div>
                                    <?php endif; ?>
                                    
                                    <?php if (!empty($instructor_data['social_facebook'])): ?>
                                    <div class="col-md-6">
                                        <div class="d-flex align-items-center p-3 bg-light rounded">
                                            <div class="social-icon me-3">
                                                <i class="fab fa-facebook text-primary" style="font-size: 1.5rem; width: 24px;"></i>
                                            </div>
                                            <div class="flex-grow-1">
                                                <h6 class="mb-1">Facebook</h6>
                                                <a href="<?= htmlspecialchars($instructor_data['social_facebook']) ?>" 
                                                   target="_blank" 
                                                   class="text-muted text-decoration-none small"
                                                   style="word-break: break-all;">
                                                    <?= htmlspecialchars($instructor_data['social_facebook']) ?>
                                                    <i class="fas fa-external-link-alt ms-1" style="font-size: 0.7rem;"></i>
                                                </a>
                                            </div>
                                        </div>
                                    </div>
                                    <?php endif; ?>
                                    
                                    <?php if (!empty($instructor_data['social_youtube'])): ?>
                                    <div class="col-md-6">
                                        <div class="d-flex align-items-center p-3 bg-light rounded">
                                            <div class="social-icon me-3">
                                                <i class="fab fa-youtube text-danger" style="font-size: 1.5rem; width: 24px;"></i>
                                            </div>
                                            <div class="flex-grow-1">
                                                <h6 class="mb-1">YouTube</h6>
                                                <a href="<?= htmlspecialchars($instructor_data['social_youtube']) ?>" 
                                                   target="_blank" 
                                                   class="text-muted text-decoration-none small"
                                                   style="word-break: break-all;">
                                                    <?= htmlspecialchars($instructor_data['social_youtube']) ?>
                                                    <i class="fas fa-external-link-alt ms-1" style="font-size: 0.7rem;"></i>
                                                </a>
                                            </div>
                                        </div>
                                    </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                            <?php endif; ?>
                            
                            <h6 class="mb-3">Areas of Expertise:</h6>
                            <div class="expertise-tags">
                                <span class="badge bg-light text-dark me-2 mb-2">Professional Training</span>
                                <span class="badge bg-light text-dark me-2 mb-2">Industry Experience</span>
                                <span class="badge bg-light text-dark me-2 mb-2">Practical Applications</span>
                                <span class="badge bg-light text-dark me-2 mb-2">Career Development</span>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Related Courses Section -->
                <?php if (!empty($related_courses)): ?>
                <div class="card border-0 shadow-sm mb-4">
                    <div class="card-body p-4">
                        <h4 class="mb-3">
                            <i class="fas fa-graduation-cap me-2 text-primary"></i>
                            Related Courses You Might Like
                        </h4>
                        <div class="row g-3">
                            <?php 
                            $course_levels = ['Beginner', 'Intermediate', 'Advanced'];
                            $level_colors = ['success', 'warning', 'danger'];
                            foreach (array_slice($related_courses, 0, 3) as $index => $related): 
                                $level = $course_levels[$index % 3];
                                $level_color = $level_colors[$index % 3];
                            ?>
                            <div class="col-md-4">
                                <div class="card border-1 h-100" style="border-color: #e9ecef !important;">
                                    <div class="position-relative">
                                        <?php if (!empty($related['thumbnail'])): ?>
                                            <img src="<?php echo htmlspecialchars($related['thumbnail']); ?>" 
                                                 class="card-img-top" 
                                                 alt="<?php echo htmlspecialchars($related['title']); ?>"
                                                 style="height: 120px; object-fit: cover;">
                                        <?php else: ?>
                                            <div class="card-img-top bg-gradient-primary d-flex align-items-center justify-content-center" 
                                                 style="height: 120px;">
                                                <i class="fas fa-play-circle text-white" style="font-size: 1.5rem;"></i>
                                            </div>
                                        <?php endif; ?>
                                        <span class="badge bg-<?php echo $level_color; ?> position-absolute" 
                                              style="top: 8px; right: 8px; font-size: 0.65rem;">
                                            <?php echo $level; ?>
                                        </span>
                                        <div class="course-overlay">
                                            <a href="<?php echo IdEncryption::generateUrl('course.php', $related['course_id']); ?>" 
                                               class="btn btn-light btn-sm" style="font-size: 0.7rem;">
                                                <i class="fas fa-eye me-1"></i>View
                                            </a>
                                        </div>
                                    </div>
                                    <div class="card-body p-3" style="min-height: 140px;">
                                        <h6 class="card-title mb-2 fw-bold" style="font-size: 0.9rem; line-height: 1.2; overflow: hidden; display: -webkit-box; -webkit-line-clamp: 2; -webkit-box-orient: vertical;">
                                            <a href="<?php echo IdEncryption::generateUrl('course.php', $related['course_id']); ?>" 
                                               class="text-decoration-none text-dark">
                                                <?php echo htmlspecialchars($related['title']); ?>
                                            </a>
                                        </h6>
                                        <p class="card-text text-muted mb-2" 
                                           style="font-size: 0.75rem; line-height: 1.3; overflow: hidden; display: -webkit-box; -webkit-line-clamp: 2; -webkit-box-orient: vertical;">
                                            <?php echo htmlspecialchars(substr($related['description'], 0, 60)); ?>
                                        </p>
                                        <div class="mt-auto">
                                            <div class="d-flex justify-content-between align-items-center mb-1">
                                                <small class="text-muted" style="font-size: 0.7rem; overflow: hidden; white-space: nowrap; text-overflow: ellipsis;">
                                                    <i class="fas fa-user me-1"></i>
                                                    <?php echo htmlspecialchars(substr($related['instructor_name'] ?: 'Instructor', 0, 12)); ?>
                                                </small>
                                            </div>
                                            <div class="text-center">
                                                <span class="fw-bold text-primary" style="font-size: 0.9rem;">
                                                    <?php echo $related['price_formatted']; ?>
                                                </span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
            </div>

                <!-- Hidden Course Card for Cart System -->
                <div class="d-none course-card" data-course-id="<?php echo IdEncryption::encrypt($course_id); ?>">
                    <div class="course-thumbnail">
                        <img src="<?php echo htmlspecialchars($course['thumbnail'] ?? ''); ?>" alt="<?php echo htmlspecialchars($course['title']); ?>">
                    </div>
                    <div class="course-content">
                        <h3 class="course-title"><?php echo htmlspecialchars($course['title']); ?></h3>
                        <div class="course-instructor"><?php echo htmlspecialchars($course['instructor_name']); ?></div>
                        <div class="course-price" data-price="<?php echo $course['price']; ?>">
                            <span class="price"><?php echo $course['price_formatted']; ?></span>
                        </div>
                        <div class="course-level" data-level="Professional">Professional</div>
                    </div>
                </div>

                <!-- Enrollment Sidebar -->
            <div class="col-lg-4">
                <div class="card border-0 shadow sticky-top" style="top: 2rem;">
                    <div class="card-body p-0">
                        <!-- Course Image -->
                        <div class="position-relative">
                            <?php if (!empty($course['thumbnail'])): ?>
                            <img src="<?php echo htmlspecialchars($course['thumbnail']); ?>" alt="Course" class="card-img-top" style="height: 200px; object-fit: cover;">
                            <?php else: ?>
                            <div style="height: 200px; background: linear-gradient(135deg, #667eea, #764ba2);" class="d-flex align-items-center justify-content-center text-white">
                                <i class="fas fa-graduation-cap fa-3x"></i>
                            </div>
                            <?php endif; ?>
                            <div class="position-absolute top-0 end-0 m-3">
                                <button class="btn btn-light btn-sm rounded-circle"></button>
                            </div>
                        </div>
                        
                        <!-- Course Info -->
                        <div class="p-4">
                            <!-- Debug course data -->
                            <?php
                            echo "<!-- DEBUG COURSE DATA:\n";
                            echo "Title: " . ($course['title'] ?? 'NULL') . "\n";
                            echo "Price: " . ($course['price'] ?? 'NULL') . "\n";
                            echo "Thumbnail: " . ($course['thumbnail'] ?? 'NULL') . "\n";
                            echo "Instructor: " . ($course['instructor_name'] ?? 'NULL') . "\n";
                            echo "Level: " . ($course['level'] ?? 'NULL') . "\n";
                            echo "Description: " . substr($course['description'] ?? 'NULL', 0, 50) . "...\n";
                            echo "-->\n";
                            ?>
                            
                            <div class="text-center mb-4">
                                <h2 class="text-primary fw-bold mb-2"><?php echo $course['price_formatted']; ?></h2>
                                <p class="text-muted mb-0">Learning this Course is</p>
                                <h4 class="text-success"><?php echo ($course['price'] > 0) ? $course['price_formatted'] : 'Absolutely Free!'; ?></h4>
                            </div>
                            
                            <?php if ($is_enrolled): ?>
                            <a href="<?php echo IdEncryption::generateUrl('student/course-detail.php', $course_id); ?>" class="btn btn-success w-100 py-3 mb-3 fw-bold">
                                Continue Learning
                            </a>
                            <?php else: ?>
                                <button class="btn btn-warning w-100 py-3 mb-3 fw-bold pa-course-cart-btn" data-course-id="<?php echo IdEncryption::encrypt($course_id); ?>">
                                    <i class="fas fa-shopping-cart me-2"></i>Add to Cart
                                </button>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <?php endif; ?>


    
    <!-- Footer -->
    <footer class="bg-white py-4 mt-5 border-top">
        <div class="container">
            <div class="row align-items-center">
                <div class="col-md-6 text-center text-md-start mb-3 mb-md-0">
                    <img src="assets/images/logo.jpeg" alt="Panadite Academy Logo" height="40" class="me-2">
                    <span class="text-muted">© <?php echo date('Y'); ?> Panadite Academy. All rights reserved.</span>
                </div>
                <div class="col-md-6 text-center text-md-end">
                    <a href="#" class="text-decoration-none text-muted me-3"><i class="fab fa-facebook-f"></i></a>
                    <a href="#" class="text-decoration-none text-muted me-3"><i class="fab fa-twitter"></i></a>
                    <a href="#" class="text-decoration-none text-muted me-3"><i class="fab fa-instagram"></i></a>
                    <a href="#" class="text-decoration-none text-muted"><i class="fab fa-linkedin-in"></i></a>
                </div>
            </div>
        </div>
    </footer>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    
    <!-- Tab Functionality CSS and JavaScript -->
    <style>
        .tab-content-panel { transition: opacity 0.3s ease; }
        .nav-link.active { background-color: #4e73df !important; color: white !important; }
    </style>
    
    <script>
        // Tab Switching Function
        function showTab(tabName) {
            // Hide all tab content panels
            document.querySelectorAll('.tab-content-panel').forEach(panel => {
                panel.style.display = 'none';
            });
            
            // Remove active class from all nav links
            document.querySelectorAll('#courseTabs .nav-link').forEach(link => {
                link.classList.remove('active');
            });
            
            // Show selected tab content
            const selectedPanel = document.getElementById(tabName + '-content');
            if (selectedPanel) {
                selectedPanel.style.display = 'block';
            }
            
            // Add active class to clicked nav link
            const selectedNavLink = document.querySelector(`#courseTabs .nav-link[data-tab="${tabName}"]`);
            if (selectedNavLink) {
                selectedNavLink.classList.add('active');
            }
            
            return false;
        }
        
        // Initialize tabs on page load
        document.addEventListener('DOMContentLoaded', function() {
            // Default to overview tab
            showTab('overview');
        });
        
        // Add any custom scripts here
        document.addEventListener('DOMContentLoaded', function() {
            // Simple animation trigger - just add the 'animate-start' class to trigger animations
            setTimeout(() => {
                const animatedElements = document.querySelectorAll('[class*="animate-"]');
                animatedElements.forEach((el) => {
                    el.classList.add('animate-start');
                });
            }, 100); // Small delay to ensure page is loaded

            // Typing Animation
            const typedTextElement = document.querySelector('.typed-text');
            const cursorElement = document.querySelector('.cursor');
            
            if (typedTextElement && cursorElement) {
                // Clear any existing content first
                typedTextElement.textContent = '';
                
                // Course-specific words to cycle through (shorter phrases to prevent wrapping)
                const words = [
                    "professional skills",
                    "tomorrow's future", 
                    "in-demand expertise",
                    "career growth",
                    "industry knowledge"
                ];
                
                let wordIndex = 0;
                let charIndex = 0;
                let isDeleting = false;
                
                function typeWords() {
                    const currentWord = words[wordIndex];
                    
                    if (isDeleting) {
                        // Remove characters
                        typedTextElement.textContent = currentWord.substring(0, charIndex);
                        charIndex--;
                        cursorElement.classList.add('typing');
                        
                        if (charIndex < 0) {
                            charIndex = 0;
                            isDeleting = false;
                            wordIndex = (wordIndex + 1) % words.length;
                            setTimeout(typeWords, 500); // Pause before typing next word
                        } else {
                            setTimeout(typeWords, 50); // Faster deletion
                        }
                    } else {
                        // Add characters
                        typedTextElement.textContent = currentWord.substring(0, charIndex + 1);
                        charIndex++;
                        cursorElement.classList.add('typing');
                        
                        if (charIndex === currentWord.length) {
                            cursorElement.classList.remove('typing');
                            isDeleting = true;
                            setTimeout(typeWords, 2000); // Pause before deleting
                        } else {
                            setTimeout(typeWords, 100); // Typing speed
                        }
                    }
                }
                
                // Start typing animation
                setTimeout(typeWords, 1000);
            }
            
            // Debug console log to verify script is loading
            console.log('Course page scripts loaded, enrollment button is:', document.getElementById('enrollButton'));
            
            // Add a click event listener directly to ensure we catch all clicks
            const enrollButton = document.getElementById('enrollButton');
            if (enrollButton) {
                console.log('Adding click handler to enrollment button');
                enrollButton.addEventListener('click', function() {
                    const courseId = this.getAttribute('data-course-id');
                    console.log('Button clicked! Course ID from attribute:', courseId);
                });
            }
        });
        
    </script>
    
    <!-- Enrollment Script -->
    <script>
        // Make function globally available
        function enrollInCourse(courseId) {
            console.log('Enrollment function called for course:', courseId);
            
            // Get the enrollment button and message div
            const enrollButton = document.getElementById('enrollButton');
            const messageDiv = document.getElementById('enrollmentMessage');
            
            // Save original button text for restoration later
            const originalButtonText = enrollButton.innerHTML;
            
            // Double check course ID
            if (!courseId) {
                courseId = enrollButton.getAttribute('data-course-id');
                console.log('Button clicked! Course ID from attribute:', courseId);
            }
            
            // Disable the button to prevent multiple clicks
            enrollButton.disabled = true;
            enrollButton.innerHTML = '<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Processing...';
            
            // Show the message div as processing
            messageDiv.classList.remove('d-none', 'alert-success', 'alert-danger');
            messageDiv.classList.add('alert-info');
            messageDiv.textContent = 'Processing your enrollment...';
            
            // Log the enrollment attempt
            console.log('Sending enrollment request for course ID:', courseId);
            
            // Create form data object
            const formData = new FormData();
            formData.append('course_id', courseId);
            
            // Use native fetch API with local path
            fetch('./process-enrollment-fixed.php', {
                method: 'POST',
                headers: {
                    'Accept': 'application/json'
                },
                body: formData
            })
            .then(response => {
                console.log('Server response status:', response.status);
                
                // Handle non-200 responses
                if (!response.ok) {
                    throw new Error('Server returned ' + response.status + ' ' + response.statusText);
                }
                
                return response.text().then(text => {
                    // Check if the response is empty
                    if (!text || text.trim() === '') {
                        throw new Error('Empty response from server');
                    }
                    
                    console.log('Raw response text:', text);
                    
                    // Check if text looks like HTML instead of JSON
                    if (text.includes('<!DOCTYPE html>') || text.indexOf('<br') === 0) {
                        throw new Error('Received HTML error page instead of JSON');
                    }
                    
                    try {
                        return JSON.parse(text);
                    } catch (e) {
                        console.log('JSON parse error:', e);
                        throw new Error('Failed to parse server response: ' + e.message);
                    }
                });
            })
            .then(data => {
                console.log('Response data:', data);
                messageDiv.classList.remove('d-none', 'alert-success', 'alert-danger', 'alert-info');
                
                if (data.success) {
                    messageDiv.classList.add('alert-success');
                    messageDiv.textContent = data.message;
                    
                    // Redirect if needed
                    if (data.redirect) {
                        setTimeout(() => {
                            window.location.href = data.redirect;
                        }, 1500);
                    }
                } else {
                    messageDiv.classList.add('alert-danger');
                    messageDiv.textContent = data.message || 'An error occurred. Check the console for details.';
                    
                    // Re-enable button if there was an error
                    enrollButton.disabled = false;
                    enrollButton.innerHTML = originalButtonText;
                }
            })
            .catch(error => {
                console.error('Fetch Error:', error);
                messageDiv.classList.remove('d-none', 'alert-info');
                messageDiv.classList.add('alert-danger');
                messageDiv.textContent = 'Error: ' + error.message;
                
                // Re-enable button
                enrollButton.disabled = false;
                enrollButton.innerHTML = originalButtonText;
            });
        }
    </script>

    <!-- Panadite Academy Guest Cart System -->
    <script src="js/guest-cart-system.js"></script>
    
    <!-- Cart System Initialization -->
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Initialize cart system
            if (typeof PanaditeCartSystem !== 'undefined') {
                window.paCartSystem = new PanaditeCartSystem();
                console.log('Cart system initialized for course page');
                
                // Update cart counter on page load
                window.paCartSystem.updateCartCounter();
            } else {
                console.error('PanaditeCartSystem not loaded');
            }
        });
    </script>
</body>
</html>
